/*             This file is part of the New World OS project
--                   Copyright (C) 2006  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
-- NWOS is free software;  you can redistribute it and/or modify it under the
-- terms of the GNU General Public License  as published by the Free Software
-- Foundation; either version 2, or (at your option) any later version.  This
-- software is distributed with the hope that it will be useful,  but WITHOUT
-- ANY WARRANTY;  without  even the  implied warranty  of MERCHANTABILITY  or
-- FITNESS FOR A PARTICULAR PURPOSE.   See the GNU General Public License for
-- more  details.  You should have received a copy  of the GNU General Public
-- License along with this package;  see the file LICENSE.  If not, write to:
--
--      Free Software Foundation, Inc.
--      59 Temple Place - Suite 330
--      Boston, MA 02111-1307, USA.
--
-- $Log: check_block_maps.c,v $
-- Revision 1.4  2006/11/11 12:01:01  jsedwards
-- Update e-mail address to something that works.
--
-- Revision 1.3  2006/11/02 11:49:28  jsedwards
-- Fixed all cases where 'z' was used as a format for 'off64_t' values because
-- the older compiler complains.
--
-- Revision 1.2  2006/10/26 01:51:22  jsedwards
-- Merged alpha_05_branch back into main trunk.
--
-- Revision 1.1.2.9  2006/10/22 16:02:58  jsedwards
-- Add print statement to print used in final blocks.
--
-- Revision 1.1.2.8  2006/10/22 14:19:23  jsedwards
-- Changed so that blocks_on_disk and reserved_public_blocks are read off
-- disk instead of defined constants.
--
-- Revision 1.1.2.7  2006/10/18 13:18:51  jsedwards
-- Changed printf formats for off64_t to 'z' instead of 'll'.
--
-- Revision 1.1.2.6  2006/10/15 12:30:02  jsedwards
-- Change to print blocks used in each print instead of a running total.
--
-- Revision 1.1.2.5  2006/10/15 12:13:14  jsedwards
-- Changed to correctly account for the first chunk on the disk as all used.
--
-- Revision 1.1.2.4  2006/10/12 12:09:57  jsedwards
-- Added kludge to add in stats for the section of the disk (reserved public)
-- that it currently skips over.  Also included the block maps themselves.
--
-- Revision 1.1.2.3  2006/10/10 07:32:36  jsedwards
-- Move bit map defines to objectify_private.h instead of spread all over.
--
-- Revision 1.1.2.2  2006/10/09 13:13:14  jsedwards
-- Change bit map size from 256 to 8192 (to improve speed, disk_usage program
-- took over an hour to run with 256 byte bit maps).
--
-- Revision 1.1.2.1  2006/10/08 13:44:09  jsedwards
-- Program to verify the 256 byte block maps are correct on the disk.
--
*/


#define _LARGEFILE64_SOURCE

#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <unistd.h>

#include "objectify_private.h"

static void print_usage(char *program)
{
    fprintf(stderr, "usage: %s\n", program);
}


int main(int argc, char* argv[])
{
    int obj_file_desc;
    off64_t ref;
    char ref_str[128];
    uint8 *chunk;
    int i;
    int j;
    size_t bytes_read;
    int num_blocks;
    int bit_num;
    int byte_num;
    int blocks_used = 0;
    int blocks_free = 0;
    int blocks_total;
    int last_total;
    uint32 blocks_on_disk;
    uint32 reserved_public_blocks;
    uint8 header[16];

    if (argc != 1)
    {
	print_usage(argv[0]);
	exit(1);
    }

    obj_file_desc = open(DEFAULT_FILE, O_RDONLY | O_LARGEFILE);

    if (obj_file_desc < 0)
    {
	perror(DEFAULT_FILE);
	exit(1);
    }

    if (read(obj_file_desc, header, sizeof(header)) != sizeof(header))
    {
	sprintf(ref_str, "reading header from: %s", DEFAULT_FILE);
	perror(ref_str);
	close(obj_file_desc);
	exit(1);
    }
    
    if (memcmp(&header[0], MAGIC_NUMBER, 4) != 0)
    {
	fprintf(stderr, "Missing magic number in disk header\n");
	exit(1);
    }

    if (memcmp(&header[4], VERSION_STRING, 4) != 0)
    {
	fprintf(stderr, "Incorrect version string in disk header\n");
	exit(1);
    }

    reserved_public_blocks = ((uint32)header[8] << 24) | ((uint32)header[9]<< 16) | ((uint32)header[10] << 8) | ((uint32)header[11]);

    blocks_on_disk = ((uint32)header[12] << 24) | ((uint32)header[13]<< 16) | ((uint32)header[14] << 8) | ((uint32)header[15]);

    assert(BLOCKS_IN_CHUNK <= reserved_public_blocks && reserved_public_blocks <= NUM_PUBLIC_IDS);
    assert(reserved_public_blocks % BLOCKS_IN_CHUNK == 0);
    assert(blocks_on_disk > reserved_public_blocks);


    /* allocate buffer for reading chunks */

    chunk = malloc(CHUNK_SIZE);

    assert(chunk != NULL);

    num_blocks = 0;

    /* since we are skipping the first 268435456 blocks add those in here */

    blocks_used = BLOCKS_IN_CHUNK;  /* the whole first chunk is used by default (not a normal chunk) */

    /* count all the block maps in the reserved public blocks (don't scan them for now because they aren't used */
    /* subtract one for the first chunk that we counted as all used above */
    blocks_used += ((reserved_public_blocks / BLOCKS_IN_CHUNK) - 1) * BIT_MAP_BLOCKS;

    /* blocks_free = (reserved_public_blocks / BLOCKS_IN_CHUNK) * (BLOCKS_IN_CHUNK - BIT_MAP_BLOCKS); */

    blocks_free = reserved_public_blocks - blocks_used;

    last_total = 0;

    for (ref = BLOCKS_IN_CHUNK; ref < blocks_on_disk; ref += BLOCKS_IN_CHUNK)
    {
	if (ref % 4194304 == 0)
	{
	    printf("%u - %u  used: %d\n", (uint32)(ref / 4194304), (uint32)ref, blocks_used - last_total);
	    fflush(stdout);
	    last_total = blocks_used;
	}

	if (lseek64(obj_file_desc, ref << 8, SEEK_SET) < 0)
	{
	    sprintf(ref_str, "lseek64 ref:%08x", (uint32)ref);
	    perror(ref_str);
	    exit(1);
	}

	bytes_read = read(obj_file_desc, chunk, CHUNK_SIZE);

	blocks_used += BIT_MAP_BLOCKS;

	for (i = 8192; i < bytes_read; i += 256)
	{
	    for (j = 0; j < 256; j++)
	    {
		if (chunk[i+j] != 0) break;
	    }

	    byte_num = (i / 256) / 8;
	    bit_num = (i / 256) % 8;

	    if (j < 256)   /* block not empty */
	    {
		blocks_used++;

		if ((chunk[byte_num] & (0x80 >> bit_num)) == 0)
		{
		    printf("bit clear that shouldn't be, chunk: %x  offset: %05x  byte: %d bit: %d\n", (uint32)ref, i, byte_num, bit_num);
//		    for (j = 0; j < 256; j++) printf("%02x%c", chunk[(byte_num&0xffffff00)+j], j%32 == 31 ? '\n' : ' ');
		    for (j = 0; j < 8192; j++) printf("%02x%c", chunk[j], j%32 == 31 ? '\n' : ' ');
		    printf("\n");
		    for (j = 0; j < 256; j++) printf("%02x%c", chunk[i+j], j%32 == 31 ? '\n' : ' ');
		    printf("\n");
		    exit(1);
		}
	    }
	    else
	    {
		blocks_free++;

		if ((chunk[byte_num] & (0x80 >> bit_num)) != 0)
		{
		    printf("bit set that shouldn't be, chunk: %x  offset: %05x  byte: %d bit: %d\n", (uint32)ref, i, byte_num, bit_num);
//		    for (j = 0; j < 256; j++) printf("%02x%c", chunk[(byte_num&0xffffff00)+j], j%32 == 31 ? '\n' : ' ');
		    for (j = 0; j < 8192; j++) printf("%02x%c", chunk[j], j%32 == 31 ? '\n' : ' ');
		    printf("\n");
		    for (j = 0; j < 256; j++) printf("%02x%c", chunk[i+j], j%32 == 31 ? '\n' : ' ');
		    printf("\n");
		    exit(1);
		}
	    }
	}

	if (bytes_read != CHUNK_SIZE)
	{
	    printf("last block read: %08x\n", (uint32)(ref + bytes_read));
	    break;
	}
    }

    printf("%u+ - %u  used: %d\n", (uint32)(ref / 4194304), (uint32)ref, blocks_used - last_total);
    fflush(stdout);

    close(obj_file_desc);

    blocks_total = blocks_used + blocks_free;

    printf("blocks total: %d  used: %d  free: %d  usage: %.1f%%\n",
	   blocks_total, blocks_used, blocks_free, ((float)blocks_used / (float)blocks_free) );

    return 0;
}

