/*
    Numdiff - compare putatively similar files, 
    ignoring small numeric differences
    Copyright (C) 2005-2007  Ivano Primi  <ivprimi@libero.it>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include<stdio.h>
#include<stdlib.h>
#include<string.h>
#include<limits.h>
#include<errno.h>
#include<ctype.h>
#ifdef ENABLE_NLS
#include<locale.h>
#endif
#include"ndfilter.h"
#include"numdiff.h" /* Only for LINE_INTERR definition */
#include"getopt.h"

extern int errno;

/* See io.c */
extern char* read_line (FILE* pf, int* errcode);

static char* strjoin (const char* s, const char* t)
{
  char* st;

  if (!s && !t)
    return NULL;
  else if (!t)
    {
      if ( (st = (char*) malloc ((strlen(s)+1)*sizeof(char))) )
	strcpy (st, s);
    }
  else if (!s)
    {
      if ( (st = (char*) malloc ((strlen(t)+1)*sizeof(char))) )
	strcpy (st, t);
    }
  else
    {
      if ( (st = (char*) malloc ((strlen(s)+strlen(t)+1)*sizeof(char))) )
	{
	  strcpy (st, s);
	  strcat (st, t);
	}
    }
  return st;
}

/*
  `table' must be the address of a `flg_array' structure.
*/
static int init_flg_array (flg_array* table)
{
  if ((table))
    {
      table->ptr = NULL;
      table->size = table->len = 0;
      return 0;
    }
  else
    return -1;
}

/*
  `table' must be the address of a `flg_array' structure.
*/
static int print_flg_array (FILE* fp, const flg_array* table)
{
  size_t n;
  unsigned short i;
  unsigned char byte, elem;

  if (!table || !table->ptr)
    return (fputs (_("<The array is empty>\n"), fp) == EOF);
  else
    {
      int outerror = 0;

      for (n = 0; !outerror && n < table->len; n += 4U)
	{
	  byte = table->ptr[n/4U];
	  for (i = 0; i < 4U; i++)
	    {
	      if ( (elem = (byte & 0x03 << 2*i) >> 2 * i) )
		{
		  if ( fprintf (fp, "%u", elem) < 0 )
		    {
		      outerror = 1;
		      break;
		    }
		}
	      else
		break;
	    }
	}
      if (!outerror)
	outerror = (fputc ('\n', fp) == EOF);
      return outerror;
    }
}

/*
  array must be the address of a `flg_array' structure.
*/
static int addnewelem (flg_array* array, unsigned char elem)
{
  if (!array)
    return -1;
  if ( !array->ptr )
    {
      if ( !(array->ptr = calloc (CHUNK_ALLOC, sizeof (unsigned char))) )
	return -1;
      else
	{
	  array->size = CHUNK_ALLOC;
	  array->ptr[0] = elem;
	  array->len = 1;
#ifdef __MEMDEBUG__
	  fprintf (stderr, "size = %6zu, len = %6zu, string:",
		   array->size, array->len);
	  print_flg_array (stderr, array);
#endif
	  return 0;
	}
    }
  else
    {
      if (array->len == 4 * array->size - 1)
	{
	  unsigned char* p;

	  if ( !(p = realloc (array->ptr, array->size + CHUNK_ALLOC)) )
	    return -1;
	  else
	    {
	      array->ptr = p;
	      array->size += CHUNK_ALLOC;
	      array->ptr[array->len / 4U] |= elem << 2 * (array->len % 4U);
	      array->len++; /* Now array->len == 4 * array->"old"size */
	      for (p = array->ptr + array->len / 4; p < array->ptr + array->size; *p = 0, p++);
#ifdef __MEMDEBUG__
	  fprintf (stderr, "size = %6zu, len = %6zu, string:",
		   array->size, array->len);
	  print_flg_array (stderr, array);
#endif
	      return 0;
	    }
	}
      else
	{
	  array->ptr[array->len / 4U] |= elem << 2 * (array->len % 4U);
	  array->len++;
#ifdef __MEMDEBUG__
	  fprintf (stderr, "size = %6zu, len = %6zu, string:",
		   array->size, array->len);
	  print_flg_array (stderr, array);
#endif
	  return 0;
	}
    }
}

/*
  array must be the address of a `flg_array' structure.
*/
static void destroy_flg_array (flg_array* array)
{
  if ((array) && (array->ptr))
    {
      free ((void*)array->ptr);
      array->ptr = NULL;
      array->size = array->len = 0;
    }
} 

static int exec_sdiff (flg_array *table, 
		       const char* command, const char* fn1, const char* fn2, 
		       unsigned short symb_pos, unsigned short tab_wd)
{
  char *line_buffer;
  char *cmdline, *ptr;
  unsigned long pos;
  int errcode;
  FILE* pf;
  
  cmdline = (char*) calloc (strlen(command) + strlen(fn1) + strlen(fn2) + 3,
			    sizeof(char));
  if (!cmdline)
    return OUT_OF_MEM;
  else
    {
      sprintf (cmdline, "%s %s %s", command, fn1, fn2);
      if ( !(pf = popen (cmdline, "r")) )
	return EXEC_ERR;
      while ( (line_buffer = read_line(pf, &errcode), errcode <= LINE_INTERR) )
	{
#ifdef __DEBUG__
	  fputs (line_buffer, stderr);
#endif
	  if (*line_buffer == '\n') /* empty line */
	    {
	      addnewelem (table, 3);
	      continue;
	    }
	  for (ptr = line_buffer, pos = 0; 
	       *ptr != '\0' && pos < symb_pos; ptr++)
	    {
#ifdef __DEBUG__
	      fprintf (stderr, "\'%c\' %lu\n", *ptr, pos); 
#endif
	      pos = (*ptr != '\t') ? pos + 1 : (pos / tab_wd + 1) * tab_wd;
	    }
	  if (pos > symb_pos)
	    {
#ifdef __DEBUG__
	      fprintf (stderr, "\'%c\' %lu\n", *(ptr-1), pos); 
#endif
	      if (*(ptr-1) == '\t')
		addnewelem (table, 3);
	      else
		return FORMAT_ERR;
	    }
	  else /* pos == symb_pos || *ptr == '\0' */
	    {
#ifdef __DEBUG__
		fprintf (stderr, "\'%c\' %lu\n", *ptr, pos); 
#endif
	      switch (*ptr)
		{
		case '<':
		case '(':
		  /* I have found a line which is only in the first file */
		  addnewelem (table, 1);
		  break;
		case '>':
		case ')':
		  /* I have found a line which is only in the 2nd file */
		  addnewelem (table, 2); 
		  break;
		case '|':
		case ' ':
		case '\t':
		case '/':
		case '\\':
		  /* I have found a line which is present in both the files */
		  addnewelem (table, 3);
		  break;
		default:
		  /* Unrecognized symbol, probably you have to try another value of */
		  /* symb_pos.                                                      */
		  return FORMAT_ERR;
		}
	    }
	  if ((line_buffer))
	    free((void*)line_buffer);
	}
      if ( pclose (pf) == -1 )
	return EXEC_ERR;
      else
	return OK;
    }
}

static int fileclose (FILE* fp)
{
  if ((fp))
    return fclose (fp);
  else
    return 0;
}

static int make_new_files (const flg_array* table, int prln,
			   const char* orig1, const char* orig2,
			   const char* new1, const char* new2)
{
  FILE *nfp1, *nfp2, *fp1, *fp2;
  char *line1, *line2;
  size_t n;
  unsigned long ln1, ln2;
  unsigned short i;
  int error, errcode;
  unsigned char byte, elem;

  fp1 = fopen (orig1, "r");
  fp2 = fopen (orig2, "r");
  nfp1 = fopen (new1, "w");
  nfp2 = fopen (new2, "w");
  if (!fp1 || !fp2 || !nfp1 || !nfp2)
    {
      fileclose (fp1);
      fileclose (fp2);
      fileclose (nfp1);
      fileclose (nfp2);
      if (!fp1)
	return OPEN_ERR_ORIG1;
      if (!fp2)
	return OPEN_ERR_ORIG2;
      if (!nfp1)
	return OPEN_ERR_NEW1;
      if (!nfp2)
	return OPEN_ERR_NEW2;
    }
  for (n = 0, ln1 = ln2 = 1, error = 0; 
       !error && n < table->len; n += 4U)
    {
      byte = table->ptr[n/4U];
      for (i = 0; i < 4U; i++)
	{
	  elem = (byte & 0x03 << 2*i) >> 2 * i;
	  switch (elem)
	    {
	    case 1:
	      /* Line only in orig1 */
	      if ( (line1 = read_line(fp1, &errcode), errcode <= LINE_INTERR) )
		{
		  if ((prln))
		    {
		      if ( fprintf (nfp1, "%lu: %s", ln1, line1) < 0 )
			error = WRITE_ERR_NEW1;
		    }
		  else
		    {
		      if (fputs (line1, nfp1) == EOF)
			error = WRITE_ERR_NEW1;
		    }
		  if (errcode == LINE_INTERR && putc (NEW_LINE, nfp1) == EOF)
		    error = WRITE_ERR_NEW1;
		  if (fputs (EMPTY_LINE, nfp2) == EOF)
		    error = WRITE_ERR_NEW2;
		  if ((line1))
		    free((void*)line1);
		  ln1++;
		}
	      else
		error = READ_ERR_ORIG1;
	      break;
	    case 2:
	      /* Line only in orig2 */
	      if ( (line2 = read_line(fp2, &errcode), errcode <= LINE_INTERR) )
		{
		  if ((prln))
		    {
		      if ( fprintf (nfp2, "%lu: %s", ln2, line2) < 0 )
			error = WRITE_ERR_NEW2;
		    }
		  else
		    {
		      if (fputs (line2, nfp2) == EOF)
			error = WRITE_ERR_NEW2;
		    }
		  if (errcode == LINE_INTERR && putc (NEW_LINE, nfp2) == EOF)
		    error = WRITE_ERR_NEW2;
		  if (fputs (EMPTY_LINE, nfp1) == EOF)
		    error = WRITE_ERR_NEW1;
		  if ((line2))
		    free((void*)line2);
		  ln2++;
		}
	      else
		error = READ_ERR_ORIG2;
	      break;
	    case 3:
	      /* Line both in orig1 and orig2 */
	      if ( (line1 = read_line(fp1, &errcode), errcode <= LINE_INTERR) )
		{
		  if ((prln))
		    {
		      if ( fprintf (nfp1, "%lu: %s", ln1, line1) < 0 )
			error = WRITE_ERR_NEW1;
		    }
		  else
		    {
		      if (fputs (line1, nfp1) == EOF)
			error = WRITE_ERR_NEW1;
		    }
		  if (errcode == LINE_INTERR && putc (NEW_LINE, nfp1) == EOF)
		    error = WRITE_ERR_NEW1;
		  if ((line1))
		    free((void*)line1);
		  ln1++;
		}
	      else
		error = READ_ERR_ORIG1;
	      if ( (line2 = read_line(fp2, &errcode), errcode <= LINE_INTERR) )
		{
		  if ((prln))
		    {
		      if ( fprintf (nfp2, "%lu: %s", ln2, line2) < 0 )
			error = WRITE_ERR_NEW2;
		    }
		  else
		    {
		      if (fputs (line2, nfp2) == EOF)
			error = WRITE_ERR_NEW2;
		    }
		  if (errcode == LINE_INTERR && putc (NEW_LINE, nfp2) == EOF)
		    error = WRITE_ERR_NEW2;
		  if ((line2))
		    free((void*)line2);
		  ln2++;
		}
	      else
		error = READ_ERR_ORIG2;
	      break;
	    case 0:
	      break;
	    }
	  if ((error) || !elem)
	    break;
	}
    }
  fileclose (fp1);
  fileclose (fp2);
  fileclose (nfp1);
  fileclose (nfp2);
  return error;
}

static void print_ndversion (const char* progname)
{
  printf ("%s %s\n", progname, VERSION);
  printf (_("Copyright (C) 2005-2007  %s <ivprimi@libero.it>\n"), 
	  /* TRANSLATORS: This is a proper name.  See the gettext
	     manual, section Names.
	     Pronounciation is like "evaa-no pree-me".  */
	  _("Ivano Primi"));
  puts (_("This is free software; see the source for copying conditions."));
  puts (_("There is NO warranty; not even for MERCHANTABILITY or\nFITNESS FOR A PARTICULAR PURPOSE.\n"));
}

static void print_ndhelp (const char* progname)
{
  puts (_("Usage:"));
  printf ("%s -h|v   %s\n\n", progname, _("or"));
  printf ("%s %s\n", progname, "[-s sffx][-e \"opts\"][-p col][-t tabwd][-n][-d][-l path][-o path] FILE1 FILE2");
  /* %%% */
  printf (_("\nCompare the given files by recalling \"sdiff\",\nand use the results of this comparison to create\na new couple of files, which hopefully should be correctly\nsynchronized for a later comparison through \"numdiff\"\n\n"));
  printf ("-s sffx   %s\n          %s\n", 
	  _("Specify the suffix to add to the old name\n\t  when creating each of the two new files"),
	  _("(The default suffix is \".new\")"));
  printf ("-e \"opts\" %s\n          %s\n", 
	  _("Specify the options to put on the command line\n\t  when calling \"sdiff\""),
	  _("(The default behavior is executing \"sdiff\" with no options)"));
  printf ("-p col    %s\n          %s\n", 
	  _("Specify the column number of the gutter in the \"sdiff\" output"),
	  _("(The default value is the one for GNU sdiff, i.e. 62)"));
  printf ("-t tabwd  %s\n          %s\n",
	  _("Specify the width of the tabulations in the \"sdiff\" output"),
	  _("(The default value is 8)"));
  printf ("-n        %s\n",
	  _("Print the line number at the head of each line\n\t  when writing the two new files"));
  printf ("-d        %s\n",
	  _("Print debug information on stderr"));
  printf ("-l path   %s\n",
	  _("Redirect warning and error messages from stderr to the indicated file"));
  printf ("-o path   %s\n",
	  _("Redirect output from stdout to the indicated file"));
  printf ("-h        %s\n", _("Show this help message"));
  printf ("-v        %s\n", _("Show version number, Copyright and NO-Warranty"));
  printf ("\n%s\n%s\n%s\n%s\n\n",
	  _("The two arguments after the options are the names of the files to compare."),
	  _("The complete paths of the files should be given,\na directory name is not accepted."),
	  _("These paths cannot refer to the same file."),
	  _("Exit status: 0 in case of normal termination, -1 (255) in case of error"));
}

extern char *optarg; 
extern int optind;

static int set_arguments (int argc, char* argv[], nargslist *list)
{
  const char *optstring = "hs:p:t:ne:dl:o:v";
  char *endptr;
  int optch; 
  unsigned long temp;

  /*
    We start by loading the default values
    for the user settable options.
  */
  list->optmask = 0x0;
  list->command = SDIFF_CMD;
  list->suffix = SUFFIX;
  list->tab_wd = TAB_WD;
  list->symb_pos = DEF_SYMB_POS;
  list->file1 = list->file2 = NULL;

  while ( (optch = getopt (argc, argv, optstring)) != -1 )
    {
      switch (optch)
	{
	case 'h':
	  list->optmask |= __H_MASK;
	  break;
	case 's':
	  list->optmask |= __S_MASK;
	  if (!(*optarg))
	    {
	      fprintf (stderr, _("%s: invalid argument after `-%c\' option\n"),
		       argv[0], optch);
	      return -1;
	    }
	  else
	    list->suffix = optarg;
	  break;
	case 'p':
	  list->optmask |= __P_MASK;
	  for (endptr = optarg; isspace (*endptr) != 0; endptr++);
	  if (*endptr == '-' ||
	      (temp = strtoul (optarg, &endptr, 10)) > USHRT_MAX ||
	      *endptr != '\0')
	    {
	      fprintf (stderr, _("%s: invalid argument after `-%c\' option\n"),
		       argv[0], optch);
	      return -1;
	    }
	  else
	    list->symb_pos = temp;
	  break;
	case 't':
	  list->optmask |= __T_MASK;
	  for (endptr = optarg; isspace (*endptr) != 0; endptr++);
	  if (*endptr == '-' ||
	      (temp = strtoul (optarg, &endptr, 10)) > USHRT_MAX ||
	      *endptr != '\0')
	    {
	      fprintf (stderr, _("%s: invalid argument after `-%c\' option\n"),
		       argv[0], optch);
	      return -1;
	    }
	  else
	    list->tab_wd = temp;
	  break;
	case 'n':
	  list->optmask |= __N_MASK;
	  break;
	case 'e':
	  list->optmask |= __E_MASK;
	  if ( !(list->command = strjoin (SDIFF_CMD, optarg)) )
	    {
	      fputs (_("Fatal error: Insufficient memory for new allocation\n"), stderr);
	      return -1;
	    }
	  break;
	case 'd':
	  list->optmask |= __D_MASK;
	  break;
	case 'l':
	  if (!freopen (optarg, "w", stderr))
	    {
	      fprintf (stderr, _("%s: Cannot open file \"%s\": "),
		       argv[0], optarg);
	      perror(0);
	      return -1;
	    }
	  break;
	case 'o':
	  if (!freopen (optarg, "w", stdout))
	    {
	      fprintf (stderr, _("%s: Cannot open file \"%s\": "),
		       argv[0], optarg);
	      perror(0);
	      return -1;
	    }
	  break;
	case 'v':
	  list->optmask |= __V_MASK;
	  break;
	default:
/*  	  fprintf (stderr,  */
/*  		   _("%s: unrecognized option `-%c\' \n"), argv[0], optch);  */
	  return -1;
	}
    }
  if (!(list->optmask & (__H_MASK | __V_MASK)) && argc - optind != 2)
    {
      print_ndhelp (PACKAGE2);
      return -1;
    }
  else
    {
      if( !(list->optmask & (__H_MASK | __V_MASK)) )
	{
	  list->file1 = (const char*) argv[optind];
	  list->file2 = (const char*) argv[optind+1];
	}
      return 0;
    }
}

int main (int argc, char** argv)
{
  flg_array table;
  int errcode;
  nargslist arg_list;

#ifdef ENABLE_NLS
  setlocale (LC_CTYPE, "");
  setlocale (LC_MESSAGES, "");
#endif
  bindtextdomain (PACKAGE2, LOCALEDIR);
  textdomain (PACKAGE2);
  if ( set_arguments (argc, argv, &arg_list) != 0 )
    return -1;
  else if ( (arg_list.optmask & (__H_MASK | __V_MASK)) )
    {
      if ((arg_list.optmask & __V_MASK))
      	print_ndversion(PACKAGE2);
      if ((arg_list.optmask & __H_MASK))
      	print_ndhelp(PACKAGE2);
      if (argc > 2)
	return -1;
      else
	return 0;
    }
  else
    {
      init_flg_array (&table);
      if ( strcmp (arg_list.file1, arg_list.file2) == 0 )
	{
	  fputs (_("Cannot compare a file with itself\n\n"), stderr);
	  return -1;	  
	}
      else if ( (errcode = exec_sdiff (&table, arg_list.command, 
				       arg_list.file1, arg_list.file2,
				       arg_list.symb_pos, arg_list.tab_wd)) )
	{
	  if (errcode == EXEC_ERR)
	    fprintf (stderr, _("Error while executing\n\"%s %s %s\"\n"), 
		     arg_list.command, arg_list.file1, arg_list.file2);
	  else if (errcode == FORMAT_ERR)
	    {
	      fprintf (stderr,
		       _("Error while parsing the output of\n\"%s %s %s\",\n"),
		       arg_list.command, arg_list.file1, arg_list.file2);
	      fputs (_("you must probably reset the value of the argument for the \'-p\' option,\n"), stderr);
	      fprintf (stderr, _("its current value is %hu\n"),
		       arg_list.symb_pos);
	    }
	  else
	    /* errcode == OUT_OF_MEM */
	    fputs (_("Fatal error: Insufficient memory for new allocation\n"),
		   stderr);
	  destroy_flg_array (&table);
	  return -1;
	}
      else if (!table.ptr)
	{
	  fprintf (stderr, 
		   _("No output resulting from the execution of\n\"%s %s %s\"\n(maybe the compared files are both empty \?)\n"), 
		   arg_list.command, arg_list.file1, arg_list.file2);
	  return -1;
	}
      else
	{
	  char *new1, *new2;
	  size_t s1 = strlen (arg_list.file1) + strlen (arg_list.suffix) + 1;
	  size_t s2 = strlen (arg_list.file2) + strlen (arg_list.suffix) + 1;

	  if ((arg_list.optmask & __D_MASK))
	    print_flg_array (stdout, &table);
	  new1 = calloc (s1, sizeof(char));
	  new2 = calloc (s2, sizeof(char));
	  if (!new1 || !new2)
	    {
	      if ((new1))
		free ((void*)new1);
	      if ((new2))
		free ((void*)new2);
	      fputs (_("Fatal error: Insufficient memory for new allocation\n"),
		     stderr);
	      return -1;
	    }
	  else
	    {
	      strcpy (new1, arg_list.file1);
	      strcat (new1, arg_list.suffix);
	      strcpy (new2, arg_list.file2);
	      strcat (new2, arg_list.suffix);
	      errcode = make_new_files (&table, arg_list.optmask & __N_MASK, 
					arg_list.file1, arg_list.file2, 
					new1, new2);
	      if ((errcode))
		fputs (_("Error occurred while creating the new files\n"), stderr);
	      else
		fputs (_("New files successfully created\n"), stdout);
	      destroy_flg_array (&table);
	      return 0;
	    }
	}
    }
}

