/***************************************************************************
 *            nc_snia_dist_cov.c
 *
 *  Mon December 03 19:34:29 2012
 *  Copyright  2012  Sandro Dias Pinto Vitenti
 *  <sandro@isoftware.com.br>
 ****************************************************************************/
/*
 * numcosmo
 * Copyright (C) 2012 Sandro Dias Pinto Vitenti <sandro@isoftware.com.br>
 * 
 * numcosmo is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * numcosmo is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * SECTION:nc_snia_dist_cov
 * @title: Supernovae Distance Covariance
 * @short_description: Calculates the covariance between distance estimates
 *
 * FIXME
 * 
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif /* HAVE_CONFIG_H */
#include "build_cfg.h"

#include "nc_snia_dist_cov.h"

#define VECTOR (NCM_MODEL (dcov)->params)
#define ALPHA  (ncm_vector_get (VECTOR, NC_SNIA_DIST_COV_ALPHA))
#define BETA   (ncm_vector_get (VECTOR, NC_SNIA_DIST_COV_BETA))

enum
{
  PROP_0,
  PROP_MU_LEN,
  PROP_SIGMA_PECZ,
  PROP_SIZE,
};

G_DEFINE_TYPE (NcSNIADistCov, nc_snia_dist_cov, NCM_TYPE_MODEL);

static void
nc_snia_dist_cov_init (NcSNIADistCov *dcov)
{
  dcov->mu_len           = 0;
  dcov->magnitude        = NULL;
  dcov->width            = NULL;
  dcov->colour           = NULL;
  dcov->magnitude_width  = NULL;
  dcov->magnitude_colour = NULL;
  dcov->width_colour     = NULL;
  dcov->snia_data        = NULL;
  dcov->sigma_pecz       = 0.0;
  dcov->dataset          = NULL;
}

static void
_nc_snia_dist_cov_set_property (GObject * object, guint prop_id, const GValue * value, GParamSpec * pspec)
{
  NcSNIADistCov *dcov = NC_SNIA_DIST_COV (object);
  g_return_if_fail (NC_IS_SNIA_DIST_COV (object));

  switch (prop_id)
  {
    case PROP_MU_LEN:
      nc_snia_dist_cov_set_size (dcov, g_value_get_uint (value));
      break;
    case PROP_SIGMA_PECZ:
      dcov->sigma_pecz = g_value_get_double (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
_nc_snia_dist_cov_get_property (GObject *object, guint prop_id, GValue *value, GParamSpec *pspec)
{
  NcSNIADistCov *dcov = NC_SNIA_DIST_COV (object);
  g_return_if_fail (NC_IS_SNIA_DIST_COV (object));

  switch (prop_id)
  {
    case PROP_MU_LEN:
      g_value_set_uint (value, dcov->mu_len);
      break;
    case PROP_SIGMA_PECZ:
      g_value_set_double (value, dcov->sigma_pecz);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
nc_snia_dist_cov_dispose (GObject *object)
{
  NcSNIADistCov *dcov = NC_SNIA_DIST_COV (object);

  nc_snia_dist_cov_set_size (dcov, 0);

  /* Chain up : end */
  G_OBJECT_CLASS (nc_snia_dist_cov_parent_class)->dispose (object);
}

static void
nc_snia_dist_cov_finalize (GObject *object)
{
  
  /* Chain up : end */
  G_OBJECT_CLASS (nc_snia_dist_cov_parent_class)->finalize (object);
}

gint32 NC_SNIA_DIST_COV_ID = -1;

static void
nc_snia_dist_cov_class_init (NcSNIADistCovClass *klass)
{
  GObjectClass* object_class = G_OBJECT_CLASS (klass);
  NcmModelClass* model_class = NCM_MODEL_CLASS (klass);

  object_class->set_property = &ncm_model_class_set_property;
  object_class->get_property = &ncm_model_class_get_property;  
  object_class->dispose      = &nc_snia_dist_cov_dispose;
  object_class->finalize     = &nc_snia_dist_cov_finalize;

  model_class->set_property = &_nc_snia_dist_cov_set_property;
  model_class->get_property = &_nc_snia_dist_cov_get_property;

  g_object_class_install_property (object_class,
                                   PROP_MU_LEN,
                                   g_param_spec_uint ("mu-len",
                                                      NULL,
                                                      "Distance modulus length",
                                                      0, G_MAXUINT, 0,
                                                      G_PARAM_READWRITE | G_PARAM_CONSTRUCT |G_PARAM_STATIC_NAME | G_PARAM_STATIC_BLURB));
  g_object_class_install_property (object_class,
                                   PROP_SIGMA_PECZ,
                                   g_param_spec_double ("sigma-pecz",
                                                      NULL,
                                                      "Error from SN Ia peculiar velocity",
                                                      0.0, 1.0e1, 5.0e-4,
                                                      G_PARAM_READWRITE | G_PARAM_CONSTRUCT |G_PARAM_STATIC_NAME | G_PARAM_STATIC_BLURB));

  ncm_model_class_register_id (model_class);
  NC_SNIA_DIST_COV_ID = model_class->model_id;

  ncm_model_class_add_params (model_class, NC_SNIA_DIST_COV_SPARAM_LEN, NC_SNIA_DIST_COV_VPARAM_LEN, PROP_SIZE);
  ncm_model_class_set_name_nick (model_class, "Supernovae Ia Distance Covariance", "SNIaDistCov");

  ncm_model_class_set_sparam (model_class, NC_SNIA_DIST_COV_ALPHA, "alpha", "alpha",
                              -10.0, 10.0, 1.0e-1,
                              NC_SNIA_DIST_COV_DEFAULT_PARAMS_ABSTOL, NC_SNIA_DIST_COV_DEFAULT_ALPHA,
                              NCM_PARAM_TYPE_FIXED);

  ncm_model_class_set_sparam (model_class, NC_SNIA_DIST_COV_BETA, "beta", "beta",
                              -10.0, 10.0, 1.0e-1,
                              NC_SNIA_DIST_COV_DEFAULT_PARAMS_ABSTOL, NC_SNIA_DIST_COV_DEFAULT_BETA,
                              NCM_PARAM_TYPE_FIXED);

  ncm_model_class_set_vparam (model_class, NC_SNIA_DIST_COV_SIGMA_INT, NC_SNIA_DIST_COV_SIGMA_INT_DEFAULT_LEN, 
                              "Sigma intrisic", "sigma_int",
                              0.0, 1.0e1, 1.0e-3, 
                              NC_SNIA_DIST_COV_DEFAULT_PARAMS_ABSTOL, NC_SNIA_DIST_COV_DEFAULT_SIGMA_INT,
                              NCM_PARAM_TYPE_FIXED);

}

/**
 * nc_snia_dist_cov_new:
 * @mu_len: FIXME
 * 
 * FIXME
 * 
 * Returns: FIXME
 */
NcSNIADistCov *
nc_snia_dist_cov_new (guint mu_len)
{
  return g_object_new (NC_TYPE_SNIA_DIST_COV,
                       "mu-len", mu_len,
                       NULL);
}

/**
 * nc_snia_dist_cov_free:
 * @dcov: FIXME
 * 
 * FIXME
 * 
 */
void 
nc_snia_dist_cov_free (NcSNIADistCov *dcov)
{
  g_object_unref (dcov);
}

/**
 * nc_snia_dist_cov_clear:
 * @dcov: FIXME
 * 
 * FIXME
 * 
 */
void 
nc_snia_dist_cov_clear (NcSNIADistCov **dcov)
{
  g_clear_object (dcov);
}

/**
 * nc_snia_dist_cov_set_size:
 * @dcov: FIXME
 * @mu_len: FIXME
 * 
 * FIXME
 * 
 */
void 
nc_snia_dist_cov_set_size (NcSNIADistCov *dcov, guint mu_len)
{
  if (mu_len == 0 || mu_len != dcov->mu_len)
  {
    ncm_matrix_clear (&dcov->magnitude);
    ncm_matrix_clear (&dcov->width);
    ncm_matrix_clear (&dcov->colour);

    ncm_matrix_clear (&dcov->magnitude_width);
    ncm_matrix_clear (&dcov->magnitude_colour);
    ncm_matrix_clear (&dcov->width_colour);

    ncm_matrix_clear (&dcov->snia_data);

    if (dcov->dataset != NULL)
    {
      g_array_unref (dcov->dataset);
      dcov->dataset = NULL;
    }
  }

  if (mu_len > 0)
  {
    dcov->mu_len           = mu_len;
    dcov->magnitude        = ncm_matrix_new_sunk (mu_len, mu_len);
    dcov->width            = ncm_matrix_new_sunk (mu_len, mu_len);
    dcov->colour           = ncm_matrix_new_sunk (mu_len, mu_len);
    dcov->magnitude_width  = ncm_matrix_new_sunk (mu_len, mu_len);
    dcov->magnitude_colour = ncm_matrix_new_sunk (mu_len, mu_len);
    dcov->width_colour     = ncm_matrix_new_sunk (mu_len, mu_len);

    dcov->snia_data        = ncm_matrix_new_sunk (mu_len, NC_SNIA_DIST_COV_LENGTH);

    dcov->dataset          = g_array_sized_new (FALSE, FALSE, sizeof(guint), mu_len);

    g_array_set_size (dcov->dataset, mu_len);
  }
}

/**
 * nc_snia_dist_cov_calc:
 * @dcov: FIXME
 * @cov: FIXME
 * 
 * FIXME
 * 
 */
void 
nc_snia_dist_cov_calc (NcSNIADistCov *dcov, NcmMatrix *cov)
{
  const gdouble alpha          = ALPHA;
  const gdouble beta           = BETA;
  const gdouble alpha2         = alpha * alpha;
  const gdouble beta2          = beta * beta;
  const gdouble two_alpha_beta = 2.0 * alpha * beta;

  ncm_matrix_memcpy (cov, dcov->magnitude);

  ncm_matrix_add_mul (cov, alpha2,         dcov->width);
  ncm_matrix_add_mul (cov, beta2,          dcov->colour);
  ncm_matrix_add_mul (cov,  2.0 * alpha,   dcov->magnitude_width);
  ncm_matrix_add_mul (cov, -2.0 * beta,    dcov->magnitude_colour);
  ncm_matrix_add_mul (cov, two_alpha_beta, dcov->width_colour);
}

static void _nc_snia_dist_cov_load_snia_data (const gchar *filename, NcmMatrix *data);

/**
 * nc_snia_dist_cov_load:
 * @dcov: FIXME
 * @filename: FIXME
 * 
 * FIXME
 * 
 */
void 
nc_snia_dist_cov_load (NcSNIADistCov *dcov, const gchar *filename)
{
  GKeyFile *snia_keyfile = g_key_file_new ();
  GError *error   = NULL;
  gchar *datafile = NULL;
  guint64 mu_len;

  if (!g_key_file_load_from_file (snia_keyfile, filename, G_KEY_FILE_NONE, &error))
    g_error ("nc_snia_dist_cov_load: invalid configuration: %s\n  %s\n", 
             filename, 
             error->message);

  if (!g_key_file_has_key (snia_keyfile, 
                           NC_SNIA_DIST_COV_DATA_GROUP,
                           NC_SNIA_DIST_COV_DATA_KEY,
                           &error))
    g_error ("nc_snia_dist_cov_load: invalid %s key file, it must define at least the data file key ["NC_SNIA_DIST_COV_DATA_KEY"]", filename);
  if (!g_key_file_has_key (snia_keyfile, 
                           NC_SNIA_DIST_COV_DATA_GROUP,
                           NC_SNIA_DIST_COV_DATA_LEN_KEY,
                           &error))
    g_error ("nc_snia_dist_cov_load: invalid %s key file, it must define at least the data length file key ["NC_SNIA_DIST_COV_DATA_LEN_KEY"]", filename);

  datafile = g_key_file_get_string (snia_keyfile, 
                                    NC_SNIA_DIST_COV_DATA_GROUP,
                                    NC_SNIA_DIST_COV_DATA_KEY,
                                    &error);

  mu_len = g_key_file_get_uint64 (snia_keyfile, 
                                  NC_SNIA_DIST_COV_DATA_GROUP,
                                  NC_SNIA_DIST_COV_DATA_LEN_KEY,
                                  &error);
  
  nc_snia_dist_cov_set_size (dcov, mu_len);
  
  _nc_snia_dist_cov_load_snia_data (datafile, dcov->snia_data);
  
  g_key_file_unref (snia_keyfile);
}

static void 
_nc_snia_dist_cov_load_snia_data (const gchar *filename, NcmMatrix *data)
{
  gchar *line = NULL;
  gsize len = 0, tpos = 0;
  GError *error = NULL;
  GIOChannel *file = g_io_channel_new_file (filename, "r", &error);
  if (file == NULL)
    g_error ("_nc_snia_dist_cov_load_snia_data: cannot open file %s: %s", 
             filename, error->message);

  while (g_io_channel_read_line (file, &line, &len, &tpos, &error) != G_IO_STATUS_EOF)
  {
    gchar **itens = g_strsplit_set (line, "        ", -1);
    printf ("# name = %s number of columns %u\n", itens[0], g_strv_length (itens));
    
    g_strfreev (itens);
    g_free (line);
  }
  
  g_io_channel_unref (file);
}
