/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 2; tab-width: 2 -*-  */

/***************************************************************************
 *            ncm_stats_dist.c
 *
 *  Wed November 07 16:02:36 2018
 *  Copyright  2018  Sandro Dias Pinto Vitenti
 *  <vitenti@uel.br>
 ****************************************************************************/
/*
 * ncm_stats_dist.c
 * Copyright (C) 2018 Sandro Dias Pinto Vitenti <vitenti@uel.br>
 *
 * numcosmo is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * numcosmo is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * SECTION:ncm_stats_dist
 * @title: NcmStatsDist
 * @short_description: Abstract class for implementing N-dimensional probability distributions.
 *
 * Abstract class to reconstruct an arbitrary N-dimensional probability distribution.
 * This class provides the tools to perform a radial basis interpolation
 * in a multidimensional function using a radial basis function and then
 * generates a new sample using the interpolation function as the kernel.
 * This method generates a sample that is distributed by the original distribution,
 * but in a more simple way since the used kernels are easier to sample from.
 * For more information about radial basis interpolation,
 * check [[Radial Basis Function Interpolation, Wilna du Toit](https://core.ac.uk/download/pdf/37320748.pdf)].
 * A brief description of the radial basis interpolation method can be found below.
 *
 * Given a d-simensional function $g(x): \mathbf{R}^d \rightarrow \mathbf{R}$, a radial basis
 * function $\phi(x, \Sigma)$ is used such that
 * \begin{align}
 * \label{Interpolation_eq}
 * s(x) = \sum_i^n \lambda_i \phi(|x-x_i|, \Sigma_i), \quad x~ \in~ \mathbf{R}
 *. \end{align}
 * The variables $\lambda_i$ represent the weights and are found such that
 *  \begin{align}
 * \label{eqnnls1}
 * s(x_i) = g(x_i)
 *, \end{align}
 * being $x_i$ the sample points.
 * The values generated by $\phi(|x-x_i|, \Sigma_i)$ are displayed in a symmetric $n \times n$ matrix $\Phi$.
 * This function depends on the norm of the points and on the covariance matrix $\Sigma$ associated with each point.
 * The weights $\lambda_i$ are also organised in a matrix representation such that equation \eqref{eqnnls1} becomes
 * \begin{align}
 * \label{eqnnls}
 * G = \lambda \times \Phi
 * ,\end{align}
 * where $G$ is a matrix containing all the function values $g(xi)$. Once the Lambda matrix is found,
 * one may use $s(x)$ to sample values from $g(x)$, which is easier to do since $s(x)$ is
 * a polynomial function.
 *
 * We want $s(x)$ to be a probability distribution so we can sample from it. Therefore the Lambda matrix containing the
 * weights is seen as the probability density and it must be minimized such that its values are always positive and sum up to one. To solve equation this problem,
 * this algorithm has the tools to solve equation \eqref{eqnnls} for $\lambda$, which is a least-squares problem,
 * using the NNLS method, which can be found in nnls.c file. Thus, the algorithm can randomly choose a kernel $\phi(|x-x_i|, \Sigma_i)$ associated
 * to a probability contained in $\lambda$ and sample a point from it.
 *
 *
 * In this object, the radial basis interpolation function is not completely defined. One must choose one of the instances of the class, the
 * #NcmStatsDistKernelST object or the #NcmStatsDistKernelGauss object, which uses a multivariate Student's t function and a Gaussian function as the kernel.
 * After initializing the desired object for the interpolation function, one may use the methods of this file to generate the interpolation and to
 * sample from the new interpolated function.
 *
 * The user must provide the input the values: @over_smooth - ncm_stats_dist_set_over_smooth(), @split_frac - ncm_stats_dist_set_split_frac(),
 * @over_smooth - ncm_stats_dist_set_over_smooth(), $v(x)$ - ncm_stats_dist_prepare_interp(). The other parameters
 * must be inserted when the instance for the #NcmStatsDistKDE or the #NcmStatsDistVKDE object is initialized. To perform a calculation of this class, one
 * needs to initialize the class within one of its subclasses (#NcmStatsDistKernelGauss or #NcmStatsDistKernelST), along with the input of a child object of the class
 * #NcmStatsDistKernel. For more information about the algorithm, see the description below.
 *
 *	 -Since this class does not define what type of kernel will be used in the calculation (the fixed kernel in the #NcmStatsDistKDE class or the variable kernel in #NcmStatsDistVKDE class),
 *  one cannot compute the sample just using this instance. Also, it must be provided the function to be used as the kernel, which is implemented in the children from the class #NcmStatsDistKernel.
 *  When initializing the #NcmStatsDistKDE or #NcmStatsDistVKDE classes, the function to be used as the kernel is defined in the object initialization function.
 *
 *  -This class also needs a child object to compute the interpolation matrix $IM$ and the covariance matrices stored in @cov_decomp to perform the interpolation,
 *  which is kernel dependent and therefore also computed by the class child objects.
 *
 *  -Regarding the kernel types based on the radial basis function, $\phi(|x-x_i|)$, and how the sample points in ncm_stats_dist_sample() are generated,
 *      see the different implementations of #NcmStatsDistKernel, e.g., #NcmStatsDistKernelGauss and #NcmStatsDistKernelST
 *
 *  -Regarding how the functions ncm_stats_dist_eval() and ncm_stats_dist_eval_m2lnp() are implemented, see
 *  the different implementations of #NcmStatsDist, i.e., #NcmStatsDistKDE and #NcmStatsDistVKDE. These objects also
 *  compute the covariance matrix of each sample point and other objects needed for the least-squares problem, when
 *  computing the weights matrix ($\lambda$).
 *
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif /* HAVE_CONFIG_H */
#include "build_cfg.h"

#include "math/ncm_stats_dist.h"
#include "math/ncm_iset.h"
#include "math/ncm_rng.h"
#include "math/ncm_lapack.h"
#include "ncm_enum_types.h"

#ifndef NUMCOSMO_GIR_SCAN
#include <gsl/gsl_blas.h>
#include <gsl/gsl_min.h>
#include <gsl/gsl_multimin.h>
#include <gsl/gsl_sort.h>
#include <omp.h>
#include "levmar/levmar.h"
#endif /* NUMCOSMO_GIR_SCAN */

#include "math/ncm_stats_dist_private.h"

enum
{
  PROP_0,
  PROP_KERNEL,
  PROP_SAMPLE_SIZE,
  PROP_OVER_SMOOTH,
  PROP_CV_TYPE,
  PROP_USE_THREADS,
  PROP_SPLIT_FRAC,
  PROP_PRINT_FIT,
};

G_DEFINE_ABSTRACT_TYPE_WITH_PRIVATE (NcmStatsDist, ncm_stats_dist, G_TYPE_OBJECT);

#define NCM_NNLS_SOLVE ncm_nnls_solve

static void
ncm_stats_dist_init (NcmStatsDist *sd)
{
  NcmStatsDistPrivate * const self = sd->priv = ncm_stats_dist_get_instance_private (sd);

  self->kernel          = NULL;
  self->sample_array    = g_ptr_array_new ();
  self->weights         = NULL;
  self->wcum            = NULL;
  self->wcum_ready      = FALSE;
  self->print_fit       = FALSE;
  self->over_smooth     = 0.0;
  self->cv_type         = NCM_STATS_DIST_CV_LEN;
  self->use_threads     = FALSE;
  self->split_frac      = 0.0;
  self->min_m2lnp       = 0.0;
  self->max_m2lnp       = 0.0;
  self->href            = 0.0;
  self->rnorm           = 0.0;
  self->n_obs           = 0;
  self->n_kernels       = 0;
  self->alloc_n_obs     = 0;
  self->alloc_n_kernels = 0;
  self->alloc_subs      = FALSE;
  self->d               = 0;
  self->sampling        = g_array_new (FALSE, FALSE, sizeof (guint));
  self->nnls            = NULL;
  self->IM              = NULL;
  self->sub_IM          = NULL;
  self->sub_x           = NULL;
  self->f               = NULL;
  self->f1              = NULL;
  self->levmar_workz    = NULL;
  self->levmar_n        = 0;
  self->fmin            = gsl_multimin_fminimizer_alloc (gsl_multimin_fminimizer_nmsimplex2, 1);
  self->m2lnp_sort      = g_array_new (FALSE, FALSE, sizeof (size_t));
  self->rng             = ncm_rng_seeded_new (NULL, 0);

  g_ptr_array_set_free_func (self->sample_array, (GDestroyNotify) ncm_vector_free);
}

static void
_ncm_stats_dist_set_property (GObject *object, guint prop_id, const GValue *value, GParamSpec *pspec)
{
  NcmStatsDist *sd = NCM_STATS_DIST (object);

  /*g_return_if_fail (NCM_IS_STATS_DIST (object));*/

  switch (prop_id)
  {
    case PROP_KERNEL:
      ncm_stats_dist_set_kernel (sd, g_value_get_object (value));
      break;
    case PROP_SAMPLE_SIZE: /* LCOV_EXCL_BR_LINE */
      g_assert_not_reached ();
      break;
    case PROP_OVER_SMOOTH:
      ncm_stats_dist_set_over_smooth (sd, g_value_get_double (value));
      break;
    case PROP_CV_TYPE:
      ncm_stats_dist_set_cv_type (sd, g_value_get_enum (value));
      break;
    case PROP_USE_THREADS:
      ncm_stats_dist_set_use_threads (sd, g_value_get_boolean (value));
      break;
    case PROP_SPLIT_FRAC:
      ncm_stats_dist_set_split_frac (sd, g_value_get_double (value));
      break;
    case PROP_PRINT_FIT:
      ncm_stats_dist_set_print_fit (sd, g_value_get_boolean (value));
      break;
    default: /* LCOV_EXCL_BR_LINE */
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
_ncm_stats_dist_get_property (GObject *object, guint prop_id, GValue *value, GParamSpec *pspec)
{
  NcmStatsDist *sd                 = NCM_STATS_DIST (object);
  NcmStatsDistPrivate * const self = sd->priv;

  g_return_if_fail (NCM_IS_STATS_DIST (object));

  switch (prop_id)
  {
    case PROP_KERNEL:
      g_value_set_object (value, ncm_stats_dist_peek_kernel (sd));
      break;
    case PROP_SAMPLE_SIZE:                               /* LCOV_EXCL_BR_LINE */
      g_value_set_uint (value, self->sample_array->len); /* LCOV_EXCL_LINE */
      break;
    case PROP_OVER_SMOOTH:
      g_value_set_double (value, ncm_stats_dist_get_over_smooth (sd));
      break;
    case PROP_CV_TYPE:
      g_value_set_enum (value, ncm_stats_dist_get_cv_type (sd));
      break;
    case PROP_USE_THREADS:
      g_value_set_boolean (value, ncm_stats_dist_get_use_threads (sd));
      break;
    case PROP_SPLIT_FRAC:
      g_value_set_double (value, ncm_stats_dist_get_split_frac (sd));
      break;
    case PROP_PRINT_FIT:
      g_value_set_boolean (value, ncm_stats_dist_get_print_fit (sd));
      break;
    default: /* LCOV_EXCL_BR_LINE */
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
_ncm_stats_dist_dispose (GObject *object)
{
  NcmStatsDist *sd                 = NCM_STATS_DIST (object);
  NcmStatsDistPrivate * const self = sd->priv;

  ncm_stats_dist_kernel_clear (&self->kernel);

  g_clear_pointer (&self->sample_array, g_ptr_array_unref); /* LCOV_EXCL_BR_LINE */
  ncm_vector_clear (&self->weights);
  ncm_vector_clear (&self->wcum);

  g_clear_pointer (&self->sampling, g_array_unref); /* LCOV_EXCL_BR_LINE */

  ncm_nnls_clear (&self->nnls);

  ncm_matrix_clear (&self->IM);
  ncm_matrix_clear (&self->sub_IM);
  ncm_vector_clear (&self->sub_x);
  ncm_vector_clear (&self->f);
  ncm_vector_clear (&self->f1);

  ncm_rng_clear (&self->rng);

  g_clear_pointer (&self->m2lnp_sort, g_array_unref); /* LCOV_EXCL_BR_LINE */

  /* Chain up : end */
  G_OBJECT_CLASS (ncm_stats_dist_parent_class)->dispose (object);
}

static void
_ncm_stats_dist_finalize (GObject *object)
{
  NcmStatsDist *sd                 = NCM_STATS_DIST (object);
  NcmStatsDistPrivate * const self = sd->priv;

  self->levmar_n = 0;
  g_clear_pointer (&self->levmar_workz, g_free);
  g_clear_pointer (&self->fmin, gsl_multimin_fminimizer_free);

  /* Chain up : end */
  G_OBJECT_CLASS (ncm_stats_dist_parent_class)->finalize (object);
}

static void _ncm_stats_dist_set_dim (NcmStatsDist *sd, const guint dim);
static gdouble _ncm_stats_dist_get_href (NcmStatsDist *sd);

/* LCOV_EXCL_START these should be overwriten and never executed */

static void
_ncm_stats_dist_prepare_kernel (NcmStatsDist *sd, GPtrArray *sample_array)
{
  g_error ("method prepare_kernel not implemented by %s.", G_OBJECT_TYPE_NAME (sd));
}

static void _ncm_stats_dist_prepare (NcmStatsDist *sd);
static void _ncm_stats_dist_prepare_interp (NcmStatsDist *sd, NcmVector *m2lnp);

static void
_ncm_stats_dist_compute_IM (NcmStatsDist *sd, NcmMatrix *IM)
{
  g_error ("method compute_IM not implemented by %s.", G_OBJECT_TYPE_NAME (sd));
}

static NcmMatrix *
_ncm_stats_dist_peek_cov_decomp (NcmStatsDist *sd, guint i)
{
  g_error ("method peek_cov_decomp not implemented by %s.", G_OBJECT_TYPE_NAME (sd));

  return NULL;
}

static gdouble
_ncm_stats_dist_get_lnnorm (NcmStatsDist *sd, guint i)
{
  g_error ("method get_lnnorm not implemented by %s.", G_OBJECT_TYPE_NAME (sd));

  return 0.0;
}

static gdouble
_ncm_stats_dist_eval_weights (NcmStatsDist *sd, NcmVector *weights, NcmVector *x)
{
  g_error ("method eval_weights not implemented by %s.", G_OBJECT_TYPE_NAME (sd));

  return 0.0;
}

static gdouble
_ncm_stats_dist_eval_weights_m2lnp (NcmStatsDist *sd, NcmVector *weights, NcmVector *x)
{
  g_error ("method eval_weights_m2lnp not implemented by %s.", G_OBJECT_TYPE_NAME (sd));

  return 0.0;
}

static void _ncm_stats_dist_reset (NcmStatsDist *sd);

/* LCOV_EXCL_STOP */

static void
ncm_stats_dist_class_init (NcmStatsDistClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  object_class->set_property = &_ncm_stats_dist_set_property;
  object_class->get_property = &_ncm_stats_dist_get_property;
  object_class->dispose      = &_ncm_stats_dist_dispose;
  object_class->finalize     = &_ncm_stats_dist_finalize;

  g_object_class_install_property (object_class,
                                   PROP_KERNEL,
                                   g_param_spec_object ("kernel",
                                                        NULL,
                                                        "Interpolating kernel",
                                                        NCM_TYPE_STATS_DIST_KERNEL,
                                                        G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY | G_PARAM_STATIC_NAME | G_PARAM_STATIC_BLURB));
  g_object_class_install_property (object_class,
                                   PROP_SAMPLE_SIZE,
                                   g_param_spec_uint ("N",
                                                      NULL,
                                                      "sample size",
                                                      0, G_MAXUINT, 0,
                                                      G_PARAM_READABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_BLURB));

  g_object_class_install_property (object_class,
                                   PROP_OVER_SMOOTH,
                                   g_param_spec_double ("over-smooth",
                                                        NULL,
                                                        "Over-smooth distribution",
                                                        1.0e-5, G_MAXDOUBLE, 1.0,
                                                        G_PARAM_READWRITE | G_PARAM_CONSTRUCT | G_PARAM_STATIC_NAME | G_PARAM_STATIC_BLURB));

  g_object_class_install_property (object_class,
                                   PROP_CV_TYPE,
                                   g_param_spec_enum ("CV-type",
                                                      NULL,
                                                      "Cross-validation method",
                                                      NCM_TYPE_STATS_DIST_CV, NCM_STATS_DIST_CV_NONE,
                                                      G_PARAM_READWRITE | G_PARAM_CONSTRUCT | G_PARAM_STATIC_NAME | G_PARAM_STATIC_BLURB));

  g_object_class_install_property (object_class,
                                   PROP_USE_THREADS,
                                   g_param_spec_boolean ("use-threads",
                                                         NULL,
                                                         "Whether to use OpenMP threads during computation",
                                                         FALSE,
                                                         G_PARAM_READWRITE | G_PARAM_CONSTRUCT | G_PARAM_STATIC_NAME | G_PARAM_STATIC_BLURB));

  g_object_class_install_property (object_class,
                                   PROP_SPLIT_FRAC,
                                   g_param_spec_double ("split-frac",
                                                        NULL,
                                                        "Fraction to use in the split cross-validation",
                                                        0.10, 0.95, 0.5,
                                                        G_PARAM_READWRITE | G_PARAM_CONSTRUCT | G_PARAM_STATIC_NAME | G_PARAM_STATIC_BLURB));
  g_object_class_install_property (object_class,
                                   PROP_PRINT_FIT,
                                   g_param_spec_boolean ("print-fit",
                                                         NULL,
                                                         "Whether to print the fitting process",
                                                         FALSE,
                                                         G_PARAM_READWRITE | G_PARAM_CONSTRUCT | G_PARAM_STATIC_NAME | G_PARAM_STATIC_BLURB));


  klass->set_dim            = &_ncm_stats_dist_set_dim;
  klass->get_href           = &_ncm_stats_dist_get_href;
  klass->prepare_kernel     = &_ncm_stats_dist_prepare_kernel;
  klass->prepare            = &_ncm_stats_dist_prepare;
  klass->prepare_interp     = &_ncm_stats_dist_prepare_interp;
  klass->compute_IM         = &_ncm_stats_dist_compute_IM;
  klass->peek_cov_decomp    = &_ncm_stats_dist_peek_cov_decomp;
  klass->get_lnnorm         = &_ncm_stats_dist_get_lnnorm;
  klass->eval_weights       = &_ncm_stats_dist_eval_weights;
  klass->eval_weights_m2lnp = &_ncm_stats_dist_eval_weights_m2lnp;
  klass->reset              = &_ncm_stats_dist_reset;
}

static void
_ncm_stats_dist_set_dim (NcmStatsDist *sd, const guint dim)
{
  NcmStatsDistPrivate * const self = sd->priv;

  self->d = dim;
}

static gdouble
_ncm_stats_dist_get_href (NcmStatsDist *sd)
{
  NcmStatsDistPrivate * const self = sd->priv;

  return self->over_smooth * ncm_stats_dist_kernel_get_rot_bandwidth (self->kernel, self->n_kernels);
}

gdouble
_ncm_stats_dist_m2lnp (const gsl_vector *v, void *params)
{
  NcmStatsDist *sd                 = NCM_STATS_DIST (params);
  NcmStatsDistPrivate * const self = sd->priv;
  const double lnos                = gsl_vector_get (v, 0);
  gdouble m2lnp                    = 0.0;
  gint i;

  self->over_smooth = exp (lnos);
  self->href        = ncm_stats_dist_get_href (sd);

  #pragma omp parallel for if (self->use_threads)

  for (i = self->n_kernels; i < self->n_obs; i++)
  {
    NcmVector *x_i        = g_ptr_array_index (self->sample_array, i);
    const gdouble m2lnp_i = ncm_stats_dist_eval_m2lnp (sd, x_i);

    m2lnp += m2lnp_i;
  }

  if (self->print_fit)
    ncm_message ("# over-smooth: % 22.15g, m2lnp = % 22.15g\n",
                 self->over_smooth, m2lnp);

  return m2lnp;
}

static void
_ncm_stats_dist_prepare (NcmStatsDist *sd)
{
  NcmStatsDistClass *sd_class      = NCM_STATS_DIST_GET_CLASS (sd);
  NcmStatsDistPrivate * const self = sd->priv;

  switch (self->cv_type)
  {
    case NCM_STATS_DIST_CV_NONE:
      self->n_obs     = self->sample_array->len;
      self->n_kernels = self->sample_array->len;
      break;
    case NCM_STATS_DIST_CV_SPLIT:
    case NCM_STATS_DIST_CV_SPLIT_NOFIT:
      self->n_obs     = self->sample_array->len;
      self->n_kernels = ceil (self->sample_array->len * self->split_frac);
      break;
    default: /* LCOV_EXCL_BR_LINE */
      g_assert_not_reached ();
      break;
  }

  if (self->n_obs < self->d)
    g_error ("_ncm_stats_dist_prepare: the sample is too small.");

  sd_class->prepare_kernel (sd, self->sample_array);

  if ((self->weights == NULL) ||
      (self->n_kernels != ncm_vector_len (self->weights)))
  {
    ncm_vector_clear (&self->weights);
    ncm_vector_clear (&self->wcum);

    self->weights    = ncm_vector_new (self->n_kernels);
    self->wcum       = ncm_vector_new (self->n_kernels + 1);
    self->alloc_subs = FALSE;
  }

  self->href = ncm_stats_dist_get_href (sd);

  ncm_vector_set_all (self->weights, 1.0 / (1.0 * self->n_kernels));
  self->wcum_ready = FALSE;

  switch (self->cv_type)
  {
    case NCM_STATS_DIST_CV_NONE:
    case NCM_STATS_DIST_CV_SPLIT:
      break;
    case NCM_STATS_DIST_CV_SPLIT_NOFIT:
    {
      gdouble s     = 0.1;
      gdouble lnos  = log (self->over_smooth);
      NcmVector *x  = ncm_vector_new_data_static (&lnos, 1, 1);
      NcmVector *ss = ncm_vector_new_data_static (&s, 1, 1);
      gsl_multimin_function minex_func;

      minex_func.n      = 1;
      minex_func.f      = _ncm_stats_dist_m2lnp;
      minex_func.params = sd;

      gsl_multimin_fminimizer_set (
        self->fmin,
        &minex_func,
        ncm_vector_gsl (x),
        ncm_vector_gsl (ss));

      {
        gint iter = 0;
        gint status;

        do {
          iter++;
          status = gsl_multimin_fminimizer_iterate (self->fmin);

          if (status)
            break;

          status = gsl_multimin_test_size (self->fmin->size, 1.0e-4);
        } while (status == GSL_CONTINUE && iter < 1000);

        if (self->print_fit)
          printf ("# iter: %d, over-smooth: % 22.15g, m2lnp = % 22.15g, gsl status (%d)\n",
                  iter, self->over_smooth, self->fmin->fval, status);
      }

      break;
    }
    default: /* LCOV_EXCL_BR_LINE */
      g_assert_not_reached ();
      break;
  }
}

static void
_ncm_stats_dist_compute_IM_full (NcmStatsDist *sd)
{
  NcmStatsDistClass *sd_class      = NCM_STATS_DIST_GET_CLASS (sd);
  NcmStatsDistPrivate * const self = sd->priv;
  gint i;

  sd_class->compute_IM (sd, self->IM);

  #pragma omp parallel for if (self->use_threads)

  for (i = 0; i < self->n_obs; i++)
    ncm_matrix_mul_row (self->IM, i, 1.0 / ncm_vector_get (self->f, i));
}

typedef struct _NcmStatsDistEval
{
  NcmStatsDist *sd;
  NcmStatsDistPrivate * const self;
  NcmStatsDistClass *sd_class;
  NcmVector *residuals;
  NcmVector *m2lnp;
} NcmStatsDistEval;

static void
_ncm_stats_dist_prepare_interp_fit_nnls_f (gdouble *p, gdouble *hx, gint m, gint n, gpointer adata)
{
  NcmStatsDistEval *eval = adata;
  NcmVector *f           = ncm_vector_new_data_static (hx, n, 1);
  gdouble rnorm          = 0.0;
  gint i;

  g_assert (eval->self->n_obs == n);

  eval->self->over_smooth = exp (p[0]);
  eval->self->href        = ncm_stats_dist_get_href (eval->sd);

  _ncm_stats_dist_compute_IM_full (eval->sd);
  rnorm = NCM_NNLS_SOLVE (eval->self->nnls, eval->self->sub_IM, eval->self->sub_x, eval->self->f1);

  #pragma omp parallel for if (eval->self->use_threads)

  for (i = 0; i < eval->self->n_obs; i++)
  {
    NcmVector *x_i         = g_ptr_array_index (eval->self->sample_array, i);
    const gdouble m2lnpt_i = ncm_vector_get (eval->m2lnp, i) - eval->self->min_m2lnp;
    const gdouble m2lnpi_i = ncm_stats_dist_eval_m2lnp (eval->sd, x_i);

    /*printf ("%d %d % 22.15g % 22.15g\n", eval->self->n, i, m2lnpt_i, m2lnpi_i);*/
    /*ncm_vector_set (f, i, sqrt (fabs (m2lnpt_i - m2lnpi_i))); */
    ncm_vector_set (f, i, expm1 (-0.5 * (m2lnpi_i - m2lnpt_i)));
  }

  if (eval->self->print_fit)
    ncm_message ("# over-smooth: % 22.15g, rnorm = % 22.15g, fnorm = % 22.15g\n",
                 eval->self->over_smooth, rnorm, ncm_vector_dnrm2 (f));

  ncm_vector_free (f);
}

static void
_ncm_stats_dist_alloc_nnls (NcmStatsDist *sd, const gint nrows, const gint ncols)
{
  NcmStatsDistPrivate * const self = sd->priv;

  if ((self->nnls == NULL) ||
      ((ncm_nnls_get_nrows (self->nnls) != nrows) || (ncm_nnls_get_ncols (self->nnls) != ncols)))
  {
    ncm_nnls_clear (&self->nnls);
    self->nnls = ncm_nnls_new (nrows, ncols);
    ncm_nnls_set_umethod (self->nnls, NCM_NNLS_UMETHOD_NORMAL);

    self->alloc_subs = FALSE;
  }

  if (!self->alloc_subs)
  {
    ncm_matrix_clear (&self->sub_IM);
    ncm_vector_clear (&self->sub_x);

    self->sub_IM = ncm_matrix_get_submatrix (self->IM, 0, 0, nrows, ncols);
    self->sub_x  = ncm_vector_get_subvector (self->weights, 0, ncols);

    self->alloc_subs = TRUE;
  }
}

static void
_ncm_stats_dist_prepare_interp (NcmStatsDist *sd, NcmVector *m2lnp)
{
  NcmStatsDistPrivate * const self = sd->priv;

  _ncm_stats_dist_prepare (sd);

  g_assert_cmpuint (ncm_vector_len (m2lnp), ==, self->n_obs);
  {
    NcmStatsDistClass *sd_class = NCM_STATS_DIST_GET_CLASS (sd);
    NcmStatsDistEval eval       = {sd, self, sd_class, NULL, m2lnp};
    const gdouble dbl_limit     = 6.0;
    gint i;

    /*
     * Evaluating the right-hand-side
     */
    self->min_m2lnp = GSL_POSINF;
    self->max_m2lnp = GSL_NEGINF;

    for (i = 0; i < self->n_kernels; i++)
    {
      const gdouble m2lnp_i = ncm_vector_get (m2lnp, i);

      self->min_m2lnp = MIN (self->min_m2lnp, m2lnp_i);
      self->max_m2lnp = MAX (self->max_m2lnp, m2lnp_i);
    }

    if (self->max_m2lnp - self->min_m2lnp > -2.0 * dbl_limit * GSL_LOG_DBL_EPSILON)
    {
      gint n_cut = 0;

      g_array_set_size (self->m2lnp_sort, self->n_kernels);
      gsl_sort_index (&g_array_index (self->m2lnp_sort, size_t, 0),
                      ncm_vector_data (m2lnp),
                      ncm_vector_stride (m2lnp),
                      ncm_vector_len (m2lnp));

      for (i = 0; i < self->n_kernels; i++)
      {
        gint p                = g_array_index (self->m2lnp_sort, size_t, i);
        const gdouble m2lnp_p = ncm_vector_get (m2lnp, p);

        if (m2lnp_p - self->min_m2lnp > -2.0 * dbl_limit * GSL_LOG_DBL_EPSILON)
        {
          n_cut = i;
          break;
        }
      }

      /* printf ("n_cut %d n_kernels %d\n", n_cut, self->n_kernels); */

      /* Too many points falling outside using normal KDE using 10%
       * of the weight for the points falling outside and 90% for
       * the points falling inside.
       */
      if (n_cut < (gint) (0.5 * self->n_obs))
      {
        ncm_vector_set_all (self->weights, 0.1 / (self->n_kernels - n_cut));

        for (i = 0; i < n_cut; i++)
        {
          gint p = g_array_index (self->m2lnp_sort, size_t, i);

          ncm_vector_set (self->weights, p, 0.9 / n_cut);
        }

        return;
      }

      {
        NcmVector *m2lnp_cut        = ncm_vector_new (n_cut);
        GPtrArray *sample_array_cut = g_ptr_array_new ();
        gint j                      = 0;

        for (i = 0; i < self->n_obs; i++)
        {
          const gdouble m2lnp_i = ncm_vector_get (m2lnp, i);

          if (m2lnp_i - self->min_m2lnp <= -2.0 * dbl_limit * GSL_LOG_DBL_EPSILON)
          {
            ncm_vector_set (m2lnp_cut, j++, m2lnp_i);
            g_ptr_array_add (sample_array_cut,
                             ncm_vector_ref (g_ptr_array_index (self->sample_array, i)));
          }
        }

        g_assert (j == n_cut);

        g_ptr_array_set_size (self->sample_array, 0);

        for (i = 0; i < n_cut; i++)
        {
          g_ptr_array_add (self->sample_array, g_ptr_array_index (sample_array_cut, i));
        }

        g_ptr_array_unref (sample_array_cut);

        ncm_stats_dist_prepare_interp (sd, m2lnp_cut);

        ncm_vector_free (m2lnp_cut);
      }

      return;
    }

    /*
     * Preparing allocations
     */
    if ((self->n_obs != self->alloc_n_obs) || (self->n_kernels != self->alloc_n_kernels))
    {
      ncm_matrix_clear (&self->IM);
      ncm_vector_clear (&self->f);
      ncm_vector_clear (&self->f1);

      self->IM = ncm_matrix_new (self->n_obs, self->n_kernels);
      self->f  = ncm_vector_new (self->n_obs);
      self->f1 = ncm_vector_new (self->n_obs);

      ncm_vector_set_all (self->f1, 1.0);

      self->alloc_n_obs     = self->n_obs;
      self->alloc_n_kernels = self->n_kernels;
      self->alloc_subs      = FALSE;
    }

    ncm_vector_set_zero (self->weights);

    for (i = 0; i < self->n_obs; i++)
    {
      const gdouble m2lnp_i = ncm_vector_get (m2lnp, i);

      ncm_vector_set (self->f, i, exp (-0.5 * (m2lnp_i - self->min_m2lnp)));
    }

    if (self->n_kernels > 20000)
      g_warning ("_ncm_stats_dist_prepare_interp: very large system n = %u!", self->n_kernels);

    switch (self->cv_type)
    {
      case NCM_STATS_DIST_CV_SPLIT:
      {
        gdouble info[LM_INFO_SZ];
        gdouble opts[LM_OPTS_SZ];
        gdouble cov, ln_os, rnorm0;

        _ncm_stats_dist_alloc_nnls (sd, self->n_obs, self->n_kernels);

        if (self->levmar_n != self->n_obs)
        {
          g_clear_pointer (&self->levmar_workz, g_free);

          self->levmar_workz = g_new0 (gdouble, LM_DIF_WORKSZ (self->d, self->n_obs));
          self->levmar_n     = self->n_obs;
        }

        opts[0] = LM_INIT_MU;
        opts[1] = 1.0e-7;
        opts[2] = 1.0e-7;
        opts[3] = 1.0e-10;
        opts[4] = LM_DIFF_DELTA;

        ln_os = log (self->over_smooth);

        _ncm_stats_dist_compute_IM_full (sd);
        rnorm0 = NCM_NNLS_SOLVE (self->nnls, self->sub_IM, self->sub_x, self->f1);

        for (i = 0; i < 10; i++)
        {
          const gdouble ln_os_try = ncm_rng_gaussian_gen (self->rng, ln_os, 0.5);
          gdouble rnorm_try;

          self->over_smooth = exp (ln_os_try);
          self->href        = ncm_stats_dist_get_href (sd);

          _ncm_stats_dist_compute_IM_full (sd);
          rnorm_try = NCM_NNLS_SOLVE (self->nnls, self->sub_IM, self->sub_x, self->f1);

          if (rnorm_try < rnorm0)
          {
            ln_os  = ln_os_try;
            rnorm0 = rnorm_try;
          }
        }

        dlevmar_dif (&_ncm_stats_dist_prepare_interp_fit_nnls_f,
                     &ln_os, NULL, 1, self->n_obs,
                     10000, opts, info, self->levmar_workz, &cov, &eval);

        self->over_smooth = exp (ln_os);
        self->href        = ncm_stats_dist_get_href (sd);

        _ncm_stats_dist_compute_IM_full (sd);
        self->rnorm = NCM_NNLS_SOLVE (self->nnls, self->sub_IM, self->sub_x, self->f1);
      }
      break;
      case NCM_STATS_DIST_CV_SPLIT_NOFIT:
      case NCM_STATS_DIST_CV_NONE:
        _ncm_stats_dist_alloc_nnls (sd, self->n_obs, self->n_kernels);
        _ncm_stats_dist_compute_IM_full (sd);
        self->rnorm = NCM_NNLS_SOLVE (self->nnls, self->sub_IM, self->sub_x, self->f1);
        break;
      default: /* LCOV_EXCL_BR_LINE */
        g_assert_not_reached ();
        break;
    }
  }

  {
    const gdouble total_weight = ncm_vector_sum_cpts (self->weights);

    g_assert (total_weight > 0.0);
    ncm_vector_scale (self->weights, 1.0 / total_weight);
  }
}

static void
_ncm_stats_dist_reset (NcmStatsDist *sd)
{
  NcmStatsDistPrivate * const self = sd->priv;

  g_ptr_array_set_size (self->sample_array, 0);
}

/**
 * ncm_stats_dist_ref:
 * @sd: a #NcmStatsDist
 *
 * Increases the reference count of @sd.
 *
 * Returns: (transfer full): @sd.
 */
NcmStatsDist *
ncm_stats_dist_ref (NcmStatsDist *sd)
{
  return g_object_ref (sd);
}

/**
 * ncm_stats_dist_free:
 * @sd: a #NcmStatsDist
 *
 * Decreases the reference count of @sd.
 *
 */
void
ncm_stats_dist_free (NcmStatsDist *sd)
{
  g_object_unref (sd);
}

/**
 * ncm_stats_dist_clear:
 * @sd: a #NcmStatsDist
 *
 * Decreases the reference count of *@sd and sets the pointer *@sd to NULL.
 *
 */
void
ncm_stats_dist_clear (NcmStatsDist **sd)
{
  g_clear_object (sd);
}

/**
 * ncm_stats_dist_set_kernel:
 * @sd: a #NcmStatsDist
 * @sdk: a #NcmStatsDistKernel
 *
 * Sets the kernel to be used in the interpolation.
 * The different types of kernels are: the gaussian kernel and the studentt kernel,
 * which are under the file names ncm_stats_dist_kernel_gauss.c and ncm_stats_dist_kernel_st.c.
 */
void
ncm_stats_dist_set_kernel (NcmStatsDist *sd, NcmStatsDistKernel *sdk)
{
  NcmStatsDistPrivate * const self = sd->priv;

  ncm_stats_dist_kernel_clear (&self->kernel);
  self->kernel = ncm_stats_dist_kernel_ref (sdk);

  NCM_STATS_DIST_GET_CLASS (sd)->set_dim (sd, ncm_stats_dist_kernel_get_dim (sdk));
}

/**
 * ncm_stats_dist_peek_kernel:
 * @sd: a #NcmStatsDist
 *
 * Gets the kernel to be used in the interpolation.
 *
 * Returns: (transfer none): current #NcmStatsDistKernel used.
 */
NcmStatsDistKernel *
ncm_stats_dist_peek_kernel (NcmStatsDist *sd)
{
  NcmStatsDistPrivate * const self = sd->priv;

  return self->kernel;
}

/**
 * ncm_stats_dist_get_kernel:
 * @sd: a #NcmStatsDist
 *
 * Gets the kernel to be used in the interpolation.
 *
 * Returns: (transfer full): current #NcmStatsDistKernel used.
 */
NcmStatsDistKernel *
ncm_stats_dist_get_kernel (NcmStatsDist *sd)
{
  NcmStatsDistPrivate * const self = sd->priv;

  return ncm_stats_dist_kernel_ref (self->kernel);
}

/**
 * ncm_stats_dist_get_dim:
 * @sd: a #NcmStatsDist
 *
 * Returns: an int d, the dimension of the sample space, which is the same dimension of the used kernel.
 */
guint
ncm_stats_dist_get_dim (NcmStatsDist *sd)
{
  NcmStatsDistPrivate * const self = sd->priv;

  return self->d;
}

/**
 * ncm_stats_dist_get_sample_size:
 * @sd: a #NcmStatsDist
 *
 * After the prepare call, this function returns the size of the sample used in the
 * interpolation.
 *
 * Returns: the size of the sample used.
 */
guint
ncm_stats_dist_get_sample_size (NcmStatsDist *sd)
{
  NcmStatsDistPrivate * const self = sd->priv;

  return self->n_obs;
}

/**
 * ncm_stats_dist_get_n_kernels:
 * @sd: a #NcmStatsDist
 *
 * After the prepare call, this function returns the number of kernels used in the
 * interpolation.
 *
 * Returns: the number of kernels used.
 */
guint
ncm_stats_dist_get_n_kernels (NcmStatsDist *sd)
{
  NcmStatsDistPrivate * const self = sd->priv;

  return self->n_kernels;
}

/**
 * ncm_stats_dist_get_href:
 * @sd: a #NcmStatsDist
 *
 * Returns: a double h, the currently used @href.
 * If the object was prepared with the VKDE class, the VKDE method is called.
 * The @href value is computed by the kernel object that was called
 * in the set kernel function.
 */
gdouble
ncm_stats_dist_get_href (NcmStatsDist *sd)
{
  NcmStatsDistClass *sd_class = NCM_STATS_DIST_GET_CLASS (sd);

  return sd_class->get_href (sd);
}

/**
 * ncm_stats_dist_set_over_smooth:
 * @sd: a #NcmStatsDist
 * @over_smooth: the over-smooth factor
 *
 * Sets the over-smooth factor to @over_smooth.
 *
 */
void
ncm_stats_dist_set_over_smooth (NcmStatsDist *sd, const gdouble over_smooth)
{
  NcmStatsDistPrivate * const self = sd->priv;

  self->over_smooth = over_smooth;
}

/**
 * ncm_stats_dist_get_over_smooth:
 * @sd: a #NcmStatsDist
 *
 * Returns: a double os, the over-smooth factor.
 */
gdouble
ncm_stats_dist_get_over_smooth (NcmStatsDist *sd)
{
  NcmStatsDistPrivate * const self = sd->priv;

  return self->over_smooth;
}

/**
 * ncm_stats_dist_set_split_frac:
 * @sd: a #NcmStatsDist
 * @split_frac: the over-smooth factor
 *
 * Sets cross-correlation split fraction to @split_frac.
 * This method shall be used when the cv_type is the cv_split.
 * The split fraction determines the fraction of sample points
 * that will be left out to use the cross validation method.
 *
 */
void
ncm_stats_dist_set_split_frac (NcmStatsDist *sd, const gdouble split_frac)
{
  NcmStatsDistPrivate * const self = sd->priv;

  g_assert_cmpfloat (split_frac, >=, 0.01);
  g_assert_cmpfloat (split_frac, <=, 1.0);

  self->split_frac = split_frac;
}

/**
 * ncm_stats_dist_get_split_frac:
 * @sd: a #NcmStatsDist
 *
 * Returns: a double @split_frac, the cross-correlation split fraction.
 */
gdouble
ncm_stats_dist_get_split_frac (NcmStatsDist *sd)
{
  NcmStatsDistPrivate * const self = sd->priv;

  return self->split_frac;
}

/**
 * ncm_stats_dist_set_print_fit:
 * @sd: a #NcmStatsDist
 * @print_fit: a boolean
 *
 * Whether to print steps during the fitting process.
 *
 */
void
ncm_stats_dist_set_print_fit (NcmStatsDist *sd, const gboolean print_fit)
{
  NcmStatsDistPrivate * const self = sd->priv;

  self->print_fit = print_fit;
}

/**
 * ncm_stats_dist_get_print_fit:
 * @sd: a #NcmStatsDist
 *
 * Returns: Whether it is going to print steps during the fitting process.
 */
gboolean
ncm_stats_dist_get_print_fit (NcmStatsDist *sd)
{
  NcmStatsDistPrivate * const self = sd->priv;

  return self->print_fit;
}

/**
 * ncm_stats_dist_set_cv_type:
 * @sd: a #NcmStatsDist
 * @cv_type: a #NcmStatsDistCV
 *
 * Sets the cross-validation method to @cv_type.
 * If the selected method is none, all the sample points
 * will be used to compute the interpolation. If the cv_type is the cv_split,
 * a split fraction of the points are randomly excluded and the interpolation
 * is computed to a best fit of the remaining sample points,
 * which leads to a more point independent interpolation.
 *
 */
void
ncm_stats_dist_set_cv_type (NcmStatsDist *sd, const NcmStatsDistCV cv_type)
{
  NcmStatsDistPrivate * const self = sd->priv;

  self->cv_type = cv_type;
}

/**
 * ncm_stats_dist_get_cv_type:
 * @sd: a #NcmStatsDist
 *
 * Returns: a string @cv_type, current cross-validation method used.
 */
NcmStatsDistCV
ncm_stats_dist_get_cv_type (NcmStatsDist *sd)
{
  NcmStatsDistPrivate * const self = sd->priv;

  return self->cv_type;
}

/**
 * ncm_stats_dist_set_use_threads:
 * @sd: a #NcmStatsDist
 * @use_threads: whether to use threads
 *
 * Sets whether to use OpenMP threads during the computation.
 *
 */
void
ncm_stats_dist_set_use_threads (NcmStatsDist *sd, const gboolean use_threads)
{
  NcmStatsDistPrivate * const self = sd->priv;

  self->use_threads = use_threads;
}

/**
 * ncm_stats_dist_get_use_threads:
 * @sd: a #NcmStatsDist
 *
 * Returns: whether to use OpenMP threads during the computation.
 */
gboolean
ncm_stats_dist_get_use_threads (NcmStatsDist *sd)
{
  NcmStatsDistPrivate * const self = sd->priv;

  return self->use_threads;
}

/**
 * ncm_stats_dist_prepare_kernel: (virtual prepare_kernel)
 * @sd: a #NcmStatsDist
 * @sample_array: (element-type NcmVector): an array of #NcmVector
 *
 * Prepares the object for computations of the individuals kernels
 * and is usually part of ncm_stats_dist_prepare() and is should not
 * be called directly.
 *
 * This virtual method does not have a default implementation and
 * must be defined by the descendants.
 *
 */
void
ncm_stats_dist_prepare_kernel (NcmStatsDist *sd, GPtrArray *sample_array)
{
  NcmStatsDistClass *sd_class = NCM_STATS_DIST_GET_CLASS (sd);

  sd_class->prepare_kernel (sd, sample_array);
}

/**
 * ncm_stats_dist_prepare: (virtual prepare)
 * @sd: a #NcmStatsDist
 *
 * Prepares the object for calculations. This function prepares
 * the weight matrix and sets all the weights to 1.0/sample size.
 * It also calls the kernel_prepare function, implemented by a child,
 * and calls the get_href function.
 */
void
ncm_stats_dist_prepare (NcmStatsDist *sd)
{
  NcmStatsDistClass *sd_class = NCM_STATS_DIST_GET_CLASS (sd);

  sd_class->prepare (sd);
}

/**
 * ncm_stats_dist_prepare_interp: (virtual prepare_interp)
 * @sd: a #NcmStatsDist
 * @m2lnp: a #NcmVector containing the distribution values that will be used to compute the interpolation function.
 *
 * Prepares the object for calculations. Using the distribution values
 * at the sample points. This function calls the prepare function and
 * prepares the needed objects to compute the least squares problem.
 * The interpolation matrix IM is prepered by a child object and called in this function.
 * Then, depending on the cross validation method, the function solves the least squares problem using the ncm_nnls object.
 */
void
ncm_stats_dist_prepare_interp (NcmStatsDist *sd, NcmVector *m2lnp)
{
  NcmStatsDistClass *sd_class = NCM_STATS_DIST_GET_CLASS (sd);

  sd_class->prepare_interp (sd, m2lnp);
}

/**
 * ncm_stats_dist_eval:
 * @sd: a #NcmStatsDist
 * @x: a #NcmVector
 *
 * Evaluate the distribution at $\vec{x}=$@x. The method ncm_stats_dist_eval_m2lnp()
 * can be used to avoid underflow.
 *
 * Returns: $P(\vec{x})$.
 */
gdouble
ncm_stats_dist_eval (NcmStatsDist *sd, NcmVector *x)
{
  NcmStatsDistClass *sd_class      = NCM_STATS_DIST_GET_CLASS (sd);
  NcmStatsDistPrivate * const self = sd->priv;

  return sd_class->eval_weights (sd, self->weights, x);
}

/**
 * ncm_stats_dist_eval_m2lnp:
 * @sd: a #NcmStatsDist
 * @x: a #NcmVector
 *
 * Evaluate the distribution at $\vec{x}=$@x. This method is more
 * stable than ncm_stats_dist_eval() since it avoids underflows
 * and overflows.
 *
 * Returns: $P(\vec{x})$.
 */
gdouble
ncm_stats_dist_eval_m2lnp (NcmStatsDist *sd, NcmVector *x)
{
  NcmStatsDistClass *sd_class      = NCM_STATS_DIST_GET_CLASS (sd);
  NcmStatsDistPrivate * const self = sd->priv;

  return sd_class->eval_weights_m2lnp (sd, self->weights, x);
}

/**
 * ncm_stats_dist_kernel_choose:
 * @sd: a #NcmStatsDist
 * @rng: a #NcmRNG
 *
 * Using the pseudo-random number generator @rng chooses
 * a random kernel based on the computed weights.
 *
 */
gint
ncm_stats_dist_kernel_choose (NcmStatsDist *sd, NcmRNG *rng)
{
  NcmStatsDistPrivate * const self = sd->priv;
  gint i;

  if (!self->wcum_ready)
  {
    gdouble cum = 0.0;

    ncm_vector_set (self->wcum, 0, cum);

    for (i = 0; i < self->n_kernels; i++)
    {
      cum += ncm_vector_get (self->weights, i);
      ncm_vector_set (self->wcum, i + 1, cum);
    }

    ncm_vector_scale (self->wcum, 1.0 / cum);
    self->wcum_ready = TRUE;
  }

  {
    const gdouble p = ncm_rng_uniform_gen (rng, 0.0, 1.0);
    gint ilo        = 0;
    gint ihi        = self->n_kernels;

    while (ihi > ilo + 1)
    {
      gint mi = (ihi + ilo) / 2;

      if (ncm_vector_fast_get (self->wcum, mi) > p)
        ihi = mi;
      else
        ilo = mi;
    }

    i = ilo;
  }

  return i;
}

/**
 * ncm_stats_dist_sample:
 * @sd: a #NcmStatsDist
 * @x: a #NcmVector
 * @rng: a #NcmRNG
 *
 * Using the pseudo-random number generator @rng generates a
 * point from the distribution and copy it to @x.
 *
 */
void
ncm_stats_dist_sample (NcmStatsDist *sd, NcmVector *x, NcmRNG *rng)
{
  NcmStatsDistPrivate * const self = sd->priv;
  const gint i                     = ncm_stats_dist_kernel_choose (sd, rng);
  NcmVector *x_i                   = g_ptr_array_index (self->sample_array, i);
  NcmMatrix *cov_U                 = ncm_stats_dist_peek_cov_decomp (sd, i);

  ncm_stats_dist_kernel_sample (self->kernel, cov_U, self->href, x_i, x, rng);
}

/**
 * ncm_stats_dist_get_rnorm:
 * @sd: a #NcmStatsDist
 *
 * Gets the value of the last $\chi^2$ fit obtained
 * when computing the interpolation through
 * ncm_stats_dist_prepare_interp().
 *
 * Returns: a double, the value of the $\chi^2$.
 */
gdouble
ncm_stats_dist_get_rnorm (NcmStatsDist *sd)
{
  NcmStatsDistPrivate * const self = sd->priv;

  return self->rnorm * self->rnorm;
}

/**
 * ncm_stats_dist_add_obs:
 * @sd: a #NcmStatsDist
 * @y: a #NcmVector
 *
 * Adds a new point @y to the sample with weight 1.0.
 * This function must be called to insert an initial sample into the object, so the interpolation can be computed.
 *
 */
void
ncm_stats_dist_add_obs (NcmStatsDist *sd, NcmVector *x)
{
  NcmStatsDistPrivate * const self = sd->priv;

  g_ptr_array_add (self->sample_array, ncm_vector_dup (x));
}

/**
 * ncm_stats_dist_peek_sample_array:
 * @sd: a #NcmStatsDist
 *
 * Returns: (transfer none) (element-type NcmVector): current sample array.
 */
GPtrArray *
ncm_stats_dist_peek_sample_array (NcmStatsDist *sd)
{
  NcmStatsDistPrivate * const self = sd->priv;

  return self->sample_array;
}

/**
 * ncm_stats_dist_peek_cov_decomp: (virtual peek_cov_decomp)
 * @sd: a #NcmStatsDist
 * @i: kernel index
 *
 * Gets the covariance matrix associated with the @i-th
 * kernel.
 *
 * Returns: (transfer none): Cholesky decomposition of the @i-th covariance matrix.
 */
NcmMatrix *
ncm_stats_dist_peek_cov_decomp (NcmStatsDist *sd, guint i)
{
  NcmStatsDistClass *sd_class = NCM_STATS_DIST_GET_CLASS (sd);

  return sd_class->peek_cov_decomp (sd, i);
}

/**
 * ncm_stats_dist_get_lnnorm: (virtual get_lnnorm)
 * @sd: a #NcmStatsDist
 * @i: kernel index
 *
 * Gets the logarithm of the @i-th kernel normalization.
 *
 * Returns: $\ln (N_i)$.
 */
gdouble
ncm_stats_dist_get_lnnorm (NcmStatsDist *sd, guint i)
{
  NcmStatsDistClass *sd_class = NCM_STATS_DIST_GET_CLASS (sd);

  return sd_class->get_lnnorm (sd, i);
}

/**
 * ncm_stats_dist_peek_weights:
 * @sd: a #NcmStatsDist
 *
 * Returns: (transfer none): current kernel weights vector.
 */
NcmVector *
ncm_stats_dist_peek_weights (NcmStatsDist *sd)
{
  NcmStatsDistPrivate * const self = sd->priv;

  return self->weights;
}

/**
 * ncm_stats_dist_reset: (virtual reset)
 * @sd: a #NcmStatsDist
 *
 * Reset the object discarding all added points.
 *
 */
void
ncm_stats_dist_reset (NcmStatsDist *sd)
{
  NcmStatsDistClass *sd_class = NCM_STATS_DIST_GET_CLASS (sd);

  sd_class->reset (sd);
}

/**
 * ncm_stats_dist_get_Ki:
 * @sd: a #NcmStatsDist
 * @i: kernel index
 * @y_i: (out callee-allocates) (transfer full): kernel location
 * @cov_i: (out callee-allocates) (transfer full): kernel covariance U
 * @n_i: (out): kernel normalization
 * @w_i: (out): kernel weight
 *
 * Return all information about the @i-th kernel.
 *
 */
void
ncm_stats_dist_get_Ki (NcmStatsDist *sd, const guint i, NcmVector **y_i, NcmMatrix **cov_i, gdouble *n_i, gdouble *w_i)
{
  NcmStatsDistPrivate * const self = sd->priv;
  NcmMatrix *cov_decomp            = ncm_stats_dist_peek_cov_decomp (sd, i);
  const gdouble lnnorm             = ncm_stats_dist_get_lnnorm (sd, i);
  const gdouble href               = ncm_stats_dist_get_href (sd);

  g_assert (i < ncm_stats_dist_get_sample_size (sd));

  y_i[0]   = ncm_vector_dup (g_ptr_array_index (self->sample_array, i));
  cov_i[0] = ncm_matrix_dup (cov_decomp);
  n_i[0]   = exp (lnnorm);
  w_i[0]   = ncm_vector_get (self->weights, i);

  ncm_matrix_triang_to_sym (cov_decomp, 'U', TRUE, cov_i[0]);

  ncm_matrix_scale (cov_i[0], href * href);
}

