/*  This file is part of NI.

    NI is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    NI is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with NI; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    Copyright (C) 2006 Michael Jumper
 */

#include <fstream>
#include <iostream>

#include "config.h"

#ifdef HAVE_LIBREADLINE
#include <stdio.h>

// readline.h
#ifdef HAVE_READLINE_READLINE_H
#include <readline/readline.h>
#else
#ifdef HAVE_READLINE_H
#include <readline.h>
#endif
#endif

// history.h
#ifdef HAVE_READLINE_HISTORY_H
#include <readline/history.h>
#else
#ifdef HAVE_HISTORY_H
#include <history.h>
#endif
#endif

#endif

#include "altt.h"
#include "parser.h"
#include "flags.h"

#define BUFF_SIZE 20480

using namespace std;


void usage() {
	cerr <<
"\
Usage: ni [OPTIONS] [SCRIPT FILE [PARAMETERS FOR SCRIPT...]]\n\
(Interactive mode is entered when no file is specified.)\n\
\n\
  -v            Increase verbosity.\n\
  -sN           Use simplification level N: 2 is highest, 0 is minimal.\n\
  -h, --help    Display this message\n\
\n\
      --print-large    Allow truth tables, etc. involving more than 26\n\
                       variables to be printed\n\
\n\
";
}


int main(int argc, char** argv) {

    #ifdef HAVE_LIBREADLINE
    rl_bind_key ('\t', rl_insert);
    #endif

    // Parse command-line options.
    while (argc > 1) {

	    if (argv[1][0] != '-')
		    break;
	    
	    if (strcmp(argv[1], "-h") == 0 || strcmp(argv[1], "--help") == 0) {
		    usage();
		    return 0;
	    }
	    else if (strcmp(argv[1], "--print-large") == 0) {
		    size_override = 1;
	    }
	    else if (strcmp(argv[1], "-v") == 0) {
		    verbosity++;
	    }
	    else if (strncmp(argv[1], "-s", 2) == 0) {
		    simplify_level = atoi(argv[1] + 2);
	    }
	    else {
		    usage();
		    return 1;
	    }
	    
	    argc--;
	    argv = &(argv[1]);	// Shift arguments left.
    }

    // Create global constants (0, 1, ?, and the result which starts as 0)
    one  = new_altt(ST_BIT1, NULL, 0, ST_NULL, NULL, 0);
    zero = new_altt(ST_BIT0, NULL, 0, ST_NULL, NULL, 0);
    unknown = new_altt(ST_BITU, NULL, 0, ST_NULL, NULL, 0);

    general_ptr result = {ID_ALTT, zero};
    storage["ANS"] = result;
   
    // Read from file
    if (argc >= 2) {

        int parm = 1;
        for (int i=2; i<argc; i++) {
            char buff[1024];
            sprintf(buff, "%i", parm);
            parseline(storage[buff], argv[i]);
            parm++;
        }
        
        ifstream inf(argv[1]);
	if (inf.fail()) {
		cerr << "ERROR: Could not open input file: " << argv[1] << '\n';
		return 1;
	}

	inf.seekg(0, ios::beg);
	int start = inf.tellg();
	inf.seekg(0, ios::end);
	int end = inf.tellg();
	int fs = end-start;

	inf.seekg(0, ios::beg);
	
        char buffer[BUFF_SIZE];
        int l=1;
        for (;;) {
            inf.getline(buffer, BUFF_SIZE);
            if (inf.eof())
                break;

            if (verbosity >= 2) {
		    cerr << "Parsing line " << l << " (~" << inf.tellg() * 100 / fs << "% done): " << '\n';
		    cerr << buffer << '\n';
		    cerr << '\n';
	    }
            
            parseline(result, buffer);
            l++;
            
        }
        return 0;
    }
    
    // Print copyright notice (interactive mode)
    cout <<
"\n\
\tNI (Not Intractable) version 0.2.3 (almost-stable)\n\
\thttp://www.nongnu.org/not-intractable\n\
\tDistributed under the GNU General Public License.\n\
\tSee the file COPYING for more information.\n\
\tCopyright (C) 2006 Michael Jumper\n\
\n\
\tType quit to exit.\n\
\n\
";

    #ifndef HAVE_LIBREADLINE
    char rl[BUFF_SIZE];	// Buffer (for when readline support is compiled out)
    #endif
    for (;;) {

        // Get a line
	#ifdef HAVE_LIBREADLINE
        char* rl = readline("NI> ");
        if (rl == NULL)
            return 0;

	#ifdef HAVE_READLINE_HISTORY
        if (*rl != 0)
            add_history(rl);
	#endif
	
	#else
	cout << "NI> ";
	cin.getline(rl, BUFF_SIZE);
	#endif
	
	if (strncmp(skipws(rl), "quit", 4) == 0)
		return 0;
	
        parseline(result, rl);
        
	#ifdef HAVE_LIBREADLINE
        free(rl);
	#endif
        
    }

    return 0;
    
}
