/* $Id: servercb.c,v 1.27 2004/12/22 23:15:04 ali Exp $
 * Copyright (C) 2002, 2003, 2004  Slash'EM Development Team
 * Copyright (C) 2004  J. Ali Harlow
 *
 * This file is part of NetHack Proxy.
 *
 * NetHack Proxy is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of the
 * License, or (at your option) any later version.
 *
 * NetHack Proxy is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with NetHack Proxy; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307   
 * USA
 *
 * Alternatively (at your option) you may instead choose to redistribute
 * and/or modify NetHack Proxy under the terms of the NetHack General
 * Public License.
 *
 * You should have receieved a copy of the NetHack General Public License
 * along with NetHack Proxy; if not, download a copy from
 * http://www.nethack.org/common/license.html
 */

#include "config.h"
#include "compat.h"
#include <nhproxy/system.h>
#include <nhproxy/xdr.h>
#include <nhproxy/md5.h>
#include <nhproxy/common.h>
#include <nhproxy/server.h>

struct nhproxy_serv_callbacks *nhproxy__serv;

extern int nhproxy__authorized;

static void NHPROXY_FDECL(callback_display_inventory, \
			(unsigned short, NhProxyXdr *, NhProxyXdr *));
static void NHPROXY_FDECL(callback_dlbh_fopen, \
			(unsigned short, NhProxyXdr *, NhProxyXdr *));
static void NHPROXY_FDECL(callback_dlbh_fgets, \
			(unsigned short, NhProxyXdr *, NhProxyXdr *));
static void NHPROXY_FDECL(callback_dlbh_fread, \
			(unsigned short, NhProxyXdr *, NhProxyXdr *));
static void NHPROXY_FDECL(callback_dlbh_fwrite, \
			(unsigned short, NhProxyXdr *, NhProxyXdr *));
static void NHPROXY_FDECL(callback_dlbh_fclose, \
			(unsigned short, NhProxyXdr *, NhProxyXdr *));
static void NHPROXY_FDECL(callback_dlbh_fmd5sum, \
			(unsigned short, NhProxyXdr *, NhProxyXdr *));
static void NHPROXY_FDECL(callback_flush_screen, \
			(unsigned short, NhProxyXdr *, NhProxyXdr *));
static void NHPROXY_FDECL(callback_doredraw, \
			(unsigned short, NhProxyXdr *, NhProxyXdr *));
static void NHPROXY_FDECL(callback_interface_mode, \
			(unsigned short, NhProxyXdr *, NhProxyXdr *));
static void NHPROXY_FDECL(callback_parse_options, \
			(unsigned short, NhProxyXdr *, NhProxyXdr *));
static void NHPROXY_FDECL(callback_get_option, \
			(unsigned short, NhProxyXdr *, NhProxyXdr *));
static void NHPROXY_FDECL(callback_get_player_choices, \
			(unsigned short, NhProxyXdr *, NhProxyXdr *));
static void NHPROXY_FDECL(callback_get_valid_selections, \
			(unsigned short, NhProxyXdr *, NhProxyXdr *));
static void NHPROXY_FDECL(callback_quit_game, \
			(unsigned short, NhProxyXdr *, NhProxyXdr *));
static void NHPROXY_FDECL(callback_display_score, \
			(unsigned short, NhProxyXdr *, NhProxyXdr *));
static void NHPROXY_FDECL(callback_doset, \
			(unsigned short, NhProxyXdr *, NhProxyXdr *));
static void NHPROXY_FDECL(callback_get_extended_commands, \
			(unsigned short, NhProxyXdr *, NhProxyXdr *));
static void NHPROXY_FDECL(callback_map_menu_cmd, \
			(unsigned short, NhProxyXdr *, NhProxyXdr *));
static void NHPROXY_FDECL(callback_get_standard_winid, \
			(unsigned short, NhProxyXdr *, NhProxyXdr *));
static void NHPROXY_FDECL(callback_get_tilesets, \
			(unsigned short, NhProxyXdr *, NhProxyXdr *));
static void NHPROXY_FDECL(callback_get_glyph_mapping, \
			(unsigned short, NhProxyXdr *, NhProxyXdr *));
static void NHPROXY_FDECL(callback_get_extensions, \
			(unsigned short, NhProxyXdr *, NhProxyXdr *));
static void NHPROXY_FDECL(callback_set_option_mod_status, \
			(unsigned short, NhProxyXdr *, NhProxyXdr *));

static void
callback_display_inventory(id, request, reply)
unsigned short id;
NhProxyXdr *request, *reply;
{
    /*
     * Ignore recursive calls. They can cause games to
     * produce illegal output and have no utility.
     */
    static int busy = 0;
    if (!busy && nhproxy__authorized) {
	busy++;
	nhproxy__serv->display_inventory();
	if (!nhproxy_rpc_async_mode())
	    nhproxy_rpc_params(reply, 0);
	busy--;
    }
}

static void
callback_dlbh_fopen(id, request, reply)
unsigned short id;
NhProxyXdr *request, *reply;
{
    char *name, *mode;
    int retval;
    nhproxy_rpc_params(request, 2, NHPROXY_STRING_PTR(name),
      NHPROXY_STRING_PTR(mode));
    retval = nhproxy__authorized ? nhproxy__serv->dlbh_fopen(name, mode) : -1;
    nhproxy_rpc_params(reply, 1, NHPROXY_INT(retval));
    free(name);
    free(mode);
}

static void
callback_dlbh_fgets(id, request, reply)
unsigned short id;
NhProxyXdr *request, *reply;
{
    int fh, len;
    char *retval;
    char *buf;
    extern char *dlbh_fgets();
    nhproxy_rpc_params(request, 2, NHPROXY_INT_PTR(len), NHPROXY_INT_PTR(fh));
    if (len > 120*1024)
	len = 120*1024;		/* Sub-protocol 2 limits at 128Kb - 12 bytes */
    buf = (char *)malloc(len);
    if (!buf && len > 512) {
	len = 512;
	buf = (char *)malloc(len);
    }
    retval = buf && nhproxy__authorized ?
      nhproxy__serv->dlbh_fgets(buf, len, fh) : (char *)0;
    nhproxy_rpc_params(reply, 1, NHPROXY_STRING(retval ? retval : ""));
    free(buf);
}

static void
callback_dlbh_fread(id, request, reply)
unsigned short id;
NhProxyXdr *request, *reply;
{
    int fh, len, nb;
    char *buf;
    nhproxy_rpc_params(request, 2, NHPROXY_INT_PTR(len), NHPROXY_INT_PTR(fh));
    if (len > 120*1024)
	len = 120*1024;		/* Sub-protocol 2 limits at 128Kb - 12 bytes */
    buf = (char *)malloc(len);
    if (!buf && len > 512) {
	len = 512;
	buf = (char *)malloc(len);
    }
    nb = buf && nhproxy__authorized ?
      nhproxy__serv->dlbh_fread(buf, 1, len, fh) : -1;
    nhproxy_rpc_params(reply, 2,
      NHPROXY_INT(nb < 0), NHPROXY_BYTES(buf, nb >= 0 ? nb : 0));
    free(buf);
}

extern FILE *nhproxy__config_fp;

static void
callback_dlbh_fwrite(id, request, reply)
unsigned short id;
NhProxyXdr *request, *reply;
{
    int fh, nb, retval;
    char *buf = (char *)0;
    nhproxy_rpc_params(request, 2, NHPROXY_INT_PTR(fh),
      NHPROXY_BYTES_PTR(buf, nb));
    if (!fh && nhproxy__config_fp && nhproxy__authorized)
	retval = fwrite(buf, nb, 1, nhproxy__config_fp) != 1;
    else
	retval = -1;
    free(buf);
    nhproxy_rpc_params(reply, 1, NHPROXY_INT(retval));
}

static void
callback_dlbh_fclose(id, request, reply)
unsigned short id;
NhProxyXdr *request, *reply;
{
    int fh, retval;
    nhproxy_rpc_params(request, 1, NHPROXY_INT_PTR(fh));
    retval = nhproxy__authorized ? nhproxy__serv->dlbh_fclose(fh) : -1;
    nhproxy_rpc_params(reply, 1, NHPROXY_INT(retval));
}

static void
callback_dlbh_fmd5sum(id, request, reply)
unsigned short id;
NhProxyXdr *request, *reply;
{
    int retval = 0;
    int fh, nb, i;
    char *name;
    char *buf;
    nhproxy_md5_state_t md5;
    nhproxy_md5_byte_t digest[16];
    char md5sum[33];
    nhproxy_rpc_params(request, 1, NHPROXY_STRING_PTR(name));
    buf = (char *)malloc(8128);
    if (!buf || !nhproxy__authorized) {
	fh = -1;
	retval = -1;
    } else {
	fh = nhproxy__serv->dlbh_fopen(name, "rb");
	if (fh < 0)
	    retval = -1;
    }
    free(name);
    if (!retval) {
	nhproxy_md5_init(&md5);
	do {
	    nb = nhproxy__serv->dlbh_fread(buf, 1, 8128, fh);
	    if (nb > 0)
		nhproxy_md5_append(&md5, buf, nb);
	} while (nb > 0);
	if (nb < 0)
	    retval = -1;
	nhproxy_md5_finish(&md5, digest);
    }
    if (fh >= 0)
	(void)nhproxy__serv->dlbh_fclose(fh);
    free(buf);
    if (retval)
	md5sum[0]='\0';
    else {
	for(i = 0; i < 16; i++)
	    sprintf(md5sum + 2 * i, "%02x", digest[i]);
	md5sum[2 * i] = '\0';
    }
    nhproxy_rpc_params(reply, 2, NHPROXY_INT(retval), NHPROXY_STRING(md5sum));
}

static void
callback_flush_screen(id, request, reply)
unsigned short id;
NhProxyXdr *request, *reply;
{
    if (nhproxy__authorized)
	nhproxy__serv->flush_screen();
    nhproxy_rpc_params(reply, 0);
}

static void
callback_doredraw(id, request, reply)
unsigned short id;
NhProxyXdr *request, *reply;
{
    if (nhproxy__authorized)
	nhproxy__serv->doredraw();
    if (!nhproxy_rpc_async_mode())
	nhproxy_rpc_params(reply, 0);
}

static void
callback_interface_mode(id, request, reply)
unsigned short id;
NhProxyXdr *request, *reply;
{
    long mode;
    nhproxy_rpc_params(request, 1, NHPROXY_LONG_PTR(mode));
    if (nhproxy__authorized)
	nhproxy__serv->interface_mode(mode);
    if (!nhproxy_rpc_async_mode())
	nhproxy_rpc_params(reply, 0);
}

static void
callback_parse_options(id, request, reply)
unsigned short id;
NhProxyXdr *request, *reply;
{
    char *opts;
    nhproxy_rpc_params(request, 1, NHPROXY_STRING_PTR(opts));
    if (nhproxy__authorized)
	nhproxy__serv->parse_options(opts);
    free(opts);
    nhproxy_rpc_params(reply, 1, NHPROXY_INT(0));
}

static void
callback_get_option(id, request, reply)
unsigned short id;
NhProxyXdr *request, *reply;
{
    char *opt, *value;
    nhproxy_rpc_params(request, 1, NHPROXY_STRING_PTR(opt));
    value = nhproxy__authorized ? nhproxy__serv->get_option(opt) : "";
    free(opt);
    nhproxy_rpc_params(reply, 1, NHPROXY_STRING(value));
}

static void
callback_get_player_choices(id, request, reply)
unsigned short id;
NhProxyXdr *request, *reply;
{
    struct nhproxy_cb_get_player_choices_res *choices;
    choices = nhproxy__serv->get_player_choices();
    nhproxy_rpc_params(reply, 1,
      NHPROXY_XDRF(nhproxy_cb_xdr_get_player_choices_res, choices));
    nhproxy__serv->free_player_choices(choices);
}

static void
callback_get_valid_selections(id, request, reply)
unsigned short id;
NhProxyXdr *request, *reply;
{
    struct nhproxy_cb_get_valid_selections_res *vs;
    vs = nhproxy__serv->get_valid_selections();
    nhproxy_rpc_params(reply, 1,
      NHPROXY_XDRF(nhproxy_cb_xdr_get_valid_selections_res, vs));
    nhproxy__serv->free_valid_selections(vs);
}

static void
callback_quit_game(id, request, reply)
unsigned short id;
NhProxyXdr *request, *reply;
{
    nhproxy__serv->quit_game();
    /* Not reached */
    nhproxy_rpc_params(reply, 0);
}

static void
callback_display_score(id, request, reply)
unsigned short id;
NhProxyXdr *request, *reply;
{
    if (nhproxy__authorized)
	nhproxy__serv->display_score();
    nhproxy_rpc_params(reply, 0);
}

static void
callback_doset(id, request, reply)
unsigned short id;
NhProxyXdr *request, *reply;
{
    if (nhproxy__authorized)
	nhproxy__serv->doset();
    nhproxy_rpc_params(reply, 0);
}

static void
callback_get_extended_commands(id, request, reply)
unsigned short id;
NhProxyXdr *request, *reply;
{
    struct nhproxy_cb_get_extended_commands_res *list;
    list = nhproxy__serv->get_extended_commands();
    nhproxy_rpc_params(reply, 1,
      NHPROXY_XDRF(nhproxy_cb_xdr_get_extended_commands_res, list));
    nhproxy__serv->free_extended_commands(list);
}

static void
callback_map_menu_cmd(id, request, reply)
unsigned short id;
NhProxyXdr *request, *reply;
{
    int ch, retval;
    nhproxy_rpc_params(request, 1, NHPROXY_INT_PTR(ch));
    retval = nhproxy__authorized ? nhproxy__serv->map_menu_cmd(ch) : ch;
    nhproxy_rpc_params(reply, 1, NHPROXY_INT(retval));
}

static void
callback_get_standard_winid(id, request, reply)
unsigned short id;
NhProxyXdr *request, *reply;
{
    char *window;
    int retval;
    nhproxy_rpc_params(request, 1, NHPROXY_STRING_PTR(window));
    retval = nhproxy__authorized ?
      nhproxy__serv->get_standard_winid(window) : -1;
    nhproxy_rpc_params(reply, 1, NHPROXY_INT(retval));
}

static void
callback_get_tilesets(id, request, reply)
unsigned short id;
NhProxyXdr *request, *reply;
{
    struct nhproxy_cb_get_tilesets_res *list;
    list = nhproxy__serv->get_tilesets();
    nhproxy_rpc_params(reply, 1, NHPROXY_XDRF(nhproxy_cb_xdr_get_tilesets_res,
      list));
    nhproxy__serv->free_tilesets(list);
}

static void
callback_get_glyph_mapping(id, request, reply)
unsigned short id;
NhProxyXdr *request, *reply;
{
    struct nhproxy_cb_get_glyph_mapping_res *mapping;
    mapping = nhproxy__serv->get_glyph_mapping();
    nhproxy_rpc_params(reply, 1,
      NHPROXY_XDRF(nhproxy_cb_xdr_get_glyph_mapping_res, mapping));
    nhproxy__serv->free_glyph_mapping(mapping);
}

static void
callback_get_extensions(id, request, reply)
unsigned short id;
NhProxyXdr *request, *reply;
{
    int i;
    extern struct nhproxy_extension *nhproxy__extents;
    struct nhproxy_cb_get_extensions_res list;
    for(i = 0; nhproxy__extents[i].name; i++)
	;
    list.extensions = (struct nhproxy_cb_get_extensions_res_extension *)
      malloc(i * sizeof(*list.extensions));
    list.n_extensions = list.extensions ? i : 0;
    for(i = 0; i < list.n_extensions; i++) {
	list.extensions[i].name = nhproxy__extents[i].name;
	list.extensions[i].version = nhproxy__extents[i].version;
	list.extensions[i].no_procedures = nhproxy__extents[i].no_procedures;
    }
    nhproxy_rpc_params(reply, 1, NHPROXY_XDRF(nhproxy_cb_xdr_get_extensions_res,
      &list));
    free(list.extensions);
}

static void
callback_set_option_mod_status(id, request, reply)
unsigned short id;
NhProxyXdr *request, *reply;
{
    char *optnam;
    int status;
    nhproxy_rpc_params(request, 2, NHPROXY_STRING_PTR(optnam),
      NHPROXY_INT_PTR(status));
    if (nhproxy__authorized)
	nhproxy__serv->set_option_mod_status(optnam, status);
    free(optnam);
    if (!nhproxy_rpc_async_mode())
	nhproxy_rpc_params(reply, 0);
}

struct nhproxy_rpc_services nhproxy__callbacks[] = {
    NHPROXY_EXT_CID_DISPLAY_INVENTORY,		callback_display_inventory,
    NHPROXY_EXT_CID_DLBH_FOPEN,			callback_dlbh_fopen,
    NHPROXY_EXT_CID_DLBH_FGETS,			callback_dlbh_fgets,
    NHPROXY_EXT_CID_DLBH_FREAD,			callback_dlbh_fread,
    NHPROXY_EXT_CID_DLBH_FWRITE,		callback_dlbh_fwrite,
    NHPROXY_EXT_CID_DLBH_FCLOSE,		callback_dlbh_fclose,
    NHPROXY_EXT_CID_DLBH_FMD5SUM,		callback_dlbh_fmd5sum,
    NHPROXY_EXT_CID_FLUSH_SCREEN,		callback_flush_screen,
    NHPROXY_EXT_CID_DOREDRAW,			callback_doredraw,
    NHPROXY_EXT_CID_INTERFACE_MODE,		callback_interface_mode,
    NHPROXY_EXT_CID_PARSE_OPTIONS,		callback_parse_options,
    NHPROXY_EXT_CID_GET_OPTION,			callback_get_option,
    NHPROXY_EXT_CID_GET_PLAYER_CHOICES,		callback_get_player_choices,
    NHPROXY_EXT_CID_GET_VALID_SELECTIONS,	callback_get_valid_selections,
    NHPROXY_EXT_CID_QUIT_GAME,			callback_quit_game,
    NHPROXY_EXT_CID_DISPLAY_SCORE,		callback_display_score,
    NHPROXY_EXT_CID_DOSET,			callback_doset,
    NHPROXY_EXT_CID_GET_EXTENDED_COMMANDS,	callback_get_extended_commands,
    NHPROXY_EXT_CID_MAP_MENU_CMD,		callback_map_menu_cmd,
    NHPROXY_EXT_CID_GET_STANDARD_WINID,		callback_get_standard_winid,
    NHPROXY_EXT_CID_GET_TILESETS,		callback_get_tilesets,
    NHPROXY_EXT_CID_GET_GLYPH_MAPPING,		callback_get_glyph_mapping,
    NHPROXY_EXT_CID_GET_EXTENSIONS,		callback_get_extensions,
    NHPROXY_EXT_CID_SET_OPTION_MOD_STATUS,	callback_set_option_mod_status,
    0, NULL,
};
