/* $Id: nblock.c,v 1.2 2004/12/22 23:15:03 ali Exp $
 * Copyright (C) 2003  Slash'EM Development Team
 * Copyright (C) 2004  J. Ali Harlow
 *
 * This file is part of NetHack Proxy.
 *
 * NetHack Proxy is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of the
 * License, or (at your option) any later version.
 *
 * NetHack Proxy is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with NetHack Proxy; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307   
 * USA
 *
 * Alternatively (at your option) you may instead choose to redistribute
 * and/or modify NetHack Proxy under the terms of the NetHack General
 * Public License.
 *
 * You should have receieved a copy of the NetHack General Public License
 * along with NetHack Proxy; if not, download a copy from
 * http://www.nethack.org/common/license.html
 */

/* NhExt: Non-blocking support using threads & mutexes */

/* #define DEBUG */

#include "config.h"
#ifdef DEBUG
#include <stdio.h>
#endif
#if STDC_HEADERS
#include <stdlib.h>
#endif
#include "compat.h"
#include <nhproxy/system.h>
#include <nhproxy/xdr.h>

#if MSWIN_API
#include <windows.h>
#include <process.h>

static int
check_res(res)
int res;
{
    return res == WAIT_ABANDONED || res == WAIT_OBJECT_0 ? 1 :
      res == WAIT_TIMEOUT ? 0 : -1;
}
#define DEFINE_LOCK(mutex)	HANDLE mutex
#define INIT_LOCK(mutex)	(mutex = CreateMutex(NULL, FALSE, NULL))
#define AQUIRE_LOCK_(mutex)	check_res(WaitForSingleObject(mutex, INFINITE))
#define TRY_LOCK_(mutex)	check_res(WaitForSingleObject(mutex, 0))
#define RELEASE_LOCK_(mutex)	ReleaseMutex(mutex)
#define FREE_LOCK(mutex)	CloseHandle(mutex)
#else	/* MSWIN_API */
#include <pthread.h>
#if HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <errno.h>

#define DEFINE_LOCK(mutex)	pthread_mutex_t mutex
#define INIT_LOCK(mutex)	(!pthread_mutex_init(&(mutex), NULL))
#define AQUIRE_LOCK_(mutex)	(!pthread_mutex_lock(&(mutex)))
#define TRY_LOCK_(mutex)	(!pthread_mutex_trylock(&(mutex)))
#define RELEASE_LOCK_(mutex)	(!pthread_mutex_unlock(&(mutex)))
#define FREE_LOCK(mutex)	(!pthread_mutex_destroy(&(mutex)))
#endif	/* MSWIN_API */

#ifdef DEBUG
#if MSWIN_API
#define debug_line(str)		fprintf(stderr, "[%X] %s\n", \
				  GetCurrentThreadId(), str)
#else
#define debug_line(str)		fprintf(stderr, "%s\n", str)
#endif
static int
debug_line_res(res, mutex, verb)
int res;
const char *mutex;
const char *verb;
{
    char buf[100];
    if (res == 1)
	sprintf(buf, "Lock %s %sd", mutex, verb);
    else if (res == 0)
	sprintf(buf, "Lock %s not %sd", mutex, verb);
    else {
	sprintf(buf, "%s of lock %s produced error", verb, mutex);
	if (buf[0] >= 'a' && buf[0] <= 'z') {
	    buf[0] -= 'a';
	    buf[0] += 'A';
	}
    }
    debug_line(buf);
    return res;
}
#define AQUIRE_LOCK(mutex)	(debug_line("Aquiring lock " #mutex), \
				  debug_line_res(AQUIRE_LOCK_(mutex), \
				  #mutex, "aquire"))
#define TRY_LOCK(mutex)		debug_line_res(TRY_LOCK_(mutex), \
				  #mutex, "aquire")
#define RELEASE_LOCK(mutex)	debug_line_res(RELEASE_LOCK_(mutex), \
				  #mutex, "release")
#else
#define AQUIRE_LOCK(mutex)	AQUIRE_LOCK_(mutex)
#define TRY_LOCK(mutex)		TRY_LOCK_(mutex)
#define RELEASE_LOCK(mutex)	RELEASE_LOCK_(mutex)
#endif

#define NHPROXY_NB_PENDING	1
#define NHPROXY_NB_CLOSED		2
#define NHPROXY_NB_ERROR		4

struct NhProxyNB_ {
    unsigned int flags;
    nhproxy_io_func func;
    nhproxy_genericptr_t handle;
#if MSWIN_API
    HANDLE thread;
#else
    pthread_t thread;
#endif
    DEFINE_LOCK(m_A);
    DEFINE_LOCK(m_B);
    DEFINE_LOCK(m_C);
    struct {
	nhproxy_genericptr_t buffer;
	int bytes;
    } cmd;
    int res;
};

#if MSWIN_API
static unsigned  __stdcall
read_thread(data)
nhproxy_genericptr_t data;
{
    NhProxyNB *nb = (NhProxyNB *)data;
#else
static nhproxy_genericptr_t
read_thread(data)
nhproxy_genericptr_t data;
{
    NhProxyNB *nb = (NhProxyNB *)data;
#endif
    nhproxy_genericptr_t buffer = data;		/* Any non-zero value */
    int bytes;
#ifdef DEBUG
    debug_line("read_thread starts");
#endif
    if (AQUIRE_LOCK(nb->m_B)) {
	for(;;) {
	    if (!AQUIRE_LOCK(nb->m_C))
		break;
	    buffer = nb->cmd.buffer;
	    bytes = nb->cmd.bytes;
	    if (!RELEASE_LOCK(nb->m_B))
		break;
	    if (!AQUIRE_LOCK(nb->m_A))
		break;
	    if (!RELEASE_LOCK(nb->m_C))
		break;
	    if (!buffer)
		break;
	    if (!AQUIRE_LOCK(nb->m_B))
		break;
#ifdef DEBUG
	    debug_line("Issuing read call");
#endif
	    nb->res = (*nb->func)(nb->handle, buffer, bytes);
#ifdef DEBUG
	    debug_line("Read call returns");
#endif
	    if (!RELEASE_LOCK(nb->m_A))
		break;
	}
    }
    (void)RELEASE_LOCK(nb->m_A);
    if (!buffer) {
	/* Controlled exit - we're responsible for cleaning up */
	(void)FREE_LOCK(nb->m_A);
	(void)FREE_LOCK(nb->m_B);
	(void)FREE_LOCK(nb->m_C);
#if MSWIN_API
	CloseHandle(nb->thread);
#endif
#ifdef DEBUG
	debug_line("read_thread terminates");
#endif
	free(nb);
	return 0;
    } else {
	(void)RELEASE_LOCK(nb->m_B);
	(void)RELEASE_LOCK(nb->m_C);
#ifdef DEBUG
	debug_line("read_thread aborts");
#endif
#if MSWIN_API
	return 1;
#else
	return (nhproxy_genericptr_t)1;
#endif
    }
}

/*
 * Sequence of events for a non-blocking read:
 *
 *	Read thread			Main thread
 *	-----------			-----------
 *	B  C Waiting for cmd		AC - nhproxy_nb_read called
 *	B  C Waiting for cmd		AC - Writes cmd
 *	BC - Copying cmd		A  B Issues cmd
 *	C  A				AB -
 *	AC -				B  C Waiting for cmd to be actioned
 *	A  B Ready to action cmd	BC -
 *	AB - In underlying system	C  - Returns to caller
 *	AB - In underlying system	C  A nhproxy_nb_read re-called
 *	B  - Result recorded		AC - Result available to caller
 *
 * (The first column for each thread is the locks currently aquired. The second
 * column is the locks currently being waited for.)
 *
 * Sequence of events for a blocking read:
 *
 *	Read thread			Main thread
 *	-----------			-----------
 *	B  C Waiting for cmd		AC - nhproxy_nb_read called
 *	B  C Waiting for cmd		AC - Writes cmd
 *	BC - Copying cmd		A  B Issues cmd
 *	C  A				AB -
 *	AC -				B  C Waiting for cmd to be actioned
 *	A  B Ready to action cmd	BC -
 *	AB - In underlying system	C  A Waiting for results
 *	B  - Result recorded		AC - Returns to caller
 *
 * The mutexes also protect certain fields as follows:
 *	A - res
 *	B -
 *	C - cmd
 */

NhProxyNB *
nhproxy_nb_open(func, handle)
nhproxy_io_func func;
nhproxy_genericptr_t handle;
{
    NhProxyNB *nb;
    int retval;
    nb = malloc(sizeof(*nb));
    if (!nb) {
#ifdef DEBUG
	debug_line("nhproxy_nb_open failing (not enough memory)");
#endif
	return NULL;
    }
    nb->func = func;
    nb->handle = handle;
    if (!INIT_LOCK(nb->m_A)) {
	free(nb);
#ifdef DEBUG
	debug_line("nhproxy_nb_open failing (can't init lock A)");
#endif
	return NULL;
    }
    if (!INIT_LOCK(nb->m_B)) {
	FREE_LOCK(nb->m_A);
#ifdef DEBUG
	debug_line("nhproxy_nb_open failing (can't init lock B)");
#endif
	free(nb);
	return NULL;
    }
    if (!INIT_LOCK(nb->m_C)) {
	FREE_LOCK(nb->m_A);
	FREE_LOCK(nb->m_B);
#ifdef DEBUG
	debug_line("nhproxy_nb_open failing (can't init lock C)");
#endif
	free(nb);
	return NULL;
    }
    if (!AQUIRE_LOCK(nb->m_A) || !AQUIRE_LOCK(nb->m_C)) {
#ifdef DEBUG
	debug_line("nhproxy_nb_open failing (can't aquire locks A & C)");
#endif
	goto out;
    }
    nb->flags = 0;
#if MSWIN_API
    nb->thread = (HANDLE)_beginthreadex(NULL, 0, read_thread, nb, 0, NULL);
    if (!nb->thread) {
#else
    if (pthread_create(&nb->thread, NULL, read_thread, nb) == EAGAIN) {
#endif
#ifdef DEBUG
	debug_line("nhproxy_nb_open failing (can't create read thread)");
#endif
out:	RELEASE_LOCK(nb->m_A);
	RELEASE_LOCK(nb->m_C);
	FREE_LOCK(nb->m_A);
	FREE_LOCK(nb->m_B);
	FREE_LOCK(nb->m_C);
	free(nb);
	return NULL;
    }
    /*
     * We must wait for the read thread to start and aquire the B mutex
     * or the synchronization will fail.
     */
    while ((retval = TRY_LOCK(nb->m_B)) > 0) {
	if (!RELEASE_LOCK(nb->m_B)) {
#if MSWIN_API
	    TerminateThread(nb->thread, -1);
	    CloseHandle(nb->thread);
#else
	    pthread_cancel(nb->thread);
#endif
#ifdef DEBUG
	    debug_line("nhproxy_nb_open failing (can't release lock B)");
#endif
	    goto out;
	}
#if MSWIN_API
	Sleep(0);			/* Relinquish time slice */
#else
# ifdef _POSIX_PRIORITY_SCHEDULING
	sched_yield();			/* Relinquish time slice */
# elif HAVE_SLEEP
	sleep(0);
# endif
#endif
    }
    if (retval < 0) {
	nhproxy_nb_close(nb);
#ifdef DEBUG
	debug_line(
	  "nhproxy_nb_open failing (error in trying to aquire lock B)");
#endif
	return NULL;
    }
    return nb;
}

nhproxy_bool_t
nhproxy_nb_close(nb)
NhProxyNB *nb;
{
    int retval;
    if (nb->flags & NHPROXY_NB_CLOSED)
	return nhproxy_false;
    nb->flags |= NHPROXY_NB_CLOSED;
    /*
     * Mutex B should always be owned by the read thread unless we have aborted
     * in the middle of a call to nhproxy_nb_read.
     */
    retval = TRY_LOCK(nb->m_B);
    if (retval == 1) {
#if MSWIN_API
	DWORD code;
	if (GetExitCodeThread(nb->thread, &code) == STILL_ACTIVE) {
	    /*
	     * Something has gone drastically wrong. Clean up as best we can.
	     */
	    TerminateThread(nb->thread, 1);
	    code = 1;
	}
#else
	nhproxy_genericptr_t code;
	if (pthread_join(nb->thread, &code)) {
	    /*
	     * Something has gone drastically wrong. Clean up as best we can.
	     */
	    pthread_cancel(nb->thread);
	    code = (nhproxy_genericptr_t)1;
	}
#endif
	if (code) {
	    /* Read thread has aborted. Clean up */
#if MSWIN_API
	    CloseHandle(nb->thread);
#endif
	    RELEASE_LOCK(nb->m_A);
	    RELEASE_LOCK(nb->m_B);
	    RELEASE_LOCK(nb->m_C);
	    FREE_LOCK(nb->m_A);
	    FREE_LOCK(nb->m_B);
	    FREE_LOCK(nb->m_C);
	    free(nb);
	} else {
	    /* read thread has terminated and cleaned up (shouldn't happen) */
	    RELEASE_LOCK(nb->m_B);
	}
	return nhproxy_false;
    }
    nb->cmd.buffer = NULL;
    nb->cmd.bytes = 0;
    if (!(nb->flags & NHPROXY_NB_PENDING))
	RELEASE_LOCK(nb->m_A);
    RELEASE_LOCK(nb->m_C);
    /* The read thread is responsible for cleaning up - if a read is pending
     * this will be after it finishes.
     */
    return nhproxy_true;
}

int
nhproxy_nb_read(nb, buf, bytes, blocking)
NhProxyNB *nb;
char *buf;
int bytes, blocking;
{
    int retval;
#ifdef DEBUG
    debug_line("nhproxy_nb_read called");
#endif
    if (nb->flags & NHPROXY_NB_ERROR)
	return -1;
    if (!(nb->flags & NHPROXY_NB_PENDING)) {
	nb->cmd.buffer = buf;
	nb->cmd.bytes = bytes;
	if (!RELEASE_LOCK(nb->m_C) || !AQUIRE_LOCK(nb->m_B) ||
	  !RELEASE_LOCK(nb->m_A) || !AQUIRE_LOCK(nb->m_C) ||
	  !RELEASE_LOCK(nb->m_B)) {
	    nb->flags |= NHPROXY_NB_ERROR;
#ifdef DEBUG
	    debug_line("nhproxy_nb_read failing with hard error");
#endif
	    return -1;
	}
	if (!blocking) {
	    nb->flags |= NHPROXY_NB_PENDING;
#ifdef DEBUG
	    debug_line("nhproxy_nb_read returning PENDING");
#endif
	    return -2;
	} else {
	    if (!AQUIRE_LOCK(nb->m_A)) {
		nb->flags |= NHPROXY_NB_ERROR;
#ifdef DEBUG
		debug_line("nhproxy_nb_read failing with hard error");
#endif
		return -1;
	    }
#ifdef DEBUG
	    debug_line("nhproxy_nb_read returns result");
#endif
	    return nb->res >= 0 ? nb->res : -1;
	}
    } else {
	if (buf != nb->cmd.buffer || bytes < nb->cmd.bytes) {
#ifdef DEBUG
	    debug_line("nhproxy_nb_read failing with soft error (INVALID)");
#endif
	    return -1;
	}
	if (!blocking) {
	    retval = TRY_LOCK(nb->m_A);
	    if (retval == 1) {
		nb->flags &= ~NHPROXY_NB_PENDING;
#ifdef DEBUG
		debug_line("nhproxy_nb_read returns result");
#endif
		return nb->res >= 0 ? nb->res : -1;
	    } else if (retval) {
		nb->flags |= NHPROXY_NB_ERROR;
#ifdef DEBUG
		debug_line("nhproxy_nb_read failing with hard error");
#endif
		return -1;
	    }
	    else {
#ifdef DEBUG
		debug_line("nhproxy_nb_read returning PENDING");
#endif
		return -2;
	    }
	} else {
	    if (!AQUIRE_LOCK(nb->m_A)) {
		nb->flags |= NHPROXY_NB_ERROR;
#ifdef DEBUG
		debug_line("nhproxy_nb_read failing with hard error");
#endif
		return -1;
	    }
	    nb->flags &= ~NHPROXY_NB_PENDING;
#ifdef DEBUG
	    debug_line("nhproxy_nb_read returns result");
#endif
	    return nb->res >= 0 ? nb->res : -1;
	}
    }
}
