/* Server.java
 * Copyright (C) 2007, 2008 Gustav Behm <gbehm (at) kth.se>
 *
 * This file is part of Netzack.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */


package se.kth.netzack;
import se.kth.netzack.*;
import java.net.*;
import java.lang.*;
import java.io.*;
import java.util.*;
import java.nio.*;





public class Server {

    Game game;

    public static InetAddress GROUP;
    public static int PUBLIC_PORT;

    public static InetAddress localAddress;

    public int id = 0;

    public int echos = 0;


    PublicChannel pub;
    ServerThread server;


    Server(Game g, String group, int port) {
	PUBLIC_PORT = port;
	try {
	    GROUP = InetAddress.getByName(group);

	    for (Enumeration interfaces = NetworkInterface.getNetworkInterfaces(); interfaces.hasMoreElements() ;) {
		for(Enumeration addresses = ((NetworkInterface)interfaces.nextElement()).getInetAddresses(); addresses.hasMoreElements(); ) {
		    InetAddress i = (InetAddress)addresses.nextElement();
		    if(!i.isLinkLocalAddress() && !i.isAnyLocalAddress() && !i.isLoopbackAddress()) {
			localAddress = i;

			break;
		    }
		}
	    }
	}
	catch(SocketException e) {
	    Netzack.debug(e);
	}
	catch(UnknownHostException e) {
	    Netzack.debug(e);
	}

	game = g;

	if(localAddress == null) {
	    Netzack.error("Couldn't find local address");
	}


	Object lock = new Object();
	synchronized(lock) {
	    pub = new PublicChannel(this, lock);
	    try {
		lock.wait();
	    }
	    catch(InterruptedException e) {

	    }

	    server = new ServerThread(this, lock);
	    try {
		lock.wait();
	    }
	    catch(InterruptedException e) {

	    }

	}




    }



    public void peerConnected(int enteree) {
	dispatchPeerConnected(enteree);
    }

    public void peerQuit(int quitee, boolean normal) {
	//reestablish(quitee);
	if(normal) {
	    server.checkConnections();

	    dispatchPeerQuit(quitee);
	}
	else {
	    server.reconnected(quitee);
	}
    }







    public void reconnect(int to, InetAddress addr, int port) {
	server.reconnect(to, addr, port);
    }

    public void reconnected(int to) {
	server.reconnected(to);
    }

    public void reestablish(int to) {
	Netzack.debug("Trying reestablish connection to #" + to + "'s peers.");
	pub.send(Packet.reconnectPacket(id, to, getPort(), localAddress.getAddress()));
    }






    public void play() {

	echos = 0;
	pub.send(Packet.echoPacket(id, true));

	try{
	    Thread.sleep(1000);
	}
	catch(InterruptedException e) {

	}

	if(echos == 0) {
	    Netzack.verbose("Starting game...");
	    id = 1;
	    game.thisIsMe(id);
	    game.newRound();
	}
	else {
	    Netzack.verbose("Joining game...");
	    game.alone = false;
	}


    }



    //PacketListeners

    private Vector packetListeners = new Vector(5);

    public void addPacketListener(PacketListener l) {
	packetListeners.add(l);
    }

    public void dispatch(Packet packet) {
	if(packet.getMsg() == Message.NONE)
	    return;

	server.dispatch(packet);

	packet.readMessage();

	Iterator itr = packetListeners.iterator();
	while(itr.hasNext()) {
	    PacketListener l = (PacketListener)itr.next();
	    l.gotPacket(this, packet);
	}

	packet.release();
    }




    //PeerListeners

    private Vector peerListeners = new Vector(5);

    public void addPeerListener(PeerListener l) {
	peerListeners.add(l);
    }

    public void dispatchPeerQuit(int quitee) {
	Iterator itr = peerListeners.iterator();
	while(itr.hasNext()) {
	    PeerListener l = (PeerListener)itr.next();
	    l.peerQuit(quitee);
	}
    }

    public void dispatchPeerConnected(int enteree) {
	Iterator itr = peerListeners.iterator();
	while(itr.hasNext()) {
	    PeerListener l = (PeerListener)itr.next();
	    l.peerConnected(enteree);
	}

    }









    public int getPort() {
	return server.getPort();
    }


    public void connect(InetAddress addr, int port) {
	server.connect(addr, port);

	if(id == 0)
	    send(Packet.helloPacket(id));
    }







    //PacketBuffer management

    public final static List packetBuffers = Collections.synchronizedList(new ArrayList(10));

    public static PacketBuffer getPacketBuffer() {
	return getPacketBuffer(false);
    }

    public static PacketBuffer getPacketBuffer(boolean allocate) {
	synchronized(packetBuffers) {
	    Iterator itr = packetBuffers.iterator();
	    while(itr.hasNext()) {
		PacketBuffer b = (PacketBuffer)itr.next();
		if(b.ref <= 0) {
		    b.buffer.clear();
		    b.addRef();
		    return b;
		}
	    }

	    if(!allocate) {
		Runtime r = Runtime.getRuntime();
		r.gc();

		return getPacketBuffer(true);
	    }

	    packetBuffers.add(new PacketBuffer());
	}

	return getPacketBuffer(true);
    }




    //Method to generate an id

    public int generateId() {
	return game.generateId();
    }




    //Aquire number of peers

    public int peers() {
	echos = 0;
	pub.send(Packet.echoPacket(id, false));

	try{
	    Thread.sleep(1000);
	}
	catch(InterruptedException e) {

	}

	return echos;
    }



    //Methods for sending packets

    public void send(Packet p) {
	server.send(p);
	p.release();
    }

    public void send(Message m, Object[] vars) {
	Packet packet = new Packet(null, m, vars);

	server.send(packet);

	packet.release();
    }



}



