/* Game.java
 * Copyright (C) 2007, 2008 Gustav Behm <gbehm (at) kth.se>
 *
 * This file is part of Netzack.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */


package se.kth.netzack;
import se.kth.netzack.*;
import java.util.*;
import java.nio.*;
import java.awt.event.*;
import java.awt.*;
import javax.swing.Timer;
import java.net.*;



class Game extends Thread implements PacketListener, KeyListener, DeathListener, ActionListener, PeerListener  {

    public Map players = Collections.synchronizedMap(new HashMap());

    public int me = 0;
    int highestIdSeen = 1;

    boolean running = true;

    int FPS = 16;

    String name;
    Color color;


    private final Timer timer;

    private boolean released = false;
    private KeyEvent releaseEvent;

    public boolean alone = true;
    public boolean waiting = false;

    ZFrame frame;

    Server server;



    ActionListener taskPerformer = new ActionListener() {
	    public void actionPerformed(ActionEvent evt) {
		server.send(Packet.roundPacket(me, true));

		startRound();
		waiting = false;
	    }
	};
    Timer t = new Timer(1500, taskPerformer);




    Game(int fps, String group, int port, String n, Color c) {
	FPS = fps;
	timer = new Timer(1, this);
	frame = new ZFrame(this);
	server = new Server(this, group, port);

	server.addPeerListener(this);
	server.addPacketListener(this);
	frame.canvas.addKeyListener(this);
	frame.addKeyListener(this);

	t.setRepeats(false);




	if(n.length() == 0) {
	    try {
		InetAddress local = InetAddress.getLocalHost();
		name = System.getProperty("user.name") + "@" + local.getHostName();
	    }
	    catch (UnknownHostException e) {
		name = "Captain-O Fancy-Pants";
	    }
	}
	else
	    name = n;

	color = c;

	server.play();

    }


    public void setNickname(String n) {
	if(n.length() != 0)
	    name = n;
    }

    public void setColor(Color c) {
	Player p;
	synchronized(players) {
	    p = (Player)players.get((Object)new Integer(me));
	    if(p != null) {
		p.worm.color = c;
	    }
	}
	color = c;
    }

    public int generateId() {
	return ++highestIdSeen;
    }

    public void seeId(int i) {
	if(i > highestIdSeen)
	    highestIdSeen = i;
    }

    public void thisIsMe(int i) {
	me = i;
	seeId(me);
	Netzack.verbose("I'm #" + me + ".");

	synchronized(players) {
	    Player p = new Player(me, name, color);
	    p.worm.color = color;
	    p.addDeathListener(this);
	    players.put((Object)new Integer(me), (Object)p);
	}

    }

    public void death(Player p) {
	if(p.id == me) {
	    frame.console("You died...");
	}
    }


    public void peerConnected(int enteree) {
	if(alone) {
	    synchronized(players) {
		Player p = (Player)players.get((Object)new Integer(me));
		p.isPlaying = false;
	    }

	    newRound();
	}
    }

    public void peerQuit(int quitee) {
	synchronized(players) {
	    Player p = (Player)players.get((Object)new Integer(quitee));
	    if(p != null) {
		frame.console(p.name + " left the game. Bastard.");
		players.remove((Object)new Integer(quitee));
	    }
	}

	if(players.size() == 1) {
	    alone = true;
	}
    }



    public void iWon() {
	frame.console("You won...");
	synchronized(players) {
	    Player p = (Player)players.get((Object)new Integer(me));
	    p.score++;
	    p.isPlaying = false;
	}

	newRound();
    }


    public void newRound() {
	server.send(Packet.roundPacket(me, false));

	nextPosition();

	t.start();
	waiting = true;
    }


    private boolean startRound() {
	if(players.isEmpty())
	    return false;

	synchronized(players) {
	    Iterator itr = players.values().iterator();
	    while(itr.hasNext()) {
		Player p = (Player)itr.next();
		if(p.isPlaying)
		    return false;
	    }

	    itr = players.values().iterator();
	    while(itr.hasNext()) {
		Player p = (Player)itr.next();
		p.startPlaying();
	    }

	}

	frame.clear();
	frame.updateScore();

	return true;
    }



    private void nextPosition() {
	frame.updateScore();


	Player p;
	synchronized(players) {
	    p = (Player)players.get((Object)new Integer(me));
	    if(p != null) {
		p.nextX = (int)Math.round(Math.random()*(Config.X-100)) + 50;
		p.nextY = (int)Math.round(Math.random()*(Config.Y-100)) + 50;
		p.nextA = Math.random()*2*Math.PI;
		p.isReady = true;
	    }
	}

	if(p != null) {
	    server.send(Packet.posPacket(me,
					 p.nextX,
					 p.nextY,
					 p.nextA,
					 color.getRGB(),
					 p.score,
					 name));
	}
    }






    public void gotPacket(Server s, Packet p) {
	Message msg = p.getMsg();

	if(msg.equals(Message.POS)) {
	    //Get data
	    int id = p.readInt();
	    int x = p.readInt();
	    int y = p.readInt();
	    double  a = p.readDouble();
	    Color c = new Color(p.readInt());
	    int score = p.readInt();

	    String nickname = p.readString();

	    seeId(id);


	    //Add data to players array.

	    synchronized(players) {
		Player player = (Player)players.get((Object)new Integer(id));

		if(player == null) {
		    frame.console(nickname + " has joined.");
		    alone = false;
		    player = new Player(id, nickname, c);
		    players.put((Object)new Integer(id), (Object)player);
		}

		player.nextX = x;
		player.nextY = y;
		player.nextA = a;

		player.score = score;

		player.isReady = true;
	    }

	}
	else if(msg.equals(Message.DIR)) {
	    //Get data
	    int id = p.readInt();

	    if(id != me) {
		int dir = p.readInt();

		double x = p.readDouble();
		double y = p.readDouble();
		double a = p.readDouble();


		turn(id, Direction.generate(dir), x, y, a);
	    }

	}
	else if(msg.equals(Message.ROUND)) {
	    int id = p.readInt();
	    boolean start = p.readBoolean();

	    if(start) {
		//Netzack.debug("New round starting...");
		startRound();
	    }
	    else {
		//Netzack.debug("New round about to start...");
		nextPosition();

		synchronized(players) {
		    Iterator itr = players.values().iterator();
		    while(itr.hasNext()) {
			Player player = (Player)itr.next();
			player.isPlaying = false;
		    }
		}
	    }
	}
    }





    public void turn(int id, Direction dir, double x, double y, double a) {
	synchronized(players) {
	    Player p = (Player)players.get((Object)new Integer(id));

	    if(p != null) {
		p.worm.x = x;
		p.worm.y = y;
		p.worm.a = a;
		p.turn(dir);

	    }
	}
    }






    public void keyPressed(KeyEvent e) {
	// Implemented with inspiration form http://bugs.sun.com/bugdatabase/view_bug.do?bug_id=4153069
	// Mostly with thanks to Ekipur <wwg (at) arco.in-berlin.de>

	released = false;
	timer.stop();


	synchronized(players) {
	    Player p = (Player)players.get((Object)new Integer(me));

	    if(p != null && p.worm.da == Direction.none) {

		int k = e.getKeyCode();
		if(k == KeyEvent.VK_LEFT && k != KeyEvent.VK_RIGHT) {
		    p.worm.turn(Direction.left);
		    server.send(Packet.dirPacket(me, Direction.left, p.worm.x, p.worm.y, p.worm.a));
		}
		else if(k != KeyEvent.VK_LEFT && k == KeyEvent.VK_RIGHT) {
		    p.worm.turn(Direction.right);
		    server.send(Packet.dirPacket(me, Direction.right, p.worm.x, p.worm.y, p.worm.a));
		}


	    }
	}

	e.consume();
    }

    public void keyReleased(KeyEvent e) {
	// Implemented with inspiration form http://bugs.sun.com/bugdatabase/view_bug.do?bug_id=4153069
	// Mostly with thanks to Ekipur <wwg (at) arco.in-berlin.de>

	if (!released) {
	    releaseEvent = e;
	    timer.restart();
	}
	else {
	    synchronized(players) {
		Player p = (Player)players.get((Object)new Integer(me));

		if(p != null) {

		    int k = e.getKeyCode();
		    if(k == KeyEvent.VK_LEFT) {
			p.worm.turn(Direction.none);
			server.send(Packet.dirPacket(me, Direction.none, p.worm.x, p.worm.y, p.worm.a));
		    }
		    if(k == KeyEvent.VK_RIGHT) {
			p.worm.turn(Direction.none);
			server.send(Packet.dirPacket(me, Direction.none, p.worm.x, p.worm.y, p.worm.a));
		    }
		    if(k == KeyEvent.VK_ESCAPE) {
			frame.dispose();
			System.exit(0);
		    }
		    else if(k == KeyEvent.VK_L) {
			frame.viewLicense();
		    }


		}
	    }
	}
	e.consume();
    }

    public void actionPerformed(ActionEvent e) {
	// Implemented with inspiration form http://bugs.sun.com/bugdatabase/view_bug.do?bug_id=4153069
	// Mostly with thanks to Ekipur <wwg (at) arco.in-berlin.de>

	released = true;
	timer.stop();
	keyReleased(releaseEvent);
    }

    public void keyTyped(KeyEvent e) {


    }





    long prev = 0;
    boolean check = false;

    public void run() {

	long start = System.currentTimeMillis();
	double n = 0;
	int playing = 0;



	ActionListener loop = new ActionListener() {
		public void actionPerformed(ActionEvent e) {

		    long now = e.getWhen();
		    int playing = frame.paintWorld(now - prev);

		    prev = now;

		    if(waiting)
			return;

		    if(!check && playing > 1) {
			check = true;
		    }
		    else if(check && playing == 1) {
			synchronized(players) {
			    Player p = (Player)players.get((Object)new Integer(me));
			    if(p.isPlaying) {
				iWon();
			    }
			    check = false;
			}
		    }
		    else if(alone && playing == 0 && players.size() == 1) {
			synchronized(players) {
			    Player p = (Player)players.get((Object)new Integer(me));
			    if(p != null && !p.isPlaying) {
				newRound();

			    }
			}
		    }
		}
	    };

	Timer f = new Timer(1000/FPS, loop);

	f.start();

	/*


	try{
	    long next = 0;
	    long now = 0;
	    long prev = start;
	    now = System.currentTimeMillis();
	    while(running) {

		next = start+Math.round(1000*n/FPS);
		if(now < next)
		    sleep(next - now);
		else if(n % 10 == 1)
		    Netzack.debug("Lagging");

		now = System.currentTimeMillis();
		playing = frame.paintWorld(now - prev);

		prev = now;
		n++;

		if(waiting)
		    continue;

		if(!check && playing > 1) {
		    check = true;
		}
		else if(check && playing == 1) {
		    synchronized(players) {
			Player p = (Player)players.get((Object)new Integer(me));
			if(p.isPlaying) {
			    iWon();
			}
			check = false;
		    }
		}
		else if(alone && playing == 0 && players.size() == 1) {
		    synchronized(players) {
			Player p = (Player)players.get((Object)new Integer(me));
			if(p != null && !p.isPlaying) {
			    newRound();

			}
		    }
		}
	    }

	}
	catch(InterruptedException e) {

	}*/

    }


}
