/**
 * WebAdmin
 * Copyright (C) 2006 Netwosix Team
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License 
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 *
 * Linking WebAdmin statically or dynamically with other modules is making
 * a combined work based on WebAdmin. Thus, the terms and conditions of the 
 * GNU General Public License cover the whole combination.
 *
 * In addition, as a special exception, the copyright holders of
 * WebAdmin give you permission to combine WebAdmin with free software
 * programs or libraries that are released under the GNU LGPL and with
 * code included in the standard release of OpenSSL under the OpenSSL
 * License and SSLeay License (or modified versions of such code, with
 * unchanged licenses). You may copy and distribute such a system
 * following the terms of the GNU GPL for WebAdmin and the licenses of
 * the other code concerned, provided that you include the source code of
 * that other code when and as the GNU GPL requires distribution of
 * source code.
 *
 * Note that people who make modified versions of WebAdmin are not obligated
 * to grant this special exception for their modified versions; 
 * it is their choice whether to do so. The GNU General Public License 
 * gives permission to release a modified version without this exception; 
 * this exception also makes it possible to release a modified 
 * version which carries forward this exception.
 */

#ifndef CONNECTION_H_
#define CONNECTION_H_


typedef struct _AlConnectionManager AlConnectionManager;
typedef struct _AlConnection        AlConnection;


/**
 */
typedef void(*al_connection_manager_delete_func)(AlConnectionManager *);


/**
 */
typedef AlConnection*(*al_connection_manager_wait_func)(AlConnectionManager *);


/**
 */
typedef AlConnection*(*al_connection_manager_connect_func)(AlConnectionManager *, const char *);


/**
 */
typedef void(*al_connection_close_func)(AlConnection *);


/**
 */
typedef void(*al_connection_delete_func)(AlConnection *);


/**
 */
typedef AlString*(*al_connection_read_func)(AlConnection *);


/**
 */
typedef void(*al_connection_send_func)(AlConnection *, const AlString *);


/**
 */
struct _AlConnectionManager
{

	al_connection_manager_wait_func     wait;
	al_connection_manager_delete_func   del;
	al_connection_manager_connect_func  conn;

	int                                 fork;
	void                              * data;

};


/**
 */
AlConnectionManager * al_connection_manager_new(al_connection_manager_wait_func wait,
						al_connection_manager_delete_func del,
						al_connection_manager_connect_func conn, 
						int fork, void * data);


/**
 * Delete a connection manager and close all open conenction.
 *
 * @param connection_manager the connection manager
 */
void al_connection_manager_delete(AlConnectionManager * connection_manager);


/**
 */
#define al_connection_manager_get_data(connection_manager,type) \
 ((type*)((connection_manager)->data))


/**
 */
struct _AlConnection
{

	AlConnectionManager       * connection_manager; 

	al_connection_read_func     read;
	al_connection_send_func     send;
	al_connection_close_func    close;
	al_connection_delete_func   del;
	
	void                      * data;
	
};


/**
 */
AlConnection * al_connection_new(AlConnectionManager * connection_manager,
				al_connection_read_func read,
				al_connection_send_func send,
				al_connection_close_func close,
				al_connection_delete_func del,
				void * data);


/**
 * Free the connection.
 *
 * @param connection connection to delete
 */
void al_connection_delete(AlConnection * connection);


/**
 */
#define al_connection_get_data(connection,type) \
 ((type*)((connection)->data))


/**
 * Wait for a new connection.
 *
 * @param connection_manager the connection manager
 *
 * @return return the new connection or NULL if there is no more new connection.
 */
#define al_connection_manager_wait(connection_manager)        \
 (connection_manager->wait(connection_manager))


/**
 * Close an open connection.
 *
 * @param connection connection to close
 */
#define al_connection_close(connection)                         \
 (connection->close(connection))

 
/**
 * Return true if the connection manager need a fork when
 * a new connection is estabilished.
 *
 * @param connection_manager the connection manager
 *
 * @return true if a fork is needed
 */
#define al_connection_manager_need_fork(connection_manager)      \
 (connection_manager->fork)


/**
 * Read a packet from a connection.
 *
 * @param connection connection to use for reading
 *
 * @return the new packet or NULL if there is no more data
 */
#define al_connection_read(connection)                           \
 (connection->read(connection))


/**
 * Send a packet at the given connection.
 *
 * @param connection connection to use for sendig
 * @param packet packet to send
 */
#define al_connection_send(connection, packet)                   \
 (connection->send(connection, packet))


/**
 * Connect to a server.
 *
 * @param connection_manager the connection manager
 * @param server server to connect
 *
 * @return the new connection or NULL if fail
 */
#define al_connection_manager_connect(connection_manager,server)  \
 (connection_manager->conn(connection_manager,server))


/**
 * Create a connection object from it's name
 *
 * @param connection_type type of connection. If NULL is 
 * take from the attribute "connection" in node
 * @param supported NULL terminated list of supported connection
 * @param node node with connection properties
 *
 * @return the connection manager or NULL if vail
 */
AlConnectionManager * al_connection_manager_from_string(const char * connection_type, const char * supported[], AlXmlNode * node);


/**
 * If the connection type passed is "list" list the supported connection and exit.
 *
 * @param type connection type
 * @param version program version
 * @param supported supported connection
 */
void al_connection_supported_list(const char * type, const char * version, const char * supported[]);


/**
 * List of supported connection.
 */
#ifdef AL_SSL
	static const char * al_connection_supported[] = { "stdio", "tcp", "ssl", NULL };
#else
	static const char * al_connection_supported[] = { "stdio", "tcp", NULL };
#endif


#endif /* CONNECTION_H_ */
