/**
 * WebAdmin
 * Copyright (C) 2006 Netwosix Team
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License 
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 *
 * Linking WebAdmin statically or dynamically with other modules is making
 * a combined work based on WebAdmin. Thus, the terms and conditions of the 
 * GNU General Public License cover the whole combination.
 *
 * In addition, as a special exception, the copyright holders of
 * WebAdmin give you permission to combine WebAdmin with free software
 * programs or libraries that are released under the GNU LGPL and with
 * code included in the standard release of OpenSSL under the OpenSSL
 * License and SSLeay License (or modified versions of such code, with
 * unchanged licenses). You may copy and distribute such a system
 * following the terms of the GNU GPL for WebAdmin and the licenses of
 * the other code concerned, provided that you include the source code of
 * that other code when and as the GNU GPL requires distribution of
 * source code.
 *
 * Note that people who make modified versions of WebAdmin are not obligated
 * to grant this special exception for their modified versions; 
 * it is their choice whether to do so. The GNU General Public License 
 * gives permission to release a modified version without this exception; 
 * this exception also makes it possible to release a modified 
 * version which carries forward this exception.
 */

#include "agents.h"



/**
 */
AgentsResponse * agents_response_new(void)
{
	AgentsResponse * response = al_new(AgentsResponse);

	response->next = response->prev = NULL;

	response->id = NULL;
	response->type = AGENTS_RESPONSE_OK;
	response->server_name = NULL;
	response->server_data = NULL;

	return response;
}


/**
 */
void agents_response_delete(AgentsResponse * response)
{

	al_return_if_fail (response);

	al_delete(response->id);
	al_delete(response->server_name);
	al_delete(response->server_data);
	al_delete(response);

}


/**
 */
AgentsResponsePacket * agents_response_packet_new(void)
{
	AgentsResponsePacket * response_packet = al_new(AgentsResponsePacket);

	response_packet->responses = NULL;

	return response_packet;
}


/**
 */
void agents_response_packet_delete(AgentsResponsePacket * response_packet)
{

	al_return_if_fail (response_packet);

	if (response_packet->responses)
	{
		AgentsResponse * response = response_packet->responses;
		AgentsResponse * next;

		while (response)
		{
			next = response->next;
			agents_response_delete(response);
			response = next;
		}
	}
	
	al_delete(response_packet);

}


/**
 */
void agents_response_packet_response_add(AgentsResponsePacket * response_packet, AgentsResponse * response)
{

	al_return_if_fail (response_packet && response);

	if (response_packet->responses == NULL)
	{
		response_packet->responses = response;
		return;
	}


	response->next = response_packet->responses->next;
	response->prev = response_packet->responses;

	response_packet->responses->next = response;

}

	
/**
 */
AgentsResponse * agents_response_from_request(const AgentsRequest * request, AlConnectionManager * connection_manager)
{
	AlConnection   * connection = NULL;
	AgentsResponse * response;
	AlString       * s;


	al_return_if_fail (request && connection_manager);


	response = agents_response_new();
	agents_response_set_id(response, agents_request_get_id(request));
	agents_response_set_server_name(response, agents_request_get_server_name(request));

	if (al_xml_verify_from_string(agents_request_get_server_data(request)))
	{
		al_log (("Invalid request server-data\n"));
		agents_response_set_type(response, AGENTS_RESPONSE_INVALID_SERVER_DATA);
	}
	else
	{
		al_log (("processing request id    : %s\n", agents_request_get_id(request)));
		al_log (("processing server        : %s\n", agents_request_get_server_name(request)));
		al_log (("processing server data   : %s\n", agents_request_get_server_data(request)));

		if ((connection = al_connection_manager_connect(connection_manager, agents_request_get_server_name(request))) != NULL)
		{
			s = al_string_new();
			al_string_set(s, agents_request_get_server_data(request));
			al_connection_send(connection, s);
			al_string_delete(s);

			s = al_connection_read(connection);
			if (s)
			{
				if (al_xml_verify_from_string(al_string_get(s)))
				{
					syslog(LOG_ERR, "Invalid xml packet read from %s.\n", agents_request_get_server_name(request));
					agents_response_set_type(response, AGENTS_RESPONSE_INVALID_SERVER_DATA);
				}
				else
				{
					agents_response_set_server_data(response, al_string_get(s));
				}
			}
			al_string_delete(s);

			al_connection_delete(connection);
		}
		else
		{
			syslog(LOG_ERR, "Could not connect to %s\n", agents_request_get_server_name(request));
			agents_response_set_type(response, AGENTS_RESPONSE_SERVER_NOT_FOUND);
		}

	}


	return response;
}


/**
 */
AgentsResponsePacket * agents_response_packet_from_request_packet(const AgentsRequestPacket * request_packet, AlConnectionManager * connection_manager)
{
	AgentsRequest        * request;
	AgentsResponse       * response;
	AgentsResponsePacket * response_packet;


	al_return_val_if_fail (request_packet && connection_manager, NULL);


	response_packet = agents_response_packet_new();

	request = request_packet->requests;
	while (request)
	{
		response = agents_response_from_request(request, connection_manager);
		if (response)
			agents_response_packet_response_add(response_packet, response);

		request = request->next;
	}

	return response_packet;
}


/**
 */
AlXmlList * agents_response_packet_to_xml(const AgentsResponsePacket * response_packet)
{
	AlXmlList      * list = al_xml_list_new();
	AgentsResponse * response;


	al_return_val_if_fail (response_packet, NULL);

	/*
	 * <agents>
	 *	<responses>
	 *		<response id=”00000000” type=”type” server=”server”>
	 *              	<![CDATA[<agentd...]]> 
	 *              </response>
	 *	</responses>
	 * </agents>
	 */

	list->parent = al_xml_node_new();
	al_xml_node_set_name(list->parent, "agents");

	list->node = al_xml_node_new();
	al_xml_node_set_name(list->node, "responses");
	for (response = response_packet->responses;response;response = response->next)
	{

		al_xml_list_node_add(list, agents_response_to_xml(response));

	}

	return list;
}


/**
 */
AlXmlNode * agents_response_to_xml(const AgentsResponse * response)
{
	AlXmlNode * node;
	char        stype[10];


	al_return_val_if_fail (response, NULL);

	node = al_xml_node_new();
	al_xml_node_set_name(node, "response");

	al_xml_node_attribute_new(node, "id", agents_response_get_id(response));
	al_xml_node_attribute_new(node, "server", agents_response_get_server_name(response));

	if (agents_response_get_type(response) == AGENTS_RESPONSE_OK)
	{
		// Return agent data only if it is ok
		al_xml_node_set_cdata(node, agents_response_get_server_data(response));
	}

	sprintf(stype,"%d",agents_response_get_type(response));
	al_xml_node_attribute_new(node, "type", stype);

	return node;
}


/**
 */
AlString * agents_response_packet_to_xml_string(const AgentsResponsePacket * response_packet)
{
	AlXmlList * list;
	AlString  * s = NULL;


	al_return_val_if_fail(response_packet, NULL);

	list = agents_response_packet_to_xml(response_packet);
	if (list) s = al_xml_list_to_string(list, "../dtd/agents-response.dtd", "agents-response");


	return s;
}


/**
 */
AgentsResponsePacket * agents_response_packet_from_xml_string(const AlString * xml)
{
	AlXml                * x = al_xml_new_from_string(al_string_get(xml));
	AgentsResponsePacket * response_packet;


	response_packet = agents_response_packet_from_xml(x);
	al_xml_delete(x);

	return response_packet;
}


/**
 */
AgentsResponsePacket * agents_response_packet_from_xml(AlXml * xml)
{
	AgentsResponsePacket * response_packet;
	AgentsResponse       * response;
	AlXmlList            * list;
	AlXmlNode            * node;


	al_return_val_if_fail(xml, NULL);

	al_log(("agents_response_packet_from_xml:\n%s\n", xml->string));

	response_packet = agents_response_packet_new();
	list = al_xml_get_list(xml, "responses");
	if (list)
	{
		for (node = list->nodes; node; node = node->next)
		{

			response = agents_response_from_xml(node);

			if (response) agents_response_packet_response_add(response_packet, response);

		}
	}
	else
	{
		agents_response_packet_delete(response_packet);
		response_packet = NULL;
	}


	return response_packet;
}


/**
 */
AgentsResponse * agents_response_from_xml(AlXmlNode * node)
{
	AgentsResponse * ret;


	al_return_val_if_fail (node, NULL);

	ret = agents_response_new();

	agents_response_set_id(ret, al_xml_get_attribute_value(node, "id"));
	agents_response_set_server_data(ret, al_xml_get_attribute_value(node, "data"));

	return ret;
}
