/**
 * WebAdmin
 * Copyright (C) 2006 Netwosix Team
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License 
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 *
 * Linking WebAdmin statically or dynamically with other modules is making
 * a combined work based on WebAdmin. Thus, the terms and conditions of the 
 * GNU General Public License cover the whole combination.
 *
 * In addition, as a special exception, the copyright holders of
 * WebAdmin give you permission to combine WebAdmin with free software
 * programs or libraries that are released under the GNU LGPL and with
 * code included in the standard release of OpenSSL under the OpenSSL
 * License and SSLeay License (or modified versions of such code, with
 * unchanged licenses). You may copy and distribute such a system
 * following the terms of the GNU GPL for WebAdmin and the licenses of
 * the other code concerned, provided that you include the source code of
 * that other code when and as the GNU GPL requires distribution of
 * source code.
 *
 * Note that people who make modified versions of WebAdmin are not obligated
 * to grant this special exception for their modified versions; 
 * it is their choice whether to do so. The GNU General Public License 
 * gives permission to release a modified version without this exception; 
 * this exception also makes it possible to release a modified 
 * version which carries forward this exception.
 */

#include "agent.h"



/**
 */
AgentResponse * agent_response_new(void)
{
	AgentResponse * response = al_new(AgentResponse);

	response->next = response->prev = NULL;

	response->id = NULL;
	response->type = AGENT_RESPONSE_OK;
	response->agent_data = NULL;

	return response;
}


/**
 */
void agent_response_delete(AgentResponse * response)
{

	al_return_if_fail (response);

	al_delete(response->id);
	al_delete(response->agent_data);
	al_delete(response);

}


/**
 */
AgentResponsePacket * agent_response_packet_new(void)
{
	AgentResponsePacket * response_packet = al_new(AgentResponsePacket);

	response_packet->responses = NULL;

	return response_packet;
}


/**
 */
void agent_response_packet_delete(AgentResponsePacket * response_packet)
{

	al_return_if_fail (response_packet);

	if (response_packet->responses)
	{
		AgentResponse * response = response_packet->responses;
		AgentResponse * next;

		while (response)
		{
			next = response->next;
			agent_response_delete(response);
			response = next;
		}
	}
	
	al_delete(response_packet);

}


/**
 */
void agent_response_packet_response_add(AgentResponsePacket * response_packet, AgentResponse * response)
{

	al_return_if_fail (response_packet && response);

	if (response_packet->responses == NULL)
	{
		response_packet->responses = response;
		return;
	}


	response->next = response_packet->responses->next;
	response->prev = response_packet->responses;

	response_packet->responses->next = response;

}


/**
 */
AgentResponse * agent_response_from_request(const AgentRequest * request, agent_response_create_func func)
{
	AgentResponse * response;
	AlString      * data = NULL;



	al_return_if_fail (request && func);

	response = agent_response_new();
	agent_response_set_id(response, agent_request_get_id(request));

	if (al_xml_verify_from_string(agent_request_get_agent_data(request)) && agent_request_get_agent_data(request) != NULL)
	{
		al_log (("Invalid request agent-data\n"));
		agent_response_set_type(response, AGENT_RESPONSE_AGENT_INVALID_DATA);
	}
	else
	{
		al_log (("processing request id    : %s\n", agent_request_get_id(request)));
		al_log (("processing agent data    : %s\n", agent_request_get_agent_data(request)));

		data = (*func)(request, agent_request_get_agent_data(request));

		if (al_xml_verify_from_string(al_string_get(data)))
		{
			al_log (("Invalid xml response agent-data\n"));
			response->type = AGENT_RESPONSE_AGENT_INVALID_DATA;
		}
		else
		{
			agent_response_set_agent_data(response,al_string_get(data));
		}
	}


	al_string_delete(data);


	return response;
}


/**
 */
AgentResponsePacket * agent_response_packet_from_request_packet(const AgentRequestPacket * request_packet, agent_response_create_func func)
{
	AgentRequest        * request;
	AgentResponse       * response;
	AgentResponsePacket * response_packet;


	al_return_val_if_fail (request_packet, NULL);


	response_packet = agent_response_packet_new();

	request = request_packet->requests;
	while (request)
	{
		response = agent_response_from_request(request, func);
		if (response)
			agent_response_packet_response_add(response_packet, response);

		request = request->next;
	}

	return response_packet;
}


/**
 */
AlXmlList * agent_response_packet_to_xml(const AgentResponsePacket * response_packet)
{
	AlXmlList     * list = al_xml_list_new();
	AgentResponse * response;


	al_return_val_if_fail (response_packet, NULL);

	/*
	 * <agent>
	 *	<responses>
	 *		<response id=”00000000” type=”type”data=”data”/>
	 *	</responses>
	 * </agent>
	 */

	list->parent = al_xml_node_new();
	al_xml_node_set_name(list->parent, "agent");

	list->node = al_xml_node_new();
	al_xml_node_set_name(list->node, "responses");
	for (response = response_packet->responses;response;response = response->next)
	{

		al_xml_list_node_add(list, agent_response_to_xml(response));

	}

	return list;
}


/**
 */
AlXmlNode * agent_response_to_xml(const AgentResponse * response)
{
	AlXmlNode * node;
	char        stype[10];


	al_return_val_if_fail (response, NULL);

	node = al_xml_node_new();
	al_xml_node_set_name(node, "response");

	al_xml_node_attribute_new(node, "id", agent_response_get_id(response));

	if (agent_response_get_type(response) == AGENT_RESPONSE_OK)
	{
		// Return agent data only if it is ok
		al_xml_node_attribute_new(node, "data", agent_response_get_agent_data(response));
	}

	sprintf(stype,"%d",agent_response_get_type(response));
	al_xml_node_attribute_new(node, "type", stype);

	return node;
}


/**
 */
AlString * agent_response_packet_to_xml_string(const AgentResponsePacket * response_packet)
{
	AlXmlList * list;
	AlString  * s = NULL;


	al_return_val_if_fail(response_packet, NULL);

	list = agent_response_packet_to_xml(response_packet);
	if (list) s = al_xml_list_to_string(list, "../dtd/agent-response.dtd", "agent-response");


	return s;
}


/**
 */
AgentResponsePacket * agent_response_packet_from_xml_string(const AlString * xml)
{
	AlXml               * x = al_xml_new_from_string(al_string_get(xml));
	AgentResponsePacket * response_packet;


	response_packet = agent_response_packet_from_xml(x);
	al_xml_delete(x);

	return response_packet;
}


/**
 */
AgentResponsePacket * agent_response_packet_from_xml(AlXml * xml)
{
	AgentResponsePacket * response_packet;
	AgentResponse       * response;
	AlXmlList           * list;
	AlXmlNode           * node;


	al_return_val_if_fail(xml, NULL);

	al_log(("agent_response_packet_from_xml:\n%s\n", xml->string));

	response_packet = agent_response_packet_new();
	list = al_xml_get_list(xml, "responses");
	if (list)
	{
		for (node = list->nodes; node; node = node->next)
		{

			response = agent_response_from_xml(node);

			if (response) agent_response_packet_response_add(response_packet, response);

		}
	}
	else
	{
		agent_response_packet_delete(response_packet);
		response_packet = NULL;
	}


	return response_packet;
}


/**
 */
AgentResponse * agent_response_from_xml(AlXmlNode * node)
{
	AgentResponse * ret;


	al_return_val_if_fail (node, NULL);

	ret = agent_response_new();

	agent_response_set_id(ret, al_xml_get_attribute_value(node, "id"));
	agent_response_set_agent_data(ret, al_xml_get_attribute_value(node, "data"));

	return ret;
}


