/**
 * WebAdmin
 * Copyright (C) 2006 Netwosix Team
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License 
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 *
 * Linking WebAdmin statically or dynamically with other modules is making
 * a combined work based on WebAdmin. Thus, the terms and conditions of the 
 * GNU General Public License cover the whole combination.
 *
 * In addition, as a special exception, the copyright holders of
 * WebAdmin give you permission to combine WebAdmin with free software
 * programs or libraries that are released under the GNU LGPL and with
 * code included in the standard release of OpenSSL under the OpenSSL
 * License and SSLeay License (or modified versions of such code, with
 * unchanged licenses). You may copy and distribute such a system
 * following the terms of the GNU GPL for WebAdmin and the licenses of
 * the other code concerned, provided that you include the source code of
 * that other code when and as the GNU GPL requires distribution of
 * source code.
 *
 * Note that people who make modified versions of WebAdmin are not obligated
 * to grant this special exception for their modified versions; 
 * it is their choice whether to do so. The GNU General Public License 
 * gives permission to release a modified version without this exception; 
 * this exception also makes it possible to release a modified 
 * version which carries forward this exception.
 */

#include "al.h"

#include <libxml/tree.h>
#include <libxml/parser.h>


/**
 */
xmlNode * al_xml_find_node(const char * name, xmlNode * a_node);


/**
 */
xmlNode * al_xml_get_root(AlXml * xml);


/**
 */
void al_xml_search_cdata(xmlNode * cur_node, AlXmlNode * node);


/**
 */
AlXml * al_xml_new(void)
{
	AlXml * ret = al_new(AlXml);


	ret->string = ret->file = ret->parse = NULL;

	return ret;
}


/**
 */
AlXml * al_xml_new_from_string(const char * s)
{
	AlXml * ret = al_xml_new();


	if (s)
	{
		ret->string = al_strdup(s);
		ret->parse = al_strdup(s);
	}

	return ret;
}


/**
 */
AlXml * al_xml_new_from_file(const char * s)
{
	AlXml * ret = al_xml_new();


	if (s) ret->file = al_strdup(s);
	ret->parse = NULL;

	return ret;
}


/**
 */
void al_xml_delete(AlXml * xml)
{
	al_return_if_fail(xml);

	al_delete(xml->file);
	al_delete(xml->string);
	al_delete(xml);

}


/**
 */
AlXmlList * al_xml_get_list(AlXml * xml, const char * name)
{
	xmlNode            * root_element;
	xmlNode            * node;
	AlXmlList      * list = NULL;
	xmlNode            * cur_node;
	xmlAttr            * cur_attr;
	AlXmlNode      * nod;
	AlXmlAttribute * atr;


	al_return_val_if_fail(xml && name, NULL);

	root_element = al_xml_get_root(xml);
	if (root_element)
	{
		node = al_xml_find_node(name, root_element);
		if (node)
		{
			for (cur_node = node->children; cur_node; cur_node = cur_node->next)
			{
				if (cur_node->type == XML_ELEMENT_NODE)
				{
					nod = al_xml_node_new();

					al_xml_node_set_name(nod, cur_node->name);
					al_xml_search_cdata(cur_node, nod);

					if (list == NULL) 
					{
						list = al_xml_list_new();
						list->node = al_xml_node_new();
						al_xml_node_set_name(list->node, name);
						for(cur_attr = node->properties; cur_attr; cur_attr = cur_attr->next)
						{
							atr = al_xml_attribute_new();

							al_xml_attribute_set_name(atr, cur_attr->name);
							al_xml_attribute_set_value(atr, cur_attr->children->content);

							al_xml_node_attribute_add(list->node, atr);
						}
					}

					al_xml_list_node_add(list, nod);
	
					for(cur_attr = cur_node->properties; cur_attr; cur_attr = cur_attr->next)
					{
						atr = al_xml_attribute_new();

						al_xml_attribute_set_name(atr, cur_attr->name);
						al_xml_attribute_set_value(atr, cur_attr->children->content);

						al_xml_node_attribute_add(nod, atr);
					}
				}
			}
		}
	}

	return list;
}


/**
 */
void al_xml_search_cdata(xmlNode * cur_node, AlXmlNode * node)
{
	
	for (cur_node = cur_node->children; cur_node; cur_node = cur_node->next)
	{
		if (cur_node->type == XML_CDATA_SECTION_NODE)
		{
			al_xml_node_set_cdata(node, cur_node->content);
			return;
		}
	}

}


/**
 */
AlXmlNode * al_xml_get_node_in_list(AlXml * xml, const char * list_name, const char * node_name, const char * attr_name, const char * attr_value)
{
	xmlNode            * root_element;
	xmlNode            * node;
	xmlNode            * cur_node;
	xmlAttr            * cur_attr = NULL;
	AlXmlNode      * ret = NULL;
	AlXmlAttribute * atr;


	al_return_val_if_fail(xml && list_name && node_name && attr_name && attr_value, NULL);

	root_element = al_xml_get_root(xml);
	if (root_element)
	{
		node = al_xml_find_node(list_name, root_element);
		if (node)
		{
			for (cur_node = node->children; cur_node; cur_node = cur_node->next)
			{
				if (cur_node->type == XML_ELEMENT_NODE && !strcmp(cur_node->name,node_name))
				{
					for(cur_attr = cur_node->properties; cur_attr; cur_attr = cur_attr->next)
					{
						if (!strcmp(cur_attr->name,attr_name) && !strcmp(cur_attr->children->content,attr_value))
						{
							ret = al_xml_node_new();
							al_xml_node_set_name(ret, cur_node->name);
							al_xml_search_cdata(cur_node, ret);

							for(cur_attr = cur_node->properties; cur_attr; cur_attr = cur_attr->next)
							{
								atr = al_xml_attribute_new();
								al_xml_attribute_set_name(atr, cur_attr->name);
								al_xml_attribute_set_value(atr, cur_attr->children->content);

								al_xml_node_attribute_add(ret, atr);
							}

							return ret;
						}
					}
				}
			}
		}
	}


	return NULL;
}


/**
 */
AlXmlNode * al_xml_get_node(AlXml * xml, const char * name)
{
	xmlNode            * root_element;
	xmlNode            * node;
	xmlAttr            * cur_attr = NULL;
	AlXmlNode      * ret = NULL;
	AlXmlAttribute * atr;


	al_return_val_if_fail(xml && name, NULL);

	root_element = al_xml_get_root(xml);

	if (root_element)
	{
		node = al_xml_find_node(name, root_element);
		if (node)
		{
			ret = al_xml_node_new();
			al_xml_node_set_name(ret, node->name);
			al_xml_search_cdata(node, ret);

			for(cur_attr = node->properties; cur_attr; cur_attr = cur_attr->next)
			{
				atr = al_xml_attribute_new();
				al_xml_attribute_set_name(atr, cur_attr->name);
				al_xml_attribute_set_value(atr, cur_attr->children->content);

				al_xml_node_attribute_add(ret, atr);
			}

			return ret;
		}
	}


	return NULL;
}


/**
 */
char * al_xml_get_attribute_value(const AlXmlNode * node, const char * attr_name)
{
	AlXmlAttribute * attr = NULL;


	al_return_val_if_fail(node && attr_name, NULL);

	
	for (attr=node->attributes;attr;attr=attr->next)
	{
		if (!strcmp(attr->name,attr_name))
			return attr->value;
	}

	return NULL;
}


/**
 */
xmlNode * al_xml_find_node(const char * name, xmlNode * a_node)
{
	xmlNode * cur_node = NULL;
	xmlNode * ret = NULL;



	al_return_val_if_fail(name && a_node, NULL);


	for (cur_node = a_node; cur_node; cur_node = cur_node->next)
	{
		if (cur_node->type == XML_ELEMENT_NODE && !strcmp(cur_node->name,name))
			return cur_node;

		if ((ret = al_xml_find_node(name, cur_node->children)))
			return ret;
	}


	return NULL;
}


/**
 */
int al_xml_verify(AlXml * xml)
{
	int status;

	al_return_val_if_fail(xml && (xml->string || xml->file), 1);

	if (al_xml_get_root(xml) == NULL) return 2;

	if (al_spawn(AL_XML_VERIFY, AL_XML_VERIFY_USER, xml->parse, NULL, &status))
		return 3;

	return status;
}


/**
 */
int al_xml_verify_from_string(const char * xml)
{
	AlXml * x;
	int     ret = 1;

	al_return_val_if_fail(xml, 40);	

	x = al_xml_new_from_string(xml);
	ret = al_xml_verify(x);
	al_xml_delete(x);

	return ret;
}


/**
 */
xmlNode * al_xml_get_root(AlXml * xml)
{
	xmlDoc       * doc = NULL;
	xmlNode      * root = NULL;
	AlString * s;


	al_return_val_if_fail(xml, NULL);

	if (xml->file)
	{
		al_delete(xml->parse);

		if ((s = al_read_file_by_name(xml->file)) == NULL) return NULL;

		xml->parse = al_strdup(al_string_get(s));
		al_string_delete(s);
	}

	/* parse the file and get the DOM */
	doc = xmlReadMemory(xml->parse,strlen(xml->parse), NULL, NULL, 0);
	if (doc == NULL) return NULL;

	/* Get the root element node */
	root = xmlDocGetRootElement(doc);

	return root;
}


/**
 */
AlXmlAttribute * al_xml_attribute_new(void)
{
	AlXmlAttribute * ret = al_new(AlXmlAttribute);

	ret->prev = ret->next = NULL;
	ret->name = NULL;
	ret->value = NULL;

	return ret;
}


/**
 */
void al_xml_attribute_delete(AlXmlAttribute * attr)
{

	al_return_if_fail (attr);

	al_delete(attr->name);
	al_delete(attr->value);
	al_delete(attr);

}


/**
 */
AlXmlNode * al_xml_node_new(void)
{
	AlXmlNode * ret = al_new(AlXmlNode);

	ret->prev = ret->next = NULL;
	ret->name = NULL;
	ret->attributes = NULL;
	ret->cdata = NULL;

	return ret;
}


/**
 */
void al_xml_node_delete(AlXmlNode * node)
{

	al_return_if_fail (node);

	if (node->attributes)
	{
		AlXmlAttribute * attrs = node->attributes;
		AlXmlAttribute * next;

		while (attrs)
		{
			next = attrs->next;
			al_xml_attribute_delete(attrs);
			attrs = next;
		}
	}
	
	al_delete(node->name);
	al_delete(node->cdata);
	al_delete(node);

}


/**
 */
void al_xml_node_attribute_add(AlXmlNode * node, AlXmlAttribute * attr)
{
	al_return_if_fail(node && attr);


	if (node->attributes == NULL)
	{
		node->attributes = attr;
		return;
	}


	attr->next = node->attributes->next;
	attr->prev = node->attributes;

	node->attributes->next = attr;

}


/**
 */
AlXmlList * al_xml_list_new(void)
{
	AlXmlList * ret = al_new(AlXmlList);

	ret->nodes = NULL;
	ret->node = NULL;
	ret->parent = NULL;

	return ret;
}


/**
 */
void al_xml_list_delete(AlXmlList * list)
{

	al_return_if_fail (list);

	if (list->nodes)
	{
		AlXmlNode * nodes = list->nodes;
		AlXmlNode * next;

		while (nodes)
		{
			next = nodes->next;
			al_xml_node_delete(nodes);
			nodes = next;
		}

		if (list->node) al_xml_node_delete(list->node);
		if (list->parent) al_xml_node_delete(list->parent);

	}
	
	al_delete(list);

}


/**
 */
void al_xml_list_node_add(AlXmlList * list, AlXmlNode * node)
{
	al_return_if_fail(list && node);


	if (list->nodes == NULL)
	{
		list->nodes = node;
		return;
	}


	node->next = list->nodes->next;
	node->prev = list->nodes;

	list->nodes->next = node;

}


/**
 */
AlString * al_xml_list_to_string(AlXmlList * list, const char * dtd_file, const char * doctype)
{
	AlXmlNode      * d_node;
	AlXmlAttribute * d_attr;
	xmlDocPtr        doc = NULL;
	xmlNodePtr       root_node = NULL, list_node = NULL, node = NULL;
	xmlDtdPtr        dtd = NULL;
	AlString       * ret = NULL;
	xmlChar        * mem;
	int              size;


	al_return_val_if_fail(list, NULL);


	/*
	 * Creates a new document, a node and set it as a root node
	 */
	doc = xmlNewDoc(BAD_CAST "1.0");

	if (list->parent)
	{
		root_node = xmlNewNode(NULL, al_xml_node_get_name(list->parent));
		xmlDocSetRootElement(doc, root_node);
	}
	else
	{
		root_node = xmlNewNode(NULL, al_xml_node_get_name(list->node));
		xmlDocSetRootElement(doc, root_node);
	}


	/*
	 * Creates a DTD declaration. Isn't mandatory. 
	 */
	if (dtd_file)
		dtd = xmlCreateIntSubset(doc, BAD_CAST doctype, NULL, BAD_CAST dtd_file);
	
	/*
	 * xmlNewChild() creates a new node, which is "attached" as child node
	 * of root_node node. 
	 */
	list_node = (list->parent) ? xmlNewChild(root_node, NULL, list->node->name, NULL) : root_node;
	for (d_node = list->nodes;d_node;d_node=d_node->next)
	{
		node = xmlNewChild(list_node, NULL, al_xml_node_get_name(d_node), NULL);
		if (al_xml_node_get_cdata(d_node))
		{
			// Add CDATA section
			xmlNodePtr cdata_node = xmlNewChild(node, NULL, "dummy", NULL);
			cdata_node->type = XML_CDATA_SECTION_NODE;
			cdata_node->content = al_strdup(al_xml_node_get_cdata(d_node));
		}

		for (d_attr = d_node->attributes;d_attr;d_attr=d_attr->next)
		{
			xmlNewProp(node, al_xml_attribute_get_name(d_attr), al_xml_attribute_get_value(d_attr));
		}
	}


	xmlDocDumpMemory(doc, &mem, &size);
	ret = al_string_new();
	al_string_set(ret, mem);

	/* free the document */
	xmlFreeDoc(doc);

	/*
	 * Free the global variables that may
	 * have been allocated by the parser.
	 */
	xmlCleanupParser();

	return ret;
}


/**
 */
void al_xml_node_attribute_new(AlXmlNode * node, const char * name, const char * value)
{
	AlXmlAttribute * attribute;


	al_return_if_fail(node && name && value);


	attribute = al_xml_attribute_new();
	al_xml_attribute_set_name(attribute, name);
	al_xml_attribute_set_value(attribute, value);

	al_xml_node_attribute_add(node, attribute);

}
