/**
 * WebAdmin
 * Copyright (C) 2006 Netwosix Team
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License 
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 *
 * Linking WebAdmin statically or dynamically with other modules is making
 * a combined work based on WebAdmin. Thus, the terms and conditions of the 
 * GNU General Public License cover the whole combination.
 *
 * In addition, as a special exception, the copyright holders of
 * WebAdmin give you permission to combine WebAdmin with free software
 * programs or libraries that are released under the GNU LGPL and with
 * code included in the standard release of OpenSSL under the OpenSSL
 * License and SSLeay License (or modified versions of such code, with
 * unchanged licenses). You may copy and distribute such a system
 * following the terms of the GNU GPL for WebAdmin and the licenses of
 * the other code concerned, provided that you include the source code of
 * that other code when and as the GNU GPL requires distribution of
 * source code.
 *
 * Note that people who make modified versions of WebAdmin are not obligated
 * to grant this special exception for their modified versions; 
 * it is their choice whether to do so. The GNU General Public License 
 * gives permission to release a modified version without this exception; 
 * this exception also makes it possible to release a modified 
 * version which carries forward this exception.
 */

#include "al.h"



/**
 */
AlConnectionManager * al_connection_manager_new(al_connection_manager_wait_func wait,
						al_connection_manager_delete_func del,
						al_connection_manager_connect_func conn, 
						int fork, void * data)
{
	AlConnectionManager * connection = al_new(AlConnectionManager);


	connection->wait = wait;
	connection->del = del;
	connection->conn = conn;

	connection->data = data;
	connection->fork = fork;

	return connection;

}


/**
 */
void al_connection_manager_delete(AlConnectionManager * connection_manager)
{

	al_return_if_fail(connection_manager);

	if (connection_manager->del) (connection_manager->del)(connection_manager);

	al_delete(connection_manager);
}


/**
 */
AlConnection * al_connection_new(AlConnectionManager * connection_manager, 
				al_connection_read_func read, 
				al_connection_send_func send,
				al_connection_close_func close, 
				al_connection_delete_func del,
				void * data)
{
	AlConnection * connection;


	al_return_val_if_fail(connection_manager, NULL);


	connection = al_new(AlConnection);
	
	connection->read = read;
	connection->send = send;
	connection->close = close;
	connection->del = del;

	connection->data = data;
	connection->connection_manager = connection_manager;

	return connection;

}


/**
 */
void al_connection_delete(AlConnection * connection)
{
	al_return_if_fail(connection);

	if (connection->del) (connection->del)(connection);
		
	al_delete(connection);
}


/**
 */
AlConnectionManager * al_connection_manager_from_string(const char * connection_type, const char * supported[], AlXmlNode * node)
{
	AlConnectionManager * ret = NULL;
	int                   i;
	int                   port;
	char                * type = NULL;
	char                * s_port = NULL;


	type = al_strdup(connection_type);

	if (supported && type)
	{
		for (i=0;supported[i];i++)
		{
			if (!strcmp(type, supported[i]))
				break;
		}

		if (supported[i] == NULL)
			return ret;
	}

	if (type == NULL && node)
		type = al_xml_get_attribute_value(node,"type");

	if (type == NULL)
		return ret;

	if (!strcmp(type,"stdio"))
	{
		if ((ret = al_stdio_connection_manager_new()) == NULL)
			al_log (("Could not open %s connection.\n", type));
	}
	else if (!strcmp(type,"tcp"))
	{
		if (!(s_port = al_xml_get_attribute_value(node,"port")))
			al_log (("Invalid tcp port %d.\n", port));
		else
		{
			port = atoi(s_port);

			if (port <= 0 || port >= 0xFFFF)
				al_log(("Invalid tcp port %d.\n", port));
			else
				if ((ret = al_socket_connection_manager_new(port)) == NULL)
					al_log (("Could not open %s connection.\n", type));
		}
	}
#ifdef AL_SSL
	else if (!strcmp(type,"ssl"))
	{
		char * key_file;
		char * cert_file;
		char * dh_file;
		char * password;
		char * ca_list_file;
		char * s_no_peer_verify;
		int    no_peer_verify = 0;

		
		if (!(key_file = al_xml_get_attribute_value(node,"key_file")))
			al_log(("No key_file for SSL.\n"));
		
		if (!(cert_file = al_xml_get_attribute_value(node,"cert_file")))
			al_log(("No cert_file for SSL.\n"));
		
		if (!(dh_file = al_xml_get_attribute_value(node,"dh_file")))
			al_log(("No dh file for SSL.\n"));

		if (!(password = al_xml_get_attribute_value(node,"password")))
			al_log(("No password file for SSL.\n"));

		if (!(ca_list_file = al_xml_get_attribute_value(node,"ca_list_file")))
			al_log(("No ca_list file for SSL.\n"));

		if ((s_no_peer_verify = al_xml_get_attribute_value(node,"no_peer_verify")))
			no_peer_verify = !strcmp(s_no_peer_verify,"true");

		if (!(s_port = al_xml_get_attribute_value(node,"port")))
		{
			al_log (("Invalid tcp port %d.\n", port));
		}
		else
		{
			port = atoi(s_port);

			if (port <= 0 || port >= 0xFFFF)
				al_log (("Invalid tcp port %d.\n", port));
			else
				if ((ret = al_ssl_connection_manager_new(port, key_file, cert_file, dh_file, ca_list_file, password, no_peer_verify)) == NULL)
					al_log (("Could not open %s connection.\n", type));
		}

		al_delete(key_file);
		al_delete(cert_file);
		al_delete(dh_file);
		al_delete(password);
		al_delete(ca_list_file);
	}
#endif
	else
	{
		al_log (("No suitable connection found.\n"));
	}


	return ret;
}


/**
 */
void al_connection_supported_list(const char * type, const char * version, const char * supported[])
{
	int i;

	if (type && !strcmp(type,"list"))
	{
		if (version) printf("%s\n", version);

		printf("Supported connection:\n\n");

		if (supported)
		{
			for(i=0;supported[i];i++)
				printf("  %s\n", supported[i]);
		}

		al_exit(0, NULL);
	}

}
