/**
 * WebAdmin
 * Copyright (C) 2006 Netwosix Team
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License 
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 *
 * Linking WebAdmin statically or dynamically with other modules is making
 * a combined work based on WebAdmin. Thus, the terms and conditions of the 
 * GNU General Public License cover the whole combination.
 *
 * In addition, as a special exception, the copyright holders of
 * WebAdmin give you permission to combine WebAdmin with free software
 * programs or libraries that are released under the GNU LGPL and with
 * code included in the standard release of OpenSSL under the OpenSSL
 * License and SSLeay License (or modified versions of such code, with
 * unchanged licenses). You may copy and distribute such a system
 * following the terms of the GNU GPL for WebAdmin and the licenses of
 * the other code concerned, provided that you include the source code of
 * that other code when and as the GNU GPL requires distribution of
 * source code.
 *
 * Note that people who make modified versions of WebAdmin are not obligated
 * to grant this special exception for their modified versions; 
 * it is their choice whether to do so. The GNU General Public License 
 * gives permission to release a modified version without this exception; 
 * this exception also makes it possible to release a modified 
 * version which carries forward this exception.
 */

#include "agentd.h"



/**
 */
AgentdResponse * agentd_response_new(void)
{
	AgentdResponse * response = al_new(AgentdResponse);

	response->next = response->prev = NULL;

	response->id = NULL;
	response->type = AGENTD_RESPONSE_OK;
	response->agent_name = NULL;
	response->agent_data = NULL;

	return response;
}


/**
 */
void agentd_response_delete(AgentdResponse * response)
{

	al_return_if_fail (response);

	al_delete(response->id);
	al_delete(response->agent_name);
	al_delete(response->agent_data);
	al_delete(response);

}


/**
 */
AgentdResponsePacket * agentd_response_packet_new(void)
{
	AgentdResponsePacket * response_packet = al_new(AgentdResponsePacket);

	response_packet->responses = NULL;

	return response_packet;
}


/**
 */
void agentd_response_packet_delete(AgentdResponsePacket * response_packet)
{

	al_return_if_fail (response_packet);

	if (response_packet->responses)
	{
		AgentdResponse * response = response_packet->responses;
		AgentdResponse * next;

		while (response)
		{
			next = response->next;
			agentd_response_delete(response);
			response = next;
		}
	}
	
	al_delete(response_packet);

}


/**
 */
void agentd_response_packet_response_add(AgentdResponsePacket * response_packet, AgentdResponse * response)
{

	al_return_if_fail (response_packet && response);

	if (response_packet->responses == NULL)
	{
		response_packet->responses = response;
		return;
	}


	response->next = response_packet->responses->next;
	response->prev = response_packet->responses;

	response_packet->responses->next = response;

}

	
/**
 */
AgentdResponse * agentd_response_from_request(const AgentdRequest * request)
{
	AgentdResponse * response;
	AgentdAgent    * agent;


	al_return_if_fail (request);

	response = agentd_response_new();
	agentd_response_set_id(response, agentd_request_get_id(request));
	agentd_response_set_agent_name(response, agentd_request_get_agent_name(request));

	if (al_xml_verify_from_string(agentd_request_get_agent_data(request)))
	{
		al_log (("Invalid request agent-data\n"));
		agentd_response_set_type(response, AGENTD_RESPONSE_INVALID_AGENT_DATA);
	}
	else
	{
		al_log (("processing request id    : %s\n", agentd_request_get_id(request)));
		al_log (("processing agent         : %s\n", agentd_request_get_agent_name(request)));
		al_log (("processing agent data    : %s\n", agentd_request_get_agent_data(request)));

		agent = agentd_agent_find(request);
		if (agent)
		{
			al_log (("agent : %s\n",agent->name));
			al_log (("bin   : %s\n",agent->bin));
			al_log (("data  : %s\n",agentd_request_get_agent_data(request)));
		
			agentd_agent_execute(agent, request, response);
		}
		else
		{
			response->type = AGENTD_RESPONSE_AGENT_NOT_FOUND;
		}
	}


	return response;
}


/**
 */
AgentdResponsePacket * agentd_response_packet_from_request_packet(const AgentdRequestPacket * request_packet)
{
	AgentdRequest        * request;
	AgentdResponse       * response;
	AgentdResponsePacket * response_packet;


	al_return_val_if_fail (request_packet, NULL);


	response_packet = agentd_response_packet_new();

	request = request_packet->requests;
	while (request)
	{
		response = agentd_response_from_request(request);
		if (response)
			agentd_response_packet_response_add(response_packet, response);

		request = request->next;
	}

	return response_packet;
}


/**
 */
AlXmlList * agentd_response_packet_to_xml(const AgentdResponsePacket * response_packet)
{
	AlXmlList      * list = al_xml_list_new();
	AgentdResponse * response;


	al_return_val_if_fail (response_packet, NULL);

	/*
	 * <agentd>
	 *	<responses>
	 *		<response id=”00000000” type=”type” agent=”agent” data=”data”/>
	 *	</responses>
	 * </agentd>
	 */

	list->parent = al_xml_node_new();
	al_xml_node_set_name(list->parent, "agentd");

	list->node = al_xml_node_new();
	al_xml_node_set_name(list->node, "responses");
	for (response = response_packet->responses;response;response = response->next)
	{

		al_xml_list_node_add(list, agentd_response_to_xml(response));

	}

	return list;
}


/**
 */
AlXmlNode * agentd_response_to_xml(const AgentdResponse * response)
{
	AlXmlNode * node;
	char        stype[10];


	al_return_val_if_fail (response, NULL);

	node = al_xml_node_new();
	al_xml_node_set_name(node, "response");

	al_xml_node_attribute_new(node, "id", agentd_response_get_id(response));
	al_xml_node_attribute_new(node, "agent", agentd_response_get_agent_name(response));

	if (agentd_response_get_type(response) == AGENTD_RESPONSE_OK)
	{
		// Return agent data only if it is ok
		al_xml_node_attribute_new(node, "data", agentd_response_get_agent_data(response));
	}

	sprintf(stype,"%d",agentd_response_get_type(response));
	al_xml_node_attribute_new(node, "type", stype);

	return node;
}


/**
 */
AlString * agentd_response_packet_to_xml_string(const AgentdResponsePacket * response_packet)
{
	AlXmlList * list;
	AlString  * s = NULL;


	al_return_val_if_fail(response_packet, NULL);

	list = agentd_response_packet_to_xml(response_packet);
	if (list) s = al_xml_list_to_string(list, "../dtd/agentd-response.dtd", "agentd-response");


	return s;
}

