/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2011 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gee;
using Tasklets;

namespace zcd
{
    /** A UDPRequestClient to send a message to a particular NIP.
      * In turn, it is a FakeRmt object.
      */
    public class NeighbourClient : UDPRequestClient
    {
        private ISerializable unicast_id;
        private string[] devs;
        private static int next_id = 0;
        private int my_id;
        public NeighbourClient(ISerializable unicast_id, string[] devs, uint16? port=null,  bool wait_response=true)
        {
            base(port, wait_response);
            my_id = next_id++;
            this.unicast_id = unicast_id;
            this.devs = devs;
        }

        public override string[] get_devs()
        {
            return devs;
        }

        public override UDPPayload get_udp_payload(RemoteCall data)
        {
            return new UDPPayload(unicast_id, data);
        }

        public override bool is_unicast()
        {
            return true;
        }

        ~NeighbourClient()
        {
        }
    }

    /** A UDPRequestClient to send a message to all nodes of a particular Network.
      * In turn, it is a FakeRmt object.
      */
    public class BroadcastClient : UDPRequestClient
    {
        private ISerializable broadcast_id;
        private string[] devs;
        private static int next_id = 0;
        private int my_id;
        public BroadcastClient(ISerializable broadcast_id, string[] devs, uint16 port=269)
        {
            base(port, false);
            my_id = next_id++;
            this.broadcast_id = broadcast_id;
            this.devs = devs;
        }

        public override string[] get_devs()
        {
            return devs;
        }

        public override UDPPayload get_udp_payload(RemoteCall data)
        {
            return new UDPPayload(broadcast_id, data);
        }

        public override bool is_unicast()
        {
            return false;
        }

        ~BroadcastClient()
        {
        }
    }

    /** A FakeRmt object to send requests to a local dispatcher. Simulates a NeighbourClient.
      */
    public class PseudoNeighbourClient : Object, FakeRmt
    {
        private RPCDispatcher rpcdispatcher;
        private bool wait_response;
        public PseudoNeighbourClient(RPCDispatcher rpcdispatcher, bool wait_response=true)
        {
            this.rpcdispatcher = rpcdispatcher;
            this.wait_response = wait_response;
        }

        public ISerializable rmt(RemoteCall data) throws RPCError
        {
            if (wait_response)
            {
                return rpcdispatcher.dispatch(null, data);
            }
            else
            {
                // Call rpcdispatcher.marshalled_dispatch(data) in a new tasklet
                struct_helper_RPCDispatcher_marshalled_dispatch arg = struct_helper_RPCDispatcher_marshalled_dispatch();
                arg.self = rpcdispatcher;
                arg.caller = null;
                arg.data = data.serialize();
                Tasklet.spawn((Spawnable)helper_marshalled_dispatch, &arg);
                return new SerializableNone();
            }
        }
    }

    /** A FakeRmt object to send requests to a set of local dispatchers. Simulates a BroadcastClient.
      */
    public class PseudoBroadcastClient : Object, FakeRmt
    {
        private Gee.List<RPCDispatcher> rpcdispatchers;
        public PseudoBroadcastClient(Gee.List<RPCDispatcher> rpcdispatchers)
        {
            this.rpcdispatchers = rpcdispatchers;
        }

        public ISerializable rmt(RemoteCall data)
        {
            foreach (RPCDispatcher rpcdispatcher in rpcdispatchers)
            {
                // Call rpcdispatcher.marshalled_dispatch(data) in a new tasklet
                struct_helper_RPCDispatcher_marshalled_dispatch arg = struct_helper_RPCDispatcher_marshalled_dispatch();
                arg.self = rpcdispatcher;
                arg.caller = null;
                arg.data = data.serialize();
                Tasklet.spawn((Spawnable)helper_marshalled_dispatch, &arg);
            }
            return new SerializableNone();
        }
    }
}

