/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2011 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gee;
using Tasklets;

namespace zcd
{
#if log_tasklet
    private string tasklet_id()
    {
        string ret = @"$(Tasklet.self().id)";
        int len = ret.length;
        for (int i = 0; i < 5-len; i++) ret = " " + ret;
        return @"[$(ret)] ";
    }
#else
    private string tasklet_id()
    {
        return "";
    }
#endif
    internal void log_debug(string msg)     {Posix.syslog(Posix.LOG_DEBUG,
                    tasklet_id() + "DEBUG "  + msg);}
    internal void log_info(string msg)      {Posix.syslog(Posix.LOG_INFO,
                    tasklet_id() + "INFO "   + msg);}
    internal void log_notice(string msg)    {Posix.syslog(Posix.LOG_NOTICE,
                    tasklet_id() + "INFO+ "  + msg);}
    internal void log_warn(string msg)      {Posix.syslog(Posix.LOG_WARNING,
                    tasklet_id() + "INFO++ " + msg);}
    internal void log_error(string msg)     {Posix.syslog(Posix.LOG_ERR,
                    tasklet_id() + "ERROR "  + msg);}
    internal void log_critical(string msg)  {Posix.syslog(Posix.LOG_CRIT,
                    tasklet_id() + "ERROR+ " + msg);}

    /** This is an error that any remotable method could throw.
      */
    public errordomain RPCError {
        FUNCTION_NOT_REMOTABLE,
        MALFORMED_PACKET,
        NETWORK_ERROR,
        NOT_VALID_MAP_YET,
        DROP,
        SERIALIZER_ERROR,
        GENERIC
    }

    /** This class represents a RPCError or any other error that is thrown
      * by the remotable method.
      */
    public class RemotableException : Object, ISerializable
    {
        public string message;
        public string code;
        public string domain;
        
        public Variant serialize_to_variant()
        {
            Variant v0 = Serializer.string_to_variant(message);
            Variant v1 = Serializer.string_to_variant(code);
            Variant v2 = Serializer.string_to_variant(domain);
            Variant vret = Serializer.tuple_to_variant_3(v0, v1, v2);
            return vret;
        }
        
        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            Variant v0;
            Variant v1;
            Variant v2;
            Serializer.variant_to_tuple_3(v, out v0, out v1, out v2);
            message = Serializer.variant_to_string(v0);
            code = Serializer.variant_to_string(v1);
            domain = Serializer.variant_to_string(v2);
        }

        public string to_string()
        {
            return @"$domain.$code '$message'";
        }
    }

    public ISerializable filter_exception(ISerializable ret) throws RPCError
    {
        if (ret.get_type().is_a(typeof(RemotableException)))
        {
            RemotableException e = (RemotableException)ret;
            if (e.domain == "RPCError")
            {
                if (e.code == "FUNCTION_NOT_REMOTABLE")
                    throw new RPCError.FUNCTION_NOT_REMOTABLE(e.message);
                if (e.code == "MALFORMED_PACKET")
                    throw new RPCError.MALFORMED_PACKET(e.message);
                if (e.code == "NETWORK_ERROR")
                    throw new RPCError.NETWORK_ERROR(e.message);
                if (e.code == "NOT_VALID_MAP_YET")
                    throw new RPCError.NOT_VALID_MAP_YET(e.message);
                if (e.code == "SERIALIZER_ERROR")
                    throw new RPCError.SERIALIZER_ERROR(e.message);
                if (e.code == "GENERIC")
                    throw new RPCError.GENERIC(e.message);
                throw new RPCError.GENERIC(@"Unexpected RPCError code $(e.code): $(e.message)");
            }
            if (e.domain == "SerializerError")
            {
                throw new RPCError.GENERIC(@"SerializerError code $(e.code): $(e.message)");
            }
            else throw new RPCError.GENERIC(@"Unexpected error $e");
        }
        return ret;
    }

    /** This class represents a call to a remotable procedure.
      *
      * The string method_name contains something like "property1.property2.method0"
      * which means that in the remote host the Dispatcher object has a
      * property "property1", that has a property "property2", that has
      * a remotable method "method0".
      * The array _elements contains the parameters (in serialized form) of the method.
      */
    public class RemoteCall : Object, ISerializable
    {
        public string method_name;
        private ArrayList<BufferOwner> _elements;

        public RemoteCall()
        {
            _elements = new ArrayList<BufferOwner>();
        }

        public void add_parameter(ISerializable arg)
        {
            _elements_deserialized = null;
            uchar[] ser = arg.serialize();
            BufferOwner buf = new BufferOwner(ser);
            _elements.add(buf);
        }

        private static bool not_impl_equal_func (Object? a, Object? b)
        {
            string astr = "null";
            if (a != null) astr = a.get_type().name();
            string bstr = "null";
            if (b != null) bstr = b.get_type().name();
            error(@"equal_func not implemented: compare between $(astr) and $(bstr)");
        }
        private ArrayList<ISerializable> _elements_deserialized = null;
        private Gee.List<ISerializable> _elements_deserialized_ro = null;
        public Gee.List<ISerializable> parameters {
            get {
                if (_elements_deserialized == null)
                {
                    try
                    {
                        _elements_deserialized = new ArrayList<ISerializable>((EqualDataFunc)not_impl_equal_func);
                        foreach (BufferOwner buf in _elements)
                        {
                            ISerializable iser = ISerializable.deserialize(buf.buf);
                            _elements_deserialized.add(iser);
                        }
                    }
                    catch (SerializerError e)
                    {
                        // TODO vala doesnt support error throwing from a property get
                        zcd.log_error(@"RemoteCall: Caught exception while deserializing parameters $(e.domain) code $(e.code): $(e.message)");
                    }
                }
                _elements_deserialized_ro = _elements_deserialized.read_only_view;
                return _elements_deserialized_ro;
            }
        }

        public Variant serialize_to_variant()
        {
            if (_elements.size > 0)
            {
                Variant[] var_elements = new Variant[_elements.size];
                int i = 0;
                foreach (BufferOwner element in _elements)
                {
                    Variant v0 = Serializer.uchar_array_to_variant(element.buf);
                    var_elements[i++] = v0;
                }
                Variant v1 = Serializer.variant_array_to_variant(var_elements);
                Variant v2 = Serializer.string_to_variant(method_name);
                Variant vret = Serializer.tuple_to_variant(v1, v2);
                return vret;
            }
            else
            {
                Variant v2 = Serializer.string_to_variant(method_name);
                return v2;
            }
        }

        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            string vt = v.get_type_string();
            assert(vt == "(vv)" || vt == "s");
            if (vt == "(vv)")
            {
                Variant v1;
                Variant v2;
                Serializer.variant_to_tuple(v, out v1, out v2);
                method_name = Serializer.variant_to_string(v2);
                Variant[] var_elements = Serializer.variant_to_variant_array(v1);
                _elements = new ArrayList<BufferOwner>();
                for (int i = 0; i < var_elements.length; i++)
                {
                    uchar[] buff = Serializer.variant_to_uchar_array(var_elements[i]);
                    BufferOwner buf = new BufferOwner(buff);
                    _elements.add(buf);
                }
            }
            else
            {
                _elements = new ArrayList<BufferOwner>();
                method_name = Serializer.variant_to_string(v);
            }
            _elements_deserialized = null;
        }

        public string to_string()
        {
            string args = "";
            string nexttok = "";
            foreach (BufferOwner buf in _elements)
            {
                args += nexttok;
                args += "[" + ISerializable.typename(buf.buf) + "]";
                nexttok = ", ";
            }
            return @"<RemoteCall $(method_name)($args)>";
        }
    }

    /** This interface has to be derived by a class that wants to send
      *  a RPC by any suitable means.
      */
    public interface FakeRmt : Object
    {
        public abstract ISerializable rmt(RemoteCall data) throws RPCError;
    }

    /** This class is used to send a message (usually without waiting for
      *  a result) to two remote objects.
      */
    public class CoupleFakeRmt : Object, FakeRmt
    {
        private FakeRmt f1;
        private FakeRmt f2;
        public CoupleFakeRmt(FakeRmt f1, FakeRmt f2)
        {
            this.f1 = f1;
            this.f2 = f2;
        }

        public ISerializable rmt(RemoteCall data) throws RPCError
        {
            zcd.log_debug(@"$(this.get_type().name()): sending $data");
            f1.rmt(data);
            return f2.rmt(data);
        }
    }

    public struct struct_helper_RPCDispatcher_marshalled_dispatch
    {
        public RPCDispatcher self;
        public Object? caller;
        public uchar[] data;
    }
    public void * helper_marshalled_dispatch(void *v)
    {
        Tasklet.declare_self("zcd.helper_marshalled_dispatch");
        struct_helper_RPCDispatcher_marshalled_dispatch *tuple_p = (struct_helper_RPCDispatcher_marshalled_dispatch *)v;
        // The caller function has to add a reference to the ref-counted instances
        RPCDispatcher self_save = tuple_p->self;
        Object? caller_save = tuple_p->caller;
        uchar[] data_save = tuple_p->data;
        // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
        Tasklet.schedule_back();
        // The actual call
        self_save.marshalled_dispatch(caller_save, data_save);
        // void method, return null
        return null;
    }

    /** This class is used on the node that receives a remote procedure call and
      * wants to execute it.
      */
    public abstract class RPCDispatcher : Object
    {
        /** This method receives a message as a stream of bytes from a caller,
          * translates it to a RemoteCall,
          * executes the call,
          * returns the result as a stream of bytes that we can send to the caller.
          */
        public uchar[] marshalled_dispatch(Object? caller, uchar[] data)
        {
            uchar[] ret;
            try {
                ISerializable ser = ISerializable.deserialize(data);
                if (ser.get_type().is_a(typeof(RemoteCall)))
                {
                    RemoteCall rc = (RemoteCall)ser;
                    Tasklet.declare_self(rc.method_name);
                    ret = dispatch(caller, rc).serialize();
                    Tasklet.declare_finished(rc.method_name);
                }
                else
                {
                    RemotableException re = new RemotableException();
                    re.domain = "RPCError";
                    re.message = "Malformed packet.";
                    re.code = "MALFORMED_PACKET";
                    ret = re.serialize();
                }
            }
            catch (SerializerError e) {
                RemotableException re = new RemotableException();
                re.domain = "RPCError";
                re.message = "Error deserializing remote call";
                re.code = "SERIALIZER_ERROR";
                ret = re.serialize();
            }
            return ret;
        }

        /** This method adds the feature that a generic-rpc exception is made serializable.
          */
        public ISerializable dispatch(Object? caller, RemoteCall data)
        {
            try
            {
                return remoteexception_dispatch(caller, data);
            }
            catch (RPCError e)
            {
                RemotableException re = new RemotableException();
                re.domain = "RPCError";
                re.code = "GENERIC";
                re.message = e.message;
                if (e is RPCError.FUNCTION_NOT_REMOTABLE)
                {
                    re.code = "FUNCTION_NOT_REMOTABLE";
                }
                if (e is RPCError.MALFORMED_PACKET)
                {
                    re.code = "MALFORMED_PACKET";
                }
                if (e is RPCError.NETWORK_ERROR)
                {
                    re.code = "NETWORK_ERROR";
                }
                if (e is RPCError.NOT_VALID_MAP_YET)
                {
                    re.code = "NOT_VALID_MAP_YET";
                }
                if (e is RPCError.SERIALIZER_ERROR)
                {
                    re.code = "SERIALIZER_ERROR";
                }
                return re;
            }
            catch (SerializerError e)
            {
                zcd.log_debug(@"$(this.get_type().name()): will report a SerializerError");
                RemotableException re = new RemotableException();
                re.domain = "SerializerError";
                re.code = "GENERIC";
                re.message = e.message;
                return re;
            }
            // An error not expected
            catch
            {
                RemotableException re = new RemotableException();
                re.domain = "RPCError";
                re.code = "GENERIC";
                re.message = "Unspecified error";
                return re;
            }
        }

        /** Override this method for each kind of "root" dispatcher. Eg address_manager, coord_peer, andna_peer...
          * in order to add the feature that a model-specific exception is made serializable.
          * This method could be called *directly* for a dispatcher that does not need to transform
          * an exception into a remotable.
          */
        public virtual ISerializable remoteexception_dispatch(Object? caller, RemoteCall data) throws Error
        {
            // Iff there aren't any model-specific exceptions then this method may also be not overridden
            return _dispatch(caller, data);
        }

        /** Override this method for each kind of "root" dispatcher. Eg address_manager, coord_peer, andna_peer...
          *
          * This method could be called *directly* for a dispatcher that does not need to transform
          * an exception into a remotable.
          */
        public virtual ISerializable _dispatch(Object? caller, RemoteCall data) throws Error
        {
            throw new RPCError.FUNCTION_NOT_REMOTABLE("Function %s is not remotable.".printf(data.method_name));
        }
    }
}

