/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2011 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gee;
using Tasklets;

namespace zcd
{
    /** A message that uses UDP as transport.
      */
    public class UDPMessage : Object, ISerializable
    {
        public int request_id {get; private set;}
        private string message_type;
        public bool wait_response {get; private set;}
        /** data will be a instance of a serializable class, that will contain
          * the marshalled_data. Furthermore it will contain information to identify
          * a single node (if request_unicast) or a group of nodes.
          * See the abstract class UDPPayload below.
          */
        public ISerializable data {get; private set;}
        private UDPMessage(int reqid = -1)
        {
            if (reqid == -1) request_id = Random.int_range(1, int.MAX);
            else request_id = reqid;
        }

        public static UDPMessage make_request_unicast(ISerializable data, bool wait_response=true)
        {
            UDPMessage ret = new UDPMessage();
            ret.message_type = "R";
            ret.wait_response = wait_response;
            ret.data = data;
            return ret;
        }

        public static UDPMessage make_request_broadcast(ISerializable data)
        {
            UDPMessage ret = new UDPMessage();
            ret.message_type = "B";
            ret.wait_response = false;
            ret.data = data;
            return ret;
        }

        public static UDPMessage make_keep_alive(int reqid)
        {
            UDPMessage ret = new UDPMessage(reqid);
            ret.message_type = "K";
            ret.wait_response = false;
            return ret;
        }

        public static UDPMessage make_response(int reqid, ISerializable data)
        {
            UDPMessage ret = new UDPMessage(reqid);
            ret.message_type = "A";
            ret.wait_response = false;
            ret.data = data;
            return ret;
        }

        public bool is_response()
        {
            return message_type == "A";
        }

        public bool is_keepalive()
        {
            return message_type == "K";
        }

        public bool is_request_unicast()
        {
            return message_type == "R";
        }

        public bool is_request_broadcast()
        {
            return message_type == "B";
        }

        public Variant serialize_to_variant()
        {
            Variant v0 = Serializer.int_to_variant(request_id);
            Variant v1 = Serializer.string_to_variant(message_type);
            Variant v2 = Serializer.int_to_variant(wait_response?1:0);
            Variant v3 = Serializer.uchar_array_to_variant(data.serialize());
            Variant vret = Serializer.tuple_to_variant_4(v0, v1, v2, v3);
            return vret;
        }
        
        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            Variant v0;
            Variant v1;
            Variant v2;
            Variant v3;
            Serializer.variant_to_tuple_4(v, out v0, out v1, out v2, out v3);
            request_id = Serializer.variant_to_int(v0);
            message_type = Serializer.variant_to_string(v1);
            wait_response = Serializer.variant_to_int(v2) == 1;
            data = ISerializable.deserialize(Serializer.variant_to_uchar_array(v3));
        }

        public string to_string()
        {
            if (is_response())
            {
                return @"<UDPMessage ANSWER request=$(request_id) [$(data.get_type().name())]>";
            }
            if (is_keepalive())
            {
                return @"<UDPMessage KEEPALIVE request=$(request_id)>";
            }
            if (is_request_unicast())
            {
                string wait_str = wait_response ? "wait_response" : "no_wait_response";
                string data_str = "null_data";
                UDPPayload _data = data as UDPPayload;
                if (_data != null) data_str = @"$(_data)";
                return @"<UDPMessage REQUEST id=$(request_id) $(data_str) $(wait_str)>";
            }
            if (is_request_broadcast())
            {
                string wait_str = wait_response ? "wait_response" : "no_wait_response";
                string data_str = "null_data";
                UDPPayload _data = data as UDPPayload;
                if (_data != null) data_str = @"$(_data)";
                return @"<UDPMessage BROADCAST id=$(request_id) $(data_str) $(wait_str)>";
            }
            return "<UDPMessage ???>";
        }
    }

    /** A UDP message containing a RemoteCall to be dispatched and data that the callbacks can inspect
      * to see whether the message has to be delivered to one of our dispatchers
      */
    public class UDPPayload : Object, ISerializable
    {
        public uchar[] ser;
        public RemoteCall data;

        public UDPPayload(ISerializable ser, RemoteCall data)
        {
            this.ser = ser.serialize();
            this.data = data;
        }

        public Variant serialize_to_variant()
        {
            Variant v0 = Serializer.uchar_array_to_variant(ser);
            Variant v1 = data.serialize_to_variant();
            Variant vret = Serializer.tuple_to_variant(v0, v1);
            return vret;
        }
        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            Variant v0;
            Variant v1;
            Serializer.variant_to_tuple(v, out v0, out v1);
            ser = Serializer.variant_to_uchar_array(v0);
            data = (RemoteCall)Object.new(typeof(RemoteCall));
            data.deserialize_from_variant(v1);
        }

        public string to_string()
        {
            return @"<UDPPayload $data>";
        }
    }
}

