/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2014 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gee;

namespace zcd
{
    /** When you have a socket connected to a server, or when you receive
      *  a connection, you get an obscure object that implements this API.
      */
    public interface IConnectedStreamSocket : Object
    {
        public uint16 peer_port {
            get {
                return this._peer_port_getter();
            }
        }
        public abstract uint16 _peer_port_getter();

        public string peer_address {
            get {
                return this._peer_address_getter();
            }
        }
        public abstract unowned string _peer_address_getter();

        public uint16 my_port {
            get {
                return this._my_port_getter();
            }
        }
        public abstract uint16 _my_port_getter();

        public string my_address {
            get {
                return this._my_address_getter();
            }
        }
        public abstract unowned string _my_address_getter();

        /** Sends all the bytes. Returns when all the bytes have been reliably sent.
          */
        public void send(uchar[] data) throws Error
        {
            int remain = data.length;
            while (remain > 0)
            {
                int done = send_part(data, remain);
                remain -= done;
                data = data[done:done+remain];
            }
        }

        protected abstract int send_part(uchar[] data, int maxlen) throws Error;
        public abstract uchar[] recv(int maxlen) throws Error;
        public abstract void close() throws Error;
    }

    /** Use this class to make a connection to a TCP service.
      * In particular, you can wait for the connect to complete without
      *  blocking the rest of the application.
      */
    public class ClientStreamSocket : Object
    {
        private Socket s;

        public ClientStreamSocket(string? my_addr = null) throws Error
        {
            s = new Socket(SocketFamily.IPV4, SocketType.STREAM, SocketProtocol.TCP);
            if (my_addr != null)
                s.bind(new InetSocketAddress(new InetAddress.from_string(my_addr), 0), false);
        }

        /** When the method returns, use the returned object
          *  to carry on the communication. Discard this instance, instead.
          */
        public IConnectedStreamSocket socket_connect(string addr, uint16 port) throws Error
        {
            assert(s != null);
            s.connect(new InetSocketAddress(new InetAddress.from_string(addr), port));
            IConnectedStreamSocket ret = new ConnectedStreamSocket(s);
            s = null;
            return ret;
        }
    }

    class ConnectedStreamSocket : Object, IConnectedStreamSocket
    {
        private Socket s;
        private string remote_addr;
        private uint16 remote_port;
        private string local_addr;
        private uint16 local_port;
        public ConnectedStreamSocket(Socket soc) throws Error
        {
            s = soc;
            InetSocketAddress x = (InetSocketAddress)s.get_remote_address();
            remote_addr = x.address.to_string();
            remote_port = (uint16)x.port;
            InetSocketAddress y = (InetSocketAddress)s.get_local_address();
            local_addr = y.address.to_string();
            local_port = (uint16)y.port;
        }

        public uint16 _peer_port_getter()
        {
            return remote_port;
        }

        public unowned string _peer_address_getter()
        {
            return remote_addr;
        }

        public uint16 _my_port_getter()
        {
            return local_port;
        }

        public unowned string _my_address_getter()
        {
            return local_addr;
        }

        protected int send_part(uchar[] data, int maxlen) throws Error
        {
            uint8[] buffer = new uint8[maxlen];
            Posix.memcpy(buffer, data, maxlen);
            int ret = (int)s.send(buffer);
            return ret;
        }

        public uchar[] recv(int maxlen) throws Error
        {
            // public ssize_t receive (uint8[] buffer, Cancellable? cancellable = null) throws Error 
            uint8[] buffer = new uint8[maxlen];
            ssize_t len = s.receive(buffer);
            uchar[] ret = new uchar[len];
            Posix.memcpy(ret, buffer, len);
            return ret;
        }

        public void close() throws Error
        {
            s.close();
        }
    }
}

