/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2014 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gee;
using Tasklets;

namespace zcd
{
    public struct struct_helper_RPCDispatcher_marshalled_dispatch
    {
        public RPCDispatcher self;
        public Object? caller;
        public uchar[] data;
    }
    public void * helper_marshalled_dispatch(void *v)
    {
        Tasklet.declare_self("zcd.helper_marshalled_dispatch");
        struct_helper_RPCDispatcher_marshalled_dispatch *tuple_p = (struct_helper_RPCDispatcher_marshalled_dispatch *)v;
        // The caller function has to add a reference to the ref-counted instances
        RPCDispatcher self_save = tuple_p->self;
        Object? caller_save = tuple_p->caller;
        uchar[] data_save = tuple_p->data;
        // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
        Tasklet.schedule_back();
        // The actual call
        self_save.marshalled_dispatch(caller_save, data_save);
        // void method, return null
        return null;
    }

    /** This class is used on the node that receives a remote procedure call and
      * wants to execute it.
      */
    public abstract class RPCDispatcher : Object
    {
        /** This method receives a message as a stream of bytes from a caller,
          * translates it to a RemoteCall,
          * executes the call,
          * returns the result as a stream of bytes that we can send to the caller.
          */
        public uchar[] marshalled_dispatch(Object? caller, uchar[] data)
        {
            uchar[] ret;
            try {
                ISerializable ser = ISerializable.deserialize(data);
                if (ser.get_type().is_a(typeof(RemoteCall)))
                {
                    RemoteCall rc = (RemoteCall)ser;
                    Tasklet.declare_self(rc.method_name);
                    ret = dispatch(caller, rc).serialize();
                    Tasklet.declare_finished(rc.method_name);
                }
                else
                {
                    RemotableException re = new RemotableException();
                    re.domain = "RPCError";
                    re.message = "Malformed packet.";
                    re.code = "MALFORMED_PACKET";
                    ret = re.serialize();
                }
            }
            catch (SerializerError e) {
                RemotableException re = new RemotableException();
                re.domain = "RPCError";
                re.message = "Error deserializing remote call";
                re.code = "SERIALIZER_ERROR";
                ret = re.serialize();
            }
            return ret;
        }

        /** This method adds the feature that a generic-rpc exception is made serializable.
          */
        public ISerializable dispatch(Object? caller, RemoteCall data)
        {
            try
            {
                return remoteexception_dispatch(caller, data);
            }
            catch (RPCError e)
            {
                RemotableException re = new RemotableException();
                re.domain = "RPCError";
                re.code = "GENERIC";
                re.message = e.message;
                if (e is RPCError.FUNCTION_NOT_REMOTABLE)
                {
                    re.code = "FUNCTION_NOT_REMOTABLE";
                }
                if (e is RPCError.MALFORMED_PACKET)
                {
                    re.code = "MALFORMED_PACKET";
                }
                if (e is RPCError.NETWORK_ERROR)
                {
                    re.code = "NETWORK_ERROR";
                }
                if (e is RPCError.NOT_VALID_MAP_YET)
                {
                    re.code = "NOT_VALID_MAP_YET";
                }
                if (e is RPCError.SERIALIZER_ERROR)
                {
                    re.code = "SERIALIZER_ERROR";
                }
                return re;
            }
            catch (SerializerError e)
            {
                zcd.log_debug(@"$(this.get_type().name()): will report a SerializerError");
                RemotableException re = new RemotableException();
                re.domain = "SerializerError";
                re.code = "GENERIC";
                re.message = e.message;
                return re;
            }
            // An error not expected
            catch
            {
                RemotableException re = new RemotableException();
                re.domain = "RPCError";
                re.code = "GENERIC";
                re.message = "Unspecified error";
                return re;
            }
        }

        /** Override this method for each kind of "root" dispatcher. Eg address_manager, coord_peer, andna_peer...
          * in order to add the feature that a model-specific exception is made serializable.
          * This method could be called *directly* for a dispatcher that does not need to transform
          * an exception into a remotable.
          */
        public virtual ISerializable remoteexception_dispatch(Object? caller, RemoteCall data) throws Error
        {
            // Iff there aren't any model-specific exceptions then this method may also be not overridden
            return _dispatch(caller, data);
        }

        /** Override this method for each kind of "root" dispatcher. Eg address_manager, coord_peer, andna_peer...
          *
          * This method could be called *directly* for a dispatcher that does not need to transform
          * an exception into a remotable.
          */
        public virtual ISerializable _dispatch(Object? caller, RemoteCall data) throws Error
        {
            throw new RPCError.FUNCTION_NOT_REMOTABLE("Function %s is not remotable.".printf(data.method_name));
        }
    }
}

