/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2014 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Ntkresolv;
using Netsukuku.InetUtils;
using Gee;
using Dns;

namespace Dns
{
    // RFC 1035
    public enum OperationCode { NAME_TO_IP=0, IP_TO_NAME=1, STATUS=2 }
    public enum ResponseCode { NO_ERROR=0, BADFORMAT=1, SERVFAIL=2, NOTFOUND=3, NOTIMPL=4, REFUSE=5 }
    public enum RRType { A=1, PTR=12, OPT=41, QR_ALL=255}
    public enum RRClass { IN=1, QR_ANY=255}

    errordomain MessageError {
        MALFORMED_MESSAGE,
        UNSUPPORTED_MESSAGE
    }

    public class QuestionRecord : Object
    {
        public string name {get; private set;}
        public RRType rrtype {get; private set; default=RRType.A;}
        public RRClass rrclass {get; private set; default=RRClass.IN;}
        public QuestionRecord(string name, RRType rrtype, RRClass rrclass)
        {
            this.name = name;
            this.rrtype = rrtype;
            this.rrclass = rrclass;
        }
    }

    public class ResRecord : Object
    {
        public string name {get; private set;}
        public RRType rrtype {get; private set; default=RRType.A;}
        public RRClass rrclass {get; private set; default=RRClass.IN;}
        public int32 ttl {get; private set;}
        public uint8[] data {get; private set;}
        public ResRecord.from_data(string name, RRType rrtype, RRClass rrclass, int32 ttl, uint8[] data)
        {
            this.name = name;
            this.rrtype = rrtype;
            this.rrclass = rrclass;
            this.ttl = ttl;
            this.data = new uint8[data.length];
            int pos = 0;
            foreach (uint8 b in data) this.data[pos++] = b;
        }
        public ResRecord.inetaddr(string name, RRType rrtype, RRClass rrclass, int32 ttl, uint8[] data)
        {
            assert(data.length == 4);
            this.from_data(name, rrtype, rrclass, ttl, data);
        }
        public ResRecord.ptr(string name, RRType rrtype, RRClass rrclass, int32 ttl, string ptr) throws MessageError
        {
            this.name = name;
            this.rrtype = rrtype;
            this.rrclass = rrclass;
            this.ttl = ttl;
            this.data = name_dotted_to_octets(ptr);
        }
    }

    internal uint8[] name_dotted_to_octets(string name) throws MessageError
    {
        // name:  www.google.com. becomes [3]www[6]google[3]com[0]
        uint8[] buf = new uint8[name.length + 1];
        int pos = 0;
        if (name == "")
        {
            buf[pos++] = (uint8)0;
        }
        else
        {
            if (name.substring(name.length-1) != ".")
                throw new MessageError.MALFORMED_MESSAGE("missing trail dot");
            string[] parts = name.split(".");
            foreach (string part in parts)
            {
                if (part.length > 255)
                    throw new MessageError.MALFORMED_MESSAGE("name part too long");
                uint8 b = (uint8)part.length;
                buf[pos++] = b;
                foreach (uint8 c in part.data) buf[pos++] = c;
            }
        }
        return buf;
    }

    public class Message : Object
    {
        public uint16 id;
        public bool question;
        public OperationCode opcode;
        public bool authoritative_answer;
        public bool truncation;
        public bool recursion_desired;
        public bool recursion_available;
        public ResponseCode rcode; 
        public ArrayList<QuestionRecord> question_recs;
        public ArrayList<ResRecord> answer_recs;
        public ArrayList<ResRecord> authority_recs;
        public ArrayList<ResRecord> additional_recs;

        public Message.from_wire(uint8[] message, owned size_t message_length=0) throws MessageError
        {
            // we can have the message in a new array with exact size, or else we
            // can have the message in a large buffer and the exact size of the message
            // is passed as argument.
            if (message_length == 0) message_length = message.length;
            // initialize lists
            question_recs = new ArrayList<QuestionRecord>();
            answer_recs = new ArrayList<ResRecord>();
            authority_recs = new ArrayList<ResRecord>();
            additional_recs = new ArrayList<ResRecord>();
            // check size supports the size of the header
            if (message_length < 12)
                throw new MessageError.MALFORMED_MESSAGE("Few bytes");
            // id
            id = message[0] * 256 + message[1];
            // QR
            uint8 m2 = message[2];  //  |QR| Opcode    |AA|TC|RD|
            m2 &= 128;
            question = m2 == 0;
            // Opcode
            m2 = message[2];
            m2 &= 8+16+32+64;
            m2 >>= 3;
            opcode = (OperationCode)m2;
            // AA
            m2 = message[2];
            m2 &= 4;
            authoritative_answer = m2 > 0;
            // TC
            m2 = message[2];
            m2 &= 2;
            truncation = m2 > 0;
            // RD
            m2 = message[2];
            m2 &= 1;
            recursion_desired = m2 > 0;
            // RA
            uint8 m3 = message[3];  //  |RA| Z      | RCode     |
            m3 &= 128;
            recursion_available = m3 > 0;
            // RCode
            m3 = message[3];
            m3 &= 15;
            rcode = (ResponseCode)m3;
            // qdcount
            uint16 qdCount = message[4] * 256 + message[5];
            // ancount
            uint16 anCount = message[6] * 256 + message[7];
            // nscount
            uint16 nsCount = message[8] * 256 + message[9];
            // arcount
            uint16 arCount = message[10] * 256 + message[11];
            size_t pos = 12; // first question
            for (int i = 0; i < qdCount; i++)
            {
                string qname;
                RRType rrtype;
                RRClass rrclass;
                name_type_class(message, message_length, ref pos, out qname, out rrtype, out rrclass);
                // question is ready and pos is ready
                QuestionRecord qr = new QuestionRecord(qname, rrtype, rrclass);
                question_recs.add(qr);
            }
            for (int i = 0; i < anCount; i++)
            {
                ResRecord rr;
                res_record(message, message_length, ref pos, out rr);
                answer_recs.add(rr);
            }
            for (int i = 0; i < nsCount; i++)
            {
                ResRecord rr;
                res_record(message, message_length, ref pos, out rr);
                authority_recs.add(rr);
            }
            for (int i = 0; i < arCount; i++)
            {
                ResRecord rr;
                res_record(message, message_length, ref pos, out rr);
                // detect OPT Pseudo-RR (see RFC 6891)
                if (question && rr.rrtype == RRType.OPT)
                {
                    // something to do?
                }
                additional_recs.add(rr);
            }
        }

        private void
        name_type_class
        (uint8 *message,
         size_t message_length,
         ref size_t pos,
         out string name,
         out RRType rrtype,
         out RRClass rrclass) throws MessageError
        {
            /*
            +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
            |                                               |
            /                                               /
            /                     NAME                      /
            |                                               |
            +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
            |                     TYPE                      |
            +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
            |                     CLASS                     |
            +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
            */
            // name:  www.google.com is [3]www[6]google[3]com[0]
            name = "";
            while (true)
            {
                if (message_length < pos + 1)
                    throw new MessageError.MALFORMED_MESSAGE("Few bytes");
                size_t len = message[pos];
                // finish?
                if (len == 0) break;
                // TODO compression mechanism of DNS says that if the first 2 bits are 1
                //  then the rest is a offset for the rest of the name.
                if (len >= 192)
                    throw new MessageError.UNSUPPORTED_MESSAGE("compression mechanism not supported yet");
                // enough octets?
                if (message_length < pos + len + 1)
                    throw new MessageError.MALFORMED_MESSAGE("Few bytes");
                // scan this piece of name
                for (size_t j = pos + 1; j < pos + len + 1; j++)
                {
                    if (message[j] == 0)
                        throw new MessageError.MALFORMED_MESSAGE("\\0 in name");
                    char c = (char)message[j];
                    name += @"$c";
                }
                // prepare for next piece
                name += ".";
                pos += len + 1;
            }
            // this is the final [0]
            pos += 1;
            // enough octets?
            if (message_length < pos + 4)
                throw new MessageError.MALFORMED_MESSAGE("Few bytes");
            // type
            rrtype = (RRType)(message[pos++] * 256 + message[pos++]);
            // class
            rrclass = (RRClass)(message[pos++] * 256 + message[pos++]);
        }

        private void
        res_record
        (uint8 *message,
         size_t message_length,
         ref size_t pos,
         out ResRecord rr) throws MessageError
        {
            /*
            +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
            |                                               |
            /                                               /
            /                     NAME                      /
            |                                               |
            +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
            |                     TYPE                      |
            +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
            |                     CLASS                     |
            +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
            |                     TTL                       |
            |                                               |
            +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
            |                  RDLENGTH                     |
            +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--|
            /                     RDATA                     /
            /                                               /
            +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+ 
            */
            string rname;
            RRType rrtype;
            RRClass rrclass;
            name_type_class(message, message_length, ref pos, out rname, out rrtype, out rrclass);
            // question is ready and pos is ready
            // enough octets?
            if (message_length < pos + 4)
                throw new MessageError.MALFORMED_MESSAGE("Few bytes");
            // ttl
            int32 ttl = 0;
            ttl *= 256;
            ttl += message[pos++];
            ttl *= 256;
            ttl += message[pos++];
            ttl *= 256;
            ttl += message[pos++];
            ttl *= 256;
            ttl += message[pos++];
            // data
            // enough octets?
            if (message_length < pos + 2)
                throw new MessageError.MALFORMED_MESSAGE("Few bytes");
            uint16 rdlen = message[pos++] * 256 + message[pos++];
            // enough octets?
            if (message_length < pos + rdlen)
                throw new MessageError.MALFORMED_MESSAGE("Few bytes");
            uint8[] data = new uint8[rdlen];
            for (uint16 i = 0; i < rdlen; i++)
                data[i] = message[pos++];
            // ready
            rr = new ResRecord.from_data(rname, rrtype, rrclass, ttl, data);
        }

        public Message? make_response()
        {
            if (question) return new Message.response(this);
            else return null;
        }

        private Message.response(Message q)
        {
            if (q.question)
            {
                question = false;
                id = q.id;
                opcode = q.opcode;
                authoritative_answer = true;
                truncation = false;
                recursion_desired = q.recursion_desired;
                recursion_available = true;
                rcode = ResponseCode.NO_ERROR;
                // initialize lists
                question_recs = new ArrayList<QuestionRecord>();
                answer_recs = new ArrayList<ResRecord>();
                authority_recs = new ArrayList<ResRecord>();
                additional_recs = new ArrayList<ResRecord>();
                // questions
                foreach (QuestionRecord qs in q.question_recs)
                    question_recs.add(qs);
                // answers to be filled
                // authority to be filled
                // additional to be filled, except for opt pseudo rr
                ResRecord optrr = new ResRecord.from_data("", RRType.OPT, (RRClass)512, 0, new uint8[] {});
                additional_recs.add(optrr);
            }
        }

        public uint8[] to_wire()
        {
            uint8[] ret = {};
            uint8 o0;
            uint8 o1;
            uint8 o2;
            uint8 o3;
            // id
            add_uint16(id, out o0, out o1);
            ret += o0;
            ret += o1;
            // |QR| Opcode    |AA|TC|RD|
            o0 = (uint8)((int)opcode * 8);
            if (!question) o0 += 128;
            if (authoritative_answer) o0 += 4;
            if (truncation) o0 += 2;
            if (recursion_desired) o0 += 1;
            ret += o0;
            // |RA| Z      | RCode     |
            o0 = (uint8)rcode;
            if (recursion_available) o0 += 128;
            ret += o0;
            // qdcount
            uint16 qdcount = (uint16)question_recs.size;
            add_uint16(qdcount, out o0, out o1);
            ret += o0;
            ret += o1;
            // ancount
            uint16 ancount = (uint16)answer_recs.size;
            add_uint16(ancount, out o0, out o1);
            ret += o0;
            ret += o1;
            // nscount
            uint16 nscount = (uint16)authority_recs.size;
            add_uint16(nscount, out o0, out o1);
            ret += o0;
            ret += o1;
            // arcount
            uint16 arcount = (uint16)additional_recs.size;
            add_uint16(arcount, out o0, out o1);
            ret += o0;
            ret += o1;
            foreach (QuestionRecord rec in question_recs)
            {
                uint8[] buf;
                output_name_type_class(rec.name, rec.rrtype, rec.rrclass, out buf);
                foreach (uint8 b in buf) ret += b;
            }
            foreach (ResRecord rec in answer_recs)
            {
                uint8[] buf;
                output_res_record(rec, out buf);
                foreach (uint8 b in buf) ret += b;
            }
            foreach (ResRecord rec in authority_recs)
            {
                uint8[] buf;
                output_res_record(rec, out buf);
                foreach (uint8 b in buf) ret += b;
            }
            foreach (ResRecord rec in additional_recs)
            {
                uint8[] buf;
                output_res_record(rec, out buf);
                foreach (uint8 b in buf) ret += b;
            }

            return ret;
        }

        private void add_uint16(uint16 i, out uint8 o0, out uint8 o1)
        {
            o0 = (uint8)(i / 256);
            o1 = (uint8)(i - o0 * 256);
        }

        private void add_uint32(uint32 i, out uint8 o0, out uint8 o1, out uint8 o2, out uint8 o3)
        {
            uint32 remainder = i / 256;
            o3 = (uint8)(i - remainder * 256);
            i = remainder;
            remainder = i / 256;
            o2 = (uint8)(i - remainder * 256);
            i = remainder;
            remainder = i / 256;
            o1 = (uint8)(i - remainder * 256);
            o0 = (uint8)remainder;
        }

        private void
        output_name_type_class
        (string name,
         RRType rrtype,
         RRClass rrclass,
         out uint8[] buf) throws MessageError
        {
            /*
            +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
            |                                               |
            /                                               /
            /                     NAME                      /
            |                                               |
            +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
            |                     TYPE                      |
            +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
            |                     CLASS                     |
            +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
            */
            // name
            uint8[] buf_start = name_dotted_to_octets(name);
            buf = new uint8[buf_start.length + 4];
            int pos = 0;
            foreach (uint8 b in buf_start) buf[pos++] = b;
            // type
            uint8 o0;
            uint8 o1;
            uint16 itype = (uint16)rrtype;
            add_uint16(itype, out o0, out o1);
            buf[pos++] = o0;
            buf[pos++] = o1;
            // class
            uint16 iclass = (uint16)rrclass;
            add_uint16(iclass, out o0, out o1);
            buf[pos++] = o0;
            buf[pos++] = o1;
        }

        private void
        output_res_record
        (ResRecord rr,
         out uint8[] buf) throws MessageError
        {
            /*
            +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
            |                                               |
            /                                               /
            /                     NAME                      /
            |                                               |
            +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
            |                     TYPE                      |
            +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
            |                     CLASS                     |
            +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
            |                     TTL                       |
            |                                               |
            +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
            |                  RDLENGTH                     |
            +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--|
            /                     RDATA                     /
            /                                               /
            +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+ 
            */
            if (rr.data.length > 65535)
                throw new MessageError.MALFORMED_MESSAGE("data too long");
            uint8[] buf_start;
            output_name_type_class(rr.name, rr.rrtype, rr.rrclass, out buf_start);
            buf = new uint8[buf_start.length + 6 + rr.data.length];
            int pos = 0;
            foreach (uint8 b in buf_start)
                buf[pos++] = b;
            uint8 o0;
            uint8 o1;
            uint8 o2;
            uint8 o3;
            add_uint32(rr.ttl, out o0, out o1, out o2, out o3);
            buf[pos++] = o0;
            buf[pos++] = o1;
            buf[pos++] = o2;
            buf[pos++] = o3;
            add_uint16((uint16)(rr.data.length), out o0, out o1);
            buf[pos++] = o0;
            buf[pos++] = o1;
            foreach (uint8 b in rr.data)
                buf[pos++] = b;
        }
    }
}


//internal void log_debug(string msg)     {Posix.syslog(Posix.LOG_DEBUG,
//                "DEBUG "  + msg);}
internal void log_info(string msg)      {Posix.syslog(Posix.LOG_INFO,
                "INFO "   + msg);}
internal void log_notice(string msg)    {Posix.syslog(Posix.LOG_NOTICE,
                "INFO+ "  + msg);}
internal void log_warn(string msg)      {Posix.syslog(Posix.LOG_WARNING,
                "INFO++ " + msg);}
internal void log_error(string msg)     {Posix.syslog(Posix.LOG_ERR,
                "ERROR "  + msg);}
internal void log_critical(string msg)  {Posix.syslog(Posix.LOG_CRIT,
                "ERROR+ " + msg);}

//internal void log_debug_lines(string dump) {
//    foreach (string line in dump.split("\n"))
//        log_debug(line);
//}

string hex_dump(uint8[] buf, owned size_t length=0)
{
    /* dumps buf to a string. Looks like:
     * [0000] 75 6E 6B 6E 6F 77 6E 20
     *                  30 FF 00 00 00 00 39 00 unknown 0.....9.
     * (in a single line of course)
     */

    string ret = "";
    uchar *p = buf;
    string addrstr = "";
    string hexstr = "";
    string charstr = "";
    if (length == 0) length = buf.length;

    for (int n = 1; n <= length; n++)
    {
        if (n % 16 == 1)
        {
            addrstr = "%.4x".printf((uint)p-(uint)buf);
        }

        uchar c = *p;
        if (c == '\0' || "( ):-_qwertyuiopasdfghjklzxcvbnmQWERTYUIOPASDFGHJKLZXCVBNM1234567890".index_of_char(c) < 0)
            c = '.';

        // store hex str (for left side)
        hexstr += "%02X ".printf(*p);
        charstr += "%c".printf(c);

        if (n % 16 == 0)
        {
            // line completed
            ret += "[%4.4s]   %-50.50s  %s\n".printf(addrstr, hexstr, charstr);
            hexstr = "";
            charstr = "";
        }
        else if (n % 16 == 8)
        {
            // half line: add whitespaces
            hexstr += "  ";
            charstr += " ";
        }

        p++; // next byte
    }

    if (hexstr.length > 0)
    {
        // print rest of buffer if not empty
        ret += "[%4.4s]   %-50.50s  %s\n".printf(addrstr, hexstr, charstr);
    }

    return ret;
}

ArrayList<uint16> serving_ids;
Socket listening_socket;
ArrayList<SocketAddress> dns_list;

public class CleanupServingIdArg : Object
{
    public uint16 id;
    public CleanupServingIdArg(uint16 id) {this.id = id;}
    public int cleanup_serving_id()
    {
        Thread.usleep(20000000);
        serving_ids.remove(id);
        return 0;
    }
}

/** This class is associated to a thread that executes its method forward_dns().
  * Its duty is to forward a dns request to one of the nameservers that this host
  * is configured to use. If it receives a response it will send it back to the
  * client.
  */
public class ForwardDnsArg : Object
{
    public SocketAddress client_address;
    public uint8[] buffer;
    public SocketAddress dns_server;
    public ForwardDnsArg(SocketAddress client_address, uint8[] buffer, SocketAddress dns_server)
    {
        this.client_address = client_address;
        this.buffer = buffer;
        this.dns_server = dns_server;
        dns_response = null;
    }

    private uint8[] dns_response;

    public int forward_dns()
    {
        Socket s0 = new Socket(SocketFamily.IPV4, GLib.SocketType.DATAGRAM, SocketProtocol.UDP);
        s0.timeout = 60;
        try
        {
            s0.send_to(dns_server, buffer);
            SocketAddress respaddr;
            uint8 respbuf[4096];
            ssize_t buflen = s0.receive_from(out respaddr, respbuf);
            dns_response = new uint8[buflen];
            for (int i = 0; i < buflen; i++) dns_response[i] = respbuf[i];
        }
        catch (IOError e)
        {
            if (e is IOError.TIMED_OUT)
            {
            }
            else throw e;
        }

        // Now I should have a response for the client
        if (dns_response != null)
        {
            listening_socket.send_to(client_address, dns_response);
        }
        return 0;
    }
}

/** This class is associated to a thread that executes its method handle_request().
  * Its duty is to handle a dns request from a client.
  */
public class HandleRequestArg : Object
{
    public SocketAddress client_address;
    public uint8[] buffer;
    public HandleRequestArg(SocketAddress client_address, uint8[] buffer)
    {
        this.client_address = client_address;
        this.buffer = buffer;
    }

    public int handle_request()
    {
        Message query = new Message.from_wire(buffer);
        // handle only request messages
        if (! query.question) return 0;
        // check DNS or ANDNA
        if (check_forward_to_dns(query))
        {
            foreach (SocketAddress dns_server in dns_list)
            {
                ForwardDnsArg a = new ForwardDnsArg(client_address, buffer, dns_server);
                Thread<int> thread = new Thread<int>.@try ("forward_dns", a.forward_dns);
            }
        }
        else
        {
            // This is a query to ANDNA realm.
            // handle a question only once
            if (! serving_ids.contains(query.id))
            {
                serving_ids.add(query.id);
                // at most after timeout remove the id
                CleanupServingIdArg a = new CleanupServingIdArg(query.id);
                Thread<int> thread = new Thread<int>.@try ("cleanup_serving_id", a.cleanup_serving_id);
                // enclose all operations and finally remove the id
                try
                {
                    // this 'if' should be implied
                    if (query.opcode == OperationCode.NAME_TO_IP)
                    {
                        // this 'if' should be implied
                        if (query.question_recs.size > 0)
                        {
                            QuestionRecord q = query.question_recs[0];
                            if (q.rrclass == RRClass.IN
                                && q.rrtype == RRType.A)
                            {
                                Gee.List<ResRecord> r = std_qry(q);
                                Message resp = query.make_response();
                                if (r.is_empty)
                                {
                                    resp.rcode = ResponseCode.NOTFOUND;
                                }
                                else
                                {
                                    resp.answer_recs.add_all(r);
                                }
                                uint8[] dns_resp = resp.to_wire();
                                listening_socket.send_to(client_address, dns_resp);
                            }
                            else if ((q.rrclass == RRClass.IN || q.rrclass == RRClass.QR_ANY)
                                && q.rrtype == RRType.PTR
                                && q.name.length > 14
                                && q.name.substring(q.name.length-14).up() == ".IN-ADDR.ARPA.")
                            {
                                Gee.List<ResRecord> r = inv_qry(q);
                                Message resp = query.make_response();
                                if (r.is_empty)
                                {
                                    resp.rcode = ResponseCode.NOTFOUND;
                                }
                                else
                                {
                                    resp.answer_recs.add_all(r);
                                }
                                uint8[] dns_resp = resp.to_wire();
                                listening_socket.send_to(client_address, dns_resp);
                            }
                        }
                    }
                }
                finally
                {
                    if (serving_ids.contains(query.id)) serving_ids.remove(query.id);
                }
            }
        }
        return 0;
    }

    bool check_forward_to_dns(Message query)
    {
        bool dns = true;
        if (query.opcode == OperationCode.IP_TO_NAME)
        {
            // not used. DNS use format "3.2.1.10.in-addr.arpa."
            // or "1.0.0.0.0.0.0.0....0.ip6.arpa" with 32 octets.
            log_info("Got obsolete DNS request of type IP_TO_NAME");
        }
        else if (query.opcode == OperationCode.NAME_TO_IP)
        {
            // standard and inverse query
            if (query.question_recs.size > 0)
            {
                if (query.question_recs.size > 1)
                {
                    log_warn(@"Got $(query.question_recs.size) questions in a message. Only first one will be examined.");
                }
                QuestionRecord q = query.question_recs[0];
                if (q.rrclass == RRClass.IN
                    && q.rrtype == RRType.A)
                {
                    dns = q.name.length < 5 || q.name.substring(q.name.length-5).up() != ".NTK.";
                }
                else if ((q.rrclass == RRClass.IN || q.rrclass == RRClass.QR_ANY)
                    && q.rrtype == RRType.PTR
                    && q.name.length > 20
                    && q.name.substring(q.name.length-14).up() == ".IN-ADDR.ARPA.")
                {
                    dns = q.name.substring(q.name.length-17).up() != ".10.IN-ADDR.ARPA.";
                }
            }
        }
        else
        {
            // not supported in ANDNA, pass to DNS
            log_info("Got unsupported type of DNS request");
        }
        return dns;
    }

    Gee.List<ResRecord> std_qry(QuestionRecord q)
    {
        string name = q.name.substring(0, q.name.length-5);
        Gee.List<NtkAddrInfo> resp = resolv(name);
        // make a DNS response
        ArrayList<ResRecord> ret = new ArrayList<ResRecord>();
        foreach (NtkAddrInfo addrinfo in resp)
        {
            uint8[] addr = (addrinfo.address as NtkInetAddr).addr;
            ResRecord r = new ResRecord.inetaddr(q.name, RRType.A, RRClass.IN, 300 /*TODO ttl*/, addr);
            ret.add(r);
        }
        return ret;
    }

    Gee.List<ResRecord> inv_qry(QuestionRecord q)
    {
        string ipstr = q.name.substring(0, q.name.length-14);
        string[] parts = ipstr.split(".");
        string[] rev = new string[parts.length];
        int j = 0;
        for (int i = parts.length-1; i >= 0; i--)
            rev[j++] = parts[i];
        ipstr = string.joinv(".", rev);
        uint8 addr_bytes[4];
        address_v4_str_to_bytes(ipstr, addr_bytes);
        NtkAddr addr = new NtkInetAddr(addr_bytes, 0);
        Gee.List<string> resp = inverse(IpFamily.IPV4, addr);
        // make a DNS response
        ArrayList<ResRecord> ret = new ArrayList<ResRecord>();
        foreach (string ptr in resp)
        {
            ResRecord r = new ResRecord.ptr(q.name, RRType.PTR, RRClass.IN, 300 /*TODO ttl*/, ptr+".NTK.");
            ret.add(r);
        }
        return ret;
    }
}

string[] read_file(string path)
{
    string[] ret = new string[0];
    if (FileUtils.test(path, FileTest.EXISTS))
    {
        try
        {
            string contents;
            assert(FileUtils.get_contents(path, out contents));
            ret = contents.split("\n");
        }
        catch (FileError e) {error("%s: %d: %s".printf(e.domain.to_string(), e.code, e.message));}
    }
    return ret;
}

void main(string[] argv)
{
    // read configuration from resolv.conf or else exit with error
    dns_list = new ArrayList<SocketAddress>();
    string[] lines = read_file(Config.SYSCONF_DIR + "/resolv.conf");
    foreach (string line in lines)
    {
        if (Regex.match_simple("""^\s*nameserver\s+""", line))
        {
            MatchInfo m;
            Regex r = new Regex("""[0-9.]+""");
            if (! r.match(line, 0, out m)) continue;
            InetSocketAddress dns_address = new InetSocketAddress(new InetAddress.from_string(m.fetch(0)), 53);
            dns_list.add(dns_address);
        }
    }
    if (dns_list.is_empty)
    {
        error("Error reading resolv.conf: no IPv4 nameserver found");
    }

    // read configuration from ntkresolv.ini or else exit with error
    KeyFile conf = new KeyFile();
    try
    {
        conf.load_from_file(Config.SYSCONF_DIR + "/ntkresolv/ntkresolv.ini", KeyFileFlags.NONE);
    }
    catch (KeyFileError e)
    {
        error(e.message);
    }
    if (!conf.has_group("NTKRESOLV") || !conf.has_key("NTKRESOLV", "DNS_TO_ANDNA"))
    {
        error("Missing key DNS_TO_ANDNA in section [NTKRESOLV] of file ntkresolv.ini");
    }
    string dns_to_andna_address = conf.get_string("NTKRESOLV", "DNS_TO_ANDNA");
    InetSocketAddress bind_address = new InetSocketAddress(new InetAddress.from_string(dns_to_andna_address), 53);

    listening_socket = new Socket(SocketFamily.IPV4, GLib.SocketType.DATAGRAM, SocketProtocol.UDP);
    // listening_socket.set_option(Posix.SOL_SOCKET, Posix.SO_REUSEADDR, 1);
    listening_socket.bind(bind_address, true);
    serving_ids = new ArrayList<uint16>();
    while (true)
    {
        SocketAddress client_address;
        uint8 buffer[1024];
        ssize_t rec_size = listening_socket.receive_from(out client_address, buffer);
        // copy
        uint8[] buf = new uint8[rec_size];
        for (int i = 0; i < rec_size; i++) buf[i] = buffer[i];
        // thread to handle request
        HandleRequestArg a = new HandleRequestArg(client_address, buf);
        Thread<int> thread = new Thread<int>.@try ("handle_request", a.handle_request);
    }
}

