/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2014 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gee;
using Tasklets;
using zcd;
using Andns;
using Netsukuku;
using Netsukuku.InetUtils;

namespace Ntkresolv
{
#if log_tasklet
    private string tasklet_id()
    {
        string ret = @"$(Tasklet.self().id)";
        int len = ret.length;
        for (int i = 0; i < 5-len; i++) ret = " " + ret;
        return @"[$(ret)] ";
    }
#else
    private string tasklet_id()
    {
        return "";
    }
#endif
    internal void log_debug(string msg)     {Posix.syslog(Posix.LOG_DEBUG,
                    tasklet_id() + "DEBUG "  + msg);}
    internal void log_info(string msg)      {Posix.syslog(Posix.LOG_INFO,
                    tasklet_id() + "INFO "   + msg);}
    internal void log_notice(string msg)    {Posix.syslog(Posix.LOG_NOTICE,
                    tasklet_id() + "INFO+ "  + msg);}
    internal void log_warn(string msg)      {Posix.syslog(Posix.LOG_WARNING,
                    tasklet_id() + "INFO++ " + msg);}
    internal void log_error(string msg)     {Posix.syslog(Posix.LOG_ERR,
                    tasklet_id() + "ERROR "  + msg);}
    internal void log_critical(string msg)  {Posix.syslog(Posix.LOG_CRIT,
                    tasklet_id() + "ERROR+ " + msg);}

    public class NtkAddr : Object
    {
        public NtkAddr()
        {}
    }

    public class NtkInetAddr : NtkAddr
    {
        public uint8 addr[4];
        public uint16 port;
        public NtkInetAddr(uint8[] addr, uint16 port)
        {
            base();
            assert(addr.length == 4);
            for (int i = 0; i < 4; i++)
            {
                this.addr[i] = addr[i];
            }
            this.port = port;
        }
        public NtkInetAddr.from_pointer(uint8 *addr, uint16 port)
        {
            base();
            for (int i = 0; i < 4; i++)
            {
                this.addr[i] = addr[i];
            }
            this.port = port;
        }
    }

    public enum IpFamily {
        IPV4,
        IPV6,
        UNSPEC
    }
    public enum IpProtocol {
        IP,
        TCP,
        UDP
    }
    public enum SocketType {
        DATAGRAM,
        STREAM,
        UNSPEC
    }

    public class NtkAddrInfo : Object
    {
        public IpFamily family;
        public SocketType socket_type;
        public IpProtocol protocol;
        public NtkAddr address;
    }

    errordomain NtkresolvError {
        GENERIC
    }

    /** There are 2 ways to use this library. One is for applications that do not
      * use the library 'tasklet' on their own. The other is for those that do.
      *
      * 1.
      * Call directly the method 'resolv' from any thread.
      *
      * 2.
      * After having initialized the Tasklet system, call once the method 'init'.
      * Then call the method 'tasklet_resolv' from any tasklet.
      *
      * If you use this library from another library and you don't know whether
      * the calling application is using the library 'tasklet' on its own, then
      * you should call once the method 'init' and then 'tasklet_resolv'. But in
      * this case you cannot guarantee that the function is going to work in the
      * case that it is called concurrently from several threads.
      */

    public void init(bool tasklet_initialized=true)
    {
        if (!tasklet_initialized)
        {
            Tasklet.init();
        }
        // Initialize rpc library
        Serializer.init();
        // Register serializable types from rpc model
        Andns.RpcAndns.init();
        Netsukuku.RpcNtk.init();
    }

    class ResolvCall : Object
    {
        public ResolvCall
            (string node,
             string? service,
             NtkAddrInfo? _hints,
             string tcpaddress)
        {
            this.node = node;
            this.service = service;
            this._hints = _hints;
            this.tcpaddress = tcpaddress;
            completed = false;
            started = false;
        }
        public string node;
        public string? service;
        public NtkAddrInfo? _hints;
        public string tcpaddress;
        public Gee.List<NtkAddrInfo> retval;
        public Error? e=null;
        public bool completed;
        public bool started;
    }
    ArrayList<ResolvCall> lst_dispatched_resolv;

    public Gee.List<NtkAddrInfo>
    resolv
            (string node,
             string? service=null,
             NtkAddrInfo? _hints=null,
             string tcpaddress="127.0.0.1") throws Error
    {
        if (Tasklet.init())
        {
            // Succeeded: first initialization of tasklet system.
            init();
            // Here we will execute a dispatcher until we can kill the tasklet system.
            lst_dispatched_resolv = new ArrayList<ResolvCall>();
            // enqueue first request
            ResolvCall my_req = new ResolvCall(node, service, _hints, tcpaddress);
            lst_dispatched_resolv.add(my_req);
            // dispatcher
            while (true)
            {
                ArrayList<ResolvCall> todel = new ArrayList<ResolvCall>();
                foreach (ResolvCall req in lst_dispatched_resolv)
                {
                    if (req.completed) todel.add(req);
                    if (!req.started)
                    {
                        Tasklet.tasklet_callback((tpar1) => {
                            ResolvCall tasklet_req = (ResolvCall)tpar1;
                            try {
                                tasklet_req.retval = tasklet_resolv
                                        (tasklet_req.node,
                                         tasklet_req.service,
                                         tasklet_req._hints,
                                         tasklet_req.tcpaddress);
                                tasklet_req.completed = true;
                            } catch (Error e) {
                                tasklet_req.e = e;
                                tasklet_req.completed = true;
                            }
                        },
                        req);
                        req.started = true;
                    }
                }
                foreach (ResolvCall req in todel) lst_dispatched_resolv.remove(req);
                if (lst_dispatched_resolv.is_empty) break;
                Tasklet.nap(0, 1000);
            }
            Tasklet.kill();
            if (my_req.e != null) throw my_req.e;
            return my_req.retval;
        }
        else
        {
            // Failed: tasklet system already ongoing.
            // We presume that we are in another system thread, because if the caller
            //  had started on its own a tasklet system, then it would have called
            //  directly the method tasklet_resolv.
            // Here we enqueue a request to the dispatcher.
            ResolvCall my_req = new ResolvCall(node, service, _hints, tcpaddress);
            lst_dispatched_resolv.add(my_req);
            // this wait will work only if we are in another system thread.
            while (! my_req.completed) Posix.usleep(10000);
            if (my_req.e != null) throw my_req.e;
            return my_req.retval;
        }
    }

    public Gee.List<NtkAddrInfo>
    tasklet_resolv
            (string node,
             string? service=null,
             NtkAddrInfo? _hints=null,
             string tcpaddress="127.0.0.1") throws Error
    {
        // default hints
        NtkAddrInfo hints;
        if (_hints == null)
        {
            hints = new NtkAddrInfo();
            hints.family = IpFamily.IPV4;
            hints.protocol = IpProtocol.IP;
            hints.socket_type = SocketType.UNSPEC;
        }
        else hints = _hints;

        AndnaServiceKey serv_key = AndnaServiceKey.NULL_SERV_KEY;
        if (service != null && hints.protocol != IpProtocol.IP)
        {
            string proto = hints.protocol == IpProtocol.TCP ? "tcp" : "udp";
            serv_key = new AndnaServiceKey(service, proto);
        }
        AndnaResolveTCPClient client = new AndnaResolveTCPClient(tcpaddress, 53000);
        client.retry_connect = false;
        AndnaQuery query = new AndnaQuery.name_to_ip_ntk(
                false,
                false,
                AndnaQueryProtocol.TCP,
                AndnaQueryIPVersion.IPV4,
                crypto_hash(node),
                serv_key);
        int id = Random.int_range(0, 32767); // (2^15-1)
        ArrayList<NtkAddrInfo> ret = new ArrayList<NtkAddrInfo>();
        AndnaResponse resp = client.get_query_handler(id).resolve(query);
        if (resp.rcode == AndnaResponseCode.NO_DOMAIN)
        {
            // no domain => empty list
            return ret;
        }
        if (resp.rcode != AndnaResponseCode.NO_ERROR)
        {
            // quick handling: type of error in the message of a generic error
            throw new NtkresolvError.GENERIC(@"$(resp.rcode)");
        }
        if (resp.answers.size == 0)
        {
            throw new NtkresolvError.GENERIC("Response Code is NoError but list is empty.");
        }
        foreach (AndnaResponseAnswer a in resp.answers)
        {
            NtkAddrInfo ainfo = new NtkAddrInfo();
            ainfo.family = IpFamily.IPV4;
            ainfo.protocol = hints.protocol;
            ainfo.socket_type = hints.socket_type;
            ainfo.address = new NtkInetAddr(a.ipv4_ip, a.port_number);
            ret.add(ainfo);
        }
        return ret;
    }

    class InverseCall : Object
    {
        public InverseCall
            (IpFamily family,
             NtkAddr addr,
             string tcpaddress)
        {
            this.family = family;
            this.addr = addr;
            this.tcpaddress = tcpaddress;
            completed = false;
            started = false;
        }
        public IpFamily family;
        public NtkAddr addr;
        public string tcpaddress;
        public Gee.List<string> retval;
        public Error? e=null;
        public bool completed;
        public bool started;
    }
    ArrayList<InverseCall> lst_dispatched_inverse;

    public Gee.List<string>
    inverse
            (IpFamily family,
             NtkAddr addr,
             string tcpaddress="127.0.0.1") throws Error
    {
        if (Tasklet.init())
        {
            // Succeeded: first initialization of tasklet system.
            init();
            // Here we will execute a dispatcher until we can kill the tasklet system.
            lst_dispatched_inverse = new ArrayList<InverseCall>();
            // enqueue first request
            InverseCall my_req = new InverseCall(family, addr, tcpaddress);
            lst_dispatched_inverse.add(my_req);
            // dispatcher
            while (true)
            {
                ArrayList<InverseCall> todel = new ArrayList<InverseCall>();
                foreach (InverseCall req in lst_dispatched_inverse)
                {
                    if (req.completed) todel.add(req);
                    if (!req.started)
                    {
                        Tasklet.tasklet_callback((tpar1) => {
                            InverseCall tasklet_req = (InverseCall)tpar1;
                            try {
                                tasklet_req.retval = tasklet_inverse
                                        (tasklet_req.family,
                                         tasklet_req.addr,
                                         tasklet_req.tcpaddress);
                                tasklet_req.completed = true;
                            } catch (Error e) {
                                tasklet_req.e = e;
                                tasklet_req.completed = true;
                            }
                        },
                        req);
                        req.started = true;
                    }
                }
                foreach (InverseCall req in todel) lst_dispatched_inverse.remove(req);
                if (lst_dispatched_inverse.is_empty) break;
                Tasklet.nap(0, 1000);
            }
            Tasklet.kill();
            if (my_req.e != null) throw my_req.e;
            return my_req.retval;
        }
        else
        {
            // Failed: tasklet system already ongoing.
            // We presume that we are in another system thread, because if the caller
            //  had started on its own a tasklet system, then it would have called
            //  directly the method tasklet_inverse.
            // Here we enqueue a request to the dispatcher.
            InverseCall my_req = new InverseCall(family, addr, tcpaddress);
            lst_dispatched_inverse.add(my_req);
            // this wait will work only if we are in another system thread.
            while (! my_req.completed) Posix.usleep(10000);
            if (my_req.e != null) throw my_req.e;
            return my_req.retval;
        }
    }

    public Gee.List<string>
    tasklet_inverse
            (IpFamily family,
             NtkAddr addr,
             string tcpaddress="127.0.0.1") throws Error
    {
        AndnaQueryIPVersion ipversion = AndnaQueryIPVersion.IPV4;
        weak uint8[] addr_bytes = null;
        if (family == IpFamily.IPV6)
        {
            // TODO IPv6
        }
        else
        {
            addr_bytes = (addr as NtkInetAddr).addr;
        }
        
        AndnaResolveTCPClient client = new AndnaResolveTCPClient(tcpaddress, 53000);
        client.retry_connect = false;
        AndnaQuery query = new AndnaQuery.ip_to_name_ntk(
                false,
                AndnaQueryProtocol.TCP,
                ipversion,
                addr_bytes);
        int id = Random.int_range(0, 32767); // (2^15-1)
        ArrayList<string> ret = new ArrayList<string>();
        AndnaResponse resp = client.get_query_handler(id).resolve(query);
        if (resp.rcode == AndnaResponseCode.NO_DOMAIN)
        {
            // no domain => empty list
            return ret;
        }
        if (resp.rcode != AndnaResponseCode.NO_ERROR)
        {
            // quick handling: type of error in the message of a generic error
            throw new NtkresolvError.GENERIC(@"$(resp.rcode)");
        }
        if (resp.answers.size == 0)
        {
            throw new NtkresolvError.GENERIC("Response Code is NoError but list is empty.");
        }
        foreach (AndnaResponseAnswer a in resp.answers)
        {
            ret.add(a.hostname);
        }
        return ret;
    }
}
