/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2014 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

namespace Nss.Test
{
    string logger;
    const bool output = false;
    public void print_out(string s)
    {
        if (output) print(s);
    }

    public class NssTester : Object
    {
        public void set_up ()
        {
            logger = "";
        }

        public void tear_down ()
        {
            logger = "";
        }

        public void test_name_no_ntk()
        {
            // This test verifies that the module reports UNAVAIL EPERM if name does
            //  not end in .NTK.
            PosixExtra.HostEnt ret;
            PosixExtra.HostEnt *result;
            int myerrno = 0;
            char buffer[4096];
            Nss.Status risp = _nss_andna_gethostbyname2_r
                    ("name.com",
                     Posix.AF_INET,
                     &ret,
                     buffer,
                     buffer.length,
                     &result,
                     &myerrno);
            assert(risp == Nss.Status.Unavailable);
            assert(myerrno == Posix.EPERM);
        }

        public void test_name_ipv6()
        {
            // This test verifies that the module reports UNAVAIL ESRCH if IPv6
            //  is requested.
            PosixExtra.HostEnt ret;
            PosixExtra.HostEnt *result;
            int myerrno = 0;
            char buffer[4096];
            Nss.Status risp = _nss_andna_gethostbyname2_r
                    ("name.com",
                     Posix.AF_INET6,
                     &ret,
                     buffer,
                     buffer.length,
                     &result,
                     &myerrno);
            assert(risp == Nss.Status.Unavailable);
            assert(myerrno == Posix.ESRCH);
        }

        public void test_addr_no_class10()
        {
            // This test verifies that the module reports UNAVAIL EPERM if IPv4 address
            //  is not in 10.0.0.0/8 class.
            PosixExtra.HostEnt ret;
            PosixExtra.HostEnt *result;
            int myerrno = 0;
            char buffer[4096];
            Posix.InAddr addr = {};
            addr.s_addr = Posix.inet_addr("192.168.2.2");
            void *addr_p = &addr;
            Posix.socklen_t addr_len = (Posix.socklen_t) sizeof(Posix.InAddr);
            Nss.Status risp = _nss_andna_gethostbyaddr_r
                    (addr_p,
                     addr_len,
                     Posix.AF_INET,
                     &ret,
                     buffer,
                     buffer.length,
                     &result,
                     &myerrno);
            assert(risp == Nss.Status.Unavailable);
            assert(myerrno == Posix.EPERM);
        }

        public void test_addr_ipv6()
        {
            // This test verifies that the module reports UNAVAIL ESRCH if IPv6
            //  is requested.
            PosixExtra.HostEnt ret;
            PosixExtra.HostEnt *result;
            int myerrno = 0;
            char buffer[4096];
            Posix.In6Addr addr = {}; // any address would do for this test
            void *addr_p = &addr;
            Posix.socklen_t addr_len = (Posix.socklen_t) sizeof(Posix.In6Addr);
            Nss.Status risp = _nss_andna_gethostbyaddr_r
                    (addr_p,
                     addr_len,
                     Posix.AF_INET6,
                     &ret,
                     buffer,
                     buffer.length,
                     &result,
                     &myerrno);
            assert(risp == Nss.Status.Unavailable);
            assert(myerrno == Posix.ESRCH);
        }

        public static int main(string[] args)
        {
            GLib.Test.init(ref args);

            GLib.Test.add_func ("/NssModule/NameNotNtk", () => {
                var x = new NssTester();
                x.set_up();
                x.test_name_no_ntk();
                x.tear_down();
            });
            GLib.Test.add_func ("/NssModule/NameIPv6", () => {
                var x = new NssTester();
                x.set_up();
                x.test_name_ipv6();
                x.tear_down();
            });
            GLib.Test.add_func ("/NssModule/AddrNot10", () => {
                var x = new NssTester();
                x.set_up();
                x.test_addr_no_class10();
                x.tear_down();
            });
            GLib.Test.add_func ("/NssModule/AddrIPv6", () => {
                var x = new NssTester();
                x.set_up();
                x.test_addr_ipv6();
                x.tear_down();
            });
            GLib.Test.run();
            return 0;
        }
    }
}
