/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2014 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gee;
using zcd;
using Andns;
using Netsukuku;
using Netsukuku.InetUtils;

namespace Ntkresolv
{
    internal void log_debug(string msg)     {Posix.syslog(Posix.LOG_DEBUG,
                    "DEBUG "  + msg);}
    internal void log_info(string msg)      {Posix.syslog(Posix.LOG_INFO,
                    "INFO "   + msg);}
    internal void log_notice(string msg)    {Posix.syslog(Posix.LOG_NOTICE,
                    "INFO+ "  + msg);}
    internal void log_warn(string msg)      {Posix.syslog(Posix.LOG_WARNING,
                    "INFO++ " + msg);}
    internal void log_error(string msg)     {Posix.syslog(Posix.LOG_ERR,
                    "ERROR "  + msg);}
    internal void log_critical(string msg)  {Posix.syslog(Posix.LOG_CRIT,
                    "ERROR+ " + msg);}

    public class NtkAddr : Object
    {
        public NtkAddr()
        {}
    }

    public class NtkInetAddr : NtkAddr
    {
        public uint8 addr[4];
        public uint16 port;
        public NtkInetAddr(uint8[] addr, uint16 port)
        {
            base();
            assert(addr.length == 4);
            for (int i = 0; i < 4; i++)
            {
                this.addr[i] = addr[i];
            }
            this.port = port;
        }
        public NtkInetAddr.from_pointer(uint8 *addr, uint16 port)
        {
            base();
            for (int i = 0; i < 4; i++)
            {
                this.addr[i] = addr[i];
            }
            this.port = port;
        }
    }

    public enum IpFamily {
        IPV4,
        IPV6,
        UNSPEC
    }
    public enum IpProtocol {
        IP,
        TCP,
        UDP
    }
    public enum SocketType {
        DATAGRAM,
        STREAM,
        UNSPEC
    }

    public class NtkAddrInfo : Object
    {
        public IpFamily family;
        public SocketType socket_type;
        public IpProtocol protocol;
        public NtkAddr address;
    }

    errordomain NtkresolvError {
        GENERIC
    }

    /** Call directly the method 'resolv' from any thread.
      */

    private bool init_done = false;
    public void init()
    {
        if (init_done) return;
        // Initialize rpc library
        Serializer.init();
        // Register serializable types from rpc model
        Andns.RpcAndns.init();
        init_done = true;
    }

    public Gee.List<NtkAddrInfo>
    resolv
            (string node,
             string? service=null,
             NtkAddrInfo? _hints=null,
             string tcpaddress="127.0.0.1") throws Error
    {
        init();
        // default hints
        NtkAddrInfo hints;
        if (_hints == null)
        {
            hints = new NtkAddrInfo();
            hints.family = IpFamily.IPV4;
            hints.protocol = IpProtocol.IP;
            hints.socket_type = SocketType.UNSPEC;
        }
        else hints = _hints;

        AndnaServiceKey serv_key = AndnaServiceKey.NULL_SERV_KEY;
        if (service != null && hints.protocol != IpProtocol.IP)
        {
            string proto = hints.protocol == IpProtocol.TCP ? "tcp" : "udp";
            serv_key = new AndnaServiceKey(service, proto);
        }
        AndnaResolveTCPClient client = new AndnaResolveTCPClient(tcpaddress, 53000);
        client.retry_connect = false;
        AndnaQuery query = new AndnaQuery.name_to_ip_ntk(
                false,
                false,
                AndnaQueryProtocol.TCP,
                AndnaQueryIPVersion.IPV4,
                crypto_hash(node),
                serv_key);
        int id = Random.int_range(0, 32767); // (2^15-1)
        ArrayList<NtkAddrInfo> ret = new ArrayList<NtkAddrInfo>();
        AndnaResponse resp = client.get_query_handler(id).resolve(query);
        if (resp.rcode == AndnaResponseCode.NO_DOMAIN)
        {
            // no domain => empty list
            return ret;
        }
        if (resp.rcode != AndnaResponseCode.NO_ERROR)
        {
            // quick handling: type of error in the message of a generic error
            throw new NtkresolvError.GENERIC(@"$(resp.rcode)");
        }
        if (resp.answers.size == 0)
        {
            throw new NtkresolvError.GENERIC("Response Code is NoError but list is empty.");
        }
        foreach (AndnaResponseAnswer a in resp.answers)
        {
            NtkAddrInfo ainfo = new NtkAddrInfo();
            ainfo.family = IpFamily.IPV4;
            ainfo.protocol = hints.protocol;
            ainfo.socket_type = hints.socket_type;
            ainfo.address = new NtkInetAddr(a.ipv4_ip, a.port_number);
            ret.add(ainfo);
        }
        return ret;
    }

    public Gee.List<string>
    inverse
            (IpFamily family,
             NtkAddr addr,
             string tcpaddress="127.0.0.1") throws Error
    {
        init();
        AndnaQueryIPVersion ipversion = AndnaQueryIPVersion.IPV4;
        weak uint8[] addr_bytes = null;
        if (family == IpFamily.IPV6)
        {
            // TODO IPv6
        }
        else
        {
            addr_bytes = (addr as NtkInetAddr).addr;
        }
        
        AndnaResolveTCPClient client = new AndnaResolveTCPClient(tcpaddress, 53000);
        client.retry_connect = false;
        AndnaQuery query = new AndnaQuery.ip_to_name_ntk(
                false,
                AndnaQueryProtocol.TCP,
                ipversion,
                addr_bytes);
        int id = Random.int_range(0, 32767); // (2^15-1)
        ArrayList<string> ret = new ArrayList<string>();
        AndnaResponse resp = client.get_query_handler(id).resolve(query);
        if (resp.rcode == AndnaResponseCode.NO_DOMAIN)
        {
            // no domain => empty list
            return ret;
        }
        if (resp.rcode != AndnaResponseCode.NO_ERROR)
        {
            // quick handling: type of error in the message of a generic error
            throw new NtkresolvError.GENERIC(@"$(resp.rcode)");
        }
        if (resp.answers.size == 0)
        {
            throw new NtkresolvError.GENERIC("Response Code is NoError but list is empty.");
        }
        foreach (AndnaResponseAnswer a in resp.answers)
        {
            ret.add(a.hostname);
        }
        return ret;
    }
}
