/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2013 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gtk;
using Gee;
using zcd;
using Tasklets;
using Netsukuku;

namespace Monitor
{
    // position of fields in treemodel
    enum LISTSERVERS {
        STR_REPR,
        STR_PUBK,
        INT_PRIO,
        INT_WEIGHT,
        INT_PORT,
        SERVER,
        NUMCOLUMNS
    }

    public class Servers : Object
    {
        public Window win_servers {get; private set;}
        public Widget widget_root {get; private set;}
        private HashMap<AndnaServiceKey, ArrayList<AndnaPrivateConfigurationServer>> services;
        private AndnaServiceKey servicekey;
        private ArrayList<AndnaPrivateConfigurationServer> servers;
        private string name;
        private ListStore liststore_servers;
        private TreeView tv_servers;
        private Button but_add;
        private Button but_remove;
        private Button but_details;
        public Entry txt_name {get; private set;}
        public Entry txt_proto {get; private set;}
        public signal void go_back();

        public
        Servers
        (HashMap<AndnaServiceKey, ArrayList<AndnaPrivateConfigurationServer>> services,
         AndnaServiceKey servicekey,
         string name)
        {
            this.services = services;
            this.servicekey = servicekey;
            this.servers = services[servicekey];
            this.name = name;

            Builder builder = new Builder ();
            builder.add_from_resource ("/org/netsukuku/monitorradar/servers.ui");
            builder.connect_signals (this);

            win_servers = builder.get_object ("win_servers") as Window;
            widget_root = builder.get_object ("widget_root") as Widget;
            liststore_servers = builder.get_object ("liststore_servers") as ListStore;
            tv_servers = builder.get_object ("tv_servers") as TreeView;
            but_add = builder.get_object ("but_add") as Button;
            but_remove = builder.get_object ("but_remove") as Button;
            but_details = builder.get_object ("but_details") as Button;
            txt_name = builder.get_object ("txt_name") as Entry;
            txt_proto = builder.get_object ("txt_proto") as Entry;
            if (! AndnaServiceKey.equal_func(AndnaServiceKey.NULL_SERV_KEY, servicekey))
            {
                txt_name.text = servicekey.name;
                txt_proto.text = servicekey.proto;
                txt_name.can_focus = true;
                txt_proto.can_focus = true;
            }
            win_servers.destroy.connect(going_back);

            TreeSelection sel_servers = tv_servers.get_selection();
            sel_servers.set_mode(SelectionMode.SINGLE);
            sel_servers.changed.connect(() => {tv_servers_selection_changed();});

            refresh();
        }

        void going_back()
        {
            if (! AndnaServiceKey.equal_func(AndnaServiceKey.NULL_SERV_KEY, servicekey))
            {
                if (txt_name.text != servicekey.name ||
                        txt_proto.text != servicekey.proto)
                {
                    // the servers go to the new slot
                    AndnaServiceKey new_servicekey =
                            new AndnaServiceKey(txt_name.text, txt_proto.text);
                    services[new_servicekey] = servers;
                    services.unset(servicekey);
                }
            }
            go_back();
        }

        [CCode (instance_pos = -1)]
        public void add_server_clicked(Button source)
        {
            add_server();
        }

        void add_server()
        {
            // adds a server
            servers.add(new AndnaPrivateConfigurationServer(null, null, 0, 1, 1));
            refresh();
        }

        [CCode (instance_pos = -1)]
        public void remove_server_clicked(Button source)
        {
            servers.remove(selected_server);
            refresh();
        }

        private Server server;
        [CCode (instance_pos = -1)]
        public void details_server_clicked(Button source)
        {
            server = new Server(selected_server, name, @"$(servicekey)");
            Window w = server.win_server;
            w.title = "server";
            w.show_all();
            // TODO uncomment: modal is buggy in ubuntu right now
            // w.modal = true;
            // w.set_transient_for(win_servers);
            w.set_position(WindowPosition.CENTER_ON_PARENT);
            server.go_back.connect(update);
        }

        public void update(AndnaPrivateConfigurationServer rec_old,
                           AndnaPrivateConfigurationServer rec_new)
        {
            servers.remove(rec_old);
            servers.add(rec_new);
            refresh();
        }

        public void refresh()
        {
            // refresh UI
            liststore_servers.clear();
            TreeIter iter;
            servers.sort(
                    (a, b) => {
                        // negative value if a < b; zero if a = b; positive value if a > b
                        if (a.priority != b.priority) return a.priority - b.priority;
                        if (a.weight != b.weight) return b.weight - a.weight;
                        return a.port_number - b.port_number;
                    });
            foreach (AndnaPrivateConfigurationServer server in servers)
            {
                liststore_servers.append(out iter);
                string alias = "<myself>";
                string pubk = "null";
                if (server.alias != null)
                {
                    alias = server.alias;
                    pubk = @"$(server.pubk)";
                }
                liststore_servers.@set(iter,
                        LISTSERVERS.STR_REPR, @"$(alias)");
                liststore_servers.@set(iter,
                        LISTSERVERS.STR_PUBK, @"$(pubk)");
                liststore_servers.@set(iter,
                        LISTSERVERS.INT_PORT, server.port_number);
                liststore_servers.@set(iter,
                        LISTSERVERS.INT_PRIO, server.priority);
                liststore_servers.@set(iter,
                        LISTSERVERS.INT_WEIGHT, server.weight);
                liststore_servers.@set(iter,
                        LISTSERVERS.SERVER, server);
            }
        }

        private AndnaPrivateConfigurationServer? selected_server;
        void tv_servers_selection_changed()
        {
            TreePath? path;
            unowned TreeViewColumn? column;
            TreeIter iter;
            tv_servers.get_cursor(out path, out column);
            if (path != null)
            {
                if (liststore_servers.get_iter(out iter, path))
                {
                    liststore_servers.@get(iter,
                            LISTSERVERS.SERVER,
                            out selected_server);
                    if (servers.size > 1)
                    {
                        but_remove.sensitive = true;
                    }
                    else
                    {
                        but_remove.sensitive = false;
                    }
                    but_details.sensitive = true;
                }
                else
                {
                    selected_server = null;
                    but_remove.sensitive = false;
                    but_details.sensitive = false;
                }
            }
            else
            {
                selected_server = null;
                but_remove.sensitive = false;
                but_details.sensitive = false;
            }
        }
    }
}

