/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2014 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gtk;
using Gee;
using zcd;
using Netsukuku;

namespace Monitor
{
    // position of fields in treemodel
    enum LISTLOGS {
        STR_LOG,
        INT_POS,
        NUMCOLUMNS
    }

    public class NodeTaskletLogs : Object
    {
        private AddressManagerFakeRmtGetter client_getter;
        private ListStore liststore_logs;
        private InfoNode info_node;
        private SpinButton txt_id;
        private Label lbl_tasklet_name;

        public NodeTaskletLogs(Builder builder, Box box_parent, AddressManagerFakeRmtGetter client_getter)
        {
            this.client_getter = client_getter;
            builder.connect_signals (this);
            txt_id = builder.get_object ("txt_id") as SpinButton;
            lbl_tasklet_name = builder.get_object ("lbl_tasklet_name") as Label;
            liststore_logs = builder.get_object ("liststore_logs") as ListStore;
            Widget widget_tasklet_logs = builder.get_object ("widget_root") as Widget;
            widget_tasklet_logs.reparent(box_parent);

            info_node = client_getter.get_client().maproute.report_yourself();
        }

        private int impl_start_operations()
        {
            while (true)
            {
                refresh_logs();

                if (nap_until_condition(1000,
                    () => {
                        return t_op_aborting;
                    })) break;
            }
            return 0;
        }

        private Thread<int>? t_op;
        private bool t_op_aborting;
        public void start_operations()
        {
            if (t_op == null)
            {
                t_op_aborting = false;
                t_op = new Thread<int>(null, impl_start_operations);
            }
        }

        public void stop_operations()
        {
            if (t_op != null)
            {
                t_op_aborting = true;
                t_op.join();
                t_op = null;
            }
        }

        /** retrieve and display data
          */
        private void refresh_logs() throws Error
        {
            try
            {
                AsyncQueue<int> q = new AsyncQueue<int>();
                MainContext.@default().invoke(() => {
                        get_current_id(q);
                        return false;});
                int current_id = q.pop();

                // Get max id
                Gee.List<string> str_ids = client_getter.get_client().report_running_tasklets();
                int max_id = 0;
                foreach (string str_id in str_ids)
                {
                    int id = int.parse(str_id);
                    if (id > max_id) max_id = id;
                }

                // Get logs for current_id
                Gee.List<string> logs = client_getter.get_client().report_tasklet_logs(current_id);
                string name = logs.remove_at(0);

                MainContext.@default().invoke(() => {
                        set_max_id(max_id);
                        set_tasklet_name(name);
                        display_logs(logs);
                        return false;});
            }
            catch (Error e)
            {
                string e_message = e.message;
                MainContext.@default().invoke(() => {display_error(e_message); return false;});
            }
        }

        void set_tasklet_name(string name)
        {
            lbl_tasklet_name.set_text(name);
        }

        void get_current_id(AsyncQueue<int> q)
        {
            q.push(txt_id.get_value_as_int());
        }

        void set_max_id(int max_id)
        {
            txt_id.adjustment.upper = max_id;
        }

        private int previous_id = 0;
        private int last_log_pos;
        void display_logs(Gee.List<string> logs)
        {
            int current_id = txt_id.get_value_as_int();
            if (current_id != previous_id) clear_logs();
            previous_id = current_id;
            if (! logs.is_empty)
            {
                int next_pos = int.parse(logs.remove_at(0));
                foreach (string s in logs)
                {
                    if (next_pos > last_log_pos)
                    {
                        TreeIter iter;
                        liststore_logs.append(out iter);
                        liststore_logs.@set(iter, LISTLOGS.STR_LOG, s);
                        liststore_logs.@set(iter, LISTLOGS.INT_POS, next_pos);
                        last_log_pos = next_pos;
                    }
                    next_pos++;
                }
            }
        }

        void clear_logs()
        {
            liststore_logs.clear();
            last_log_pos = 0;
        }

        void display_error(string e_message)
        {
        }
    }
}
