/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2013 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gtk;
using Gee;
using zcd;
using Netsukuku;

namespace Monitor
{
    // position of fields in treemodel
    enum LISTROUTES {
        INFOROUTE,
        STR_GNODE_DEST,
        STR_GID,
        STR_REM,
        NUMCOLUMNS
    }

    // position of fields in treemodel
    enum LISTHOPS {
        STR_GNODE_HOP,
        NUMCOLUMNS
    }

    public class NodeRoutes : Object
    {
        private AddressManagerFakeRmtGetter client_getter;
        private Tasklets.Timer t_info_node;
        private InfoNode _info_node;
        private InfoNode info_node {
            get {
                if (t_info_node == null || t_info_node.is_expired())
                {
                    _info_node = client_getter.get_client().maproute.report_yourself();
                    t_info_node = new Tasklets.Timer(1000);
                }
                return _info_node;
            }
        }
        private ListStore liststore_routes;
        private ListStore liststore_hops;
        private TreeView tv_routes;

        public NodeRoutes(Builder builder, Box box_parent, AddressManagerFakeRmtGetter client_getter, NIP nip)
        {
            this.client_getter = client_getter;
            builder.connect_signals (this);
            liststore_routes = builder.get_object ("liststore_routes") as ListStore;
            liststore_hops = builder.get_object ("liststore_hops") as ListStore;
            tv_routes = builder.get_object ("tv_routes") as TreeView;
            Widget widget_routes = builder.get_object ("widget_root") as Widget;
            widget_routes.reparent(box_parent);

            TreeSelection sel = tv_routes.get_selection();
            sel.set_mode(SelectionMode.SINGLE);
            sel.changed.connect(() => {tv_routes_selection_changed();});
        }

        private int impl_start_operations()
        {
            while (true)
            {
                try { refresh_routes();
                } catch (Error e) {}

                if (nap_until_condition(1000,
                    () => {
                        return t_op_aborting;
                    })) break;
            }
            return 0;
        }

        public void tv_routes_selection_changed()
        {
            update_selected_route();
        }

        private Thread<int>? t_op;
        private bool t_op_aborting;
        public void start_operations()
        {
            if (t_op == null)
            {
                t_op_aborting = false;
                t_op = new Thread<int>(null, impl_start_operations);
            }
        }

        public void stop_operations()
        {
            if (t_op != null)
            {
                t_op_aborting = true;
                t_op.join();
                t_op = null;
            }
        }

        /** retrieve and display data
          */
        private void refresh_routes()
        {
            try
            {
                Gee.List<InfoRoute> ret = client_getter.get_client().maproute.report_routes();

                MainContext.@default().invoke(() => {
                        display_routes(ret);
                        update_selected_route();
                        return false;});
            }
            catch (Error e)
            {
                string e_message = e.message;
                MainContext.@default().invoke(() => {display_error(e_message); return false;});
            }
        }

        string uid_route(InfoRoute r)
        {
            string str_nip_part = @"$(r.dest.get_partialnip_relative_to(info_node.nip))";
            string str_gw_part = nip_to_str(info_node.levels, info_node.gsize, r.gwnip);
            return @"$(str_nip_part) via $(str_gw_part) dev $(r.gwdev)";
        }

        void display_routes(Gee.List<InfoRoute> routes)
        {
            TreeIter iter;
            foreach (InfoRoute r in routes)
            {
                // scan treemodel for it
                TreeIter? found_iter = null;
                liststore_routes.@foreach((model, path, iter) => {
                    string str_gnode_dest;
                    model.@get(iter, LISTROUTES.STR_GNODE_DEST, out str_gnode_dest);
                    if (str_gnode_dest == uid_route(r))
                    {
                        found_iter = iter;
                        return true;
                    }
                    return false;
                });
                if (found_iter != null)
                {
                    // modify a list item
                    liststore_routes.@set(found_iter,
                                          LISTROUTES.INFOROUTE, r, 
                                          LISTROUTES.STR_REM, @"$(r.rem)", 
                                          LISTROUTES.STR_GID, @"$(r.gid.ident)");
                }
                else
                {
                    // append a new list item
                    liststore_routes.append(out iter);
                    liststore_routes.@set(iter,
                                          LISTROUTES.STR_GNODE_DEST, uid_route(r),
                                          LISTROUTES.INFOROUTE, r, 
                                          LISTROUTES.STR_REM, @"$(r.rem)", 
                                          LISTROUTES.STR_GID, @"$(r.gid.ident)");
                }
            }
            // remove absent items
            TreeIter[] iters_toremove = {};
            liststore_routes.@foreach((model, path, iter) => {
                string str_gnode_dest;
                model.@get(iter, LISTROUTES.STR_GNODE_DEST, out str_gnode_dest);
                bool found = false;
                foreach (InfoRoute r in routes)
                {
                    if (str_gnode_dest == uid_route(r))
                    {
                        found = true;
                        break;
                    }
                }
                if (!found) iters_toremove += iter;
                return false;
            });
            foreach (TreeIter i in iters_toremove) liststore_routes.remove(i);
        }

        public void update_selected_route()
        {
            TreePath? path;
            unowned TreeViewColumn? column;
            TreeIter iter;
            tv_routes.get_cursor(out path, out column);
            if (path != null)
            {
                if (liststore_routes.get_iter(out iter, path))
                {
                    InfoRoute inforoute;
                    liststore_routes.@get(iter, LISTROUTES.INFOROUTE, out inforoute);
                    display_hops(inforoute);
                }
                else
                {
                    // not found
                    clear_hops();
                }
            }
            else
            {
                // not found
                clear_hops();
            }
        }

        void clear_routes()
        {
            liststore_routes.clear();
        }

        void display_hops(InfoRoute r)
        {
            TreeIter iter;
            liststore_hops.clear();
            foreach (HCoord hop in r.hops)
            {
                liststore_hops.append(out iter);
                liststore_hops.@set(iter, LISTHOPS.STR_GNODE_HOP, @"$(hop.get_partialnip_relative_to(info_node.nip))");
            }
            liststore_hops.append(out iter);
            liststore_hops.@set(iter, LISTHOPS.STR_GNODE_HOP, @"$(r.dest.get_partialnip_relative_to(info_node.nip))");
        }

        void clear_hops()
        {
            liststore_hops.clear();
        }

        void display_error(string e_message)
        {
            TreeIter iter;
            liststore_routes.clear();
            liststore_routes.append(out iter);
            liststore_routes.@set(iter, LISTROUTES.STR_GNODE_DEST, e_message);
            liststore_hops.clear();
        }
    }
}
