/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2014 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gtk;
using Gee;
using zcd;
using Netsukuku;
using Netsukuku.InetUtils;

namespace Monitor
{
    // position of fields in treemodel
    enum LISTANDNACACHE {
        STR_DOMAIN,
        INFOANDNACACHE,
        INT_TTL,
        NUMCOLUMNS
    }

    // position of fields in treemodel
    enum LISTANDNAREGISTRATION {
        STR_DOMAIN,
        INFOANDNAREGISTRATION,
        BOOL_REGISTERED,
        NUMCOLUMNS
    }

    // position of fields in treemodel
    enum LISTCACHESERVERS {
        STR_ALIAS,
        STR_PORT,
        STR_PRIORITY,
        STR_WEIGHT,
        NUMCOLUMNS
    }

    // position of fields in treemodel
    enum LISTCACHESERVICEKEY {
        STR_REPR,
        SERVERS,
        NUMCOLUMNS
    }

    public class NodeAndna : Object
    {
        private AddressManagerFakeRmtGetter client_getter;
        private InfoNode info_node;
        private InfoAndna? info_andna;
        private NIP nip;
        private Label lbl_pubk;
        private CheckButton chk_memory;
        private CheckButton chk_registering;
        private ListStore liststore_andnacache;
        private TreeView tv_andnacache;
        private ListStore liststore_andnaregistration;
        private Button but_edit_mynames;
        private Label lbl_record_pubk;
        private Label lbl_record_nip;
        private ListStore liststore_servers;
        private ListStore liststore_servicekey;
        private ComboBox cbo_servicekey;

        public NodeAndna(Builder builder, Box box_parent, AddressManagerFakeRmtGetter client_getter, NIP nip)
        {
            this.client_getter = client_getter;
            this.nip = nip;
            builder.connect_signals (this);
            lbl_pubk = builder.get_object ("lbl_pubk") as Label;
            chk_memory = builder.get_object ("chk_memory") as CheckButton;
            chk_registering = builder.get_object ("chk_registering") as CheckButton;
            liststore_andnacache = builder.get_object ("liststore_andnacache") as ListStore;
            tv_andnacache = builder.get_object ("tv_andnacache") as TreeView;
            liststore_andnaregistration = builder.get_object ("liststore_andnaregistration") as ListStore;
            but_edit_mynames = builder.get_object ("but_edit_mynames") as Button;
            lbl_record_pubk = builder.get_object ("lbl_record_pubk") as Label;
            lbl_record_nip = builder.get_object ("lbl_record_nip") as Label;
            liststore_servers = builder.get_object ("liststore_servers") as ListStore;
            liststore_servicekey = builder.get_object ("liststore_servicekey") as ListStore;
            cbo_servicekey = builder.get_object ("cbo_servicekey") as ComboBox;
            Widget widget_routes = builder.get_object ("widget_root") as Widget;
            widget_routes.reparent(box_parent);

            TreeSelection sel_andnacache = tv_andnacache.get_selection();
            sel_andnacache.set_mode(SelectionMode.SINGLE);
            sel_andnacache.changed.connect(() => {tv_andnacache_selection_changed();});

            info_node = client_getter.get_client().maproute.report_yourself();
            info_andna = null;
        }

        private int impl_start_operations()
        {
            while (true)
            {
                try { refresh_andna();
                } catch (Error e) {}

                if (nap_until_condition(1000,
                    () => {
                        return t_op_aborting;
                    })) break;
            }
            return 0;
        }

        private Thread<int>? t_op;
        private bool t_op_aborting;
        public void start_operations()
        {
            if (t_op == null)
            {
                t_op_aborting = false;
                t_op = new Thread<int>(null, impl_start_operations);
            }
        }

        public void stop_operations()
        {
            if (t_op != null)
            {
                t_op_aborting = true;
                t_op.join();
                t_op = null;
            }
        }

        /** retrieve and display data
          */
        private void refresh_andna() throws Error
        {
            try
            {
                info_andna = client_getter.get_client().andna.report_status();
                MainContext.@default().invoke(() => {
                        lbl_pubk.label = info_andna.pubk.to_string();
                        chk_memory.active = info_andna.memory_initialized;
                        chk_registering.active = info_andna.register_ongoing;
                        display_cache(info_andna.cache);
                        display_registration(info_andna.registrations);
                        return false;});
            }
            catch (Error e)
            {
                string e_message = e.message;
                MainContext.@default().invoke(() => {
                        info_andna = null;
                        display_error(e_message);
                        return false;});
            }
        }

        private NamesApp names_app;
        private HashMap<string, AndnaDomainRecord> names;
        [CCode (instance_pos = -1)]
        public void but_edit_mynames_clicked(Button source)
        {
            // get the names from the monitored node
            AndnaPrivateConfigurationList names = client_getter.get_client().andna.get_mynames();

            names_app = new NamesApp(names,
                        /*RetrievePubkDelegate:  string => PublicKey*/
                        (domain) => {
                            // retrieve publickey of registrar for hashed_hostname
                            // OR null
                            return client_getter.get_client().andna.retrieve_registrar_pubk(crypto_hash(domain));
                        });
            names_app.win_names.title = "Edit names";
            names_app.win_names.show_all();
            // TODO uncomment: modal is buggy in ubuntu right now
            // names_app.win_names.modal = true;
            // names_app.win_names.set_transient_for(win_names);
            names_app.win_names.set_position(WindowPosition.CENTER_ON_PARENT);

            names_app.go_back.connect(
                        // when the dialog returns
                        () => {
                            // send the names to the monitored node
                            client_getter.get_client().andna.set_mynames(names);
                        });
        }

        public void tv_andnacache_selection_changed()
        {
            update_selected_andnacache();
        }

        void update_selected_andnacache()
        {
            TreePath? path;
            unowned TreeViewColumn? column;
            TreeIter iter;
            tv_andnacache.get_cursor(out path, out column);
            if (path != null)
            {
                if (liststore_andnacache.get_iter(out iter, path))
                {
                    InfoAndnaCache c;
                    liststore_andnacache.@get(iter, LISTANDNACACHE.INFOANDNACACHE, out c);
                    display_cache_detail(c);
                }
                else
                {
                    // not found
                    clear_cache_detail();
                }
            }
            else
            {
                // not found
                clear_cache_detail();
            }
        }

        void clear_cache_detail()
        {
            lbl_record_pubk.label = "";
            lbl_record_nip.label = "";
            liststore_servicekey.clear();
            liststore_servers.clear();
        }

        void display_cache_detail(InfoAndnaCache c)
        {
            lbl_record_pubk.label = @"$(c.rec.pubk)";
            lbl_record_nip.label = @"$(c.rec.nip)";
            liststore_servicekey.clear();
            foreach (AndnaServiceKey k in c.rec.services.keys)
            {
                TreeIter iter;
                liststore_servicekey.append(out iter);
                liststore_servicekey.@set(iter,
                        LISTCACHESERVICEKEY.SERVERS, c.rec.services[k]);
                string str_k = "NULL";
                if (! AndnaServiceKey.equal_func(AndnaServiceKey.NULL_SERV_KEY, k))
                    str_k = @"$(k.name).$(k.proto)";
                liststore_servicekey.@set(iter,
                        LISTCACHESERVICEKEY.STR_REPR, str_k);
            }
            liststore_servers.clear();
        }

        [CCode (instance_pos = -1)]
        public void cbo_servicekey_changed(ComboBox source)
        {
            TreeIter iter;
            if (cbo_servicekey.get_active_iter(out iter))
            {
                ArrayList<AndnaServerRecord> srvrs;
                liststore_servicekey.@get(iter,
                        LISTCACHESERVICEKEY.SERVERS, out srvrs);
                liststore_servers.clear();
                foreach (AndnaServerRecord r in srvrs)
                {
                    TreeIter iter_server;
                    liststore_servers.append(out iter_server);
                    string str_hashed_alias = "NULL";
                    if (r.hashed_alias != null)
                    {
                        str_hashed_alias = @"$(r.hashed_alias)  $(r.pubk)";
                    }
                    liststore_servers.@set(iter_server,
                            LISTCACHESERVERS.STR_ALIAS, str_hashed_alias);
                    liststore_servers.@set(iter_server,
                            LISTCACHESERVERS.STR_PORT, @"$(r.port_number)");
                    liststore_servers.@set(iter_server,
                            LISTCACHESERVERS.STR_PRIORITY, @"$(r.priority)");
                    liststore_servers.@set(iter_server,
                            LISTCACHESERVERS.STR_WEIGHT, @"$(r.weight)");
                }
            }
        }

        void display_cache(Gee.List<InfoAndnaCache> records)
        {
            TreeIter iter;
            foreach (InfoAndnaCache c in records)
            {
                // scan treemodel for it
                TreeIter? found_iter = null;
                liststore_andnacache.@foreach((model, path, iter) => {
                    string iter_domain_str;
                    model.@get(iter, LISTANDNACACHE.STR_DOMAIN, out iter_domain_str);
                    if (iter_domain_str == c.domain)
                    {
                        found_iter = iter;
                        return true;
                    }
                    return false;
                });
                if (found_iter != null)
                {
                    // modify a list item
                    liststore_andnacache.@set(found_iter,
                            LISTANDNACACHE.INFOANDNACACHE, c);
                    int64 secs = c.rec.expires.get_msec_ttl() / 1000;
                    liststore_andnacache.@set(found_iter,
                            LISTANDNACACHE.INT_TTL, (int)secs);
                }
                else
                {
                    // append a new list item
                    liststore_andnacache.append(out iter);
                    liststore_andnacache.@set(iter,
                            LISTANDNACACHE.INFOANDNACACHE, c);
                    liststore_andnacache.@set(iter,
                            LISTANDNACACHE.STR_DOMAIN, c.domain);
                    int64 secs = c.rec.expires.get_msec_ttl() / 1000;
                    liststore_andnacache.@set(iter,
                            LISTANDNACACHE.INT_TTL, (int)secs);
                }
            }
            // remove absent items
            TreeIter[] iters_toremove = {};
            liststore_andnacache.@foreach((model, path, iter) => {
                string iter_domain_str;
                model.@get(iter, LISTANDNACACHE.STR_DOMAIN, out iter_domain_str);
                bool found = false;
                foreach (InfoAndnaCache c in records)
                {
                    if (iter_domain_str == c.domain)
                    {
                        found = true;
                        break;
                    }
                }
                if (!found) iters_toremove += iter;
                return false;
            });
            foreach (TreeIter i in iters_toremove) liststore_andnacache.remove(i);
        }

        void display_registration(Gee.List<InfoAndnaRegistration> records)
        {
            TreeIter iter;
            foreach (InfoAndnaRegistration c in records)
            {
                // scan treemodel for it
                TreeIter? found_iter = null;
                liststore_andnaregistration.@foreach((model, path, iter) => {
                    string iter_domain_str;
                    model.@get(iter, LISTANDNAREGISTRATION.STR_DOMAIN, out iter_domain_str);
                    if (iter_domain_str == c.domain)
                    {
                        found_iter = iter;
                        return true;
                    }
                    return false;
                });
                if (found_iter != null)
                {
                    // modify a list item
                    liststore_andnaregistration.@set(found_iter,
                            LISTANDNAREGISTRATION.INFOANDNAREGISTRATION, c);
                    liststore_andnaregistration.@set(found_iter,
                            LISTANDNAREGISTRATION.BOOL_REGISTERED, c.registered);
                }
                else
                {
                    // append a new list item
                    liststore_andnaregistration.append(out iter);
                    liststore_andnaregistration.@set(iter,
                            LISTANDNAREGISTRATION.INFOANDNAREGISTRATION, c);
                    liststore_andnaregistration.@set(iter,
                            LISTANDNAREGISTRATION.STR_DOMAIN, c.domain);
                    liststore_andnaregistration.@set(iter,
                            LISTANDNAREGISTRATION.BOOL_REGISTERED, c.registered);
                }
            }
            // remove absent items
            TreeIter[] iters_toremove = {};
            liststore_andnaregistration.@foreach((model, path, iter) => {
                string iter_domain_str;
                model.@get(iter, LISTANDNAREGISTRATION.STR_DOMAIN, out iter_domain_str);
                bool found = false;
                foreach (InfoAndnaRegistration c in records)
                {
                    if (iter_domain_str == c.domain)
                    {
                        found = true;
                        break;
                    }
                }
                if (!found) iters_toremove += iter;
                return false;
            });
            foreach (TreeIter i in iters_toremove) liststore_andnaregistration.remove(i);
        }

        void display_error(string e_message)
        {
            TreeIter iter;
            liststore_andnacache.clear();
            liststore_andnacache.append(out iter);
            liststore_andnacache.@set(iter, LISTANDNACACHE.STR_DOMAIN, e_message);
            liststore_andnaregistration.clear();
        }
    }
}
