/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2011-2014 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gee;
using Netsukuku;
using Tasklets;
using zcd;
using Ntk.Test;

namespace Netsukuku
{
#if log_tasklet
    private string tasklet_id()
    {
        return @"[$(Tasklet.self().id)] ";
    }
#else
    private string tasklet_id()
    {
        return "";
    }
#endif
    internal void log_debug(string msg) {print_out(tasklet_id()+msg+"\n");}
    internal void log_info(string msg) {print_out(tasklet_id()+msg+"\n");}
    internal void log_notice(string msg) {print_out(tasklet_id()+msg+"\n");}
    internal void log_warn(string msg) {print_out(tasklet_id()+msg+"\n");}
    internal void log_error(string msg) {print_out(tasklet_id()+msg+"\n");}
    internal void log_critical(string msg) {print_out(tasklet_id()+msg+"\n");}

    public errordomain PeerToPeerError {
        REGISTER,
        GENERIC
    }

    public class RmtPeer : Object
    {
        private PeerToPeer realdest;
        public RmtPeer(PeerToPeer peer_to_peer_service, Object? key=null, NIP? hIP=null, AggregatedNeighbour? aggregated_neighbour=null)
        {
            MapRoute mr = peer_to_peer_service.maproute;
            // I have to reach dest
            NIP dest;
            if (hIP == null)
                dest = peer_to_peer_service.h(key);
            else
                dest = hIP;
            // The best approximation alive is realdest
            AndnaCounter d = Ntk.Test.best_approximation(mr.levels, mr.gsize, dest, 1);
            if (peer_to_peer_service.get_type().is_a(typeof(Andna))) realdest = d.andna;
            if (peer_to_peer_service.get_type().is_a(typeof(Counter))) realdest = d.counter;
        }

        public virtual ISerializable rmt(RemoteCall data) throws Error
        {
            print_out(@"RmtPeer.rmt(data $(data))\n");
            print_out(@"sent to $(realdest.maproute.me)\n");
            
            return realdest._dispatch(null, data);
        }
    }

    public class PeerToPeerAll : Object
    {
        public virtual void peer_to_peer_register(PeerToPeer peer_to_peer) throws PeerToPeerError
        {
        }
    }

    public class PeerToPeer : Object
    {
        public PeerToPeer(AggregatedNeighbourManager aggregated_neighbour_manager,
                    MapRoute maproute, int pid)
        {
            this.maproute = maproute;
        }

        public bool participant {
            get {
                return true;
            }
        }

        public MapRoute maproute;

        public virtual ISerializable _dispatch (Object? caller, RemoteCall data) throws GLib.Error
        {
            return null;
        }

        public signal void map_peer_to_peer_validated();

        public virtual NIP h(Object key)
        {
            assert(false);
            return (NIP)key;
        }

        public Gee.List<int> list_ids(int center, int sign)
        {
            print_out(@"[$(this.maproute.me)]: list_ids(center $(center), sign $(sign))\n");
            ArrayList<int> ret = Ntk.Test.list_positions(center, sign, maproute.gsize);
            string strret = "";
            foreach (int r in ret) strret += @" $(r) ";
            print_out(@"[$(this.maproute.me)]: list_ids = $(strret)\n");
            return ret;
        }

        // Simulate a network with a number of nodes and a number of records
        // in the DHT database. Simulate various cases, of a new node with a
        // certain NIP. Based on the NIP of the new node the method
        // find_hook_peers of PeerToPeer has to return the NIP of the
        // participants that are first_forward, first_back and last_back at
        // a particular level and a particular num_dupl.
        // At the same time the method rmt of RmtPeer, which
        // will eventually get called with RemoteCall <"get_cache">, has to
        // return a ListISerializable backed by a Gee.List<DHTRecord> that
        // will contain records based on the NIP passed to the constructor
        // of RmtPeer. This combination should allow to test the right behaviour
        // of the method hook_to_service of DistributedHashTable.
        //
        // 
        public void find_hook_peers(out int? ret_first_forward,
                                    out int? ret_first_back,
                                    out int? ret_last_back,
                                    int lvl, int num_dupl, int timeout=120000)
        {
            print_out(@"[$(this.maproute.me)]: find_hook_peers(lvl $(lvl), num_dupl $(num_dupl))\n");
            ret_first_forward = null;
            ret_first_back = null;
            ret_last_back = null;
            // check positions from me forward
            Gee.List<int> ids_me_forward = list_ids(maproute.me.position_at(lvl), 1);
            ids_me_forward.remove_at(0);
            foreach (int _id in ids_me_forward)
            {
                int num = 0;
                try {
                    num = get_number_of_participants(lvl, _id, timeout);
                }
                catch (RPCError e) {}
                if (num > 0)
                {
                    ret_first_forward = _id;
                    break;
                }
            }
            // Maybe no nodes at all participate. It would return 3 null.
            // Otherwise...
            if (ret_first_forward != null)
            {
                // check positions from me back
                Gee.List<int> ids_me_back = list_ids(maproute.me.position_at(lvl), -1);
                ids_me_back.remove_at(0);
                int remaining = num_dupl;
                foreach (int _id in ids_me_back)
                {
                    int num = 0;
                    try {
                        num = get_number_of_participants(lvl, _id, timeout);
                    }
                    catch (RPCError e) {}
                    if (num > 0 && ret_first_back == null) ret_first_back = _id;
                    remaining -= num;
                    if (remaining <= 0)
                    {
                        ret_last_back = _id;
                        break;
                    }
                }
                if (remaining > 0)
                {
                    // not enough nodes participate.
                    ret_first_back = null;
                    // I will reach the first forward and be a replica for any
                    // record that it has.
                }
            }
            // One possible situation is that ret_first_back == ret_last_back
            // This would mean that that gnode has a number of participants
            // greater than num_dupl.
        }

        public int get_number_of_participants(int lvl, int pos, int timeout)
        {
            print_out(@"[$(this.maproute.me)]: get_number_of_participants(lvl $(lvl), pos $(pos))\n");
            int ret = 0;
            foreach (NIP n in Ntk.Test.nodes.keys)
            {
                HCoord hcn = n.get_hcoord_relative_to(maproute.me);
                if (hcn.lvl == lvl && hcn.pos == pos) ret++;
            }
            print_out(@"[$(this.maproute.me)]:   $(ret)\n");
            return ret;
        }

        public Gee.List<NIP> find_nearest_to_register(NIP hash_nip, int num_dupl)
        {
            print_out(@"[$(this.maproute.me)]: find_nearest_to_register(hash_nip $(hash_nip), num_dupl $(num_dupl))\n");
            ArrayList<NIP> ret = new ArrayList<NIP>(PartialNIP.equal_func);
            ArrayList<NIP> lst = 
                Ntk.Test.ordered_lookup(maproute.levels, maproute.gsize, hash_nip, 1);
            foreach (NIP s in lst)
            {
                if (Ntk.Test.nodes.has_key(s)) ret.add(s);
                if (ret.size >= num_dupl) break;
            }
            foreach (NIP s in ret) print_out(@"[$(this.maproute.me)]:   $(s)\n");
            return ret;
        }

        public HCoord? search_participant(NIP hIP, int path_sign=1) throws PeerToPeerError
        {
            print_out(@"[$(this.maproute.me)]: search_participant(hIP $(hIP), path_sign $(path_sign))\n");
            // Must find the participant at this moment in the network (I will check
            //  the collection Ntk.Test.nodes) starting at hIP and going path_sign.
            // Then must get HCoord of the result based on this.maproute.
            // If it's me, then null.
            NIP? first = Ntk.Test.best_approximation_nip(maproute.levels,
                         maproute.gsize, hIP, path_sign);
            if (first == null)
                print_out(@"[$(this.maproute.me)]: search_participant: first null\n");
            else
                print_out(@"[$(this.maproute.me)]: search_participant: first $(first)\n");
            if (first == null) throw new PeerToPeerError.GENERIC("No participants.");
            if (first.is_equal(maproute.me)) return null;
            return first.get_hcoord_relative_to(maproute.me);
        }

        public PartialNIP search_participant_as_nip(NIP hIP, int path_sign=1) throws PeerToPeerError
        {
            int[]positions = maproute.me.get_positions();
            HCoord? lvlpos = search_participant(hIP, path_sign);
            if (lvlpos != null)
            {
                int lvl = lvlpos.lvl;
                int pos = lvlpos.pos;
                positions[lvl] = pos;
                for (int i = lvl-1; i >= 0; i--) positions[i] = -1;
            }
            return new PartialNIP(positions);
        }
    }

    public class OptionalPeerToPeer : PeerToPeer
    {
        //public MapPeerToPeer map_peer_to_peer;
        public bool will_participate;
        public OptionalPeerToPeer(AggregatedNeighbourManager aggregated_neighbour_manager,
                    MapRoute maproute, int pid)
        {
            base(aggregated_neighbour_manager, maproute, pid);
        }

        public bool hooked = false;
        public void participate()
        {
            hooked = true;
        }
    }

    public class AggregatedNeighbourManager : Object
    {
    }

    public class AggregatedNeighbour : Object
    {
    }

    public class MapRoute : Object
    {
        public MapRoute(int l, int s, NIP n)
        {
            levels = l;
            gsize = s;
            me = n;
        }
        public int levels;
        public int gsize;
        public NIP me;
        public PartialNIP? choose_fast(Gee.List<PartialNIP> choose_from)
        {
            return choose_from[0];
        }
    }

    namespace Settings
    {
        public bool TESTING;
    }
    public ArrayList<int> valid_ids(int levels, int gsize, int lvl, PartialNIP partial_nip)
    {
        ArrayList<int> ret = new ArrayList<int>();
        for (int i = 0; i < gsize; i++) ret.add(i);
        return ret;
    }

    public string nip_to_str(int levels, int gsize, NIP nip) {return @"$(levels)_$(gsize)_$(nip)";}

    // For this test set a low number of replication
    public const int COUNTER_DUPLICATION = 3;
    // For this test set a low number of replication
    public const int ANDNA_DUPLICATION = 3;
    public const int MAX_HOSTNAMES = 256;
    public const int64 MAX_TTL_ANDNA = (int64)1000 * (int64)60 *
                                       (int64)60 * (int64)24 *
                                       (int64)30; // 30 days in millisec
    public const int64 MAX_TTL_OF_NEGATIVE = (int64)1000 * (int64)30;
                                             // 30 seconds in millisec
    public const int64 MAX_TTL_COUNTER = (int64)1000 * (int64)60 *
                                         (int64)60 * (int64)24 *
                                         (int64)30; // 30 days in millisec
    public const int64 MAX_WAIT_REFRESH_ANDNA = (int64)1000 * (int64)60 *
                                                (int64)60; // 1 hour in millisec
    public const int64 MAX_WAIT_REFRESH_COUNTER = (int64)1000 * (int64)60 *
                                                  (int64)60; // 1 hour in millisec
    public const int MAX_QUEUE_ANDNA = 5;
    // For this test set a low number of spread duplication
    public const int SPREAD_ANDNA = 3;

}

namespace Ntk.Test
{
    string logger;
    const bool output = false;
    public void print_out(string s)
    {
        if (output) print(s);
    }

    public HashMap<NIP, AndnaCounter> nodes;
    public HashMap<string, AndnaCounter> nodes_by_niptostr;

    public NIP? best_approximation_nip(int levels, int gsize, NIP hIP, int sign)
    {
        // The best approximation alive
        NIP? ret = null;
        ArrayList<NIP> lst = 
            Ntk.Test.ordered_lookup(levels, gsize, hIP, sign);
        foreach (NIP s in lst)
        {
            if (Ntk.Test.nodes.has_key(s)) {ret = s; break;}
        }
        return ret;
    }

    public AndnaCounter best_approximation(int levels, int gsize, NIP hIP, int sign)
    {
        NIP nip = best_approximation_nip(levels, gsize, hIP, sign);
        assert(nip != null);
        assert(Ntk.Test.nodes.has_key(nip));
        return Ntk.Test.nodes[nip];
    }

    public ArrayList<int> list_positions(int startpos, int sign, int gsize)
    {
        // Given gsize, startpos and sign this method will return the ordered list
        //  of all the positions (in one level) that will be contacted as peers
        ArrayList<int> ret = new ArrayList<int>();
        for (int i = 0; i < gsize; i++)
            ret.add((startpos + gsize + i * sign) % gsize);
        return ret;
    }

    public ArrayList<NIP> ordered_lookup(int levels, int gsize, NIP hIP, int sign)
    {
        // Given levels, gsize, hIP and sign this method will return the ordered list
        //  of all the NIPs that will be contacted as peers in our peertopeer
        ArrayList<NIP> tot = new ArrayList<NIP>();
        for (int lvl = 0; lvl < levels; lvl++)
        {
            ArrayList<NIP> tmp = new ArrayList<NIP>();
            foreach (int n in list_positions(hIP.position_at(lvl), sign, gsize))
            {
                if (tot.is_empty) tmp.add(new NIP({n}));
                else foreach (NIP s in tot)
                {
                    int[] n_s_pos = new int[lvl+1];
                    int[] s_pos = s.get_positions();
                    for (int l = 0; l < lvl; l++) n_s_pos[l] = s_pos[l];
                    n_s_pos[lvl] = n;
                    tmp.add(new NIP(n_s_pos));
                }
            }
            tot = tmp;
        }
        return tot;
    }

    public class CounterDebug : Counter
    {
        public CounterDebug(KeyPair keypair,
                            AggregatedNeighbourManager aggregated_neighbour_manager,
                            MapRoute maproute,
                            PeerToPeerAll peer_to_peer_all)
        {
            base(keypair, aggregated_neighbour_manager, maproute, peer_to_peer_all);
        }

        public override
        IAddressManagerRootDispatcher
        contact_registrar(string registrar_address)
        {
            print_out(@"registrar_address = $(registrar_address)\n");
            // Look at fake implementation of:
            //   nip_to_str(levels, gsize, nip);
            //  above.
            FakeClient ret = new FakeClient();
            ret.actual_andna = nodes_by_niptostr[registrar_address].andna;
            return ret;
        }
    }

    public class FakeClient : Object, IAddressManagerRootDispatcher
    {
        public Andna actual_andna;

		public unowned IAggregatedNeighbourManager _aggregated_neighbour_manager_getter ()
		{return null;}
		public unowned IAndna _andna_getter ()
		{return actual_andna;}
		public unowned IBorderNodesManager _border_nodes_manager_getter ()
		{return null;}
		public unowned ICoord _coordnode_getter ()
		{return null;}
		public unowned IEtp _etp_getter ()
		{return null;}
		public unowned IHook _hook_getter ()
		{return null;}
		public unowned IMapRoute _maproute_getter ()
		{return null;}
		public unowned IPeerToPeerAll _peer_to_peer_all_getter ()
		{return null;}
		public unowned ITunnelManager _tunnel_manager_getter ()
		{return null;}
		public Gee.List<DHTRecord> dht_list ()
		{return null;}
		public DHTRecord? dht_retr (DHTKey k)
		{return null;}
		public void dht_stor (DHTRecord rec)
		{}
		public IOptionalPeerToPeer get_optional_peer_to_peer_service (int pid)
		{return null;}
		public IPeerToPeer get_peer_to_peer_service (int pid)
		{return null;}
		public Gee.List<TaskletStats> report_tasklets_stats(int minutes)
		{return null;}
		public Gee.List<string> report_tasklet_logs(int minutes)
		{return null;}
		public Gee.List<string> report_running_tasklets()
		{return null;}
    }

    public class AndnaCounter : Object
    {
        public Andna andna;
        public Counter counter;
    }

    public class AndnaCounterTester : Object
    {
        public void set_up ()
        {
            logger = "";
        }

        public void tear_down ()
        {
            logger = "";
        }

        public void test_hook_and_replica()
        {
            print_out("test_hook_and_replica started\n");
            int int_set_1_counter = 0;
            int[] int_set_1 =
            {
                /* levels */ 3,
                /* gsize */ 4,
                /* positions */
                3, 0, 2,     /* node 2 in gnode 0 in ggnode 3 */
                  0,  /* how many names to store */
                2, 0, 1, 1,
                1, 1, 3, 1,
                0, 3, 2, 1,
                0, 2, 2, 1,
                3, 1, 3, 1,
                2, 1, 1, 0,
                3, 0, 3, 0,
                2, 0, 2, 0,
                3, 1, 1, 0,
                2, 2, 2, 0,
                0, 2, 3, 0,
                1, 1, 1, 0,
                2, 2, 1, 0,
                0, 0, 2, 0,
                2, 3, 0, 0,
                3, 3, 1, 0,
                3, 2, 2, 0,
                0, 2, 0, 0,
                3, 3, 2, 0
            };
            string[] hostnames = 
            {
                "luca.dionisi.perugia.it", "",   /* hostname, default snsd */
                // Default snsd registration means:
                //   only 1 snsd record:
                //     NULL_SERV_KEY => 1 server:
                //       no alias, 1, 1, 1
                // It is not present in the arrays below snsd_records,
                //  snsdserver_alias and snsdserver_values
                "netsukuku.org", "2",   /* hostname, 2 records snsd */
                "pegasus", "",
                "zeus", "",
                "andromeda", ""
            };
            string[] snsd_records =
            {
                "", "", "1",   /* NULL_SERV_KEY, 1 server */
                "_www", "_tcp", "2",   /* service www, 2 servers */
            };
            string[] snsdserver_alias =
            {
                "", "",   /* null */
                "luca.dionisi.perugia.it", "1",  /* alias_name, pubk# */
                "", ""
            };
            int[] snsdserver_values =
            {
                1, 1, 1,   /* priority, weight, port_number */
                1, 50, 80,
                1, 50, 80
            };
            string[] searchkeys = 
            {
                "k1", "1",       /* key, should-find */
                "kn", "1",
                "pippo", "",     /* key, should-not-find */
                "athena", "1",
                "zeus", "1",
                "dedalo", "1"
            };

            // These collections will contain the nodes in the network
            nodes = new HashMap<NIP, AndnaCounter>(PartialNIP.hash_func, PartialNIP.equal_func);
            nodes_by_niptostr = new HashMap<string, AndnaCounter>();

            NIP[] nips = {};
            KeyPair[] keypairs = {};
            int levels = int_set_1[int_set_1_counter++];
            int gsize = int_set_1[int_set_1_counter++];
            print_out("generating keys...\n");
            while (true)
            {
                int[] positions = new int[levels];
                for (int i = levels-1; i >= 0; i--)
                    positions[i] = int_set_1[int_set_1_counter++];
                NIP nip = new NIP(positions);
                nips += nip;
                KeyPair keypair = new KeyPair();
                keypairs += keypair;
                int_set_1_counter++;  // store record?
                if (int_set_1_counter >= int_set_1.length) break;
            }
            print_out(@"$(keypairs.length) keypairs done.\n");

            MapRoute[] mrs = {};
            int_set_1_counter = 2;
            int keypairs_counter = 0;
            int hostname_counter = 0;
            int snsd_record_counter = 0;
            int snsdserver_alias_counter = 0;
            int snsdserver_values_counter = 0;
            int searchkeys_counter = 0;
            foreach (NIP nip in nips)
            {
                AggregatedNeighbourManager anm = new AggregatedNeighbourManager();
                MapRoute mr = new MapRoute(levels, gsize, nip);
                mrs += mr;
                PeerToPeerAll ptpall = new PeerToPeerAll();
                KeyPair keypair = keypairs[keypairs_counter++];
                Counter counter = new CounterDebug(keypair, anm, mr, ptpall);
                Andna andna = new Andna(keypair, counter, anm, mr, ptpall);
                counter.andna = andna;
                AndnaCounter andnacounter = new AndnaCounter();
                andnacounter.andna = andna;
                andnacounter.counter = counter;
                counter.map_peer_to_peer_validated();
                andna.map_peer_to_peer_validated();
                while (!counter.hooked) Tasklet.nap(0, 10000);
                while (!andna.hooked) Tasklet.nap(0, 10000);
                nodes[mr.me] = andnacounter;
                nodes_by_niptostr[nip_to_str(levels, gsize, mr.me)] = andnacounter;

                int_set_1_counter += levels;

                // how many names do I want to store?
                int store = int_set_1[int_set_1_counter++];
                // first, the node contacts the Counter to set its pubkey and
                // names.
                ArrayList<string> myhostnames = new ArrayList<string>();
                for (int i_name = 0; i_name < store; i_name++)
                {
                    myhostnames.add(hostnames[hostname_counter+i_name*2]);
                    // * 2 because the first item is the name, then there is
                    // an item for the same record which is the number of snsd items.
                }
                foreach (string hn in myhostnames)
                    print_out(@"On Counter I will register hash for $(hn).\n");
                Gee.List<string> hashed_hostnames = CounterUtilities.hash_names(myhostnames);
                foreach (string hhn in hashed_hostnames)
                    print_out(@"On Counter I will register $(hhn).\n");
                uchar[] msg_counter = CounterUtilities.prepare_msg(mr.me, hashed_hostnames);
                uchar[] signed_counter = keypair.sign(msg_counter);
                SerializableBuffer signature_counter =
                        new SerializableBuffer((uint8[])(signed_counter));
                CounterSetDataResponse resp_counter =
                    counter.peer(null, mr.me)
                        .set_data_for_pubk
                            (mr.me,
                             hashed_hostnames,
                             keypair.pub_key.to_pubkey(),
                             signature_counter);
                print_out(@"CounterSetDataResponse $(resp_counter.response)\n");
                assert(resp_counter.response == "OK");
                Tasklet.nap(0, 10000);  // delay for replication of Counter
                // now, the node contacts the Andna for each name.
                for (int i_name = 0; i_name < store; i_name++)
                {
                    string hostname = hostnames[hostname_counter+i_name*2];
                    print_out(@"On Andna I will register $(hostname)\n");
                    string num_snsd_str = hostnames[hostname_counter+i_name*2+1];
                    HashMap<AndnaServiceKey, ArrayList<AndnaPrivateConfigurationServer>>?
                        services = null;
                    if (num_snsd_str != "")
                    {
                        int num_snsd = int.parse(num_snsd_str);
                        print_out(@"this includes $(num_snsd) scattered services\n");
                        // services[sk] = [AndnaPrivateConfigurationServer, ...]
                        services = AndnaPrivateConfiguration.make_empty_services();
                        for (int i_key = 0; i_key < num_snsd; i_key++)
                        {
                            string key_name = snsd_records[snsd_record_counter++];
                            string key_prot = snsd_records[snsd_record_counter++];
                            int num_servers = int.parse(snsd_records[snsd_record_counter++]);
                            AndnaServiceKey srvkey = AndnaServiceKey.NULL_SERV_KEY;
                            if (key_name != "")
                            {
                                srvkey = new AndnaServiceKey(key_name, key_prot);
                            }
                            print_out(@"service $(srvkey) has $(num_servers) servers\n");
                            services[srvkey] = AndnaPrivateConfigurationServer.make_empty_list();
                            for (int i_server = 0; i_server < num_servers; i_server++)
                            {
                                PublicKey? server_pubk = null;
                                string? server_alias = snsdserver_alias[snsdserver_alias_counter++];
                                if (server_alias == "")
                                {
                                    server_alias = null;
                                    snsdserver_alias_counter++;
                                }
                                else
                                {
                                    int server_pubk_index =
                                            int.parse(snsdserver_alias[snsdserver_alias_counter++]);
                                    server_pubk = keypairs[server_pubk_index].pub_key.to_pubkey();
                                }
                                int server_priority = snsdserver_values[snsdserver_values_counter++];
                                int server_weight = snsdserver_values[snsdserver_values_counter++];
                                int server_port_number = snsdserver_values[snsdserver_values_counter++];
                                AndnaPrivateConfigurationServer srv =
                                        new AndnaPrivateConfigurationServer(
                                            server_alias,
                                            server_pubk,
                                            server_port_number,
                                            server_priority,
                                            server_weight);
                                services[srvkey].add(srv);
                                print_out(@"  * $(srv)\n");
                            }
                        }
                    }
                    print_out("Now registering...\n");
                    AndnaPrivateConfiguration cfg =
                            new AndnaPrivateConfiguration(
                                hostname,
                                services);
                    AndnaDomainRequest andna_req = andna.make_domain_request(cfg);
                    uchar[] message_andna = andna_req.hash_for_signature();
                    uchar[] signed_andna = keypair.sign(message_andna);
                    SerializableBuffer signature_andna =
                            new SerializableBuffer((uint8[])(signed_andna));
                    AndnaRegisterMainResponse resp_andna =
                        andna.peer(null, new AndnaServiceHashnodeKey(andna_req.hashed_domain))
                            .register_main_for_pubk(andna_req, signature_andna);
                    print_out(@"AndnaRegisterMainResponse $(resp_andna.response)\n");
                    assert(resp_andna.response == "REGISTERED");
                    Tasklet.nap(0, 10000);  // delay for replication of Andna
                }
                
                hostname_counter += store*2;
            }
            Andna test = nodes[nips[0]].andna;
            AndnaGetServersResponse look_resp;
            // search for zeus
            print_out("Looking for zeus\n");
            look_resp = test.ask_get_servers(Andna.crypto_hash("zeus"));
            print_out("got response\n");
            print_out(@" expires in $(look_resp.expires.get_string_msec_ttl())\n");
            assert(!look_resp.response.is_not_found);
            assert(look_resp.response.servers.size == 1);
            foreach (AndnaServer s in look_resp.response.servers)
            {
                print_out(@" . $(s)\n");
            }
            // search for http:netsukuku.org
            print_out("Looking for http:netsukuku.org\n");
            look_resp = test.ask_get_servers(Andna.crypto_hash("netsukuku.org"),
                        new AndnaServiceKey("_www", "_tcp"));
            print_out("got response\n");
            print_out(@" expires in $(look_resp.expires.get_string_msec_ttl())\n");
            assert(!look_resp.response.is_not_found);
            assert(look_resp.response.servers.size == 2);
            // verify there is a resolvable alias
            NIP? found = null;
            foreach (AndnaServer s in look_resp.response.servers)
            {
                print_out(@" . $(s)\n");
                // if it is an alias...
                if (s.alias_name != null)
                {
                    // resolve alias
                    AndnaGetServersResponse resolvealias_resp =
                            test.ask_get_servers(s.alias_name,
                            AndnaServiceKey.NULL_SERV_KEY, false);
                    assert(!resolvealias_resp.response.is_not_found);
                    // search for the first nip
                    foreach (AndnaServer s2 in resolvealias_resp.response.servers)
                    {
                        if (s2.alias_name == null)
                        {
                            found = s2.registrar_nip;
                            break;
                        }
                    }
                    print_out(@"   $(found)\n");
                }
            }
            assert(found != null);
            // search for poseidon
            print_out("Looking for poseidon\n");
            look_resp = test.ask_get_servers(Andna.crypto_hash("poseidon"));
            print_out(@"got response with $(look_resp.response.servers.size) records\n");
            print_out(@" expires in $(look_resp.expires.get_string_msec_ttl())\n");
            assert(look_resp.response.is_not_found);

        }

        public static int main(string[] args)
        {
            GLib.Test.init(ref args);
            Tasklet.init();
            GLib.Test.add_func ("/AndnaCounter/HookingAndReplica", () => {
                var x = new AndnaCounterTester();
                x.set_up();
                x.test_hook_and_replica();
                x.tear_down();
            });
            GLib.Test.run();
            Tasklet.kill();
            return 0;
        }
    }
}
