/*
 *  This file is part of librpc.
 *  (c) Copyright 2013 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  librpc is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  librpc is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with librpc.  If not, see <http://www.gnu.org/licenses/>.
 */
using Gee;
void main()
{
    buildpeercode("peerbuilder.in", "out.vala");
}

    class Argument
    {
        public string argname;
        public string classname;
    }

    class Method
    {
        public string returntype;
        public string name;
        public Argument[] args;
        public bool pass_caller = false;
        public string[] errors;
    }
    
    string[] serializable_types;
    HashMap<string, ArrayList<string>> errordefs;
    string[] remotes;
    HashMap<string, ArrayList<Method>> methods;
    HashMap<string, ArrayList<Method>> methods_rmt;

    string output;
    string closepar;

    public void buildpeercode(string if_name, string of_name)
    {
        read_file_if(if_name);

        output = "";
        write_file_of();
        write_file(of_name, output);
    }

    void read_file_if(string if_name)
    {
        string[] lines = read_file(if_name);
        lines += "";
        serializable_types = {};
        errordefs = new HashMap<string, ArrayList<string>>();
        remotes = {};
        methods = new HashMap<string, ArrayList<Method>>();
        methods_rmt = new HashMap<string, ArrayList<Method>>();

        int c = 0;
        if (lines[c] != "errors:") malformed(c);
        while (Regex.match_simple("""^ [a-zA-Z_]""", lines[c+1]))
        {
            c++;
            string dom = get_keyword(lines[c]);
            errordefs[dom] = new ArrayList<string>();
            while (Regex.match_simple("""^  [a-zA-Z_]""", lines[c+1]))
            {
                c++;
                string code = get_keyword(lines[c]);
                errordefs[dom].add(code);
            }
        }
        c++;
        if (lines[c] != "serializables:") malformed(c);
        while (Regex.match_simple("""^ [a-zA-Z_]""", lines[c+1]))
        {
            c++;
            string ser = get_keyword(lines[c]);
            serializable_types += ser;
        }
        serializable_types += "RemoteCall";
        serializable_types += "SerializableBuffer";
        c++;
        if (lines[c] != "peers:") malformed(c);
        while (Regex.match_simple("""^ [a-zA-Z_]""", lines[c+1]))
        {
            c++;
            string remote = get_keyword(lines[c]);
            remotes += remote;
            methods[remote] = new ArrayList<Method>();
            c++;
            if (lines[c] != "  methods:") malformed(c);
            while (Regex.match_simple("""^   [a-zA-Z_]""", lines[c+1]))
            {
                c++;
                Method m = new Method();
                string[] meth = get_two_keywords(lines[c]);
                m.returntype = meth[0];
                m.name = meth[1];
                m.args = {};
                m.errors = {};
                methods[remote].add(m);
                c++;
                if (lines[c] != "    arguments:") malformed(c);
                while (Regex.match_simple("""^     [a-zA-Z_]""", lines[c+1]))
                {
                    c++;
                    Argument a = new Argument();
                    string[] arg = get_two_keywords(lines[c]);
                    a.classname = arg[0];
                    a.argname = arg[1];
                    m.args += a;
                }
                c++;
                if (lines[c] == "    pass_caller")
                {
                    m.pass_caller = true;
                    c++;
                }
                if (lines[c] != "    throws:") malformed(c);
                while (Regex.match_simple("""^     [a-zA-Z_]""", lines[c+1]))
                {
                    c++;
                    string err = get_keyword(lines[c]);
                    m.errors += err;
                }
            }
        }
    }

    string get_keyword(string s)
    {
        MatchInfo m;
        Regex r = new Regex("""[a-zA-Z0-9_]+""");
        if (! r.match(s, 0, out m)) return "";
        return m.fetch(0);
    }

    string[] get_two_keywords(string s)
    {
        MatchInfo m;
        // A classname can contain letters numbers and underscore.
        // It can be prefixed with a namespace.
        // It can contain generics (supports only Gee.List<xxxx>)
        // It can be nullable.
        Regex r = new Regex("""[a-zA-Z0-9_.<>\??]+ [a-zA-Z0-9_]+""");
        if (! r.match(s, 0, out m)) return " ".split(" ");
        return m.fetch(0).split(" ");
    }

    void malformed(int c)
    {
        stderr.printf(@"Input file is malformed. Line $(c).\n");
        Process.exit (1);
    }

    string[] read_file(string path)
    {
        string[] ret = new string[0];
        if (FileUtils.test(path, FileTest.EXISTS))
        {
            try
            {
                string contents;
                assert(FileUtils.get_contents(path, out contents));
                ret = contents.split("\n");
            }
            catch (FileError e) {error("%s: %d: %s".printf(e.domain.to_string(), e.code, e.message));}
        }
        return ret;
    }

    void write_file(string path, string contents)
    {
        try
        {
            assert(FileUtils.set_contents(path, contents));
        }
        catch (FileError e) {error("%s: %d: %s".printf(e.domain.to_string(), e.code, e.message));}
    }

    string method_signature(Method met)
    {
        string ret = "";
        ret += rec_line(
@"$(met.returntype)"
        );
        if (met.args.length == 0)
        {
            ret += rec_line(
@"$(met.name) ()"
            );
        }
        else
        {
            ret += rec_line(
@"$(met.name)"
            );
            string strprefix = "(";
            string strsuffix = ",";
            string prevline = "";
            foreach (Argument arg in met.args)
            {
                if (prevline != "") ret += rec_line(
@"$(prevline),"
                );
                prevline = strprefix + @"$(arg.classname) $(arg.argname)";
                strprefix = " ";
            }
            ret += rec_line(
@"$(prevline))"
            );
        }
        string str_throws = "";
        foreach (string err in met.errors) if (errordefs.keys.contains(err))
            str_throws += @", $(err)";
        ret += @"  throws RPCError$(str_throws)"; // same as rec_line without NL
        return ret;
    }

    void stub_declare_abstract_method(Method met)
    {
        string signature = method_signature(met) + ";\n";
        string writ = indentlines(signature, "                       ");
        writ = "        public abstract" + writ.substring(23);
        wr(writ);
    }

    void stub_declare_method(Method met)
    {
        string signature = method_signature(met) + "\n";
        string writ = indentlines(signature, "              ");
        writ = "        public" + writ.substring(14);
        wr(writ);
    }

    void stub_add_parameters(Argument[] args)
    {
        foreach (Argument arg in args)
        {
            if (arg.classname.length > 1 && arg.classname.substring(arg.classname.length-1) == "?")
            {
                // nullable... what?
                string nullclassname = arg.classname.substring(0, arg.classname.length-1);
                if (nullclassname in serializable_types)
                {
                    wr_line(
@"          if ($(arg.argname) == null) rc.add_parameter(new SerializableNone());"
                    );
                    wr_line(
@"          else rc.add_parameter($(arg.argname));"
                    );
                }
                else if (nullclassname == "int")
                {
                    wr_line(
@"          if ($(arg.argname) == null) rc.add_parameter(new SerializableNone());"
                    );
                    wr_line(
@"          else rc.add_parameter(new SerializableInt($(arg.argname)));"
                    );
                }
                else if (nullclassname == "bool")
                {
                    wr_line(
@"          if ($(arg.argname) == null) rc.add_parameter(new SerializableNone());"
                    );
                    wr_line(
@"          else rc.add_parameter(new SerializableBool($(arg.argname)));"
                    );
                }
                else if (nullclassname == "string")
                {
                    wr_line(
@"          if ($(arg.argname) == null) rc.add_parameter(new SerializableNone());"
                    );
                    wr_line(
@"          else rc.add_parameter(new SerializableString($(arg.argname)));"
                    );
                }
            }
            else if (arg.classname == "ISerializable")
            {
                wr_line(
@"          rc.add_parameter($(arg.argname));"
                );
            }
            else if (arg.classname in serializable_types)
            {
                wr_line(
@"          rc.add_parameter($(arg.argname));"
                );
            }
            else if (arg.classname == "int")
            {
                wr_line(
@"          rc.add_parameter(new SerializableInt($(arg.argname)));"
                );
            }
            else if (arg.classname == "bool")
            {
                wr_line(
@"          rc.add_parameter(new SerializableBool($(arg.argname)));"
                );
            }
            else if (arg.classname == "string")
            {
                wr_line(
@"          rc.add_parameter(new SerializableString($(arg.argname)));"
                );
            }
            else if (arg.classname == "Gee.List<string>")
            {
                wr_line(
@"          ListString _$(arg.argname) = new ListString.with_backer($(arg.argname));"
                );
                wr_line(
@"          rc.add_parameter(_$(arg.argname));"
                );
            }
            else if (arg.classname.length > 9 && arg.classname.substring(0, 9) == "Gee.List<")
            {
                wr_line(
@"          ListISerializable _$(arg.argname) = new ListISerializable.with_backer($(arg.argname));"
                );
                wr_line(
@"          rc.add_parameter(_$(arg.argname));"
                );
            }
        }
    }

    void stub_launch_method_firstpart(Method met)
    {
        if (met.returntype == "void")
        {
            wr_line(
@"              filter_exception("
            );
        }
        else if (met.returntype.length > 1 && met.returntype.substring(met.returntype.length-1) == "?")
        {
            // nullable... whatever
            wr_block(
"""
                ISerializable ret =
                    filter_exception(
"""
            );
        }
        else if (met.returntype == "ISerializable")
        {
            wr_line(
@"              return"
            );
            wr_line(
@"                  filter_exception("
            );
        }
        else if (met.returntype in serializable_types)
        {
            wr_line(
@"              return ($(met.returntype))"
            );
            wr_line(
@"                  filter_exception("
            );
        }
        else if (met.returntype == "int")
        {
            wr_line(
@"              return ((SerializableInt)"
            );
            wr_line(
@"                  filter_exception("
            );
        }
        else if (met.returntype == "bool")
        {
            wr_line(
@"              return ((SerializableBool)"
            );
            wr_line(
@"                  filter_exception("
            );
        }
        else if (met.returntype == "string")
        {
            wr_line(
@"              return ((SerializableString)"
            );
            wr_line(
@"                  filter_exception("
            );
        }
        else if (met.returntype == "Gee.List<string>")
        {
            wr_line(
@"              ListString ret = (ListString)"
            );
            wr_line(
@"                  filter_exception("
            );
        }
        else if (met.returntype.length > 9 && met.returntype.substring(0, 9) == "Gee.List<")
        {
            wr_line(
@"              ListISerializable ret = (ListISerializable)"
            );
            wr_line(
@"                  filter_exception("
            );
        }

        closepar = "";
        foreach (string err in met.errors) if (errordefs.keys.contains(err))
        {
            wr_line(
@"                  filter_exception_$(err)("
            );
            closepar += ")";
        }
    }

    void stub_launch_method_secondpart(Method met)
    {
        if (met.returntype == "void")
        {
            wr_line(
@"              );"
            );
        }
        else if (met.returntype.length > 1 && met.returntype.substring(met.returntype.length-1) == "?")
        {
            // nullable... what?
            string nullclassname = met.returntype.substring(0, met.returntype.length-1);
            if (nullclassname in serializable_types)
            {
                wr_block(
"""
                );
                if (ret.get_type().is_a(typeof(SerializableNone))) return null;
"""
                );
                wr_line(
@"              else return ($(nullclassname))ret;"
                );
            }
            else if (nullclassname == "int")
            {
                wr_block(
"""
                );
                if (ret.get_type().is_a(typeof(SerializableNone))) return null;
"""
                );
                wr_line(
@"              else return ((SerializableInt)ret).i;"
                );
            }
            else if (nullclassname == "bool")
            {
                wr_block(
"""
                );
                if (ret.get_type().is_a(typeof(SerializableNone))) return null;
"""
                );
                wr_line(
@"              else return ((SerializableBool)ret).b;"
                );
            }
            else if (nullclassname == "string")
            {
                wr_block(
"""
                );
                if (ret.get_type().is_a(typeof(SerializableNone))) return null;
"""
                );
                wr_line(
@"              else return ((SerializableString)ret).s;"
                );
            }
        }
        else if (met.returntype == "ISerializable")
        {
            wr_line(
@"              );"
            );
        }
        else if (met.returntype in serializable_types)
        {
            wr_line(
@"              );"
            );
        }
        else if (met.returntype == "int")
        {
            wr_line(
@"              )).i;"
            );
        }
        else if (met.returntype == "bool")
        {
            wr_line(
@"              )).b;"
            );
        }
        else if (met.returntype == "string")
        {
            wr_line(
@"              )).s;"
            );
        }
        else if (met.returntype.length > 9 && met.returntype.substring(0, 9) == "Gee.List<")
        {
            string listof = met.returntype.substring(9, met.returntype.length-10);
            wr_line(
@"              );"
            );
            wr_line(
@"              return (Gee.List<$(listof)>)ret.backed;"
            );
        }
    }


    void write_file_of()
    {
        foreach (string remote in remotes)
        {
            wr_block(
"""

"""
            );
            wr_line(
@"  public interface I$(remote)AsPeer : Object"
            );
            wr_line(
@"  {"
            );
            foreach (Method met in methods[remote])
            {
                stub_declare_abstract_method(met);
            }
            wr_line(
@"  }"
            );
        }

        foreach (string remote in remotes)
        {
            wr_block(
"""

"""
            );
            wr_line(
@"  public class Rmt$(remote)Peer : RmtPeer, I$(remote)AsPeer"
            );
            wr_line(
@"  {"
            );
            wr_line(
@"      public Rmt$(remote)Peer"
            );
            wr_block(
"""
               (PeerToPeer peer_to_peer_service,
                Object? key=null,
                NIP? hIP=null,
                AggregatedNeighbour? aggregated_neighbour=null)
        {
            base(peer_to_peer_service, key, hIP, aggregated_neighbour);
        }

"""
            );
            foreach (Method met in methods[remote])
            {
                stub_declare_method(met);
                wr_block(
"""
        {
            RemoteCall rc = new RemoteCall();
"""
                );
                wr_line(
@"          rc.method_name = \"$(met.name)\";"
                );

                stub_add_parameters(met.args);
                wr_block(
"""
            try {
"""
                );
                stub_launch_method_firstpart(met);
                wr_line(
@"                  this.rmt(rc)$(closepar)"
                );
                stub_launch_method_secondpart(met);
                wr_block(
"""
            }
            catch (RPCError e) {throw e;}
            catch (Error e)
            {
                throw new RPCError.GENERIC
                    (@"Unexpected error $(e.domain).$(e.code) '$(e.message)'");
            }
        }

"""
                );
            }
            wr_block(
"""
    }
"""
            );
        }

        foreach (string remote in remotes)
        {
            wr_block(
"""

"""
            );
            wr_line(
@"  public class $(remote) : OptionalPeerToPeer, I$(remote)AsPeer, I$(remote)"
            );
            wr_block(
"""
    {

"""
            );
            wr_line(
@"      public Rmt$(remote)Peer"
            );
            wr_block(
"""
               peer
               (NIP? hIP=null,
                Object? key=null,
                AggregatedNeighbour? aggregated_neighbour=null)
        {
            assert(hIP != null || key != null);
"""
            );
            wr_line(
@"          return new Rmt$(remote)Peer(this, key, hIP, aggregated_neighbour);"
            );
            wr_block(
"""
        }

"""
            );
            wr_block(
"""
        /** This method could be called *directly* for a dispatcher that does not need to transform
          * an exception into a remotable.
          */
        public override ISerializable _dispatch(Object? caller, RemoteCall data) throws Error
        {
            string[] pieces = data.method_name.split(".");
"""
            );
            string impl = recurse_remote_members(remote);
            wr(impl);
            wr_block(
"""
            return base._dispatch(caller, data);
        }
    }
"""
            );
        }
    }

    string recurse_remote_members(string classname)
    {
        string ret = "";
        foreach (Method m in methods[classname])
        {
            ret += rec_line(
@"          if (pieces[0] == \"$(m.name)\")"
            );
            ret += rec_block(
"""
            {
"""
            );
            ret += rec_line(
@"              if (pieces.length != 1)"
            );
            ret += rec_block(
"""
                    throw new RPCError.MALFORMED_PACKET(
"""
            );
            ret += rec_line(
@"                      \"$(m.name) is a function.\");"
            );
            ret += rec_line(
@"              if (data.parameters.size != $(m.args.length))"
            );
            ret += rec_block(
"""
                    throw new RPCError.MALFORMED_PACKET(
"""
            );
            if (m.args.length == 0)
                ret += rec_line(
@"                      \"$(m.name) wants no parameters.\");"
                );
            else if (m.args.length == 1)
                ret += rec_line(
@"                      \"$(m.name) wants 1 parameter.\");"
                );
            else
                ret += rec_line(
@"                      \"$(m.name) wants $(m.args.length) parameters.\");"
                );
            int argcount = -1;
            foreach (Argument arg in m.args)
            {
                argcount++;
                ret += rec_line(
@"              ISerializable iser$(argcount) = data.parameters[$(argcount)];"
                );
                if (arg.classname.length > 1 && arg.classname.substring(arg.classname.length-1) == "?")
                {
                    // nullable... what?
                    string nullclassname = arg.classname.substring(0, arg.classname.length-1);
                    if (nullclassname in serializable_types)
                    {
                        ret += rec_line(
@"                  if (! iser$(argcount).get_type().is_a(typeof($(nullclassname))) && ! iser$(argcount).get_type().is_a(typeof(SerializableNone)))"
                        );
                        ret += rec_line(
@"                      throw new RPCError.MALFORMED_PACKET("
                        );
                        ret += rec_line(
@"                          \"$(m.name) parameter $(argcount+1) is not a $(nullclassname)?.\");"
                        );
                        ret += rec_line(
@"                  $(nullclassname)? $(arg.argname) = null;"
                        );
                        ret += rec_line(
@"                  if (iser$(argcount).get_type().is_a(typeof($(nullclassname)))) $(arg.argname) = ($(nullclassname)) iser$(argcount);"
                        );
                    }
                    else if (nullclassname == "int")
                    {
                        ret += rec_line(
@"                  if (! iser$(argcount).get_type().is_a(typeof(SerializableInt)) && ! iser$(argcount).get_type().is_a(typeof(SerializableNone)))"
                        );
                        ret += rec_line(
@"                      throw new RPCError.MALFORMED_PACKET("
                        );
                        ret += rec_line(
@"                          \"$(m.name) parameter $(argcount+1) is not a int?.\");"
                        );
                        ret += rec_line(
@"                  int? $(arg.argname) = null;"
                        );
                        ret += rec_line(
@"                  if (iser$(argcount).get_type().is_a(typeof(SerializableInt))) $(arg.argname) = ((SerializableInt) iser$(argcount)).i;"
                        );
                    }
                    else if (nullclassname == "bool")
                    {
                        ret += rec_line(
@"                  if (! iser$(argcount).get_type().is_a(typeof(SerializableBool)) && ! iser$(argcount).get_type().is_a(typeof(SerializableNone)))"
                        );
                        ret += rec_line(
@"                      throw new RPCError.MALFORMED_PACKET("
                        );
                        ret += rec_line(
@"                          \"$(m.name) parameter $(argcount+1) is not a bool?.\");"
                        );
                        ret += rec_line(
@"                  bool? $(arg.argname) = null;"
                        );
                        ret += rec_line(
@"                  if (iser$(argcount).get_type().is_a(typeof(SerializableBool))) $(arg.argname) = ((SerializableBool) iser$(argcount)).b;"
                        );
                    }
                    else if (nullclassname == "string")
                    {
                        ret += rec_line(
@"                  if (! iser$(argcount).get_type().is_a(typeof(SerializableString)) && ! iser$(argcount).get_type().is_a(typeof(SerializableNone)))"
                        );
                        ret += rec_line(
@"                      throw new RPCError.MALFORMED_PACKET("
                        );
                        ret += rec_line(
@"                          \"$(m.name) parameter $(argcount+1) is not a string?.\");"
                        );
                        ret += rec_line(
@"                  string? $(arg.argname) = null;"
                        );
                        ret += rec_line(
@"                  if (iser$(argcount).get_type().is_a(typeof(SerializableString))) $(arg.argname) = ((SerializableString) iser$(argcount)).s;"
                        );
                    }
                }
                else if (arg.classname == "ISerializable")
                {
                    ret += rec_line(
@"              if (! iser$(argcount).get_type().is_a(typeof($(arg.classname))))"
                    );
                    ret += rec_line(
@"                  throw new RPCError.MALFORMED_PACKET("
                    );
                    ret += rec_line(
@"                      \"$(m.name) parameter $(argcount+1) is not a $(arg.classname).\");"
                    );
                    ret += rec_line(
@"              ISerializable $(arg.argname) = iser$(argcount);"
                    );
                }
                else if (arg.classname in serializable_types)
                {
                    ret += rec_line(
@"              if (! iser$(argcount).get_type().is_a(typeof($(arg.classname))))"
                    );
                    ret += rec_line(
@"                  throw new RPCError.MALFORMED_PACKET("
                    );
                    ret += rec_line(
@"                      \"$(m.name) parameter $(argcount+1) is not a $(arg.classname).\");"
                    );
                    ret += rec_line(
@"              $(arg.classname) $(arg.argname) = ($(arg.classname))iser$(argcount);"
                    );
                }
                else if (arg.classname == "int")
                {
                    ret += rec_line(
@"              if (! iser$(argcount).get_type().is_a(typeof(SerializableInt)))"
                    );
                    ret += rec_line(
@"                  throw new RPCError.MALFORMED_PACKET("
                    );
                    ret += rec_line(
@"                      \"$(m.name) parameter $(argcount+1) is not a int.\");"
                    );
                    ret += rec_line(
@"              int $(arg.argname) = ((SerializableInt)iser$(argcount)).i;"
                    );
                }
                else if (arg.classname == "bool")
                {
                    ret += rec_line(
@"              if (! iser$(argcount).get_type().is_a(typeof(SerializableBool)))"
                    );
                    ret += rec_line(
@"                  throw new RPCError.MALFORMED_PACKET("
                    );
                    ret += rec_line(
@"                      \"$(m.name) parameter $(argcount+1) is not a bool.\");"
                    );
                    ret += rec_line(
@"              bool $(arg.argname) = ((SerializableBool)iser$(argcount)).b;"
                    );
                }
                else if (arg.classname == "string")
                {
                    ret += rec_line(
@"              if (! iser$(argcount).get_type().is_a(typeof(SerializableString)))"
                    );
                    ret += rec_line(
@"                  throw new RPCError.MALFORMED_PACKET("
                    );
                    ret += rec_line(
@"                      \"$(m.name) parameter $(argcount+1) is not a string.\");"
                    );
                    ret += rec_line(
@"              string $(arg.argname) = ((SerializableString)iser$(argcount)).s;"
                    );
                }
                else if (arg.classname == "Gee.List<string>")
                {
                    ret += rec_line(
@"              if (! iser$(argcount).get_type().is_a(typeof(ListString)))"
                    );
                    ret += rec_line(
@"                  throw new RPCError.MALFORMED_PACKET("
                    );
                    ret += rec_line(
@"                      \"$(m.name) parameter $(argcount+1) is not a List<string>.\");"
                    );
                    ret += rec_line(
@"              ListString _$(arg.argname) = (ListString)iser$(argcount);"
                    );
                    ret += rec_line(
@"              Gee.List<string> $(arg.argname) = _$(arg.argname).backed;"
                    );
                }
                else if (arg.classname.length > 9 && arg.classname.substring(0, 9) == "Gee.List<")
                {
                    ret += rec_line(
@"              if (! iser$(argcount).get_type().is_a(typeof(ListISerializable)))"
                    );
                    ret += rec_line(
@"                  throw new RPCError.MALFORMED_PACKET("
                    );
                    ret += rec_line(
@"                      \"$(m.name) parameter $(argcount+1) is not a List<NIP>.\");"
                    );
                    ret += rec_line(
@"              ListISerializable _$(arg.argname) = (ListISerializable)iser$(argcount);"
                    );
                    ret += rec_line(
@"              Gee.List<NIP> $(arg.argname) = (Gee.List<NIP>)_$(arg.argname).backed;"
                    );
                }
            }
            string str_args = "";
            string str_args_next = "";
            foreach (Argument arg in m.args)
            {
                str_args += @"$(str_args_next)$(arg.argname)";
                str_args_next = ", ";
            }
            if (m.pass_caller)
                str_args += @"$(str_args_next)(CallerInfo)caller";
            string str_command = @"$(m.name)($(str_args))";
            if (m.returntype == "void")
            {
                ret += rec_line(
@"              $(str_command);"
                );
                ret += rec_line(
@"              return new SerializableNone();"
                );
            }
            else if (m.returntype.length > 1 && m.returntype.substring(m.returntype.length-1) == "?")
            {
                // nullable... what?
                string nullclassname = m.returntype.substring(0, m.returntype.length-1);
                if (nullclassname in serializable_types)
                {
                    ret += rec_line(
@"              $(nullclassname)? ret = $(str_command);"
                    );
                    ret += rec_line(
@"              if (ret == null) return new SerializableNone();"
                    );
                    ret += rec_line(
@"              return ret;"
                    );
                }
                else if (nullclassname == "int")
                {
                    ret += rec_line(
@"              int? ret = $(str_command);"
                    );
                    ret += rec_line(
@"              if (ret == null) return new SerializableNone();"
                    );
                    ret += rec_line(
@"              return new SerializableInt(ret);"
                    );
                }
                else if (nullclassname == "bool")
                {
                    ret += rec_line(
@"              bool? ret = $(str_command);"
                    );
                    ret += rec_line(
@"              if (ret == null) return new SerializableNone();"
                    );
                    ret += rec_line(
@"              return new SerializableBool(ret);"
                    );
                }
                else if (nullclassname == "string")
                {
                    ret += rec_line(
@"              string? ret = $(str_command);"
                    );
                    ret += rec_line(
@"              if (ret == null) return new SerializableNone();"
                    );
                    ret += rec_line(
@"              return new SerializableString(ret);"
                    );
                }
            }
            else if (m.returntype == "ISerializable")
            {
                ret += rec_line(
@"              return $(str_command);"
                );
            }
            else if (m.returntype in serializable_types)
            {
                ret += rec_line(
@"              return $(str_command);"
                );
            }
            else if (m.returntype == "int")
            {
                ret += rec_line(
@"              return new SerializableInt($(str_command));"
                );
            }
            else if (m.returntype == "bool")
            {
                ret += rec_line(
@"              return new SerializableBool($(str_command));"
                );
            }
            else if (m.returntype == "string")
            {
                ret += rec_line(
@"              return new SerializableString($(str_command));"
                );
            }
            else if (m.returntype == "Gee.List<string>")
            {
                ret += rec_line(
@"              Gee.List<string> _ret = $(str_command);"
                );
                ret += rec_block(
"""
                ListString ret = new ListString.with_backer(_ret);
                return ret;
"""
                );
            }
            else if (m.returntype.length > 9 && m.returntype.substring(0, 9) == "Gee.List<")
            {
                string listof = m.returntype.substring(9, m.returntype.length-10);
                ret += rec_line(
@"              Gee.List<$(listof)> _ret = $(str_command);"
                );
                ret += rec_block(
"""
                ListISerializable ret = new ListISerializable.with_backer(_ret);
                return ret;
"""
                );
            }

            ret += rec_block(
"""
            }
"""
            );
        }
        return ret;
    }

    string rec_block(string m)
    {
        // remove first NL
        return m.substring(1);
    }

    string rec_line(string m)
    {
        // add 2 blanks and final NL
        return "  " + m + "\n";
    }

    string indentlines(owned string x, string prefix = "    ")
    {
        if (x.length == 0) return x;
        bool finaleol = x.substring(x.length-1) == "\n";
        if (finaleol) x = x.substring(0, x.length-1);

        Regex regex = new Regex ("^");
        x = regex.replace (x, x.length, 0, prefix);
        Regex regex1 = new Regex ("\n");
        x = regex1.replace (x, x.length, 0, "\n" + prefix);

        if (finaleol) x = x + "\n";
        return x;
    }

    void wr_block(string m)
    {
        // remove first NL
        wr(m.substring(1));
    }

    void wr_line(string m)
    {
        // add 2 blanks and final NL
        wr("  " + m + "\n");
    }

    void wr(string m)
    {
        output += m;
    }

