/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2011 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gee;
using zcd;
using Tasklets;

namespace Netsukuku
{
    struct struct_helper_NtkNode_initialize
    {
        public NtkNode self;
    }

    struct struct_helper_NtkNode_primary_address_changed
    {
        public NtkNode self;
        AddressManager? old_addr;
        AddressManager? new_addr;
    }

    struct struct_helper_NtkNode_counter_hooked
    {
        public NtkNode self;
        AddressManager addr_man;
    }

    struct struct_helper_NtkNode_counter_registered
    {
        public NtkNode self;
        AddressManager addr_man;
    }

    struct struct_helper_NtkNode_initialize_launch_counter_participate
    {
        Addresses addresses;
    }

    struct struct_helper_NtkNode_initialize_launch_andna_participate
    {
        Addresses addresses;
    }

    public delegate void Callback_rehook();
    struct struct_helper_NtkNode_rehook
    {
        public NtkNode self;
        public AddressManager addr_man_from;
        public Gee.List<AggregatedNeighbour> passed_neighbour_list;
        public Callback_rehook? call_back;
        public string reason;
    }

    struct struct_helper_NtkNode_exit
    {
        public NtkNode self;
    }

    struct struct_helper_NtkNode_gnode_splitted
    {
        public NtkNode self;
        public AddressManager addrman;
        public Gee.List<AggregatedNeighbour> passed_neighbour_list;
        public Gee.List<int> queue_of_request_ids;
        public GNodeID actual_gid;
    }

    public class NtkNode : Object
    {
        private Addresses addresses;
        private NetworkInterfaces nics;
        private KeyPair keypair;
        private TCPServer tcpserver;
        private KrnlRoute krnl_route;
        public NtkNode()
        {

            if (Settings.BITS_PER_LEVEL == 0)
                error("Configuration error: BITS_PER_LEVEL cannot be equal to 0");

            Posix.mode_t mode = Posix.S_IRUSR|Posix.S_IWUSR|Posix.S_IXUSR|Posix.S_IRGRP|Posix.S_IXGRP|Posix.S_IROTH|Posix.S_IXOTH;
            if (! FileUtils.test(Settings.CONFIGURATION_DIR, FileTest.EXISTS))
            {
                Posix.mkdir(Settings.CONFIGURATION_DIR, mode);
            }
            if (! FileUtils.test(Settings.KEY_PAIR_DIR, FileTest.EXISTS))
            {
                Posix.mkdir(Settings.KEY_PAIR_DIR, mode);
            }
            if (! FileUtils.test(Settings.DATA_DIR, FileTest.EXISTS))
            {
                Posix.mkdir(Settings.DATA_DIR, mode);
            }

            if (FileUtils.test(Settings.KEY_PAIR_PATH, FileTest.EXISTS))
            {
                try
                {
                    keypair = new KeyPair(Settings.KEY_PAIR_PATH);
                    keypair.save_pub_key(Settings.PUB_KEY_PATH);
                }
                catch (Crypto.GCryptError e)
                {
                    error(@"Unable to save public key: $(e.message)");
                }
            }
            else
            {
                try
                {
                    keypair = new KeyPair();
                    keypair.save_pair(Settings.KEY_PAIR_PATH);
                }
                catch (Crypto.GCryptError e)
                {
                    error(@"Unable to save key pair: $(e.message)");
                }
                try
                {
                    keypair.save_pub_key(Settings.PUB_KEY_PATH);
                }
                catch (Crypto.GCryptError e)
                {
                    error(@"Unable to save public key: $(e.message)");
                }
            }

            // create a NIC for each settings.NICS - settings.EXCLUDE_NICS
            HashMap<string, NIC> _nics = new HashMap<string, NIC>();
            foreach (string dev in Settings.NICS)
            {
                if (! Settings.EXCLUDE_NICS.contains(dev))
                    _nics[dev] = NIC.create_instance(dev);
            }
            // Load the core modules
            addresses = new Addresses(keypair);
            addresses.primary_address_changed.connect(primary_address_changed);
            addresses.counter_hooked.connect(counter_hooked);
            addresses.counter_registered.connect(counter_registered);
            nics = new NetworkInterfaces();
            foreach (string nic_name in _nics.keys)
            {
                NIC nic = _nics[nic_name];
                if (Settings.FORCE_NIC_RESET)
                    nic.down();
                if (! nic.is_active)
                    nic.up();
                nics.add(nic_name, nic,
                         Addresses.udp_unicast_callback,
                         Addresses.udp_broadcast_callback);
                log_debug(@"At start handling nic $(nic_name)");
            }
            addresses.net_collision.connect(network_collision);
            addresses.gnode_splitted.connect(gnode_splitted);

            krnl_route = new KrnlRoute(addresses);
        }

        /** Called when user requests termination.
          */
        private void impl_exit() throws Error
        {
            Tasklet.declare_self("NtkNode.exit");
            ArrayList<AddressManager> addrs = new ArrayList<AddressManager>();
            foreach (AddressManager addr in addresses) addrs.add(addr);
            foreach (AddressManager addr in addrs) addresses.remove(addr);
            Posix.exit(0);
        }

        /* Decoration of microfunc */
        private static void * helper_exit(void *v) throws Error
        {
            struct_helper_NtkNode_exit *tuple_p = (struct_helper_NtkNode_exit *)v;
            // The caller function has to add a reference to the ref-counted instances
            NtkNode self_save = tuple_p->self;
            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            self_save.impl_exit();
            // void method, return null
            return null;
        }

        public void exit()
        {
            struct_helper_NtkNode_exit arg = struct_helper_NtkNode_exit();
            arg.self = this;
            Tasklet.spawn((Spawnable)helper_exit, &arg);
        }

        public void run()
        {
            initialize();
        }

        /* Launch in new tasklet counter.participate */
        private static void * initialize_launch_counter_participate(void *v)
        {
            struct_helper_NtkNode_initialize_launch_counter_participate *tuple_p = (struct_helper_NtkNode_initialize_launch_counter_participate *)v;
            // The caller function has to add a reference to the ref-counted instances
            Addresses _addresses = tuple_p->addresses;
            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            _addresses.primary_address.counter.participate();
            // void method, return null
            return null;
        }

        /* Launch in new tasklet andna.participate */
        private static void * initialize_launch_andna_participate(void *v)
        {
            struct_helper_NtkNode_initialize_launch_andna_participate *tuple_p = (struct_helper_NtkNode_initialize_launch_andna_participate *)v;
            // The caller function has to add a reference to the ref-counted instances
            Addresses _addresses = tuple_p->addresses;
            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            _addresses.primary_address.andna.participate();
            // void method, return null
            return null;
        }

        private void impl_initialize() throws Error
        {
            Tasklet.declare_self("NtkNode.initialize");
            RouteSetter route = RouteSetter.get_instance();
            route.ip_forward(true);
            tcpserver = new TCPServer(Addresses.tcp_callback);
            tcpserver.listen();
            addresses.init_network(Settings.LEVELS, (int)Math.pow(2, Settings.BITS_PER_LEVEL));
            log_debug(@"Setting levels and gsize of initial network $(addresses.levels), $(addresses.gsize)");
            // Add address as bootstrap (hook_reservation = null)
            AddressManager new_address = addresses.create(null);
            log_debug(@"First NIP (random) is $(new_address.maproute.me)");
            addresses.add(new_address);
            addresses.primary_address = new_address;
            foreach (NetworkInterfaceManager nic_man in nics)
            {
                addresses.primary_address.add_nic_manager(nic_man);
                // This will start radar.
            }
            addresses.primary_address.add_nic_manager(nics.glue_nic);
            addresses.primary_address.start_operations();

            // Now I'm also participating to service Counter and Andna
            struct_helper_NtkNode_initialize_launch_counter_participate arg = struct_helper_NtkNode_initialize_launch_counter_participate();
            arg.addresses = addresses;
            Tasklet.spawn((Spawnable)initialize_launch_counter_participate, &arg);
            struct_helper_NtkNode_initialize_launch_andna_participate arg2 = struct_helper_NtkNode_initialize_launch_andna_participate();
            arg2.addresses = addresses;
            Tasklet.spawn((Spawnable)initialize_launch_andna_participate, &arg2);
        }

        /* Decoration of microfunc */
        private static void * helper_initialize(void *v) throws Error
        {
            struct_helper_NtkNode_initialize *tuple_p = (struct_helper_NtkNode_initialize *)v;
            // The caller function has to add a reference to the ref-counted instances
            NtkNode self_save = tuple_p->self;
            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            self_save.impl_initialize();
            // void method, return null
            return null;
        }

        public void initialize()
        {
            struct_helper_NtkNode_initialize arg = struct_helper_NtkNode_initialize();
            arg.self = this;
            Tasklet.spawn((Spawnable)helper_initialize, &arg);
        }

        public void network_collision(AddressManager addr_man, Gee.List<AggregatedNeighbour> passed_neighbour_list)
        {
            // Handle network collision with a re_hook
            rehook(addr_man, passed_neighbour_list);
        }

        private void impl_gnode_splitted(AddressManager addr_man, Gee.List<AggregatedNeighbour> passed_neighbour_list,
                                    Gee.List<int> queue_of_request_ids, GNodeID actual_gid) throws Error
        {
            Tasklet.declare_self("NtkNode.gnode_splitted");
            if (addr_man.is_primary)
            {
                // Have we got a valid secondary address? ... preferably a mature one.
                AddressManager? candidate = null;
                foreach (AddressManager a in addresses)
                {
                    if (candidate == null && ! a.is_primary) candidate = a;
                    if (! a.is_primary && a.is_mature)
                    {
                        candidate = a;
                        break;
                    }
                }

                Callback_rehook propagate_gid = () => {
                    // rehooked. Now wait to be mature before propagating previous answer.");
                    // Now we have the new primary_address
                    NIP current_nip = addresses.primary_address.maproute.me;
                    Tasklets.Timer tc = new Tasklets.Timer(60000);
                    while (true)
                    {
                        try
                        {
                            if (tc.is_expired())
                            {
                                // failure: time out.
                                break;
                            }
                            if (current_nip != addresses.primary_address.maproute.me)
                            {
                                // failure: rehooked again.
                                break;
                            }
                            if (addresses.primary_address.is_mature)
                            {
                                addresses.primary_address.maproute.send_answer_gid(queue_of_request_ids, actual_gid);
                                // success: sent previous answer.
                                break;
                            }
                        }
                        catch (Error e)
                        {
                            // just log
                            log_warn(@"NtkdNode.gnode_splitted: got $(e.domain) code $(e.code): $(e.message)");
                        }
                        finally
                        {
                            ms_wait(1000);
                        }
                    }
                };

                if (candidate != null)
                {
                    // Set as primary the valid address, remove the old one, then propagate new GID.
                    addresses.primary_address = candidate;
                    addresses.remove(addr_man);
                    propagate_gid();
                }
                else
                {
                    // Handle gnode split with a re_hook, then propagate new GID.
                    rehook(addr_man, passed_neighbour_list, propagate_gid, "gnode_split");
                }
            }
            else
            {
                // If the address_manager that should rehook is not a primary_address, then it simply dies.
                addresses.remove(addr_man);
            }
        }

        private static void * helper_gnode_splitted(void *v) throws Error
        {
            struct_helper_NtkNode_gnode_splitted *tuple_p = (struct_helper_NtkNode_gnode_splitted *)v;
            // The caller function has to add a reference to the ref-counted instances
            NtkNode self_save = tuple_p->self;
            AddressManager addrman_save = tuple_p->addrman;
            Gee.List<AggregatedNeighbour> passed_neighbour_list_save = tuple_p->passed_neighbour_list;
            Gee.List<int> queue_of_request_ids_save = tuple_p->queue_of_request_ids;
            GNodeID actual_gid_save = tuple_p->actual_gid;
            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            self_save.impl_gnode_splitted(addrman_save, passed_neighbour_list_save, queue_of_request_ids_save, actual_gid_save);
            // void method, return null
            return null;
        }

        public void gnode_splitted(AddressManager addrman, Gee.List<AggregatedNeighbour> passed_neighbour_list,
                                    Gee.List<int> queue_of_request_ids, GNodeID actual_gid)
        {
            struct_helper_NtkNode_gnode_splitted arg = struct_helper_NtkNode_gnode_splitted();
            arg.self = this;
            arg.addrman = addrman;
            arg.passed_neighbour_list = passed_neighbour_list;
            arg.queue_of_request_ids = queue_of_request_ids;
            arg.actual_gid = actual_gid;
            Tasklet.spawn((Spawnable)helper_gnode_splitted, &arg);
        }

        private string me_to_ipstr()
        {
            MapRoute mr = addresses.primary_address.maproute;
            return nip_to_str(mr.levels, mr.gsize, mr.me);
        }

        private void impl_rehook(AddressManager addr_man_from, Gee.List<AggregatedNeighbour> passed_neighbour_list,
                                 Callback_rehook? call_back, string reason) throws Error
        {
            // Does a rehook with passed_neighbour_list
            NetworkID candidate_netid = passed_neighbour_list[0].netid;
            int candidate_levels = passed_neighbour_list[0].levels;
            int candidate_gsize = passed_neighbour_list[0].gsize;
            if (! addresses.primary_address.is_mature)
            {
                // rehook: ignored because it is too early.
                return;
            }
            if (reason == "collision")
            {
                foreach (AddressManager addr in addresses)
                {
                    if (addr.is_in_my_network(candidate_netid))
                    {
                        // rehook: ignored because we are already in.
                        return;
                    }
                }
            }

            HookReservation hook_reservation;
            try
            {
                hook_reservation = addr_man_from.hook.hook(passed_neighbour_list);
            }
            catch (Error e)
            {
                log_info(@"NtkdNode.rehook: tried to rehook to $(candidate_netid) but got Exception $(e.domain) $(e.code) $(e.message)");
                // TODO Note that P2P hook is non sense for strict services, such as Coord.
                if (e is RPCError.NOT_VALID_MAP_YET)
                {
                    log_info("NtkdNode.rehook: retry in 2 seconds.");
                    ms_wait(2000);
                    rehook(addr_man_from, passed_neighbour_list);
                }
                return;
            }

            addresses.init_network(candidate_levels, candidate_gsize);
            AddressManager new_address = addresses.create(hook_reservation);
            addresses.add(new_address);
            addresses.primary_address = new_address;
            foreach (NetworkInterfaceManager nic_man in nics)
            {
                addresses.primary_address.add_nic_manager(nic_man);
                // This will start radar.
            }
            addresses.primary_address.add_nic_manager(nics.glue_nic);
            addresses.primary_address.start_operations(20000);
            // This delay replaces the delay between HOOKED and HOOKED_STABLE: that is, wait
            //  for the first ETP to come in, before e.g. hooking in PeerToPeer.
            if (call_back != null)
                call_back();
        }

        private static void * helper_rehook(void *v) throws Error
        {
            Tasklet.declare_self("NtkNode.rehook dispatcher");
            struct_channel *ch_cont_p = (struct_channel *)v;
            // The caller function has to add a reference to the ref-counted instances
            Channel ch = ch_cont_p->self;
            // schedule back to the spawner; this will probably invalidate *v and *ch_cont_p.
            Tasklet.schedule_back();
            // The actual dispatcher
            while (true)
            {
                string? doing = null;
                try
                {
                    struct_helper_NtkNode_rehook tuple_p;
                    {
                        Value vv = ch.recv();
                        tuple_p = *((struct_helper_NtkNode_rehook *)(vv.get_boxed()));
                    }
                    doing = @"rehook($(tuple_p.addr_man_from.maproute.me), $(tuple_p.reason))";
                    Tasklet.declare_self(doing);
                    // The helper function should not need to copy values
                    tuple_p.self.impl_rehook(tuple_p.addr_man_from, tuple_p.passed_neighbour_list, tuple_p.call_back, tuple_p.reason);
                    // do we need to give schedule?
                    Tasklet.nap(0, 100);
                }
                catch (Error e)
                {
                    // log_warn(@"NtkNode.rehook: a dispatched microfunc reported an error: $(e.message)");
                }
                if (doing != null) Tasklet.declare_finished(doing);
            }
        }

        public void rehook(AddressManager addr_man_from, Gee.List<AggregatedNeighbour> passed_neighbour_list,
                                Callback_rehook? call_back=null, string reason="collision")
        {
            // Register (once) the spawnable function that is our dispatcher
            // and obtain a channel to drive it.
            Channel ch = TaskletDispatcher.get_channel_for_helper((Spawnable)helper_rehook);
            struct_helper_NtkNode_rehook arg = struct_helper_NtkNode_rehook();
            arg.self = this;
            arg.addr_man_from = addr_man_from;
            arg.passed_neighbour_list = passed_neighbour_list;
            arg.call_back = call_back;
            arg.reason = reason;
            // send the struct
            ch.send_async(arg);
        }

        private void impl_primary_address_changed(AddressManager? old_addr, AddressManager? new_addr) throws Error
        {
            Tasklet.declare_self("NtkNode.primary_address_changed");
            // stop counter/andna keeping
            if (old_addr != null)
            {
                old_addr.andna.stop_register_my_names();
                old_addr.counter.stop_reset_my_counter_node();
            }
        }

        /* Decoration of microfunc */
        private static void * helper_primary_address_changed(void *v) throws Error
        {
            struct_helper_NtkNode_primary_address_changed *tuple_p = (struct_helper_NtkNode_primary_address_changed *)v;
            // The caller function has to add a reference to the ref-counted instances
            NtkNode self_save = tuple_p->self;
            AddressManager? old_addr_save = tuple_p->old_addr;
            AddressManager? new_addr_save = tuple_p->new_addr;
            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            self_save.impl_primary_address_changed(old_addr_save, new_addr_save);
            // void method, return null
            return null;
        }

        void primary_address_changed(Addresses addresses, AddressManager? old_addr, AddressManager? new_addr)
        {
            struct_helper_NtkNode_primary_address_changed arg = struct_helper_NtkNode_primary_address_changed();
            arg.self = this;
            arg.old_addr = old_addr;
            arg.new_addr = new_addr;
            Tasklet.spawn((Spawnable)helper_primary_address_changed, &arg);
        }

        private void impl_counter_hooked(AddressManager addr_man) throws Error
        {
            Tasklet.declare_self("NtkNode.counter_hooked");
            if (addr_man == addresses.primary_address)
            {
                if (! addr_man.counter.reset_my_counter_node_ongoing())
                {
                    // start counter keeping
                    addr_man.counter.reset_my_counter_node();
                }
            }
        }

        /* Decoration of microfunc */
        private static void * helper_counter_hooked(void *v) throws Error
        {
            struct_helper_NtkNode_counter_hooked *tuple_p = (struct_helper_NtkNode_counter_hooked *)v;
            // The caller function has to add a reference to the ref-counted instances
            NtkNode self_save = tuple_p->self;
            AddressManager addr_man_save = tuple_p->addr_man;
            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            self_save.impl_counter_hooked(addr_man_save);
            // void method, return null
            return null;
        }

        void counter_hooked(AddressManager addr_man)
        {
            struct_helper_NtkNode_counter_hooked arg = struct_helper_NtkNode_counter_hooked();
            arg.self = this;
            arg.addr_man = addr_man;
            Tasklet.spawn((Spawnable)helper_counter_hooked, &arg);
        }

        private void impl_counter_registered(AddressManager addr_man) throws Error
        {
            Tasklet.declare_self("NtkNode.counter_registered");
            if (addr_man == addresses.primary_address)
            {
                if (! addr_man.andna.register_my_names_ongoing())
                {
                    // wait a bit for replication of records in Counter service
                    ms_wait(10000);
                    // then check again...
                    if (addr_man == addresses.primary_address)
                    {
                        if (! addr_man.andna.register_my_names_ongoing())
                        {
                            // start hostnames keeping
                            addr_man.andna.register_my_names();
                        }
                    }
                }
            }
        }

        /* Decoration of microfunc */
        private static void * helper_counter_registered(void *v) throws Error
        {
            struct_helper_NtkNode_counter_registered *tuple_p = (struct_helper_NtkNode_counter_registered *)v;
            // The caller function has to add a reference to the ref-counted instances
            NtkNode self_save = tuple_p->self;
            AddressManager addr_man_save = tuple_p->addr_man;
            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            self_save.impl_counter_registered(addr_man_save);
            // void method, return null
            return null;
        }

        void counter_registered(AddressManager addr_man)
        {
            struct_helper_NtkNode_counter_registered arg = struct_helper_NtkNode_counter_registered();
            arg.self = this;
            arg.addr_man = addr_man;
            Tasklet.spawn((Spawnable)helper_counter_registered, &arg);
        }
    }
}

