/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2011 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gee;
using zcd;
using Tasklets;

namespace Netsukuku.ImplLinux
{
    /** Launch a process and block this tasklet till it ends.
      * Returns exit status, stdout and stderr.
      */
    CommandResult exec_command(string cmdline) throws SpawnError
    {
        if (!("route flush cache" in cmdline)) log_trace(@"ImplLinux: going to execute...: $(cmdline)");

        CommandResult com_ret = Tasklet.exec_command(cmdline);

        string report = @" with exit value $(com_ret.exit_status) and stderr ";
        if (com_ret.cmderr == "") report += "null";
        else report += "NOT null";
        report = @"ImplLinux: completed$(report): $(cmdline)";
        if (com_ret.cmderr != "" || com_ret.exit_status != 0)
        {
            log_trace(report);
        }

        return com_ret;
    }

    struct struct_helper_ntk_linux_sequential_command
    {
        public string cmdline;
    }

    Channel? _sequential_command_channel = null;
    Channel get_sequential_command_channel()
    {
        if (_sequential_command_channel == null) _sequential_command_channel = Channel.find("ntk_linux_sequential_command_channel");
        if (_sequential_command_channel == null) _sequential_command_channel = new Channel("ntk_linux_sequential_command_channel");
        return (Channel)_sequential_command_channel;
    }

    void * helper_sequential_command(void *v)
    {
        Tasklet.declare_self("ImplLinux.sequential_command dispatcher");
        struct_channel *ch_cont_p = (struct_channel *)v;
        // The caller function has to add a reference to the ref-counted instances
        Channel ch = ch_cont_p->self;
        // schedule back to the spawner; this will probably invalidate *v and *ch_cont_p.
        Tasklet.schedule_back();
        // The actual dispatcher starts now. Refresh the channel for answers, in case a previous dispatcher tasklet was aborted.
        _sequential_command_channel = null;
        while (true)
        {
            string? doing = null;
            try
            {
                struct_helper_ntk_linux_sequential_command tuple_p;
                {
                    Value vv = ch.recv();
                    tuple_p = *((struct_helper_ntk_linux_sequential_command *)(vv.get_boxed()));
                }
                // The helper function should not need to copy values

                // do we need to give schedule?
                ms_wait(10);

                doing = @"$(tuple_p.cmdline)";
                Tasklet.declare_self(doing);
                string cmdout = null;
                string cmderr = null;
                int exit_status;
                try
                {
                    CommandResult com_ret = exec_command(tuple_p.cmdline);
                    cmdout = com_ret.cmdout;
                    cmderr = com_ret.cmderr;
                    exit_status = com_ret.exit_status;
                }
                catch (SpawnError e)
                {
                    exit_status = -1;
                    string x = @"Got $(e.domain) $(e.code) $(e.message)";
                    cmderr = x;
                    cmdout = "";
                }

                // do we need to give schedule?
                ms_wait(10);

                // pass the results
                get_sequential_command_channel().send(exit_status);
                get_sequential_command_channel().send(cmdout);
                get_sequential_command_channel().send(cmderr);
                // do we need to give schedule?
                Tasklet.schedule();
            }
            catch (Error e)
            {
                log_warn(@"Sequential command spawning reported an error: $(e.message)");
            }
            if (doing != null) Tasklet.declare_finished(doing);
        }
    }

    /** Launch a process and block this tasklet till it ends.
      * Makes sure that a previous call (from any other tasklet) has been completed.
      * Returns exit status, stdout and stderr.
      */
    public void sequential_command(string cmdline, out string cmdout, out string cmderr, out int exit_status)
    {
        // Register (once) the spawnable function that is our dispatcher
        // and obtain a channel to drive it.
        Channel ch = TaskletDispatcher.get_channel_for_helper((Spawnable)helper_sequential_command);
        struct_helper_ntk_linux_sequential_command arg = struct_helper_ntk_linux_sequential_command();
        arg.cmdline = cmdline;
        // send the struct, wait my turn
        ch.send(arg);
        // my request is being processed, wait for the result
        exit_status = (int)get_sequential_command_channel().recv();
        cmdout = (string)get_sequential_command_channel().recv();
        cmderr = (string)get_sequential_command_channel().recv();
    }


    public errordomain IPROUTECommandError{
        GENERIC
    }

    /** An iproute wrapper
      */
    public string iproute(string args) throws IPROUTECommandError
    {
        string cmd = Settings.IPROUTE_PATH;
        if (cmd == null || cmd == "")
        {
            // we might set it in configuration file
            Settings.IPROUTE_PATH = "ip";
            cmd = Settings.IPROUTE_PATH;
        }
        string? cmdout;
        string? cmderr;
        int exit_status;
        sequential_command(cmd + " " + args, out cmdout, out cmderr, out exit_status);
        if (exit_status != 0) throw new IPROUTECommandError.GENERIC(cmderr);
        return cmdout;
    }

    /** An iproute without exceptions
      */
    public void iproute_exp(string args)
    {
        try
        {
            iproute(args);
        }
        catch (IPROUTECommandError e)
        {
            log_warn(@"Iproute: Got following exception, but proceeding anyway: $(e.message)");
        }
    }


    public errordomain IPTABLESCommandError{
        GENERIC
    }

    /** An iptables wrapper
      */
    public string iptables(string args) throws IPTABLESCommandError
    {
        string cmd = Settings.IPTABLES_PATH;
        if (cmd == null || cmd == "")
        {
            // we might set it in configuration file
            Settings.IPTABLES_PATH = "iptables";
            cmd = Settings.IPTABLES_PATH;
        }
        string? cmdout;
        string? cmderr;
        int exit_status;
        sequential_command(cmd + " " + args, out cmdout, out cmderr, out exit_status);
        if (exit_status != 0) throw new IPTABLESCommandError.GENERIC(cmderr);
        return cmdout;
    }

    /** An iptables without exceptions
      */
    public void iptables_exp(string args)
    {
        try
        {
            iptables(args);
        }
        catch (IPTABLESCommandError e)
        {
            log_warn(@"Iptables: Got following exception, but proceeding anyway: $(e.message)");
        }
    }


    /** Check reachability of an IP
      */
    public bool check_ping(string ipstr)
    {
        string cmd = Settings.PING_PATH;
        if (cmd == null || cmd == "")
        {
            // we might set it in configuration file
            Settings.PING_PATH = "ping";
            cmd = Settings.PING_PATH;
        }
        int exit_status;
        try
        {
            CommandResult com_ret = exec_command(@"$(cmd) -c 1 $(ipstr) -W 1");
            exit_status = com_ret.exit_status;
        }
        catch (Error e) {error(@"linux: check_ping: $(e.domain.to_string()) $(e.code) $(e.message)");}
        return exit_status == 0;
    }


    void write_file(string pathname, uint8[] buf)
    {
        FileStream fout = FileStream.open(pathname, "w"); // truncate to 0 | create
        if (fout == null)
        {
            log_error(@"linux: write_file: $(strerror(Posix.errno))");
            assert(false);
        }
        fout.write(buf);
    }

    uint8[]? read_file(string pathname, int maxlen)
    {
        uint8[] buf = new uint8[maxlen];
        FileStream? fin = FileStream.open(pathname, "r"); // read from 0
        if (fin == null) return null;
        size_t len = fin.read(buf);
        uint8[] ret = new uint8[len];
        for (int i = 0; i < len; i++) ret[i] = buf[i];
        return ret;
    }


    public class LinuxNIC : NIC
    {
        private static NIC _create(string nic_name) {return new LinuxNIC(nic_name);}
        public static void register() {NIC.register_class("linux", _create);}

        public LinuxNIC(string nic_name)
        {
            base(nic_name);
            _mac = null;
        }

        private string? _mac;

        public override unowned string _mac_getter()
        {
            if (_mac == null)
            {
                try
                {
                    Regex r = new Regex("""([a-fA-F0-9]{2}[:|\-]?){6}""");
                    string cmdout = iproute(@"link show $name");
                    MatchInfo match_info;
                    r.match(cmdout, 0, out match_info);
                    _mac = match_info.fetch(0);
                }
                catch (Error e) {}
            }
            return _mac;
        }

        public override bool _is_active_getter()
        {
            try
            {
                Regex r2 = new Regex("""<.*,(UP),.*>""");
                string cmdout = iproute(@"link show $name");
                return r2.match(cmdout);
            }
            catch (Error e) {}
            return false;
        }

        public override void _up()
        {
            try {
                iproute(@"link set $name up");
            }
            catch (Error e) {error(@"LinuxNIC: _up: $(e.domain.to_string()) $(e.code) $(e.message)");}
        }

        public override void _down()
        {
            try {
                iproute(@"link set $name down");
            }
            catch (Error e) {error(@"LinuxNIC: _down: $(e.domain.to_string()) $(e.code) $(e.message)");}
        }

        public override void _add_address(string address)
        {
            try {
                iproute(@"addr add $address dev $name");
            }
            catch (Error e) {error(@"LinuxNIC: _add_address: $(e.domain.to_string()) $(e.code) $(e.message)");}
        }

        public override void _remove_address(string address)
        {
            try {
                iproute_exp(@"addr del $address/32 dev $name");
            }
            catch (Error e) {error(@"LinuxNIC: _remove_address: $(e.domain.to_string()) $(e.code) $(e.message)");}
        }

        public override void _disable_filtering()
        {
            string path0 = "/proc/sys/net/ipv4/conf/all/rp_filter";
            string path1 = @"/proc/sys/net/ipv4/conf/$(name)/rp_filter";
            string content = "0";
            try {
                write_file(path0, content.data);
            }
            catch (Error e) {
                log_error(@"rp_filter: error (ignored) writing '$content' in '$path0'");
            }
            try {
                write_file(path0, content.data);
            }
            catch (Error e) {
                log_error(@"rp_filter: error (ignored) writing '$content' in '$path1'");
            }
        }
    }

    const string RT_TABLE = "/etc/iproute2/rt_tables";
    int NTK_TABLE = 200;
    Regex? NTK_IN_RT_TABLE = null;

    /** Initialize linux policy routing db.
      */
    void init_prdb()
    {
        try
        {
            // Create table 'ntk'
            if (NTK_IN_RT_TABLE == null) NTK_IN_RT_TABLE = new Regex("\\d+\\s+ntk");
            string conf = "";
            // a path
            File ftable = File.new_for_commandline_arg(RT_TABLE);
            // load content
            string rt_table_content;
            uint8[] rt_table_content_arr;
            ftable.load_contents(null, out rt_table_content_arr, null);
            rt_table_content = (string)rt_table_content_arr;
            if (! NTK_IN_RT_TABLE.match(rt_table_content))
            {
                if (rt_table_content[rt_table_content.length-1] != '\n') conf += "\n";
                conf += "# Added by netsukuku\n";
                conf += @"$(NTK_TABLE)\tntk\n";
                // a path
                File fout = File.new_for_commandline_arg(RT_TABLE);
                // add "conf" to file
                FileOutputStream fos = fout.append_to(FileCreateFlags.NONE);
                fos.write(conf.data);
            }
            // add the table to the policy routing db, if it's not yet
            string pres = iproute("rule list");
            if (! (" lookup ntk " in pres))
                iproute("rule add table ntk");
        }
        catch (Error e)
        {
            error(@"LinuxNic.init_prdb: Caught exception $(e.domain) code $(e.code): $(e.message)");
        }
    }

    struct struct_helper_LinuxRoute_activate_multipath
    {
        public LinuxRoute self;
    }

    public class LinuxRoute : RouteSetter
    {
        private static RouteSetter _create()
        {
            // this is executed just once.
            if (! Settings.TESTING)
                init_prdb();
            return new LinuxRoute();
        }
        public static void register() {RouteSetter.register_class("linux", _create);}

        /** mac_table[w.upper()] is present (and has value n)
          *   iff a routing table exists for packets coming from w
          * w is a MAC string, eg '6a:b8:1e:cf:1d:4f'
          */
        private HashMap<string, int> mac_table;

        public LinuxRoute()
        {
            base();
            mac_table = new HashMap<string, int>();
        }

        private void impl_activate_multipath() throws Error
        {
            while (true)
            {
                try {
                    ms_wait(1000);
                    iproute("route flush cache");
                }
                catch (Error e)
                {
                    log_error(@"LinuxRoute: Exception while keep_flushing: $(e.message)");
                }
            }
        }

        private static void * helper_activate_multipath(void *v) throws Error
        {
            Tasklet.declare_self("LinuxRoute.activate_multipath dispatcher");
            struct_channel *ch_cont_p = (struct_channel *)v;
            // The caller function has to add a reference to the ref-counted instances
            Channel ch = ch_cont_p->self;
            // schedule back to the spawner; this will probably invalidate *v and *ch_cont_p.
            Tasklet.schedule_back();
            // The actual dispatcher
            while (true)
            {
                string? doing = null;
                try
                {
                    struct_helper_LinuxRoute_activate_multipath tuple_p;
                    {
                        Value vv = ch.recv();
                        tuple_p = *((struct_helper_LinuxRoute_activate_multipath *)(vv.get_boxed()));
                    }
                    doing = "activate_multipath()";
                    Tasklet.declare_self(doing);
                    // The helper function should not need to copy values
                    tuple_p.self.impl_activate_multipath();
                    // do we need to give schedule?
                    Tasklet.nap(0, 100);
                }
                catch (Error e)
                {
                    log_warn(@"LinuxRoute: activate_multipath microfunc reported an error: $(e.message)");
                }
                if (doing != null) Tasklet.declare_finished(doing);
            }
        }

        public override void activate_multipath()
        {
            // Register (once) the spawnable function that is our dispatcher
            // and obtain a channel to drive it.
            Channel ch = TaskletDispatcher.get_channel_for_helper((Spawnable)helper_activate_multipath);
            struct_helper_LinuxRoute_activate_multipath arg = struct_helper_LinuxRoute_activate_multipath();
            arg.self = this;
            // send the struct
            ch.send_async(arg);
        }

        /** Makes sure that a routing table exists for packets coming from macaddr.
          * Returns the number of that table.
          */
        public int _table_for_macaddr(string _macaddr) throws Error
        {
            string macaddr = _macaddr.down();
            int idn;
            if (! mac_table.has_key(macaddr))
            {
                // Find first integer not used, starting from 199 going back
                // TODO: is the number ok?
                idn = 199;
                while (mac_table.values.contains(idn)) idn--;
                // Add a routing table for mac prev_hop with number idn
                iptables(@"-t mangle -A PREROUTING -m mac --mac-source $macaddr -j MARK --set-mark $idn");
                iproute(@"rule add fwmark $idn table $idn");
                mac_table[macaddr] = idn;
            }
            else
            {
                idn = mac_table[macaddr];
            }
            return idn;
        }

        /** Removes all routing table created for packets coming from any macaddr
          */
        public void _table_for_macaddr_remove_all() throws Error
        {
            ArrayList<string> all_macs = new ArrayList<string>();
            all_macs.add_all(mac_table.keys);
            foreach (string k in all_macs)
                _table_for_macaddr_remove(k);
        }

        /** Makes sure that a routing table doesn't exist anymore for packets coming from macaddr
          */
        public void _table_for_macaddr_remove(string _macaddr) throws Error
        {
            string macaddr = _macaddr.down();
            if (mac_table.has_key(macaddr))
            {
                int idn = mac_table[macaddr];
                // Remove the association to routing table with number idn for mac prev_hop
                iptables(@"-t mangle -D PREROUTING -m mac --mac-source $macaddr -j MARK --set-mark $idn");
                iproute(@"rule del fwmark $idn table $idn");
                mac_table.unset(macaddr);
            }
        }

        /** Returns proper iproute command arguments to add/change/delete routes
          */
        public string _modify_routes_cmd(string command, string ip, string cidr, RouteSolutions? route_solutions, string? table=null)
        {
            string cmd = @"route $command $(ip)/$(cidr)";

            if (table != null) cmd += @" table $table";
            else cmd += " table ntk";

            // cmd += " protocol ntk";

            if (route_solutions != null)
            {
                if (route_solutions.pref_src != null)
                    cmd += @" src $(route_solutions.pref_src)";
                foreach (RouteSolution rs in route_solutions.values)
                    cmd += @" nexthop via $(rs.gateway) dev $(rs.dev) weight $(1+rs.weight*255/1000)";
            }

            return cmd;
        }

        /** Returns proper iproute command arguments to add/change/delete a neighbour
          */
        public string _modify_neighbour_cmd(string command, string ip, string? dev, string? pref_src)
        {
            string cmd = @"route $command $(ip)";

            if (dev != null) cmd += @" dev $dev";
            cmd += " table ntk";
            if (pref_src != null) cmd += @" src $pref_src";

            // cmd += " protocol ntk";

            return cmd;
        }


        /** Add a route (that was non existent) towards ip/cidr
          * for packets generated by this host.
          * The gateways are route_solutions
          */
        protected override void _add_outgoing_route(string ip, string cidr, RouteSolutions route_solutions)
        {
            if (cidr == "32" && route_solutions.contains_gw(ip)) return;
            iproute_exp(_modify_routes_cmd("add", ip, cidr, route_solutions));
        }

        /** Add a route (that was non existent) towards ip/cidr
          * for packets coming from MAC prev_hop.
          * The gateways are route_solutions
          */
        protected override void _add_forwarding_route(string ip, string cidr, string prev_hop, RouteSolutions route_solutions) throws Error
        {
            if (cidr == "32" && route_solutions.contains_gw(ip)) return;
            int idn = _table_for_macaddr(prev_hop);
            iproute_exp(_modify_routes_cmd("add", ip, cidr, route_solutions, idn.to_string()));
        }

        /** Add a route (that was non existent) towards ip/cidr
          * for packets coming from MAC prev_hop.
          * In fact this route is UNREACHABLE
          */
        protected override void _add_forwarding_route_unreachable(string ip, string cidr, string prev_hop) throws Error
        {
            int idn = _table_for_macaddr(prev_hop);
            iproute_exp(@"route add table $idn unreachable $ip/$cidr");
        }

        /** Add a route (that was non existent) towards ip/cidr
          * for packets coming from MAC prev_hop.
          * In fact this route is DROP
          */
        protected override void _add_forwarding_route_drop(string ip, string cidr, string prev_hop) throws Error
        {
            int idn = _table_for_macaddr(prev_hop);
            iproute_exp(@"route add table $idn blackhole $ip/$cidr");
        }

        /** Change a route (that was existent) towards ip/cidr
          * for packets generated by this host.
          * The gateways are route_solutions.
          * The old gateways were old_route_solutions
          */
        protected override void _change_outgoing_route(string ip, string cidr, RouteSolutions route_solutions, RouteSolutions old_route_solutions)
        {
            if (cidr == "32")
            {
                if (route_solutions.contains_gw(ip))
                {
                    if (old_route_solutions.contains_gw(ip))
                    {
                        return;
                    }
                    else
                    {
                        // delete old
                        iproute_exp(_modify_routes_cmd("del", ip, cidr, null));
                        return;
                    }
                }
                else
                {
                    if (old_route_solutions.contains_gw(ip))
                    {
                        // add new
                        iproute_exp(_modify_routes_cmd("add", ip, cidr, route_solutions));
                        return;
                    }
                    // else go on...
                }
            }
            iproute_exp(_modify_routes_cmd("change", ip, cidr, route_solutions));
        }

        /** Change a route (that was existent) towards ip/cidr
          * for packets coming from MAC prev_hop.
          * The gateways are route_solutions.
          * The old gateways were old_route_solutions
          */
        protected override void _change_forwarding_route(string ip, string cidr, string prev_hop,
                RouteSolutions route_solutions, RouteSolutions old_route_solutions) throws Error
        {
            int idn = _table_for_macaddr(prev_hop);
            if (cidr == "32")
            {
                if (route_solutions.contains_gw(ip))
                {
                    if (old_route_solutions.contains_gw(ip))
                    {
                        return;
                    }
                    else
                    {
                        // delete old
                        iproute_exp(_modify_routes_cmd("del", ip, cidr, null, idn.to_string()));
                        return;
                    }
                }
                else
                {
                    if (old_route_solutions.contains_gw(ip))
                    {
                        // add new
                        iproute_exp(_modify_routes_cmd("add", ip, cidr, route_solutions, idn.to_string()));
                        return;
                    }
                    // else go on...
                }
            }
            iproute_exp(_modify_routes_cmd("change", ip, cidr, route_solutions, idn.to_string()));
        }

        /** Change a route (that was existent) towards ip/cidr
          * for packets coming from MAC prev_hop.
          * In fact this route is UNREACHABLE
          * The old gateways were old_route_solutions
          */
        protected override void _change_forwarding_route_unreachable(string ip, string cidr, string prev_hop,
                RouteSolutions old_route_solutions) throws Error
        {
            int idn = _table_for_macaddr(prev_hop);
            if (cidr == "32" && old_route_solutions.contains_gw(ip))
            {
                // add new
                iproute_exp(@"route add table $idn unreachable $ip/$cidr");
                return;
            }
            iproute_exp(@"route change table $idn unreachable $ip/$cidr");
        }

        /** Change a route (that was existent) towards ip/cidr
          * for packets coming from MAC prev_hop.
          * In fact this route is DROP
          * The old gateways were old_route_solutions
          */
        protected override void _change_forwarding_route_drop(string ip, string cidr, string prev_hop,
                RouteSolutions old_route_solutions) throws Error
        {
            int idn = _table_for_macaddr(prev_hop);
            if (cidr == "32" && old_route_solutions.contains_gw(ip))
            {
                // add new
                iproute_exp(@"route add table $idn blackhole $ip/$cidr");
                return;
            }
            iproute_exp(@"route change table $idn blackhole $ip/$cidr");
        }

        /** Change a route (that was existent) towards ip/cidr
          * for packets coming from MAC prev_hop.
          * The gateways are route_solutions.
          * Previously the route was UNREACHABLE
          */
        protected override void _change_forwarding_route_was_unreachable(string ip, string cidr, string prev_hop,
                RouteSolutions route_solutions) throws Error
        {
            int idn = _table_for_macaddr(prev_hop);
            if (cidr == "32" && route_solutions.contains_gw(ip))
            {
                // delete old unreachable
                iproute_exp(@"route del table $idn unreachable $ip/$cidr");
                return;
            }
            iproute_exp(_modify_routes_cmd("change", ip, cidr, route_solutions, idn.to_string()));
        }

        /** Change a route (that was existent) towards ip/cidr
          * for packets coming from MAC prev_hop.
          * In fact now this route is DROP
          * Previously the route was UNREACHABLE
          */
        protected override void _change_forwarding_route_drop_was_unreachable(string ip, string cidr,
                string prev_hop) throws Error
        {
            int idn = _table_for_macaddr(prev_hop);
            iproute_exp(@"route change table $idn blackhole $ip/$cidr");
        }

        /** Change a route (that was existent) towards ip/cidr
          * for packets coming from MAC prev_hop.
          * The gateways are route_solutions.
          * Previously the route was DROP
          */
        protected override void _change_forwarding_route_was_drop(string ip, string cidr, string prev_hop,
                RouteSolutions route_solutions) throws Error
        {
            int idn = _table_for_macaddr(prev_hop);
            if (cidr == "32" && route_solutions.contains_gw(ip))
            {
                // delete old blackhole
                iproute_exp(@"route del table $idn blackhole $ip/$cidr");
                return;
            }
            iproute_exp(_modify_routes_cmd("change", ip, cidr, route_solutions, idn.to_string()));
        }

        /** Change a route (that was existent) towards ip/cidr
          * for packets coming from MAC prev_hop.
          * In fact now this route is UNREACHABLE
          * Previously the route was DROP
          */
        protected override void _change_forwarding_route_unreachable_was_drop(string ip, string cidr,
                string prev_hop) throws Error
        {
            int idn = _table_for_macaddr(prev_hop);
            iproute_exp(@"route change table $idn unreachable $ip/$cidr");
        }

        /** Remove a route (that was existent) towards ip/cidr
          * for packets generated by this host.
          * The old gateways were old_route_solutions
          */
        protected override void _remove_outgoing_route(string ip, string cidr, RouteSolutions old_route_solutions)
        {
            iproute_exp(_modify_routes_cmd("del", ip, cidr, null));
        }

        /** Remove a route (that was existent) towards ip/cidr
          * for packets coming from MAC prev_hop.
          * The old gateways were old_route_solutions
          */
        protected override void _remove_forwarding_route(string ip, string cidr, string prev_hop,
                RouteSolutions old_route_solutions) throws Error
        {
            int idn = _table_for_macaddr(prev_hop);
            iproute_exp(_modify_routes_cmd("del", ip, cidr, null, idn.to_string()));
        }

        /** Remove a route (that was existent) towards ip/cidr
          * for packets coming from MAC prev_hop.
          * Previously the route was UNREACHABLE
          */
        protected override void _remove_forwarding_route_was_unreachable(string ip, string cidr,
                string prev_hop) throws Error
        {
            int idn = _table_for_macaddr(prev_hop);
            iproute_exp(_modify_routes_cmd("del", ip, cidr, null, idn.to_string()));
        }

        /** Remove a route (that was existent) towards ip/cidr
          * for packets coming from MAC prev_hop.
          * Previously the route was DROP
          */
        protected override void _remove_forwarding_route_was_drop(string ip, string cidr, string prev_hop) throws Error
        {
            int idn = _table_for_macaddr(prev_hop);
            iproute_exp(_modify_routes_cmd("del", ip, cidr, null, idn.to_string()));
        }

        /** Stop forwarding (in fact it means delete all rules for...)
          * packets coming from MAC prev_hop.
          */
        protected override void _forward_no_more_from(string _prev_hop) throws Error
        {
            string prev_hop = _prev_hop.down();
            if (mac_table.has_key(prev_hop))
            {
                int idn = _table_for_macaddr(prev_hop);
                try
                {
                    iproute(@"route flush table $idn");
                }
                catch (Error e)
                {
                    if (!e.message.contains("Nothing to flush"))
                    {
                        log_error(@"ip route flush table $idn: reported $(e.message). Perhaps better to abort?");
                    }
                }
                _table_for_macaddr_remove(prev_hop);
            }
        }

        /** Delete all specific rules for any prev_hop.
          */
        protected override void _forward_no_more() throws Error
        {
            _table_for_macaddr_remove_all();
        }

        /** Adds a new neighbour with corresponding properties.
          */
        protected override void _add_neighbour(string ip, string dev, string pref_src)
        {
            iproute_exp(_modify_neighbour_cmd("add", ip, dev, pref_src));
        }

        /** Edits the neighbour with the corresponding properties.
          */
        protected override void _change_neighbour(string ip, string dev, string pref_src, string old_dev)
        {
            iproute_exp(_modify_neighbour_cmd("change", ip, dev, pref_src));
        }

        /** Removes the neighbour with the corresponding properties.
          */
        protected override void _delete_neighbour(string ip, string old_dev)
        {
            iproute_exp(_modify_neighbour_cmd("del", ip, null, null));
        }

        /** IGS management **/
        
        protected void _disable_rp_filter(string nicname)
        {
            string path = @"/proc/sys/net/ipv4/conf/$(nicname)/rp_filter";
            string content = "0";
            try {
                write_file(path, content.data);
            }
            catch (Error e) {
                log_error(@"rp_filter: error (ignored) writing '$content' in '$path'");
            }
        
        }

        protected override bool _igs_update_rules(string prev_mode, bool prev_announce_myself, ArrayList<TunnelItem> prev_list_tunnels,
                    string mode, bool announce_myself, ArrayList<TunnelItem> list_tunnels, string pref_src) throws Error
        {
            // An implementor returns false iff it is unable to implement this feature.
            // mode in ['NONE', 'SHARE', 'USE', 'BOTH']
            // TunnelItem is a pair (ipstr, rem)
            // ArrayList has been created with    new ArrayList<TunnelItem>(TunnelItem.equal_func)
            // prev_* are previous values.

            try {
                CommandResult com_ret = exec_command("modprobe ipip");
            }
            catch (Error e) {}

            string dbg_list_tunnels = "{";
            foreach (TunnelItem tun in list_tunnels) dbg_list_tunnels += @"$tun,";
            dbg_list_tunnels += "}";

            assert(mode != "BOTH");

            if (prev_mode == "NONE" && mode == "SHARE")
            {
                iptables_exp("-t nat -A POSTROUTING -s 10.0.0.0/8 ! -d 10.0.0.0/8 -j MASQUERADE");
                iproute_exp("tunnel add ntk-share-inet mode ipip");
                iproute_exp("link set tunl0 up");
                _disable_rp_filter("tunl0");
            }

            if (mode == "NONE" && prev_mode == "SHARE")
            {
                iptables_exp("-t nat -D POSTROUTING -s 10.0.0.0/8 ! -d 10.0.0.0/8 -j MASQUERADE");
                iproute_exp("link set tunl0 down");
                iproute_exp("tunnel del ntk-share-inet mode ipip");
            }

            if (prev_mode == "NONE" && mode == "USE" && ! list_tunnels.is_empty)
            {
                // use up to 10 predefined values for weight
                int[] weights = {1, 3, 6, 10, 30, 60, 100, 150, 200, 255};
                // TODO improve the choice of values based on the
                //   REMs of the inet-gateways and experience on the field.
                int i = 0;
                foreach (TunnelItem tun in list_tunnels)
                {
                    string ipstr = tun.ipstr;
                    string dev = @"ntk-to-inet-$i";
                    iproute_exp(@"tunnel add $dev mode ipip remote $ipstr");
                    iproute_exp(@"link set $dev up");
                    _disable_rp_filter("tunl0");
                    _disable_rp_filter(dev);
                    iproute_exp(@"addr add $pref_src dev $dev");
                    i++;
                }

                iptables(@"-t mangle -A OUTPUT -m conntrack --ctstate ESTABLISHED,RELATED  -j CONNMARK --restore-mark");
                i = 0;
                foreach (TunnelItem tun in list_tunnels)
                {
                    string ipstr = tun.ipstr;
                    string dev = @"ntk-to-inet-$i";
                    int tableid = 99 - i;
                    iptables_exp(@"-t mangle -A POSTROUTING -o $dev -m conntrack --ctstate NEW -j CONNMARK --set-mark $tableid");
                    iproute_exp(@"rule add fwmark $tableid lookup $tableid");
                    iproute_exp(@"route add table $tableid default via $ipstr dev $dev src $pref_src onlink");
                    i++;
                }

                string cmd = "route add default ";
                i = 0;
                foreach (TunnelItem tun in list_tunnels)
                {
                    string ipstr = tun.ipstr;
                    string dev = @"ntk-to-inet-$i";
                    int weight = weights[list_tunnels.size - 1 - i];
                    cmd += @"nexthop via $ipstr dev $dev weight $weight onlink ";
                    i++;
                }
                iproute_exp(cmd);

            }

            if (mode == "NONE" && prev_mode == "USE" && ! prev_list_tunnels.is_empty)
            {
                // use up to 10 predefined values for weight
                int[] weights = {1, 3, 6, 10, 30, 60, 100, 150, 200, 255};
                // TODO improve the choice of values based on the
                //   REMs of the inet-gateways and experience on the field.
                int i = 0;
                string cmd = "route del default ";
                foreach (TunnelItem tun in prev_list_tunnels)
                {
                    string ipstr = tun.ipstr;
                    string dev = @"ntk-to-inet-$i";
                    int weight = weights[list_tunnels.size - 1 - i];
                    cmd += @"nexthop via $ipstr dev $dev weight $weight onlink ";
                    i++;
                }
                iproute_exp(cmd);

                i = 0;
                foreach (TunnelItem tun in prev_list_tunnels)
                {
                    string ipstr = tun.ipstr;
                    string dev = @"ntk-to-inet-$i";
                    int tableid = 99 - i;
                    iptables_exp(@"-t mangle -D POSTROUTING -o $dev -m conntrack --ctstate NEW -j CONNMARK --set-mark $tableid");
                    iproute_exp(@"rule del fwmark $tableid lookup $tableid");
                    iproute_exp(@"route del table $tableid default via $ipstr dev $dev src $pref_src onlink");
                    i++;
                }
                iptables(@"-t mangle -D OUTPUT -m conntrack --ctstate ESTABLISHED,RELATED  -j CONNMARK --restore-mark");

                i = 0;
                foreach (TunnelItem tun in prev_list_tunnels)
                {
                    string ipstr = tun.ipstr;
                    string dev = @"ntk-to-inet-$i";
                    iproute_exp(@"addr del $pref_src dev $dev");
                    iproute_exp(@"link set $dev down");
                    iproute_exp(@"tunnel dek $dev mode ipip remote $ipstr");
                    i++;
                }
            }

            // mode == 'USE' and prev_mode == 'USE' and prev_list_tunnels != list_tunnels
            if (mode == "USE" && prev_mode == "USE"
                && ! (prev_list_tunnels.contains_all(list_tunnels) && list_tunnels.contains_all(prev_list_tunnels)))
            {
                // use up to 10 predefined values for weight
                int[] weights = {1, 3, 6, 10, 30, 60, 100, 150, 200, 255};
                // TODO improve the choice of values based on the
                //   REMs of the inet-gateways and experience on the field.
                if (! prev_list_tunnels.is_empty)
                {
                    int i = 0;
                    string cmd = "route del default ";
                    foreach (TunnelItem tun in prev_list_tunnels)
                    {
                        string ipstr = tun.ipstr;
                        string dev = @"ntk-to-inet-$i";
                        int weight = weights[list_tunnels.size - 1 - i];
                        cmd += @"nexthop via $ipstr dev $dev weight $weight onlink ";
                        i++;
                    }
                    iproute_exp(cmd);

                    i = 0;
                    foreach (TunnelItem tun in prev_list_tunnels)
                    {
                        string ipstr = tun.ipstr;
                        string dev = @"ntk-to-inet-$i";
                        int tableid = 99 - i;
                        iptables_exp(@"-t mangle -D POSTROUTING -o $dev -m conntrack --ctstate NEW -j CONNMARK --set-mark $tableid");
                        iproute_exp(@"rule del fwmark $tableid lookup $tableid");
                        iproute_exp(@"route del table $tableid default via $ipstr dev $dev src $pref_src onlink");
                        i++;
                    }
                    iptables(@"-t mangle -D OUTPUT -m conntrack --ctstate ESTABLISHED,RELATED  -j CONNMARK --restore-mark");

                    i = 0;
                    foreach (TunnelItem tun in prev_list_tunnels)
                    {
                        string ipstr = tun.ipstr;
                        string dev = @"ntk-to-inet-$i";
                        iproute_exp(@"addr del $pref_src dev $dev");
                        iproute_exp(@"link set $dev down");
                        iproute_exp(@"tunnel dek $dev mode ipip remote $ipstr");
                        i++;
                    }
                }
                if (! list_tunnels.is_empty)
                {
                    int i = 0;
                    foreach (TunnelItem tun in list_tunnels)
                    {
                        string ipstr = tun.ipstr;
                        string dev = @"ntk-to-inet-$i";
                        iproute_exp(@"tunnel add $dev mode ipip remote $ipstr");
                        iproute_exp(@"link set $dev up");
                        _disable_rp_filter("tunl0");
                        _disable_rp_filter(dev);
                        iproute_exp(@"addr add $pref_src dev $dev");
                        i++;
                    }

                    iptables(@"-t mangle -A OUTPUT -m conntrack --ctstate ESTABLISHED,RELATED  -j CONNMARK --restore-mark");
                    i = 0;
                    foreach (TunnelItem tun in list_tunnels)
                    {
                        string ipstr = tun.ipstr;
                        string dev = @"ntk-to-inet-$i";
                        int tableid = 99 - i;
                        iptables_exp(@"-t mangle -A POSTROUTING -o $dev -m conntrack --ctstate NEW -j CONNMARK --set-mark $tableid");
                        iproute_exp(@"rule add fwmark $tableid lookup $tableid");
                        iproute_exp(@"route add table $tableid default via $ipstr dev $dev src $pref_src onlink");
                        i++;
                    }

                    string cmd = "route add default ";
                    i = 0;
                    foreach (TunnelItem tun in list_tunnels)
                    {
                        string ipstr = tun.ipstr;
                        string dev = @"ntk-to-inet-$i";
                        int weight = weights[list_tunnels.size - 1 - i];
                        cmd += @"nexthop via $ipstr dev $dev weight $weight onlink ";
                        i++;
                    }
                    iproute_exp(cmd);
                }
            }

            return true;
        }

        protected override bool check_ping(string ipstr)
        {
            return ImplLinux.check_ping(ipstr);
        }
        public bool impl_check_ping(string ipstr)
        {
            return this.check_ping(ipstr);
        }

        // initializations

        // An implementor returns false iff it is unable to implement this feature.
        protected override bool _reset_routes(string ip_whole_network, string cidr_whole_network)
        {
            try
            {
                iproute("route flush table ntk");
            }
            catch (Error e)
            {
                // It could result in "Nothing to flush". It's ok.
                assert(e.message.contains("Nothing to flush"));
            }
            iproute_exp(@"route add table ntk unreachable $(ip_whole_network)/$(cidr_whole_network)");
            ArrayList<int> done = new ArrayList<int>();
            foreach (int idn in mac_table.values)
            {
                if (! done.contains(idn))
                {
                    done.add(idn);
                    try
                    {
                        iproute(@"route flush table $idn");
                    }
                    catch (Error e)
                    {
                        // It could result in "Nothing to flush". It's ok.
                        assert(e.message.contains("Nothing to flush"));
                    }
                }
            }
            return true;
        }

        public override void ip_forward(bool enable)
        {
            string path = "/proc/sys/net/ipv4/ip_forward";
            string content = enable ? "1" : "0";
            try {
                write_file(path, content.data);
            }
            catch (Error e) {
                log_error(@"ip_forward: error (ignored) writing '$content' in '$path'");
            }
        }

    }

    public class LinuxConnections : Connections
    {
        private static Connections _create() {return new LinuxConnections();}
        public static void register() {Connections.register_class("linux", _create);}

        public override bool active_tcp_connections(string ip)
        {
            // TODO issue netstat
            log_info(@"Connections: issue: netstat -an | grep ^tcp | grep ESTABLISHED | grep $ip:");
            return false;
        }
    }

    /** Managing tunnels via tincd in linux
      */
    public class LinuxTunnel : BaseTunnel
    {
        private static BaseTunnel _create() {return new LinuxTunnel();}
        public static void register() {BaseTunnel.register_class("linux", _create);}

        private LinkedList<int> port_pool;
        private HashMap<string, int> used_ports;

        public LinuxTunnel()
        {
            base();
            // We can have multiple tincd daemons, as long as they use different ports:
            port_pool = new LinkedList<int>();
            for (int i = 50655; i < 50680; i++) port_pool.offer(i);
            // used_ports: key is nic_name, value is port
            used_ports = new HashMap<string, int>();
        }

        public override string serve(CallbackSendDelegate cb_send, CallbackRecvDelegate cb_recv, string my_address, string dest_address) throws TunnelError, RPCError
        {
            // Serves a tunnel.
            // If all goes well, it returns the name of the new NIC.

            // Get a port
            if (port_pool.is_empty)
                throw new TunnelError.GENERIC("tincd: no more free ports. too many instances?");
            int my_port = port_pool.poll();
            // Receive nic_name from the peer.
            string nic_name = ((SerializableString)cb_recv()).s;
            used_ports[nic_name] = my_port;
            // Configure tincd (using a script)
            string? cmdout;
            string? cmderr;
            int exit_status;
            sequential_command(@"/etc/netsukuku/tinc configure_server $nic_name $my_address $my_port", out cmdout, out cmderr, out exit_status);
            if (exit_status != 0)
                throw new TunnelError.GENERIC(@"tincd: $cmderr");
            // Read the key-file
            uint8[]? key = read_file(@"/etc/tinc/$(nic_name)/hosts/node_server", 2000);
            if (key == null)
                throw new TunnelError.GENERIC("tincd: key file not generated.");
            // Receive the peer's key.
            SerializableBuffer peer_key = (SerializableBuffer)cb_recv();
            // Pass to the peer.
            cb_send(new SerializableBuffer(key));
            // Save the peer's key-file
            write_file(@"/etc/tinc/$(nic_name)/hosts/node_client", peer_key.buffer);
            // We are done configuring, now launch the tincd
            sequential_command(@"/etc/netsukuku/tinc start $nic_name $my_address $my_port", out cmdout, out cmderr, out exit_status);
            if (exit_status != 0)
                throw new TunnelError.GENERIC(@"tincd: $cmderr");
            // All done.
            return nic_name;
        }

        public override string request(CallbackSendDelegate cb_send, CallbackRecvDelegate cb_recv, string my_address, string dest_address) throws TunnelError, RPCError
        {
            // Requests a tunnel.
            // If all goes well, it returns the name of the new NIC.

            // Get a port
            if (port_pool.is_empty)
                throw new TunnelError.GENERIC("tincd: no more free ports. too many instances?");
            int my_port = port_pool.poll();
            // Generate a random ID for the virtual collision domain
            int tunnel_id = Random.int_range(0, ((int)Math.pow(2, 32))-1);
            string nic_name = @"net_$(tunnel_id)";
            used_ports[nic_name] = my_port;
            // Pass to the peer.
            cb_send(new SerializableString(nic_name));
            // Configure tincd (using a script)
            string? cmdout;
            string? cmderr;
            int exit_status;
            sequential_command(@"/etc/netsukuku/tinc configure_client $nic_name $my_address $my_port", out cmdout, out cmderr, out exit_status);
            if (exit_status != 0)
                throw new TunnelError.GENERIC(@"tincd: $cmderr");
            // Read the key-file
            uint8[]? key = read_file(@"/etc/tinc/$(nic_name)/hosts/node_client", 2000);
            if (key == null)
                throw new TunnelError.GENERIC("tincd: key file not generated.");
            // Pass to the peer.
            cb_send(new SerializableBuffer(key));
            // Receive the peer's key.
            SerializableBuffer peer_key = (SerializableBuffer)cb_recv();
            // Save the peer's key-file
            write_file(@"/etc/tinc/$(nic_name)/hosts/node_server", peer_key.buffer);
            // We are done configuring, now launch the tincd
            sequential_command(@"/etc/netsukuku/tinc start $nic_name $my_address $my_port", out cmdout, out cmderr, out exit_status);
            if (exit_status != 0)
                throw new TunnelError.GENERIC(@"tincd: $cmderr");
            // All done.
            return nic_name;
        }

        public override void close(string nic_name, string my_address, string dest_address) throws TunnelError
        {
            // Closes a tunnel.

            string? cmdout;
            string? cmderr;
            int exit_status;
            sequential_command(@"/etc/netsukuku/tinc stop $nic_name none none", out cmdout, out cmderr, out exit_status);
            if (exit_status != 0)
                throw new TunnelError.GENERIC(@"tincd: $cmderr");
            sequential_command(@"/etc/netsukuku/tinc remove $nic_name none none", out cmdout, out cmderr, out exit_status);
            if (exit_status != 0)
                throw new TunnelError.GENERIC(@"tincd: $cmderr");
            int my_port = used_ports[nic_name];
            port_pool.offer(my_port);
            used_ports.unset(nic_name);
        }
    }
}

