/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2011 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gee;
using zcd;
using Tasklets;

namespace Netsukuku
{
    struct struct_helper_KrnlRoute_dispatched
    {
        public KrnlRoute self;
    }

    /** Listens to MapRoute generated events, and updates the kernel table
      */
    public class KrnlRoute : Object
    {
        private Addresses addresses;
        private bool multipath;
        private bool running;
        private bool enqueued;
        private ArrayList<string> updated_node_macs;

        public KrnlRoute(Addresses addresses)
        {
            this.addresses = addresses;
            multipath = Settings.MULTIPATH;

            addresses.network_reset.connect(network_reset);

            // To handle neighbours
            addresses.aggregated_neighbour_new_before.connect(impl_aggregated_neighbour_new);
            addresses.aggregated_neighbour_deleted_after.connect(impl_aggregated_neighbour_deleted);
            addresses.aggregated_neighbour_rem_chged_before.connect(impl_aggregated_neighbour_rem_chged);

            // To handle routes
            addresses.routes_updated.connect(impl_routes_updated);
            addresses.sig_is_mature.connect(impl_sig_is_mature);

            // To handle incoming packet forwarding
            addresses.incoming_node_updated.connect(impl_incoming_node_updated);

            running = false;
            enqueued = false;
            updated_node_macs = new ArrayList<string>();

            if (multipath) RouteSetter.get_instance().activate_multipath();
        }

        /** We have no routes
          */
        private void network_reset()
        {
            try
            {
                RouteSetter kroute = RouteSetter.get_instance();
                string ipstr_whole_network;
                string bits_whole_network;
                // let's represent the outermost gnode:
                int levels = addresses.levels;
                int gsize = addresses.gsize;
                int[] gnode_pos = new int[levels];
                for (int i = 0; i < levels; i++) gnode_pos[i] = -1;
                PartialNIP gnode = new PartialNIP(gnode_pos);
                // the outermost gnode in terms of CIDR
                gnode_to_ip_cidr(gnode, gsize, out ipstr_whole_network, out bits_whole_network);
                kroute.reset_routes(ipstr_whole_network, bits_whole_network);
                kroute.forward_no_more();
            }
            catch (Error e)
            {
                error(@"KrnlRoute: Caught exception while network_reset $(e.domain) code $(e.code): $(e.message)");
            }
        }

        /**
          * The rules that this class has to keep satisfied are:
          * given that we are x;
          * at any time, for any v ∈ V,
          *     if Bestᵗ (x → v) = xy...v > 0
          *         x maintains this RULE: destination = v → gateway = y
          *     ∀w ∈ x*
          *         if w ∈ Bestᵗ (x → v) AND Bestᵗ (x → w̃ → v) = xz...v > 0
          *             x maintains this RULE: source w, destination = v → gateway = z
          */

        private string me_to_ipstr()
        {
            MapRoute mr = addresses.primary_address.maproute;
            return nip_to_str(mr.levels, mr.gsize, mr.me);
        }

        private void impl_incoming_node_updated(AddressManager addrman, string mac)
        {
            updated_node_macs.add(mac);
            set_current_neighbours_routes_and_forwards();
        }
        private void impl_routes_updated(AddressManager addrman, HCoord lvl_pos)
        {
            set_current_neighbours_routes_and_forwards();
        }
        private void impl_sig_is_mature(AddressManager mature_addrman)
        {
            set_current_neighbours_routes_and_forwards();
        }
        private void impl_aggregated_neighbour_new(AggregatedNeighbour aggregated_neighbour)
        {
            set_current_neighbours_routes_and_forwards();
        }
        private void impl_aggregated_neighbour_deleted(AggregatedNeighbour aggregated_neighbour)
        {
            set_current_neighbours_routes_and_forwards();
        }
        private void impl_aggregated_neighbour_rem_chged(AggregatedNeighbour aggregated_neighbour, REM old_rem)
        {
            set_current_neighbours_routes_and_forwards();
        }

        private void set_current_neighbours_routes_and_forwards()
        {
            // If another tasklet is already running this function, then just
            //  enqueue another run when that ends. But do not enqueue more than
            //  once.
            if (enqueued) return;
            if (running)
            {
                enqueued = true;
                return;
            }
            // Start a tasklet to do the duties.
            running = true;
            dispatched_set_current_neighbours_routes_and_forwards();
        }

        private void impl_dispatched_set_current_neighbours_routes_and_forwards() throws Error
        {
            while (true)
            {
                once_set_current_neighbours_routes_and_forwards();
                // Test if another run has been enqueued
                if (enqueued)
                {
                    enqueued = false;
                }
                else
                {
                    running = false;
                    break;
                }
            }
        }

        private static void * helper_dispatched_set_current_neighbours_routes_and_forwards(void *v) throws Error
        {
            Tasklet.declare_self("KrnlRoute.neighbours_routes_and_forwards dispatcher");
            struct_channel *ch_cont_p = (struct_channel *)v;
            // The caller function has to add a reference to the ref-counted instances
            Channel ch = ch_cont_p->self;
            // schedule back to the spawner; this will probably invalidate *v and *ch_cont_p.
            Tasklet.schedule_back();
            // The actual dispatcher
            while (true)
            {
                string? doing = null;
                try
                {
                    struct_helper_KrnlRoute_dispatched tuple_p;
                    {
                        Value vv = ch.recv();
                        tuple_p = *((struct_helper_KrnlRoute_dispatched *)(vv.get_boxed()));
                    }
                    doing = "set()";
                    Tasklet.declare_self(doing);
                    // The helper function should not need to copy values
                    tuple_p.self.impl_dispatched_set_current_neighbours_routes_and_forwards();
                    // do we need to give schedule?
                    Tasklet.nap(0, 100);
                }
                catch (Error e)
                {
                    log_warn(@"KrnlRoute: set_current: running dispatcher reported an error: $(e.message)");
                }
                if (doing != null) Tasklet.declare_finished(doing);
            }
        }

        private void dispatched_set_current_neighbours_routes_and_forwards()
        {
            // Register (once) the spawnable function that is our dispatcher
            // and obtain a channel to drive it.
            Channel ch = TaskletDispatcher.get_channel_for_helper((Spawnable)helper_dispatched_set_current_neighbours_routes_and_forwards);
            struct_helper_KrnlRoute_dispatched arg = struct_helper_KrnlRoute_dispatched();
            arg.self = this;
            // send the struct
            ch.send_async(arg);
        }

        private void once_set_current_neighbours_routes_and_forwards()
        {
            try
            {
                // BEGIN this section is executed entirely without passing schedule.
                ArrayList<NIP> nips_for_me = new ArrayList<NIP>(PartialNIP.equal_func);
                foreach (AddressManager addr_man in addresses)
                    if (addr_man.do_i_participate_in_routing_tables() && addr_man.is_mature)
                        nips_for_me.add(addr_man.maproute.me);
                CondensedNeighbours all_neighbours = addresses.get_condensed_neighbours();
                CondensedMapRoute all_map_routes = addresses.get_condensed_map_route();
                // update CondensedPathChooser(s) based on CondensedNeighbour(s)
                foreach (PartialNIP k in all_map_routes.routes.keys)
                {
                    CondensedPathChooser v = all_map_routes.routes[k];
                    foreach (CloneRoute p in v.paths)
                        foreach (CondensedNeighbour condensed_neighbour in all_neighbours.neighbours_list)
                            if (p.gwnip.is_equal(condensed_neighbour.nip))
                            {
                                p.gwdev = condensed_neighbour.bestdev;
                                p.gwrem = condensed_neighbour.bestrem;
                                p.rem = p.rem_at_gw.add_segment(p.gwrem);
                                break;
                            }
                }
                CondensedIncomingNodeContainer all_incoming_nodes = addresses.get_condensed_incoming_nodes();
                RouteSetter kroute = RouteSetter.get_instance();
                Collection<Variant> initial_krnl_routes_table = kroute.get_known_destinations();
                Gee.Map<string, string> initial_krnl_neighbours_table = kroute.get_known_neighbours();
                // END section to be executed entirely without passing schedule.

                // Handle neighbours and routes:
                // Remove all routes (managed by ntkd) that are not in all_map_routes
                foreach (Variant ipcidr_v in initial_krnl_routes_table)
                {
                    string ipstr;
                    string bits;
                    ipcidr_v.get("(ss)", out ipstr, out bits);
                    PartialNIP prefix = ip_cidr_to_gnode(addresses.levels, addresses.gsize, ipstr, bits);
                    //CondensedPathChooser
                    if (! all_map_routes.routes.has_key(prefix))
                    {
                        // Remove the route destination = k (and sub-routes)
                        kroute.outgoing_routes(ipstr, bits, null);
                    }
                }
                // Remove all neighbours (managed by ntkd) that are not in all_neighbours
                foreach (string ipstr in initial_krnl_neighbours_table.keys)
                {
                    bool found = false;
                    foreach (CondensedNeighbour n in all_neighbours.neighbours_list)
                    {
                        if (ipstr == nip_to_str(n.levels, n.gsize, n.nip))
                        {
                            found = true;
                            break;
                        }
                    }
                    if (! found)
                    {
                        // Remove the neighbour ipstr
                        kroute.delete_neighbour(ipstr);
                    }
                }
                // Add all neighbours
                foreach (CondensedNeighbour condensed_neighbour in all_neighbours.neighbours_list)
                {
                    string ipstr = nip_to_str(condensed_neighbour.levels, condensed_neighbour.gsize,
                                              condensed_neighbour.nip);
                    if (initial_krnl_neighbours_table.has_key(ipstr))
                    {
                        if (initial_krnl_neighbours_table[ipstr] != condensed_neighbour.bestdev)
                        {
                            // Change neighbour
                            kroute.change_neighbour(ipstr, condensed_neighbour.bestdev, me_to_ipstr());
                        }
                    }
                    else
                    {
                        // Add neighbour
                        kroute.add_neighbour(ipstr, condensed_neighbour.bestdev, me_to_ipstr());
                    }
                }

                foreach (PartialNIP prefix in all_map_routes.routes.keys)
                {
                    CondensedPathChooser v = all_map_routes.routes[prefix];
                    string ipstr;
                    string bits;
                    gnode_to_ip_cidr(prefix, addresses.gsize, out ipstr, out bits);
                    Gee.List<CloneRoute> paths = v.all_paths_without_any(nips_for_me);
                    if (paths.is_empty)
                    {
                        // RULE: destination = prefix → unknown
                        kroute.outgoing_routes(ipstr, bits, null);
                    }
                    else
                    {
                        // RULE: destination = prefix → gateways = paths
                        CloneRoute[] dead_paths = {};
                        CloneRoute[] almost_dead_paths = {};
                        CloneRoute[] rtt_paths = {};
                        CloneRoute[] remaining_paths = {};
                        foreach (CloneRoute path in paths)
                        {
                            if (path.rem.get_type() == typeof(DeadREM)) dead_paths += path;
                            else if (path.rem.get_type() == typeof(AlmostDeadREM)) almost_dead_paths += path;
                            else if (path.rem.get_type() == typeof(RTT)) rtt_paths += path;
                            else remaining_paths += path;
                        }

                        // Our logical weights range from 1 to 1000. Each "o.s. network adapter" will adjust.
                        ArrayList<int> weights = new ArrayList<int>();
                        ArrayList<string> gateways = new ArrayList<string>();
                        ArrayList<string> devs = new ArrayList<string>();
                        // Rtt
                        if (rtt_paths.length > 0)
                        {
                            ArrayList<int> rtts = new ArrayList<int>();
                            int min_rtts = -1;
                            foreach (CloneRoute path in rtt_paths)
                            {
                                RTT path_rem = (RTT)path.rem;
                                rtts.add(path_rem.delay + 1);
                                if (min_rtts == -1 || min_rtts > path_rem.delay + 1)
                                    min_rtts = path_rem.delay + 1;
                            }
                            float scale = (float)(min_rtts * 999);
                            foreach (int rtt in rtts) weights.add((int)(scale / rtt) + 1);
                            foreach (CloneRoute path in rtt_paths) gateways.add(path.gwipstr);
                            foreach (CloneRoute path in rtt_paths) devs.add(path.gwdev);
                        }
                        // TODO Here we handle correctly only where rem is a RTT. Must support other metrics. Always 500 for now.
                        foreach (CloneRoute path in remaining_paths)
                        {
                            gateways.add(path.gwipstr);
                            devs.add(path.gwdev);
                            weights.add(500);
                        }
                        // AlmostDead. Always 1.
                        foreach (CloneRoute path in almost_dead_paths)
                        {
                            gateways.add(path.gwipstr);
                            devs.add(path.gwdev);
                            weights.add(1);
                        }

                        RouteSolutions route_solutions = new RouteSolutions(gateways, devs, weights, me_to_ipstr());
                        kroute.outgoing_routes(ipstr, bits, route_solutions);
                    }
                }

                // Handle incoming packet forwarding:
                ArrayList<string> macs_to_test = updated_node_macs;
                updated_node_macs = new ArrayList<string>();
                foreach (string mac in macs_to_test) check_dead_incoming_routes(mac, all_incoming_nodes);

                // We need to forward packets from all known macs through paths that don't contain any of these nips.
                foreach (PartialNIP prefix in all_map_routes.routes.keys)
                {
                    CondensedPathChooser v = all_map_routes.routes[prefix];
                    string ipstr;
                    string bits;
                    gnode_to_ip_cidr(prefix, addresses.gsize, out ipstr, out bits);
                    Gee.List<CloneRoute> def_paths = v.all_paths();
                    if (! def_paths.is_empty)
                    {
                        foreach (string mac in all_incoming_nodes.get_all_macs())
                        {
                            ArrayList<NIP> nips_for_mac = new ArrayList<NIP>(PartialNIP.equal_func);
                            foreach (CondensedIncomingNode incoming in all_incoming_nodes.get_nodes_with_mac(mac))
                                nips_for_mac.add(incoming.nip);
                            ArrayList<NIP> nips_for_mac_and_me = new ArrayList<NIP>(PartialNIP.equal_func);
                            nips_for_mac_and_me.add_all(nips_for_mac);
                            nips_for_mac_and_me.add_all(nips_for_me);
                            Gee.List<CloneRoute> paths = v.all_paths_without_any(nips_for_mac_and_me);
                            if (! paths.is_empty)
                            {
                                // RULE: destination = prefix, prev = mac → gateways = paths
                                CloneRoute[] dead_paths = {};
                                CloneRoute[] almost_dead_paths = {};
                                CloneRoute[] rtt_paths = {};
                                CloneRoute[] remaining_paths = {};
                                foreach (CloneRoute path in paths)
                                {
                                    if (path.rem.get_type() == typeof(DeadREM)) dead_paths += path;
                                    else if (path.rem.get_type() == typeof(AlmostDeadREM)) almost_dead_paths += path;
                                    else if (path.rem.get_type() == typeof(RTT)) rtt_paths += path;
                                    else remaining_paths += path;
                                }

                                // Our logical weights range from 1 to 1000. Each "o.s. network adapter" will adjust.
                                ArrayList<int> weights = new ArrayList<int>();
                                ArrayList<string> gateways = new ArrayList<string>();
                                ArrayList<string> devs = new ArrayList<string>();
                                // Rtt
                                if (rtt_paths.length > 0)
                                {
                                    ArrayList<int> rtts = new ArrayList<int>();
                                    int min_rtts = -1;
                                    foreach (CloneRoute path in rtt_paths)
                                    {
                                        RTT path_rem = (RTT)path.rem;
                                        rtts.add(path_rem.delay + 1);
                                        if (min_rtts == -1 || min_rtts > path_rem.delay + 1)
                                            min_rtts = path_rem.delay + 1;
                                    }
                                    float scale = (float)(min_rtts * 999);
                                    foreach (int rtt in rtts) weights.add((int)(scale / rtt) + 1);
                                    foreach (CloneRoute path in rtt_paths) gateways.add(path.gwipstr);
                                    foreach (CloneRoute path in rtt_paths) devs.add(path.gwdev);
                                }
                                // TODO Here we handle correctly only where rem is a RTT. Must support other metrics. Always 500 for now.
                                foreach (CloneRoute path in remaining_paths)
                                {
                                    gateways.add(path.gwipstr);
                                    devs.add(path.gwdev);
                                    weights.add(500);
                                }
                                // AlmostDead. Always 1.
                                foreach (CloneRoute path in almost_dead_paths)
                                {
                                    gateways.add(path.gwipstr);
                                    devs.add(path.gwdev);
                                    weights.add(1);
                                }
                                
                                RouteSolutions route_solutions = new RouteSolutions(gateways, devs, weights, me_to_ipstr());
                                kroute.forwarding_routes(ipstr, bits, mac, route_solutions);
                            }
                            else
                            {
                                // RULE: source incoming, destination = prefix → host/network UNREACHABLE.
                                kroute.forwarding_routes(ipstr, bits, mac, null);
                            }
                        }
                    }
                    else
                    {
                        log_error("An item in get_condensed_map_route() is with no (non-dead) paths:");
                        log_error(@"prefix = $(prefix)");
                        foreach (CloneRoute r in v.paths)
                            log_error(@"a path = $(r)");
                    }
                }
            }
            catch (Error e)
            {
                log_warn(@"KrnlRoute: set_current: running once reported an error: $(e.message)");
            }
        }

        private void check_dead_incoming_routes(string mac, CondensedIncomingNodeContainer all_incoming_nodes) throws Error
        {
            // if we got an event INCOMING_NODE_UPDATED _and_ that MAC is missing
            if (! all_incoming_nodes.get_all_macs().contains(mac))
            {
                RouteSetter.get_instance().forward_no_more_from(mac);
            }
        }
    }
}

