/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2011 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gee;
using zcd;
using Tasklets;

namespace Netsukuku
{
    struct struct_helper_BorderNodesManager_run
    {
        public BorderNodesManager self;
        public int level_of_gnode;
    }

    struct struct_helper_BorderNodesManager_tunnel_created
    {
        public BorderNodesManager self;
        public AuxiliaryAddressManager sec_addr_man;
        public string peer_addr;
        public string nic_name;
        public int level_of_gnode;
    }

    struct struct_helper_BorderNodesManager_wait_for_tunnel
    {
        public BorderNodesManager self;
        public int level_of_gnode;
    }

    struct struct_helper_BorderNodesManager_remove_aux_address_manager
    {
        public BorderNodesManager self;
        public AuxiliaryAddressManager addr_man;
    }

    /** This class handles things that a border-node has to do
      */
    public class BorderNodesManager : Object, IBorderNodesManager
    {
        public weak AddressManager address_manager {get; private set;}
        public weak MapRoute maproute {get; private set;}
        private weak AggregatedNeighbourManager aggregated_neighbour_manager;
        private weak Coord coordnode;
        private HashMap<int, Tasklet> running_handle;
        private HashMap<int, int> sent_messages;
        private HashMap<int, bool> waiting_tunnel;
        private HashMap<int, BnodeTunnelingAddressManager> waiting_tunnel_2;
        private HashMap<int, bool> running_tunnel;

        public BorderNodesManager(AddressManager address_manager, MapRoute maproute, AggregatedNeighbourManager aggregated_neighbour_manager, Coord coordnode)
        {
            this.address_manager = address_manager;
            this.maproute = maproute;
            this.aggregated_neighbour_manager = aggregated_neighbour_manager;
            this.coordnode = coordnode;
            running_handle = new HashMap<int, Tasklet>();
            sent_messages = new HashMap<int, int>();
            waiting_tunnel = new HashMap<int, bool>();
            waiting_tunnel_2 = new HashMap<int, BnodeTunnelingAddressManager>();
            running_tunnel = new HashMap<int, bool>();
        }

        public virtual void start_operations()
        {
            // Override in BnodeTunneling, do nothing
            if (running_handle.is_empty)
            {
                for (int level_of_gnode = 1; level_of_gnode < maproute.levels; level_of_gnode++)
                {
                    run(level_of_gnode);
                }
            }
        }

        public virtual void stop_operations()
        {
            // Override in BnodeTunneling, do nothing
            ms_wait(100);  // to make sure the micros have started
            foreach (Tasklet t in running_handle.values)
            {
                t.abort();
            }
            running_handle = new HashMap<int, Tasklet>();
        }

        public virtual bool is_border_node(int level_of_gnode)
        {
            // Override in BnodeTunneling, return False
            bool at_least_one_inside = false;
            bool at_least_one_outside = false;
            Gee.List<AggregatedNeighbour> x = aggregated_neighbour_manager.neighbour_list(/*in_my_network*/true);
            AggregatedNeighbour[] real_main_neighbours = {};
            foreach (AggregatedNeighbour n in x)
                if (!n.is_local())
                    real_main_neighbours += n;
            foreach (AggregatedNeighbour aggregated_neighbour in real_main_neighbours)
            {
                if (maproute.nip_cmp(aggregated_neighbour.nip.get_positions()) == level_of_gnode-1)
                    at_least_one_inside = true;
                if (maproute.nip_cmp(aggregated_neighbour.nip.get_positions()) >= level_of_gnode)
                    at_least_one_outside = true;
            }
            return at_least_one_inside && at_least_one_outside;
        }

        public Gee.List<InfoBorderNode> report_bordernodes_status()
        {
            ArrayList<InfoBorderNode> ret = new ArrayList<InfoBorderNode>();
            foreach (int level_of_gnode in running_handle.keys)
            {
                if (is_border_node(level_of_gnode))
                {
                    InfoBorderNode info = new InfoBorderNode();
                    info.gnode = maproute.me.get_gnode_at_level(level_of_gnode);
                    info.number_messages = sent_messages[level_of_gnode];
                    info.has_tunnel = has_tunnel(level_of_gnode);
                    info.is_willing = is_willing();
                    ret.add(info);
                }
            }
            return ret;
        }

        /** Tasklet to let a Coordinator know if I become a border node.
          */
        protected virtual void impl_run(int level_of_gnode) throws Error
        {
            Tasklet.declare_self("BorderNodesManager.run");
            // Override in BnodeTunneling, do nothing
            running_handle[level_of_gnode] = Tasklet.self();
            sent_messages[level_of_gnode] = 0;
            PartialNIP gnode = maproute.me.get_gnode_at_level(level_of_gnode);
            bool last_is_border = false;
            bool last_has_tunnel = false;
            bool last_is_willing = false;
            Tasklets.Timer expiration_last_message = new Tasklets.Timer(long.MAX);
            while (true)
            {
                ms_wait(10000);
                bool is_border = is_border_node(level_of_gnode);
                bool current_is_border = false;
                bool current_has_tunnel = false;
                bool current_is_willing = false;
                if (is_border)
                {
                    current_is_border = is_border;
                    current_has_tunnel = has_tunnel(level_of_gnode);
                    current_is_willing = is_willing();
                }
                if (expiration_last_message.is_expired() || 
                        last_is_border != current_is_border || 
                        last_has_tunnel != current_has_tunnel || 
                        last_is_willing != current_is_willing)
                {
                    Coord_hkey key = new Coord_hkey();
                    key.level_of_gnode = level_of_gnode;
                    key.nip = maproute.me;
                    RmtCoordPeer peer = coordnode.peer(null, key);
                    bool interested = peer.register_bnode(gnode, maproute.me, current_is_border, current_has_tunnel, current_is_willing);
                    sent_messages[level_of_gnode] += 1;
                    if (!interested)
                    {
                        last_is_border = false;
                        last_has_tunnel = false;
                        last_is_willing = false;
                        expiration_last_message = new Tasklets.Timer(long.MAX);
                        ms_wait(3600000);  // 1 hour
                    }
                    else
                    {
                        last_is_border = current_is_border;
                        last_has_tunnel = current_has_tunnel;
                        last_is_willing = current_is_willing;
                        if (is_border && has_tunnel(level_of_gnode))
                        {
                            expiration_last_message = new Tasklets.Timer(280000);  // almost 5 minutes
                        }
                        else
                        {
                            expiration_last_message = new Tasklets.Timer(long.MAX);
                        }
                    }
                }
            }
        }

        private void impl_caller_run(int level_of_gnode) throws Error
        {
            impl_run(level_of_gnode);
        }

        private static void * helper_run(void *v) throws Error
        {
            struct_helper_BorderNodesManager_run *tuple_p = (struct_helper_BorderNodesManager_run *)v;
            // The caller function has to add a reference to the ref-counted instances
            BorderNodesManager self_save = tuple_p->self;
            // The caller function has to copy the value of byvalue parameters
            int level_of_gnode_save = tuple_p->level_of_gnode;
            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            self_save.impl_caller_run(level_of_gnode_save);
            // void method, return null
            return null;
        }

        public void run(int level_of_gnode)
        {
            struct_helper_BorderNodesManager_run arg = struct_helper_BorderNodesManager_run();
            arg.self = this;
            arg.level_of_gnode = level_of_gnode;
            Tasklet.spawn((Spawnable)helper_run, &arg);
        }

        /** Called in a autonomous_address from a Coordinator.
          * Get distance from other nodes inside 'gnode'
          */
        public Gee.List<PairNipDistance> get_distances(PartialNIP gnode,
                        Gee.List<NIP> list_of_nips, CallerInfo? _rpc_caller = null) throws BorderNodesError
        {
            assert(_rpc_caller != null);
            assert(_rpc_caller.get_type().is_a(typeof(CallerInfo)));
            CallerInfo rpc_caller = (CallerInfo)_rpc_caller;

            NIP caller_nip = str_to_nip(maproute.levels, maproute.gsize, rpc_caller.caller_ip);
            int level_of_gnode = gnode.level_of_gnode();
            if (!caller_nip.belongs_to(gnode))
                throw new BorderNodesError.WRONG_GNODE("Caller node is not member of requested gnode");
            if (!maproute.me.belongs_to(gnode))
                throw new BorderNodesError.WRONG_GNODE("This node is not member of requested gnode");
            if (!is_border_node(level_of_gnode))
                throw new BorderNodesError.NOT_BORDER_NODE("This node is not a border node of requested gnode");
            if (!is_willing())
                throw new BorderNodesError.WILL_NOT_TUNNEL("This node is not willing to do tunnel");
            // Now, get distances:
            ArrayList<PairNipDistance> ret = new ArrayList<PairNipDistance>(PairNipDistance.equal_func);
            foreach (NIP nip in list_of_nips)
            {
                if (!nip.belongs_to(gnode))
                {
                    // do not evaluate, wrong gnode
                    continue;
                }
                HCoord hc = maproute.nip_to_lvlid(nip);
                int dist = maproute.get_distance(hc.lvl, hc.pos);
                ret.add(new PairNipDistance(nip, dist));
            }
            return ret;
        }

        /** Called in a autonomous_address from a Coordinator.
          * Obtain a new address to create a virtual link in order to increase connectivity inside 'gnode'
          */
        public NIP get_new_address(PartialNIP gnode, NIP? peer_nip=null, CallerInfo? _rpc_caller = null) throws BorderNodesError
        {
            assert(_rpc_caller != null);
            assert(_rpc_caller.get_type().is_a(typeof(CallerInfo)));
            CallerInfo rpc_caller = (CallerInfo)_rpc_caller;

            NIP caller_nip = str_to_nip(maproute.levels, maproute.gsize, rpc_caller.caller_ip);
            int level_of_gnode = gnode.level_of_gnode();
            if (!caller_nip.belongs_to(gnode))
                throw new BorderNodesError.WRONG_GNODE("Caller node is not member of requested gnode");
            if (!maproute.me.belongs_to(gnode))
                throw new BorderNodesError.WRONG_GNODE("This node is not member of requested gnode");
            if (!is_border_node(level_of_gnode))
                throw new BorderNodesError.NOT_BORDER_NODE("This node is not a border node of requested gnode");
            if (!is_willing())
                throw new BorderNodesError.WILL_NOT_TUNNEL("This node is not willing to do tunnel");
            // We are now preparing to make a tunnel
            waiting_tunnel[level_of_gnode] = true;
            // Make sure that within a certain time we got a tunnel from this address
            wait_for_tunnel(level_of_gnode);
            // We want a new address in this network, and we want to hook
            //  in a gnode outside my current gnode at level_of_gnode
            AddressManager this_addr_man = address_manager;
            Gee.List<AggregatedNeighbour> x = this_addr_man.aggregated_neighbour_manager.neighbour_list(/*in_my_network*/true);
            ArrayList<AggregatedNeighbour> neighbours = new ArrayList<AggregatedNeighbour>(AggregatedNeighbour.equal_func);
            foreach (AggregatedNeighbour neighbour in x)
                if (this_addr_man.maproute.nip_to_lvlid(neighbour.nip).lvl >= level_of_gnode)
                    neighbours.add(neighbour);
            HookReservation hook_reservation;
            try
            {
                hook_reservation = this_addr_man.hook.hook(neighbours);
            }
            catch (HookingError e)
            {
                throw new BorderNodesError.WILL_NOT_TUNNEL(@"Could not get a valid address: $(e.message)");
            }
            BnodeTunnelingAddressManager new_addr_man = new BnodeTunnelingAddressManager(
                    this_addr_man.levels, this_addr_man.gsize,
                    this_addr_man.keypair, hook_reservation,
                    this_addr_man.create_new_nic, this_addr_man, gnode, peer_nip);
            address_manager.auxiliary_addresses.add(new_addr_man);
            address_manager.auxiliary_address_manager_new(new_addr_man);
            foreach (NetworkInterfaceManager nic_man in this_addr_man.nics)
                if (nic_man.to_be_copied)
                    new_addr_man.add_nic_manager(nic_man);
            // Wait for the first ETP to come in from some neighbour.
            new_addr_man.start_operations(20000);  // the new tasklet will start in 20 secs
            // Did I succed in time?
            if (!waiting_tunnel.has_key(level_of_gnode))
            {
                remove_aux_address_manager(new_addr_man);
                throw new BorderNodesError.TIMEOUT("Timeout");
            }
            // Yes, I did.
            waiting_tunnel_2[level_of_gnode] = new_addr_man;
            // Return new address
            return new_addr_man.maproute.me;
        }

        /** Called in a autonomous_address from a Coordinator.
          * Pass the peer nip to the previously created auxiliary address manager.
          */
        public void assign_peer_nip(NIP nip_x_secondary, NIP nip_y_secondary)
        {
            foreach (AddressManager addr_man in address_manager.auxiliary_addresses)
            {
                if (addr_man.maproute.me.is_equal(nip_x_secondary))
                {
                    ((BnodeTunnelingAddressManager)addr_man).peer_nip = nip_y_secondary;
                    ((BnodeTunnelingBorderNodesManager)addr_man.border_nodes_manager).make_tunnel();
                }
            }
        }

        private void impl_wait_for_tunnel(int level_of_gnode) throws Error
        {
            Tasklet.declare_self("BorderNodesManager.wait_for_tunnel");
            ms_wait(300000);
            if (waiting_tunnel[level_of_gnode] && !waiting_tunnel_2.has_key(level_of_gnode))
            {
                // not yet obtained a secondary address.
                // we do nothing, just remove the value from the dict,
                // and the tasklet in get_new_address will know that the
                // new address has to be immediately removed.
            }
            else if (!waiting_tunnel[level_of_gnode] && !waiting_tunnel_2.has_key(level_of_gnode))
            {
                // obtained a secondary address AND the tunnel
                // we do nothing, just remove the value from the dict.
            }
            else
            {
                // obtained a secondary address, but not yet the tunnel
                BnodeTunnelingAddressManager sec_addr_man = waiting_tunnel_2[level_of_gnode];
                // failed tunneling. try to remove address_manager
                remove_aux_address_manager(sec_addr_man);
            }
            waiting_tunnel.unset(level_of_gnode);
            if (waiting_tunnel_2.has_key(level_of_gnode)) waiting_tunnel_2.unset(level_of_gnode);
        }

        private static void * helper_wait_for_tunnel(void *v) throws Error
        {
            struct_helper_BorderNodesManager_wait_for_tunnel *tuple_p = (struct_helper_BorderNodesManager_wait_for_tunnel *)v;
            // The caller function has to add a reference to the ref-counted instances
            BorderNodesManager self_save = tuple_p->self;
            // The caller function has to copy the value of byvalue parameters
            int level_of_gnode_save = tuple_p->level_of_gnode;
            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            self_save.impl_wait_for_tunnel(level_of_gnode_save);
            // void method, return null
            return null;
        }

        public void wait_for_tunnel(int level_of_gnode)
        {
            struct_helper_BorderNodesManager_wait_for_tunnel arg = struct_helper_BorderNodesManager_wait_for_tunnel();
            arg.self = this;
            arg.level_of_gnode = level_of_gnode;
            Tasklet.spawn((Spawnable)helper_wait_for_tunnel, &arg);
        }

        private void impl_remove_aux_address_manager(AuxiliaryAddressManager addr_man) throws Error
        {
            Tasklet.declare_self("BorderNodesManager.remove_aux_address_manager");
            address_manager.auxiliary_addresses.remove(addr_man);
            address_manager.auxiliary_address_manager_deleted(addr_man);
        }

        private static void * helper_remove_aux_address_manager(void *v) throws Error
        {
            struct_helper_BorderNodesManager_remove_aux_address_manager *tuple_p = (struct_helper_BorderNodesManager_remove_aux_address_manager *)v;
            // The caller function has to add a reference to the ref-counted instances
            BorderNodesManager self_save = tuple_p->self;
            AuxiliaryAddressManager addr_man_save = tuple_p->addr_man;
            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            self_save.impl_remove_aux_address_manager(addr_man_save);
            // void method, return null
            return null;
        }

        public void remove_aux_address_manager(AuxiliaryAddressManager addr_man)
        {
            struct_helper_BorderNodesManager_remove_aux_address_manager arg = struct_helper_BorderNodesManager_remove_aux_address_manager();
            arg.self = this;
            arg.addr_man = addr_man;
            Tasklet.spawn((Spawnable)helper_remove_aux_address_manager, &arg);
        }

        private void impl_tunnel_created(AuxiliaryAddressManager sec_addr_man, string peer_addr, string nic_name, int level_of_gnode) throws Error
        {
            Tasklet.declare_self("BorderNodesManager.tunnel_created");
            // Check that we did it in time
            if (!waiting_tunnel.has_key(level_of_gnode))
            {
                // too late:
                try
                {
                    ((BnodeTunnelingBorderNodesManager)sec_addr_man.border_nodes_manager).remove_tunnel(peer_addr, nic_name);
                }
                catch (Error e)
                {
                    log_warn(@"BorderNodesManager.tunnel_created: exception while trying to remove: $(e.domain): $(e.code): $(e.message)");
                }
                remove_aux_address_manager(sec_addr_man);
                return;
            }
            // Confirm that we got a tunnel from this address manager
            waiting_tunnel[level_of_gnode] = false;
            try
            {
                running_tunnel[level_of_gnode] = true;
                // Add new nic_name to this address
                AddressManager this_addr_man = address_manager;
                NIC nic_class = this_addr_man.create_new_nic(nic_name);
                TunneledNetworkInterfaceManager nic_man = new TunneledNetworkInterfaceManager(
                        nic_name, nic_class,
                        Addresses.udp_unicast_callback,
                        Addresses.udp_broadcast_callback);
                this_addr_man.add_nic_manager(nic_man);
                // check that I see a neighbour through nic_man.
                int unused = 0;
                while (unused < 100)
                {
                    ms_wait(1000);
                    var nm = this_addr_man.neighbour_managers[nic_name];
                    if (! nm.neighbour_list().is_empty) unused = 0;
                    else unused++;
                }
                // remove
                this_addr_man.remove_nic_manager(nic_man);
                try
                {
                    ((BnodeTunnelingBorderNodesManager)sec_addr_man.border_nodes_manager).remove_tunnel(peer_addr, nic_name);
                }
                catch (Error e)
                {
                    log_warn(@"BorderNodesManager.tunnel_created: exception while trying to remove: $(e.domain): $(e.code): $(e.message)");
                }
                remove_aux_address_manager(sec_addr_man);
            }
            finally
            {
                running_tunnel.unset(level_of_gnode);
            }
        }

        private static void * helper_tunnel_created(void *v) throws Error
        {
            struct_helper_BorderNodesManager_tunnel_created *tuple_p = (struct_helper_BorderNodesManager_tunnel_created *)v;
            // The caller function has to add a reference to the ref-counted instances
            BorderNodesManager self_save = tuple_p->self;
            AuxiliaryAddressManager sec_addr_man_save = tuple_p->sec_addr_man;
            // The caller function has to copy the value of byvalue parameters
            string peer_addr_save = tuple_p->peer_addr;
            string nic_name_save = tuple_p->nic_name;
            int level_of_gnode_save = tuple_p->level_of_gnode;
            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            self_save.impl_tunnel_created(sec_addr_man_save, peer_addr_save, nic_name_save, level_of_gnode_save);
            // void method, return null
            return null;
        }

        /** Called in a autonomous_address from a local auxiliary address to confirm that we got a tunnel
          */
        public void tunnel_created(AuxiliaryAddressManager sec_addr_man, string peer_addr, string nic_name, int level_of_gnode)
        {
            struct_helper_BorderNodesManager_tunnel_created arg = struct_helper_BorderNodesManager_tunnel_created();
            arg.self = this;
            arg.sec_addr_man = sec_addr_man;
            arg.peer_addr = peer_addr;
            arg.nic_name = nic_name;
            arg.level_of_gnode = level_of_gnode;
            Tasklet.spawn((Spawnable)helper_tunnel_created, &arg);
        }

        /** Called in a autonomous_address to see if it has a auxiliary address doing tunnel
          *  (or preparing to do) for this gnode
          */
        public bool has_tunnel(int level_of_gnode)
        {
            return running_tunnel.has_key(level_of_gnode) || waiting_tunnel.has_key(level_of_gnode);
        }

        /** Called in a autonomous_address to see if it has a auxiliary address doing tunnel
          *  (or preparing to do) at any level
          */
        public bool has_any_tunnel()
        {
            for (int level_of_gnode = 1; level_of_gnode < maproute.levels; level_of_gnode++)
            {
                if (this.has_tunnel(level_of_gnode)) return true;
            }
            return false;
        }

        /** Called in a autonomous_address to see if it is willing to do tunnel
          */
        public bool is_willing()
        {
            // If I have already a running tunnel, I refuse.
            if (this.has_any_tunnel()) return false;
            // If settings specify so, I refuse.
            if (Settings.DO_NOT_TUNNEL) return false;
            // TODO Detect the computational power of this node and the bandwidth available
            //  in the best nic. Use that to decide whether it is reasonable to provide a tunnel.
            //  At the moment, we pretend it is ok.
            return true;
        }

    }
}

