/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2014 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gee;
using Tasklets;
using zcd;
using Netsukuku.InetUtils;
using Netsukuku;

namespace Andns
{
    public class AndnsServer : Object
    {
        public static void start_operations()
        {
            andna_resolve = new AndnaResolve();
            tcpserver = new TCPServer(tcp_callback, 53000);
            tcpserver.listen();
        }

        public static void stop_operations()
        {
            tcpserver.stop();
        }

        private static void tcp_callback(CallerInfo caller,
                                              TCPRequest tcprequest,
                                              out RPCDispatcher? rpcdispatcher,
                                              out uchar[] data,
                                              out uchar[] response)
        {
            rpcdispatcher = new AndnaResolveDispatcher(andna_resolve);
            data = tcprequest.data.serialize();
            response = null;
        }

        private static TCPServer tcpserver;
        private static AndnaResolve andna_resolve;
    }

    public class AndnaResolve : Object, IAndnaResolveRootDispatcher
    {
        private HashMap<int, AndnaQueryHandler> handlers;

        public AndnaResolve()
        {
            handlers = new HashMap<int, AndnaQueryHandler>();
        }

        public IAndnaQueryHandler get_query_handler(int request_id)
        {
            IAndnaQueryHandler ret;
            if (! handlers.has_key(request_id))
            {
                AndnaQueryHandler handler = new AndnaQueryHandler(request_id);
                handlers[request_id] = handler;
                handler.finished.connect(() => {
                    handlers.unset(request_id);
                });
                ret = handler;
            }
            else
            {
                // Already serving this request. Ignore.
                ret = new RPCErrorThrower(@"AndnaResolve: Already serving request $(request_id). Ignore.");
            }
            return ret;
        }
    }

    /** This class is used to simulate a RPCError thrown by a remote method that should
      *  normally return another remote object.
      * When a client calls a remote method of this kind, the client immediately obtain
      *  a stub that is used to call another remote method. Only when it is sent to the
      *  remote site we can see that the first method would have thrown an error.
      */
    public class RPCErrorThrower : Object, IAndnaQueryHandler
    {
        public string message {get; private set;}
        public RPCErrorThrower(string message)
        {
            this.message = message;
        }

        public AndnaResponse resolve(AndnaQuery query) throws RPCError
        {
            throw new RPCError.GENERIC(message);
        }
    }

    public class AndnaQueryHandler : Object, IAndnaQueryHandler
    {
        public int request_id {get; private set;}
        public signal void finished();

        public AndnaQueryHandler(int request_id)
        {
            this.request_id = request_id;
        }

        public AndnaResponse resolve(AndnaQuery query)
        {
            try
            {
                AndnaResponse ret = null;

                if (query.query_type == AndnaQueryType.NAME_TO_IP &&
                    query.ip_realm == AndnaQueryRealm.NTK)
                {
                    AddressManager addrman = Addresses.get_addresses_instance().primary_address;
                    AndnaGetServersResponse resp = addrman.andna.ask_get_servers(
                            query.hashed_domain,
                            query.service);
                    ArrayList<AndnaResponseAnswer> answers =
                            new ArrayList<AndnaResponseAnswer>(AndnaResponseAnswer.equal_func);
                    AndnaResponseCode rcode = AndnaResponseCode.NO_ERROR;
                    if (resp.response.is_not_found)  rcode = AndnaResponseCode.NO_DOMAIN;
                    else
                    {
                        foreach (AndnaServer s in resp.response.servers)
                        {
                            if (s.registrar_nip != null)
                            {
                                if (query.ip_version == AndnaQueryIPVersion.IPV4)
                                {
                                    MapRoute mr = addrman.maproute;
                                    uint8[] ipv4_ip =
                                            nip_to_octets(mr.levels,
                                                          mr.gsize,
                                                          s.registrar_nip);
                                    AndnaResponseAnswer ans =
                                            new AndnaResponseAnswer.name_to_ip
                                            (s.weight,
                                             s.priority,
                                             ipv4_ip,
                                             (uint16)s.port_number);
                                    answers.add(ans);
                                }
                                else
                                {
                                    // TODO IPv6
                                }
                            }
                            else
                            {
                                AndnaResponseAnswer ans =
                                        new AndnaResponseAnswer.name_to_alias
                                        (s.weight,
                                         s.priority,
                                         s.alias_name,
                                         (uint16)s.port_number);
                                answers.add(ans);
                            }
                        }
                    }
                    ret = new AndnaResponse(
                            rcode,
                            answers);
                }
                else if (query.query_type == AndnaQueryType.IP_TO_NAME &&
                    query.ip_realm == AndnaQueryRealm.NTK)
                {
                    if (query.ip_version == AndnaQueryIPVersion.IPV4)
                    {
                        string dest_addr = InetUtils.address_v4_bytes_to_str(query.ipv4_ip);
                        AddressManager addrman = Addresses.get_addresses_instance().primary_address;
                        MapRoute mr = addrman.maproute;
                        NIP nip = str_to_nip(mr.levels, mr.gsize, dest_addr);
                        IAddressManagerRootDispatcher client = 
                                new AddressManagerTCPClient(dest_addr);

                        Gee.List<string> resp = new ArrayList<string>(); // empty default
                        // give a timeout
                        Tasklet t = Tasklet.tasklet_callback(
                                () => {
                                    try {
                                        resp = client.andna
                                                .get_your_hostnames(nip);
                                    }
                                    catch (Error e) {
                                        // error trying to get hostnames
                                    }
                                });
                        bool finished = Tasklet.nap_until_condition(
                                () => {
                                    return t.is_dead();
                                }, 4000);
                        if (! finished)
                        {
                            t.abort();
                        }
                        
                        ArrayList<AndnaResponseAnswer> answers =
                                new ArrayList<AndnaResponseAnswer>(AndnaResponseAnswer.equal_func);
                        AndnaResponseCode rcode = AndnaResponseCode.NO_ERROR;
                        if (resp.is_empty) rcode = AndnaResponseCode.NO_DOMAIN;
                        foreach (string hn in resp)
                        {
                            AndnaResponseAnswer ans =
                                    new AndnaResponseAnswer.ip_to_name(hn);
                            answers.add(ans);
                        }
                        ret = new AndnaResponse(
                                rcode,
                                answers);
                    }
                    else
                    {
                        // TODO IPv6
                    }
                }
                else
                {
                    ret = new AndnaResponse(
                            AndnaResponseCode.NOT_IMPL,
                            new ArrayList<AndnaResponseAnswer>(AndnaResponseAnswer.equal_func));

                }
                finished();
                return ret;
            }
            catch (Error e)
            {
                finished();
                return new AndnaResponse(
                        AndnaResponseCode.SERV_FAIL,
                        new ArrayList<AndnaResponseAnswer>(AndnaResponseAnswer.equal_func));
            }
        }
    }
}
