/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2011 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gee;
using zcd;
using Tasklets;

namespace Netsukuku
{
	public errordomain AggregatedNeighbourError {
		DUPLICATE_INSERT,
		MISSING_REMOVE
	}

    struct struct_helper_AggregatedNeighbour_tcp_client_collect
    {
        public AggregatedNeighbour self;
        public AddressManagerTCPClient tcp_client;
    }

    /** This class simply represent a neighbour.
      * Each instance of this class represents a neighbour as seen by any number of nics
      *  of this node associated to a certain address.
      */
    public class AggregatedNeighbour : Object
    {
        /** nip: neighbour's nip;
          * nodeid: id of the node
          *       nip + nodeid = unique key.
          * levels, gsize, netid: attributes of the network
          * interfaces: a dict<nic_name,rem>
          * macs: a sequence of string: the MAC of the nics of the neighbour that I can see through my nics "interfaces";
          *     there could be more than one.
          * rpcdispatcher is valorized for neighbours that represent one of my addresses. it is used to make an instance of PseudoNeighbourClient.
          */
        public int levels;
        public int gsize;
        public NIP nip;
        public int nodeid;
        public NetworkID netid;
        public bool is_primary;
        public bool is_auxiliary;
        public HashMap<string, REM>? interfaces;
        public Gee.List<string> macs;
        private RPCDispatcher? rpcdispatcher;
        public int mod_seq_num;
        private AddressManagerTCPClient? _tcp_client;

        private AggregatedNeighbour(int levels, int gsize, NIP nip,
                                   int nodeid, NetworkID netid, bool is_primary, bool is_auxiliary,
                                   HashMap<string, REM>? interfaces,
                                   Gee.List<string>? macs,
                                   RPCDispatcher? rpcdispatcher)
        {
            this.levels = levels;
            this.gsize = gsize;
            this.nip = nip;
            this.nodeid = nodeid;
            this.netid = netid;
            this.is_primary = is_primary;
            this.is_auxiliary = is_auxiliary;

            if (interfaces != null)
            {
                this.interfaces = new HashMap<string, REM>();
                foreach (string k in interfaces.keys)
                    this.interfaces[k] = interfaces[k];
            }
            else this.interfaces = null;
            this.macs = new ArrayList<string>();
            if (macs != null)
                foreach (string s in macs)
                    this.macs.add(s);

            this.rpcdispatcher = rpcdispatcher;
            mod_seq_num = 0;
            _tcp_client = null;
        }

        public AggregatedNeighbour.local(int levels, int gsize, NIP nip,
                                   int nodeid, NetworkID netid, bool is_primary, bool is_auxiliary,
                                   RPCDispatcher rpcdispatcher)
        {
            this(levels, gsize, nip, nodeid, netid, is_primary, is_auxiliary, null, null, rpcdispatcher);
        }

        public AggregatedNeighbour.real(int levels, int gsize, NIP nip,
                                   int nodeid, NetworkID netid, bool is_primary, bool is_auxiliary,
                                   HashMap<string, REM> interfaces,
                                   Gee.List<string> macs)
        {
            this(levels, gsize, nip, nodeid, netid, is_primary, is_auxiliary, interfaces, macs, null);
        }

        public bool is_local()
        {
            return rpcdispatcher != null;
        }

        public void get_best_interface(out string? nic_name, out REM rem)
        {
            nic_name = null;
            if (is_local())
            {
                // it is locally reacheable
                rem = new NullREM();
                return;
            }
            if (interfaces == null || interfaces.size == 0)
            {
                // it is unreacheable
                rem = new DeadREM();
                return;
            }
            foreach (string local_nic_name in interfaces.keys)
            {
                REM local_rem = interfaces[local_nic_name];
                if (nic_name == null || interfaces[nic_name].compare_to(local_rem) < 0)
                {
                    nic_name = local_nic_name;
                }
            }
            rem = interfaces[nic_name];
            return;
        }

        private string _dev;
        private REM _rem;
        public string dev {
            get {
                REM temp_rem;
                get_best_interface(out _dev, out temp_rem);
                return _dev;
            }
        }
        public REM rem {
            get {
                string temp_dev;
                get_best_interface(out temp_dev, out _rem);
                return _rem;
            }
        }

        public void activate_tcp(bool activate=true, bool force=false)
        {
            if (activate)
            {
                if (_tcp_client == null || force)
                {
                    string ipstr = nip_to_str(levels, gsize, nip);
                    _tcp_client = new AddressManagerTCPClient(ipstr);
                }
            }
            else
                _tcp_client = null;
        }

        public AddressManagerTCPClient tcp_client {
            get {
                if (_tcp_client != null && _tcp_client.calling)
                {
                    string ipstr = nip_to_str(levels, gsize, nip);
                    tcp_client_collect(_tcp_client);
                    _tcp_client = new AddressManagerTCPClient(ipstr);
                }
                return _tcp_client;
            }
        }

        private void impl_tcp_client_collect(AddressManagerTCPClient tcp_client) throws Error
        {
            Tasklet.declare_self("AggregatedNeighbour.tcp_client_collect");
            try
            {
                while (tcp_client.calling) Tasklet.nap(0, 1000);
                tcp_client.close();
            }
            catch (RPCError e) {}
        }

        /* Decoration of microfunc */
        private static void * helper_tcp_client_collect(void *v) throws Error
        {
            struct_helper_AggregatedNeighbour_tcp_client_collect *tuple_p = (struct_helper_AggregatedNeighbour_tcp_client_collect *)v;
            // The caller function has to add a reference to the ref-counted instances
            AggregatedNeighbour self_save = tuple_p->self;
            AddressManagerTCPClient tcp_client_save = tuple_p->tcp_client;
            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            self_save.impl_tcp_client_collect(tcp_client_save);
            // void method, return null
            return null;
        }

        public void tcp_client_collect(AddressManagerTCPClient tcp_client)
        {
            struct_helper_AggregatedNeighbour_tcp_client_collect arg = struct_helper_AggregatedNeighbour_tcp_client_collect();
            arg.self = this;
            arg.tcp_client = tcp_client;
            Tasklet.spawn((Spawnable)helper_tcp_client_collect, &arg);
        }

        public AddressManagerFakeRmt create_neighbour_client(bool response=true)
        {
            if (rpcdispatcher != null)
            {
                return new AddressManagerPseudoNeighbourClient(rpcdispatcher, response);
            }
            else
            {
                string[] devs = interfaces.keys.to_array();
                UnicastID ucid = new UnicastID();
                ucid.nip = nip;
                ucid.nodeid = nodeid;
                return new AddressManagerNeighbourClient(ucid, devs, null, response);
            }
        }
        private AddressManagerFakeRmt? _neighbour_client = null;
        public AddressManagerFakeRmt neighbour_client {
            get {
                if (_neighbour_client == null) _neighbour_client = create_neighbour_client();
                return _neighbour_client;
            }
        }

        public static bool equal_func(AggregatedNeighbour? a, AggregatedNeighbour? b)
        {
            if (a == b) return true;
            if (a == null || b == null) return false;
            if (a.nodeid != b.nodeid) return false;
            if (! PartialNIP.equal_func(a.nip, b.nip)) return false;
            return true;
        }

        public static uint hash_func(AggregatedNeighbour a)
        {
            return @"$(a.nodeid)".hash();
        }

        public string to_string()
        {
            string ipstr = nip_to_str(levels, gsize, nip);
            string ret = @"<AggregatedNeighbour($(ipstr), id $(nodeid) in $(netid)): ";
            if (is_local()) ret += "local>";
            else ret += @"$(rem) through $(dev)>";
            return ret;
        }
    }

    struct struct_helper_AggregatedNeighbourManager_serialized_store_XXX_neighbour
    {
        public AggregatedNeighbourManager self;
        public int function_code;
        // 1 : serialized_store_add_neighbour
        // 2 : serialized_store_delete_neighbour
        // 3 : serialized_store_changed_neighbour
        public AggregatedNeighbour val;
        public string key;
        public string dev;
        public REM? rem;
        public Gee.List<string>? macs;
        public NetworkID? netid;
        public bool? is_primary;
        public bool? is_auxiliary;
        public bool delete_dev;
    }

    public class AggregatedNeighbourManager : Object, IAggregatedNeighbourManager
    {
        protected AddressManager address_manager;
        private HashMap<string, AggregatedNeighbour> tnip_nodeid_table;
        private string myself;

        public AggregatedNeighbourManager(AddressManager address_manager)
        {
            this.address_manager = address_manager;
            // tnip_nodeid_table. This is a dict whose key is a string constructed on a nip and a nodeid.
            // The values of this dict are instances of AggregatedNeighbour.
            tnip_nodeid_table = new HashMap<string, AggregatedNeighbour>();

            myself = tnip_nodeid_key(address_manager.maproute.me, address_manager.get_my_id());
        }

        private string tnip_nodeid_key(NIP key_nip, int key_nodeid)
        {
            string ret = "";
            int[] nip = key_nip.get_positions();
            foreach (int i in nip) ret += "%d_".printf(i);
            ret += "%d".printf(key_nodeid);
            return ret;
        }
        private void key_tnip_nodeid(string key, out NIP key_nip, out int key_nodeid)
        {
            string[] nums = key.split("_");
            int[] pos = new int[nums.length - 1];
            for (int i = 0; i < pos.length; i++) pos[i] = int.parse(nums[i]);
            key_nodeid = int.parse(nums[nums.length - 1]);
            key_nip = new NIP(pos);
        }

        public signal void aggregated_neighbour_new(AggregatedNeighbour aggregated_neighbour);
        public signal void aggregated_neighbour_new_before(AggregatedNeighbour aggregated_neighbour);
        public signal void aggregated_neighbour_deleted(AggregatedNeighbour aggregated_neighbour);
        public signal void aggregated_neighbour_deleted_after(AggregatedNeighbour aggregated_neighbour);
        public signal void aggregated_neighbour_rem_chged(AggregatedNeighbour aggregated_neighbour, REM old_rem);
        public signal void aggregated_neighbour_rem_chged_before(AggregatedNeighbour aggregated_neighbour, REM old_rem);
        public signal void aggregated_neighbour_going_new(AggregatedNeighbour aggregated_neighbour);
        public signal void aggregated_neighbour_going_deleted(AggregatedNeighbour aggregated_neighbour);
        public signal void aggregated_neighbour_going_rem_chged(AggregatedNeighbour aggregated_neighbour);
        public signal void aggregated_neighbour_colliding_new(AggregatedNeighbour aggregated_neighbour);
        public signal void aggregated_neighbour_colliding_deleted(AggregatedNeighbour aggregated_neighbour);
        public signal void aggregated_neighbour_colliding_rem_chged(AggregatedNeighbour aggregated_neighbour, REM old_rem);
        public signal void aggregated_neighbour_colliding_going_new(AggregatedNeighbour aggregated_neighbour);
        public signal void aggregated_neighbour_colliding_going_deleted(AggregatedNeighbour aggregated_neighbour);
        public signal void aggregated_neighbour_colliding_going_rem_chged(AggregatedNeighbour aggregated_neighbour);

        public Gee.List<InfoNeighbour> report_neighbours()
        {
            ArrayList<InfoNeighbour> ret = new ArrayList<InfoNeighbour>();
            foreach (AggregatedNeighbour aggregated_neighbour in neighbour_list())
            {
                InfoNeighbour info = new InfoNeighbour(aggregated_neighbour.netid,
                                                       aggregated_neighbour.levels,
                                                       aggregated_neighbour.gsize,
                                                       aggregated_neighbour.nip);
                ret.add(info);
            }
            return ret;
        }

        /** Radar functions **/
        public void reply(int radar_id, NIP nip, int nodeid, NetworkID netid, CallerInfo? _rpc_caller = null) throws RPCError
        {
            assert(_rpc_caller != null);
            assert(_rpc_caller.get_type().is_a(typeof(CallerInfo)));
            CallerInfo rpc_caller = (CallerInfo)_rpc_caller;
            // We choose if we want to be a gateway for this nip,nodeid,netid. If so, we reply to its radar with a unicast
            //  message with no response. The message is replied to all Radar instances, just the originator will use it.
            if (address_manager.do_i_act_as_gateway_for(nip, nodeid, netid))
            {
                ArrayList<string> my_macs = new ArrayList<string>();
                foreach (NetworkInterfaceManager nic in address_manager.nics)
                {
                    if (nic.to_be_managed && nic.nic_name == rpc_caller.dev)
                    {
                        my_macs.add(nic.nic_class.mac);
                    }
                }
                // the nic could be unmanaged by a certain address
                if (! my_macs.is_empty)
                {
                    MapRoute maproute = address_manager.maproute;
                    string[] devs = {rpc_caller.dev};
                    UnicastID ucid = new UnicastID();
                    ucid.nip = nip;
                    ucid.nodeid = nodeid;
                    var neighbour_client = new AddressManagerNeighbourClient(ucid, devs, null, false);
                    neighbour_client.aggregated_neighbour_manager.time_register(radar_id,
                                maproute.levels, maproute.gsize, maproute.me,
                                address_manager.get_my_id(),
                                address_manager.get_main_netid(),
                                my_macs[0],
                                address_manager.is_primary,
                                address_manager.is_auxiliary);
                }
            }
        }

        public void time_register(int radar_id, int levels, int gsize, NIP nip, int nodeid,
                    NetworkID netid, string mac, bool is_primary, bool is_auxiliary)
        {
            foreach (NeighbourManager neighbour_manager in address_manager.neighbour_managers.values)
            {
                neighbour_manager.radar.time_register(radar_id, levels, gsize, nip, nodeid, netid, mac, is_primary, is_auxiliary);
            }
        }

        public void declare_dead(NIP nip, int nodeid)
        {
            foreach (NeighbourManager neighbour_manager in address_manager.neighbour_managers.values)
            {
                neighbour_manager.declare_dead(nip, nodeid);
            }
            store_delete_neighbour(tnip_nodeid_key(nip, nodeid));
        }

        /** Inserts this neighbour in our data structures.
          */
        private void memorize(AggregatedNeighbour val) throws AggregatedNeighbourError
        {
            // key = (tuple(val.nip), val.nodeid)
            // key should not be already in tnip_nodeid_table.

            // ATTENTION: this method MUST NOT pass schedule until the end.

            string skey = tnip_nodeid_key(val.nip, val.nodeid);
            if (tnip_nodeid_table.has_key(skey))
                throw new AggregatedNeighbourError.DUPLICATE_INSERT("Key was already present");
            tnip_nodeid_table[skey] = val;
        }

        /** Removes this neighbour in our data structures.
          */
        private void unmemorize(string skey) throws AggregatedNeighbourError
        {
            // key: pair tuple(nip), nodeid
            // key should be in tnip_nodeid_table.

            // ATTENTION: this method MUST NOT pass schedule until the end.

            if (! tnip_nodeid_table.has_key(skey))
                throw new AggregatedNeighbourError.MISSING_REMOVE("Key was not present");
            tnip_nodeid_table.unset(skey);
        }

        /** return a AggregatedNeighbour object from its nip and nodeid
          */
        public AggregatedNeighbour? key_to_neighbour(NIP nip, int nodeid)
        {
            string skey = tnip_nodeid_key(nip, nodeid);
            if (! tnip_nodeid_table.has_key(skey)) return null;
            else return tnip_nodeid_table[skey];
        }

        /** Returns the list of neighbours.
          * If with_this_netid is not None, then returns only nodes
          *   that are in the network with this netid.
          * Else, if in_my_network is not None, then returns only nodes
          *   that are in my network (compared with all the GID of last level).
          * Else all the neighbours are returned.
          */
        public Gee.List<AggregatedNeighbour> neighbour_list(bool? in_my_network=null, NetworkID? with_this_netid=null)
        {
            ArrayList<AggregatedNeighbour> nlist = new ArrayList<AggregatedNeighbour>(AggregatedNeighbour.equal_func);
            foreach (string key in tnip_nodeid_table.keys)
            {
                AggregatedNeighbour val = tnip_nodeid_table[key];
                bool wanted = true;
                if (with_this_netid != null)
                    wanted = val.netid.is_same_network(with_this_netid);
                else if (in_my_network != null)
                    wanted = address_manager.is_in_my_network(val.netid);
                if (wanted)
                    nlist.add(val);
            }
            return nlist;
        }

        private void serialized_store_add_neighbour(string key, AggregatedNeighbour val) throws Error
        {
            // Check that it did NOT exist; otherwise exit.
            if (tnip_nodeid_table.has_key(key)) return;

            // First, we emit a signal to notify that we are
            //  gonna make a change. The handler shouldn't be a microfunc.
            going_add_neighbour(val);

            // We update the data structures.
            memorize(val);

            // Then we emit a signal to notify the change.
            add_neighbour(key);
        }

        private void serialized_store_delete_neighbour(string key) throws Error
        {
            // Check that it did exist and is real; otherwise exit.
            if (! tnip_nodeid_table.has_key(key)) return;

            // First, we emit a signal to notify that we are
            //  gonna make a change. The handler shouldn't be a microfunc.
            AggregatedNeighbour old_val = tnip_nodeid_table[key];
            going_delete_neighbour(old_val);

            // We update the data structures.
            unmemorize(key);

            // Then we emit a signal to notify the change.
            delete_neighbour(key, old_val);
        }

        private void serialized_store_changed_neighbour(string key, string dev,
                                                        REM? rem, Gee.List<string>? macs, NetworkID? netid,
                                                        bool? is_primary, bool? is_auxiliary, bool delete_dev) throws Error
        {
            // Check that it did exist and is real; otherwise exit.
            if (! tnip_nodeid_table.has_key(key)) return;

            // A particular NIC has seen a change in this neighbour. We signal
            //  that this AggregatedNeighbour is going to change. Then we process
            //  the change and signal the new current bestdev (which could be the same)

            // First, we emit a signal to notify that we are
            //  gonna make a change. The handler shouldn't be a microfunc.
            AggregatedNeighbour old_val = tnip_nodeid_table[key];
            going_rem_change_neighbour(old_val);

            // We update the data structures.
            REM old_rem = old_val.rem;
            if (! old_val.is_local())
            {
                if (delete_dev)
                {
                    if (old_val.interfaces.has_key(dev))
                    {
                        old_val.interfaces.unset(dev);
                    }
                }
                else
                {
                    old_val.interfaces[dev] = rem;
                    foreach (string mac in macs)
                    {
                        if (! old_val.macs.contains(mac))
                        {
                            old_val.macs.add(mac);
                        }
                    }
                }
            }
            if (netid != null) old_val.netid = netid;
            if (is_primary != null) old_val.is_primary = is_primary;

            // Then we emit a signal to notify the change.
            if (old_rem.compare_to(old_val.rem) != 0)
                rem_change_neighbour(key, old_rem);
        }

        /* Decoration of several microfunc with one unique dispatcher */
        private static void * helper_serialized_store_XXX_neighbour(void *v) throws Error
        {
            Tasklet.declare_self("AggregatedNeighbourManager.serialized_store dispatcher");
            struct_channel *ch_cont_p = (struct_channel *)v;
            // The caller function has to add a reference to the ref-counted instances
            Channel ch = ch_cont_p->self;
            // schedule back to the spawner; this will probably invalidate *v and *ch_cont_p.
            Tasklet.schedule_back();
            // The actual dispatcher
            while (true)
            {
                string? doing = null;
                struct_helper_AggregatedNeighbourManager_serialized_store_XXX_neighbour tuple_p;
                {
                    Value vv = ch.recv();
                    tuple_p = *((struct_helper_AggregatedNeighbourManager_serialized_store_XXX_neighbour *)(vv.get_boxed()));
                }
                try
                {
                    // The helper function should not need to copy values
                    if (tuple_p.function_code == 1)
                    {
                        doing = @"add_neighbour($(tuple_p.key))";
                        Tasklet.declare_self(doing);
                        tuple_p.self.serialized_store_add_neighbour(
                                tuple_p.key, tuple_p.val);
                    }
                    if (tuple_p.function_code == 2)
                    {
                        doing = @"delete_neighbour($(tuple_p.key))";
                        Tasklet.declare_self(doing);
                        tuple_p.self.serialized_store_delete_neighbour(
                                tuple_p.key);
                    }
                    if (tuple_p.function_code == 3)
                    {
                        doing = @"changed_neighbour($(tuple_p.key))";
                        Tasklet.declare_self(doing);
                        tuple_p.self.serialized_store_changed_neighbour(
                                tuple_p.key, tuple_p.dev,
                                tuple_p.rem, tuple_p.macs, tuple_p.netid,
                                tuple_p.is_primary, tuple_p.is_auxiliary, tuple_p.delete_dev);
                    }
                }
                catch (Error e)
                {
                    string part1 = "serialized_store_XXX_neighbour";
                    if (tuple_p.function_code == 1) part1 = "serialized_store_add_neighbour";
                    if (tuple_p.function_code == 2) part1 = "serialized_store_delete_neighbour";
                    if (tuple_p.function_code == 3) part1 = "serialized_store_changed_neighbour";
                    log_warn(@"AggregatedNeighbourManager: $part1 reported an error: $(e.message)");
                }
                if (doing != null) Tasklet.declare_finished(doing);
            }
        }

        protected virtual void store_add_neighbour(string key, AggregatedNeighbour val)
        {
            // Register (once) the spawnable function that is our dispatcher
            // and obtain a channel to drive it.
            Channel ch = TaskletDispatcher.get_channel_for_helper((Spawnable)helper_serialized_store_XXX_neighbour);
            struct_helper_AggregatedNeighbourManager_serialized_store_XXX_neighbour arg = 
                        struct_helper_AggregatedNeighbourManager_serialized_store_XXX_neighbour();
            arg.self = this;
            arg.function_code = 1;
            arg.key = key;
            arg.val = val;
            // send the struct
            ch.send_async(arg);
        }

        private void store_delete_neighbour(string key)
        {
            // Register (once) the spawnable function that is our dispatcher
            // and obtain a channel to drive it.
            Channel ch = TaskletDispatcher.get_channel_for_helper((Spawnable)helper_serialized_store_XXX_neighbour);
            struct_helper_AggregatedNeighbourManager_serialized_store_XXX_neighbour arg = 
                        struct_helper_AggregatedNeighbourManager_serialized_store_XXX_neighbour();
            arg.self = this;
            arg.function_code = 2;
            arg.key = key;
            // send the struct
            ch.send_async(arg);
        }

        private void store_changed_neighbour(string key, string dev,
                                                        REM? rem, Gee.List<string>? macs, NetworkID? netid,
                                                        bool? is_primary, bool? is_auxiliary, bool delete_dev=false)
        {
            // Register (once) the spawnable function that is our dispatcher
            // and obtain a channel to drive it.
            Channel ch = TaskletDispatcher.get_channel_for_helper((Spawnable)helper_serialized_store_XXX_neighbour);
            struct_helper_AggregatedNeighbourManager_serialized_store_XXX_neighbour arg = 
                        struct_helper_AggregatedNeighbourManager_serialized_store_XXX_neighbour();
            arg.self = this;
            arg.function_code = 3;
            arg.key = key;
            arg.dev = dev;
            arg.rem = rem;
            arg.macs = macs;
            arg.netid = netid;
            arg.is_primary = is_primary;
            arg.is_auxiliary = is_auxiliary;
            arg.delete_dev = delete_dev;
            // send the struct
            ch.send_async(arg);
        }

        /** Take the list of known neighbours from all the NICs associated and build
          *  my internal data structures.
          */
        public void initialize()
        {
            HashMap<string, ArrayList<Neighbour>> map_list_neighbour = new HashMap<string, ArrayList<Neighbour>>();
            HashMap<string, AggregatedNeighbour> map_aggregated_neighbour = new HashMap<string, AggregatedNeighbour>();
            foreach (NeighbourManager neighbour_manager in address_manager.neighbour_managers.values)
            {
                foreach (Neighbour neighbour in neighbour_manager.neighbour_list())
                {
                    NIP key_nip = neighbour.nip;
                    int key_nodeid = neighbour.nodeid;
                    string key = tnip_nodeid_key(key_nip, key_nodeid);
                    // ignore myself, as signaled by GlueNetworkInterfaceManager
                    if (key != myself)
                    {
                        if (! map_list_neighbour.has_key(key))
                            map_list_neighbour[key] = new ArrayList<Neighbour>(Neighbour.equal_func);
                        map_list_neighbour[key].add(neighbour);
                    }
                }
            }
            foreach (string key in map_list_neighbour.keys)
            {
                ArrayList<Neighbour> nlist = map_list_neighbour[key];
                HashMap<string, REM> interfaces = new HashMap<string, REM>();
                ArrayList<string> macs = new ArrayList<string>();
                // this block of variables will surely be overwritten
                int levels = 0;
                int gsize = 0;
                NetworkID? netid = null;
                bool is_primary = false;
                bool is_auxiliary = false;
                NIP? nip = null;
                int nodeid = 0;
                // end block
                key_tnip_nodeid(key, out nip, out nodeid);
                AggregatedNeighbour? aggregated_neighbour = null;
                foreach (Neighbour neighbour in nlist)
                {
                    levels = neighbour.levels;
                    gsize = neighbour.gsize;
                    netid = neighbour.netid;
                    is_primary = neighbour.is_primary;
                    is_auxiliary = neighbour.is_auxiliary;
                    if (neighbour.is_local())
                    {
                        aggregated_neighbour = new AggregatedNeighbour.local(levels, gsize, nip,
                                   nodeid, netid, is_primary, is_auxiliary, neighbour.rpcdispatcher);
                        break;
                    }
                    interfaces[neighbour.dev] = neighbour.rem;
                    foreach (string mac in neighbour.get_macs())
                        if (! macs.contains(mac))
                            macs.add(mac);
                }
                if (aggregated_neighbour == null)
                {
                    aggregated_neighbour = new AggregatedNeighbour.real(levels, gsize, nip,
                                   nodeid, netid, is_primary, is_auxiliary, interfaces, macs);
                }
                map_aggregated_neighbour[key] = aggregated_neighbour;
            }

            // For each single AggregatedNeighbour, call store_add_neighbour.
            foreach (string key in map_aggregated_neighbour.keys)
                store_add_neighbour(key, map_aggregated_neighbour[key]);
        }
        
        /** A neighbour is going to be added in a nic.
          */
        public void nic_going_add_neighbour(Neighbour neighbour)
        {
            // This listener, which is not microfunc, is called and executed
            //  just before a NeighbourManager (associated to a NIC) adds
            //  a Neighbour.
            // Right now, we don't use it.
        }
        
        /** A neighbour is going to be rem_chged in a nic.
          */
        public void nic_going_rem_change_neighbour(Neighbour neighbour)
        {
            // This listener, which is not microfunc, is called and executed
            //  just before a NeighbourManager (associated to a NIC) changes
            //  a Neighbour's REM.
            // Right now, we don't use it.
        }
        
        /** A neighbour is going to be deleted in a nic.
          */
        public void nic_going_delete_neighbour(Neighbour neighbour)
        {
            // This listener, which is not microfunc, is called and executed
            //  just before a NeighbourManager (associated to a NIC) deletes
            //  a Neighbour.
            // Right now, we don't use it.
        }

        /** A neighbour has been added in a nic.
          */
        public void nic_add_neighbour(Neighbour neighbour)
        {
            // This listener, which could be a microfunc, is called
            //  right after a NeighbourManager (associated to a NIC) has
            //  added a Neighbour.
            NIP key_nip = neighbour.nip;
            int key_nodeid = neighbour.nodeid;
            string key = tnip_nodeid_key(key_nip, key_nodeid);
            // ignore myself, as signaled by GlueNetworkInterfaceManager
            if (key != myself)
            {
                // is it already known?
                if (tnip_nodeid_table.has_key(key))
                {
                    store_changed_neighbour(key, neighbour.dev, neighbour.rem, 
                            neighbour.get_macs(), neighbour.netid,
                            neighbour.is_primary, neighbour.is_auxiliary);
                }
                else
                {
                    AggregatedNeighbour val;
                    if (neighbour.is_local())
                    {
                        val = new AggregatedNeighbour.local(neighbour.levels,
                                            neighbour.gsize,
                                            key_nip, key_nodeid,
                                            neighbour.netid, neighbour.is_primary, neighbour.is_auxiliary,
                                            neighbour.rpcdispatcher);
                    }
                    else
                    {
                        HashMap<string, REM> interfaces = new HashMap<string, REM>();
                        interfaces[neighbour.dev] = neighbour.rem;
                        val = new AggregatedNeighbour.real(neighbour.levels,
                                            neighbour.gsize,
                                            key_nip, key_nodeid,
                                            neighbour.netid, neighbour.is_primary, neighbour.is_auxiliary,
                                            interfaces, neighbour.get_macs());
                    }
                    store_add_neighbour(key, val);
                }
            }
        }
        
        /** A neighbour has been rem_chged in a nic.
          */
        public void nic_rem_change_neighbour(Neighbour neighbour, REM old_rem)
        {
            // This listener, which could be a microfunc, is called
            //  right after a NeighbourManager (associated to a NIC) has
            //  changed a Neighbour's attributes.
            NIP key_nip = neighbour.nip;
            int key_nodeid = neighbour.nodeid;
            string key = tnip_nodeid_key(key_nip, key_nodeid);
            // ignore myself, as signaled by GlueNetworkInterfaceManager
            if (key != myself)
            {
                // is it already known?
                if (tnip_nodeid_table.has_key(key))
                {
                    store_changed_neighbour(key, neighbour.dev, neighbour.rem, 
                            neighbour.get_macs(), neighbour.netid,
                            neighbour.is_primary, neighbour.is_auxiliary);
                }
                else
                {
                    AggregatedNeighbour val;
                    if (neighbour.is_local())
                    {
                        val = new AggregatedNeighbour.local(neighbour.levels,
                                            neighbour.gsize,
                                            key_nip, key_nodeid,
                                            neighbour.netid, neighbour.is_primary, neighbour.is_auxiliary,
                                            neighbour.rpcdispatcher);
                    }
                    else
                    {
                        HashMap<string, REM> interfaces = new HashMap<string, REM>();
                        interfaces[neighbour.dev] = neighbour.rem;
                        val = new AggregatedNeighbour.real(neighbour.levels,
                                            neighbour.gsize,
                                            key_nip, key_nodeid,
                                            neighbour.netid, neighbour.is_primary, neighbour.is_auxiliary,
                                            interfaces, neighbour.get_macs());
                    }
                    store_add_neighbour(key, val);
                }
            }
        }
        
        /** A neighbour has been deleted in a nic.
          */
        public void nic_delete_neighbour(Neighbour neighbour)
        {
            // This listener, which could be a microfunc, is called
            //  right after a NeighbourManager (associated to a NIC) has
            //  deleted a Neighbour.
            NIP key_nip = neighbour.nip;
            int key_nodeid = neighbour.nodeid;
            string key = tnip_nodeid_key(key_nip, key_nodeid);
            // ignore myself, as signaled by GlueNetworkInterfaceManager
            if (key != myself)
            {
                // is it unknown? it shouldn't happen, though
                if (! tnip_nodeid_table.has_key(key)) return;
                AggregatedNeighbour aggregated_neighbour = tnip_nodeid_table[key];
                // if the neighbour is a local address, remove without further ado
                if (neighbour.is_local())
                    store_delete_neighbour(key);
                else
                {
                    // didn't it have already that nic? it shouldn't happen, though
                    if (! aggregated_neighbour.interfaces.has_key(neighbour.dev)) return;
                    // does it have other nics?
                    if (aggregated_neighbour.interfaces.size == 1)
                        store_delete_neighbour(key);
                    else
                        store_changed_neighbour(key, neighbour.dev, null, null, null, null, null, true);
                }
            }
        }

        /** Emits signal BEFORE a new neighbour.
          */
        public void going_add_neighbour(AggregatedNeighbour aggregated_neighbour)
        {
            string ipstr = nip_to_str(aggregated_neighbour.levels, aggregated_neighbour.gsize, aggregated_neighbour.nip);
            if (address_manager.is_in_my_network(aggregated_neighbour.netid))
            {
                // emit a signal notifying we are going to add a node
                aggregated_neighbour_going_new(aggregated_neighbour);
            }
            else
            {
                // emit a signal notifying we are going to add a node
                aggregated_neighbour_colliding_going_new(aggregated_neighbour);
            }
        }

        /** Emits signal BEFORE a dead neighbour.
          */
        public void going_delete_neighbour(AggregatedNeighbour aggregated_neighbour)
        {
            string ipstr = nip_to_str(aggregated_neighbour.levels, aggregated_neighbour.gsize, aggregated_neighbour.nip);
            if (address_manager.is_in_my_network(aggregated_neighbour.netid))
            {
                // emit a signal notifying we are going to delete a node
                aggregated_neighbour_going_deleted(aggregated_neighbour);
            }
            else
            {
                // emit a signal notifying we are going to delete a node
                aggregated_neighbour_colliding_going_deleted(aggregated_neighbour);
            }
        }

        /** Emits signal BEFORE a changed rem neighbour.
          */
        public void going_rem_change_neighbour(AggregatedNeighbour aggregated_neighbour)
        {
            string ipstr = nip_to_str(aggregated_neighbour.levels, aggregated_neighbour.gsize, aggregated_neighbour.nip);
            if (address_manager.is_in_my_network(aggregated_neighbour.netid))
            {
                // emit a signal notifying we are going to chage rem of a node
                aggregated_neighbour_going_rem_chged(aggregated_neighbour);
            }
            else
            {
                // emit a signal notifying we are going to chage rem of a node
                aggregated_neighbour_colliding_going_rem_chged(aggregated_neighbour);
            }
        }

        /** Emits signal for a new neighbour.
          */
        public void add_neighbour(string key)
        {
            AggregatedNeighbour aggregated_neighbour = tnip_nodeid_table[key];
            string ipstr = nip_to_str(aggregated_neighbour.levels, aggregated_neighbour.gsize, aggregated_neighbour.nip);
            if (address_manager.is_in_my_network(aggregated_neighbour.netid))
            {
                log_debug(@"AggregatedNeighbourManager: emit AGGREGATED_NEIGHBOUR_NEW for ip $(ipstr), netid $(aggregated_neighbour.netid) dev $(aggregated_neighbour.dev)");
                log_verbose(@"AggregatedNeighbourManager: emit AGGREGATED_NEIGHBOUR_NEW for ip $(ipstr), netid $(aggregated_neighbour.netid) dev $(aggregated_neighbour.dev)");
                // emit a signal notifying we added a node
                aggregated_neighbour_new_before(aggregated_neighbour);
                aggregated_neighbour_new(aggregated_neighbour);
            }
            else
            {
                log_debug(@"AggregatedNeighbourManager: emit AGGREGATED_NEIGHBOUR_COLLIDING_NEW for ip $(ipstr), netid $(aggregated_neighbour.netid) dev $(aggregated_neighbour.dev)");
                log_verbose(@"AggregatedNeighbourManager: emit AGGREGATED_NEIGHBOUR_COLLIDING_NEW for ip $(ipstr), netid $(aggregated_neighbour.netid) dev $(aggregated_neighbour.dev)");
                // emit a signal notifying we added a node
                aggregated_neighbour_colliding_new(aggregated_neighbour);
            }
        }

        /** Emits signal for a dead neighbour.
          */
        public void delete_neighbour(string key, AggregatedNeighbour old_val)
        {
            AggregatedNeighbour aggregated_neighbour = old_val;
            string ipstr = nip_to_str(old_val.levels, old_val.gsize, old_val.nip);
            if (address_manager.is_in_my_network(aggregated_neighbour.netid))
            {
                log_debug(@"AggregatedNeighbourManager: emit AGGREGATED_NEIGHBOUR_DELETED for ip $(ipstr), netid $(old_val.netid) dev $(old_val.dev)");
                log_verbose(@"AggregatedNeighbourManager: emit AGGREGATED_NEIGHBOUR_DELETED for ip $(ipstr), netid $(old_val.netid) dev $(old_val.dev)");
                // emit a signal notifying we deleted a node
                aggregated_neighbour_deleted(old_val);
                aggregated_neighbour_deleted_after(old_val);
            }
            else
            {
                // emit a signal notifying we deleted a node
                aggregated_neighbour_colliding_deleted(old_val);
            }
        }

        /** Emits signal for a changed rem neighbour.
          */
        public void rem_change_neighbour(string key, REM old_rem)
        {
            AggregatedNeighbour aggregated_neighbour = tnip_nodeid_table[key];
            string ipstr = nip_to_str(aggregated_neighbour.levels, aggregated_neighbour.gsize, aggregated_neighbour.nip);
            if (address_manager.is_in_my_network(aggregated_neighbour.netid))
            {
                // emit a signal notifying the node's rtt changed
                aggregated_neighbour_rem_chged_before(aggregated_neighbour, old_rem);
                aggregated_neighbour_rem_chged(aggregated_neighbour, old_rem);
            }
            else
            {
                // emit a signal notifying the node's rtt changed
                aggregated_neighbour_colliding_rem_chged(aggregated_neighbour, old_rem);
            }
        }
    }
}

