/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2011 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gee;
using zcd;
using Tasklets;

namespace Netsukuku
{
    /** The class IGSManagerDispatcher will be instantiated in each AddressManager.
      * Forward the call to the remotable method got_list_inetgw to the one instance of IGSManager
      * only if we are the primary AddressManager.
      */
    public class IGSManagerDispatcher : Object
    {
        private AddressManager address_manager;
        private /*IGSManager*/ Object igs_manager;

        public IGSManagerDispatcher(AddressManager address_manager, /*IGSManager*/ Object igs_manager)
        {
            this.address_manager = address_manager;
            this.igs_manager = igs_manager;
            // TODO  self.remotable_funcs = [self.got_list_inetgw]
        }

// TODO
/*
    def got_list_inetgw(self, *args, **kargs):
        if self.address_manager.is_primary:
            self.igs_manager.got_list_inetgw(*args, **kargs)
*/
    }

    struct struct_helper_AddressManager_start_operations
    {
        public AddressManager self;
        public int delay;
    }

    struct struct_helper_AddressManager_stop_operations
    {
        public AddressManager self;
    }

    public class AddressManager : Object, IAddressManagerRootDispatcher
    {
        public AddressManager(int levels,
                              int gsize,
                              KeyPair keypair,
                              HookReservation? hook_reservation,
                              CreateNicDelegate create_new_nic)
        {
            _is_primary = false;
            auxiliary_addresses = new ArrayList<AddressManager>();
            this.levels = levels;
            this.gsize = gsize;
            this.keypair = keypair;
            this.create_new_nic = create_new_nic;
            operative = true;
            is_mature = false;
            
            NIP nip;
            GNodeID[] id_myself;
            CoordinatorKnowledgeSet coordinator_knowledge_set;
            _get_data_from_hook_reservation(levels, gsize, hook_reservation,
                        out nip, out id_myself, out coordinator_knowledge_set);

            incoming_nodes = create_IncomingNodes();
            maproute = create_MapRoute(levels, gsize, nip, id_myself);
            aggregated_neighbour_manager = create_AggregatedNeighbourManager();
            etp = create_Etp(aggregated_neighbour_manager, maproute);
            peer_to_peer_all = create_PeerToPeerAll(aggregated_neighbour_manager, maproute);
            coordnode = create_Coord(aggregated_neighbour_manager, maproute, peer_to_peer_all, coordinator_knowledge_set);
            counter = create_Counter(keypair, aggregated_neighbour_manager, maproute, peer_to_peer_all);
            andna = create_Andna(keypair, counter, aggregated_neighbour_manager, maproute, peer_to_peer_all);
            counter.andna = andna;
            tunnel_manager = create_TunnelManager(nip_to_str(maproute.levels, maproute.gsize, maproute.me));
            hook = create_Hook(maproute, coordnode);
            border_nodes_manager = create_BorderNodesManager(maproute, aggregated_neighbour_manager, coordnode);
            migration_manager = create_MigrationManager(maproute, aggregated_neighbour_manager);
            igs_manager = null;
            dht_service = new DistributedHashTable(aggregated_neighbour_manager, maproute, peer_to_peer_all);

            // We register handlers to signals
            routes_updated.connect(maproute.evaluate_changed_netid);
            maproute.id_gnode_changed.connect(adv_mods);

            nics = new ArrayList<NetworkInterfaceManager>();
            neighbour_managers = new HashMap<string, NeighbourManager>();
            // Repeat signals
            maproute.gnode_splitted.connect(repeat_gnode_splitted);
            maproute.routes_updated.connect(repeat_routes_updated);
            aggregated_neighbour_manager.aggregated_neighbour_new.connect(repeat_aggregated_neighbour_new);
            aggregated_neighbour_manager.aggregated_neighbour_new_before.connect(repeat_aggregated_neighbour_new_before);
            aggregated_neighbour_manager.aggregated_neighbour_deleted.connect(repeat_aggregated_neighbour_deleted);
            aggregated_neighbour_manager.aggregated_neighbour_deleted_after.connect(repeat_aggregated_neighbour_deleted_after);
            aggregated_neighbour_manager.aggregated_neighbour_rem_chged.connect(repeat_aggregated_neighbour_rem_chged);
            aggregated_neighbour_manager.aggregated_neighbour_rem_chged_before.connect(repeat_aggregated_neighbour_rem_chged_before);
            aggregated_neighbour_manager.aggregated_neighbour_going_new.connect(repeat_aggregated_neighbour_going_new);
            aggregated_neighbour_manager.aggregated_neighbour_going_deleted.connect(repeat_aggregated_neighbour_going_deleted);
            aggregated_neighbour_manager.aggregated_neighbour_going_rem_chged.connect(repeat_aggregated_neighbour_going_rem_chged);
            aggregated_neighbour_manager.aggregated_neighbour_colliding_new.connect(repeat_aggregated_neighbour_colliding_new);
            aggregated_neighbour_manager.aggregated_neighbour_colliding_deleted.connect(repeat_aggregated_neighbour_colliding_deleted);
            aggregated_neighbour_manager.aggregated_neighbour_colliding_rem_chged.connect(repeat_aggregated_neighbour_colliding_rem_chged);
            aggregated_neighbour_manager.aggregated_neighbour_colliding_going_new.connect(repeat_aggregated_neighbour_colliding_going_new);
            aggregated_neighbour_manager.aggregated_neighbour_colliding_going_deleted.connect(repeat_aggregated_neighbour_colliding_going_deleted);
            aggregated_neighbour_manager.aggregated_neighbour_colliding_going_rem_chged.connect(repeat_aggregated_neighbour_colliding_going_rem_chged);
            etp.sent_etp.connect(repeat_sent_etp);
            etp.net_collision.connect(repeat_net_collision);
            incoming_nodes.incoming_node_updated.connect(repeat_incoming_node_updated);
            counter.counter_hooked.connect(repeat_counter_hooked);
            counter.counter_registered.connect(repeat_counter_registered);
            andna.andna_hooked.connect(repeat_andna_hooked);
            andna.andna_registered.connect(repeat_andna_registered);
        }

        private void adv_mods(MapRoute mr, int level)
        {
            // a change in gnode's id has to be communicated
            etp.advertise_modification_to_gnodeid();
        }

        protected virtual void _get_data_from_hook_reservation(int levels,
                                 int gsize, HookReservation? hook_reservation, out NIP nip,
                                 out GNodeID[] id_myself, out CoordinatorKnowledgeSet coordinator_knowledge_set)
        {
            // Internal method called by the constructor.
            // It may be overriden, this is useful for testing/debugging purpose.

            // Create instances of the classes that we need in order to manage an address
            PartialNIP partial_nip;
            int elderliness;
            GNodeID[] gids;
            if (hook_reservation == null)
            {
                // hook_reservation is None means choose a new valid address in a fresh new network.
                int[] partial_nip_positions = new int[levels];
                for (int i = 0; i < levels; i++) partial_nip_positions[i] = -1;
                partial_nip = new PartialNIP(partial_nip_positions);
                elderliness = 0;
                gids = new GNodeID[0];
                coordinator_knowledge_set = new CoordinatorKnowledgeSet();
            }
            else
            {
                // initial data from hook_reservation object
                int[] partial_nip_positions = hook_reservation.gnode.get_positions();
                int given_up_to_level = hook_reservation.gnode.level_of_gnode();
                partial_nip_positions[given_up_to_level-1] = hook_reservation.pos;
                partial_nip = new PartialNIP(partial_nip_positions);
                elderliness = hook_reservation.elderliness;
                gids = hook_reservation.gids;
                coordinator_knowledge_set = hook_reservation.coordinator_knowledge_set;
            }
            // len(gids) = l -> we hooked in a gnode of level levels-l
            int new_ident = Random.int_range(0, (int)Math.pow(2,32) - 1);
            // ID_myself = [GNodeid(0+ID), GNodeid(0+0+ID), ... GNodeid(NA+0+ID), GNodeid(?+?+?), ..., GNodeid(?)]  levels+1 elements
            id_myself = new GNodeID[levels+1];
            for (int i = 0; i < (levels - gids.length); i++) id_myself[i] = new GNodeID(0, 0, new_ident);
            id_myself[levels - gids.length] = new GNodeID(elderliness, 0, new_ident);
            for (int i = 0; i < gids.length; i++) id_myself[i + levels - gids.length + 1] = gids[i];
            nip = complete_nip(partial_nip, gsize, levels);
        }

        /** Test methods for peer_to_peer
          */

        public void dht_stor(DHTRecord rec)
        {
            this.dht_service.peer(null, rec.key).store(rec);
        }

        public DHTRecord? dht_retr(DHTKey k)
        {
            return this.dht_service.peer(null, k).retrieve(k);
        }

        public Gee.List<DHTRecord> dht_list()
        {
            return this.dht_service.peer(maproute.me).get_cache();
        }

        /** Report statistics about tasklet usage
          */
        public Gee.List<TaskletStats> report_tasklets_stats(int minutes)
        {
            Gee.List<TaskletStats> ret =
             (Gee.List<TaskletStats>) Tasklets.get_tasklet_stats();

            ArrayList<int> topurge = new ArrayList<int>();
            int64 timeout = (int64)1000 * (int64)60 * (int64)minutes;
            Tasklets.Timer threshold = new Tasklets.Timer(-timeout);
            foreach (TaskletStats s in ret)
            {
                if (s.status == Status.ENDED ||
                    s.status == Status.ABORTED)
                {
                    if (! s.tasklet_ended.is_younger(threshold)) topurge.add(s.id);
                }
            }
            Tasklets.purge_tasklet_stats(topurge);
            ret = (Gee.List<TaskletStats>) Tasklets.get_tasklet_stats();

            return ret;
        }

        /** Get recent logs for a given tasklet (id)
          * 1st item is name for tasklet 'id'.
          *  As special case, name="STOPPED" means the tasklet is stopped and no more logs will come from there.
          * 2nd item (if existent) is the pos of first log.
          * Following items are logs.
          */
        public Gee.List<string> report_tasklet_logs(int id)
        {
            return Tasklet.get_logs(id);
        }

        /** Get list of running tasklets' id.
          */
        public Gee.List<string> report_running_tasklets()
        {
            ArrayList<string> ret = new ArrayList<string>();
            Tasklet.nap(0, 1000);
            Gee.List<TaskletStats> stats =
             (Gee.List<TaskletStats>) Tasklets.get_tasklet_stats();
            foreach (TaskletStats s in stats)
            {
                if (s.status == Status.SPAWNED || s.status == Status.STARTED)
                {
                    ret.add(@"$(s.id)");
                }
            }
            return ret;
        }

        /** This method returns a valid random NIP inside a gnode.
          */
        public NIP complete_nip(PartialNIP gnode, int gsize, int levels)
        {
            int[] full_pos = new int[levels];
            int given_up_to_level = gnode.level_of_gnode();
            for (int i = given_up_to_level; i < levels; i++) full_pos[i] = gnode.position_at(i);
            for (int l = given_up_to_level-1; l >= 0; l--)
            {
                ArrayList<int> range = valid_ids(levels, gsize, l, gnode);
                int choice = Random.int_range(0, range.size);
                full_pos[l] = range.get(choice);
            }
            return new NIP(full_pos);
        }

        public unowned IBorderNodesManager _border_nodes_manager_getter()
        {
            return border_nodes_manager;
        }
        public unowned IMapRoute _maproute_getter()
        {
            return maproute;
        }
        public unowned ITunnelManager _tunnel_manager_getter()
        {
            return tunnel_manager;
        }
        public unowned IEtp _etp_getter()
        {
            return etp;
        }
        public unowned IHook _hook_getter()
        {
            return hook;
        }
        public unowned IAggregatedNeighbourManager _aggregated_neighbour_manager_getter()
        {
            return aggregated_neighbour_manager;
        }
        public unowned IPeerToPeerAll _peer_to_peer_all_getter()
        {
            return peer_to_peer_all;
        }
        public unowned ICoord _coordnode_getter()
        {
            return coordnode;
        }
        public unowned IAndna _andna_getter()
        {
            return andna;
        }

        public IPeerToPeer get_peer_to_peer_service(int pid)
        {
            return peer_to_peer_all.pid_get(pid);
        }
        public IOptionalPeerToPeer get_optional_peer_to_peer_service(int pid)
        {
            return (OptionalPeerToPeer)peer_to_peer_all.pid_get(pid);
        }

        private bool _is_primary;
        public bool is_primary {
            get {
                return _is_primary;
            }
            set {
                bool changed = _is_primary != value;
                _is_primary = value;
                if (changed) is_primary_changed();
                if (!value)
                {
                    // leave all optional peer_to_peer services
                    peer_to_peer_all.leave_all_optional_peer_to_peer();
                }
            }
        }

        // An instance of this class is NOT auxiliary, but a derived class could override.
        public virtual bool is_auxiliary {
            get {
                return false;
            }
        }

        // An instance of this class IS autonomous, but a derived class could override.
        public virtual bool is_autonomous {
            get {
                return true;
            }
        }

        public ArrayList<AddressManager> auxiliary_addresses;
        public int levels;
        public int gsize;
        public KeyPair keypair;
        public CreateNicDelegate create_new_nic;
        public bool operative;
        public bool is_mature;
        public IncomingNodes incoming_nodes;
        public MapRoute maproute;
        public AggregatedNeighbourManager aggregated_neighbour_manager;
        public Etp etp;
        public PeerToPeerAll peer_to_peer_all;
        public Coord coordnode;
        public Counter counter;
        public Andna andna;
        public TunnelManager tunnel_manager;
        public Hook hook;
        public BorderNodesManager border_nodes_manager;
        public MigrationManager migration_manager;
        public IGSManagerDispatcher? igs_manager;
        public DistributedHashTable dht_service;

        public ArrayList<NetworkInterfaceManager> nics;
        public HashMap<string, NeighbourManager> neighbour_managers;

        public signal void routes_updated(HCoord lvl_pos);
        public signal void gnode_splitted(Gee.List<AggregatedNeighbour> passed_neighbour_list, Gee.List<int> queue_of_request_ids, GNodeID actual_gid);
        public signal void aggregated_neighbour_new(AggregatedNeighbour aggregated_neighbour);
        public signal void aggregated_neighbour_new_before(AggregatedNeighbour aggregated_neighbour);
        public signal void aggregated_neighbour_deleted(AggregatedNeighbour aggregated_neighbour);
        public signal void aggregated_neighbour_deleted_after(AggregatedNeighbour aggregated_neighbour);
        public signal void aggregated_neighbour_rem_chged(AggregatedNeighbour aggregated_neighbour, REM old_rem);
        public signal void aggregated_neighbour_rem_chged_before(AggregatedNeighbour aggregated_neighbour, REM old_rem);
        public signal void aggregated_neighbour_going_new(AggregatedNeighbour aggregated_neighbour);
        public signal void aggregated_neighbour_going_deleted(AggregatedNeighbour aggregated_neighbour);
        public signal void aggregated_neighbour_going_rem_chged(AggregatedNeighbour aggregated_neighbour);
        public signal void aggregated_neighbour_colliding_new(AggregatedNeighbour aggregated_neighbour);
        public signal void aggregated_neighbour_colliding_deleted(AggregatedNeighbour aggregated_neighbour);
        public signal void aggregated_neighbour_colliding_rem_chged(AggregatedNeighbour aggregated_neighbour, REM old_rem);
        public signal void aggregated_neighbour_colliding_going_new(AggregatedNeighbour aggregated_neighbour);
        public signal void aggregated_neighbour_colliding_going_deleted(AggregatedNeighbour aggregated_neighbour);
        public signal void aggregated_neighbour_colliding_going_rem_chged(AggregatedNeighbour aggregated_neighbour);
        public signal void net_collision(Gee.List<AggregatedNeighbour> others);
        public signal void sent_etp(NIP nip_caller, NetworkID netid_caller,
                                Gee.List<string> macs_caller, ExtendedTracerPacket etp);
        public signal void incoming_node_updated(string mac);
        public signal void counter_hooked();
        public signal void andna_hooked();
        public signal void counter_registered();
        public signal void andna_registered(string domain);
        public signal void sig_is_mature(AddressManager mature_addrman);
        public signal void is_primary_changed();
        public signal void auxiliary_address_manager_new(AddressManager addrman);
        public signal void auxiliary_address_manager_deleted(AddressManager addrman);
        public signal void address_manager_new(HookReservation hook_reservation, AddressManager old_addr_man, ref AddressManager return_value);
        public signal void request_be_primary(AddressManager addrman);
        public signal void address_manager_delete(AddressManager addrman);

        void repeat_gnode_splitted(MapRoute maproute, Gee.List<AggregatedNeighbour> passed_neighbour_list,Gee.List<int> queue_of_request_ids, GNodeID actual_gid)
        {
            gnode_splitted(passed_neighbour_list, queue_of_request_ids, actual_gid);
        }
        void repeat_routes_updated(MapRoute maproute, HCoord lvl_pos)
        {
            routes_updated(lvl_pos);
        }
        void repeat_aggregated_neighbour_new(AggregatedNeighbourManager aggregated_neighbour_manager, AggregatedNeighbour aggregated_neighbour)
        {
            aggregated_neighbour_new(aggregated_neighbour);
        }
        void repeat_aggregated_neighbour_new_before(AggregatedNeighbourManager aggregated_neighbour_manager, AggregatedNeighbour aggregated_neighbour)
        {
            aggregated_neighbour_new_before(aggregated_neighbour);
        }
        void repeat_aggregated_neighbour_deleted(AggregatedNeighbourManager aggregated_neighbour_manager, AggregatedNeighbour aggregated_neighbour)
        {
            aggregated_neighbour_deleted(aggregated_neighbour);
        }
        void repeat_aggregated_neighbour_deleted_after(AggregatedNeighbourManager aggregated_neighbour_manager, AggregatedNeighbour aggregated_neighbour)
        {
            aggregated_neighbour_deleted_after(aggregated_neighbour);
        }
        void repeat_aggregated_neighbour_rem_chged(AggregatedNeighbourManager aggregated_neighbour_manager, AggregatedNeighbour aggregated_neighbour, REM old_rem)
        {
            aggregated_neighbour_rem_chged(aggregated_neighbour, old_rem);
        }
        void repeat_aggregated_neighbour_rem_chged_before(AggregatedNeighbourManager aggregated_neighbour_manager, AggregatedNeighbour aggregated_neighbour, REM old_rem)
        {
            aggregated_neighbour_rem_chged_before(aggregated_neighbour, old_rem);
        }
        void repeat_aggregated_neighbour_going_new(AggregatedNeighbourManager aggregated_neighbour_manager, AggregatedNeighbour aggregated_neighbour)
        {
            aggregated_neighbour_going_new(aggregated_neighbour);
        }
        void repeat_aggregated_neighbour_going_deleted(AggregatedNeighbourManager aggregated_neighbour_manager, AggregatedNeighbour aggregated_neighbour)
        {
            aggregated_neighbour_going_deleted(aggregated_neighbour);
        }
        void repeat_aggregated_neighbour_going_rem_chged(AggregatedNeighbourManager aggregated_neighbour_manager, AggregatedNeighbour aggregated_neighbour)
        {
            aggregated_neighbour_going_rem_chged(aggregated_neighbour);
        }
        void repeat_aggregated_neighbour_colliding_new(AggregatedNeighbourManager aggregated_neighbour_manager, AggregatedNeighbour aggregated_neighbour)
        {
            aggregated_neighbour_colliding_new(aggregated_neighbour);
        }
        void repeat_aggregated_neighbour_colliding_deleted(AggregatedNeighbourManager aggregated_neighbour_manager, AggregatedNeighbour aggregated_neighbour)
        {
            aggregated_neighbour_colliding_deleted(aggregated_neighbour);
        }
        void repeat_aggregated_neighbour_colliding_rem_chged(AggregatedNeighbourManager aggregated_neighbour_manager, AggregatedNeighbour aggregated_neighbour, REM old_rem)
        {
            aggregated_neighbour_colliding_rem_chged(aggregated_neighbour, old_rem);
        }
        void repeat_aggregated_neighbour_colliding_going_new(AggregatedNeighbourManager aggregated_neighbour_manager, AggregatedNeighbour aggregated_neighbour)
        {
            aggregated_neighbour_colliding_going_new(aggregated_neighbour);
        }
        void repeat_aggregated_neighbour_colliding_going_deleted(AggregatedNeighbourManager aggregated_neighbour_manager, AggregatedNeighbour aggregated_neighbour)
        {
            aggregated_neighbour_colliding_going_deleted(aggregated_neighbour);
        }
        void repeat_aggregated_neighbour_colliding_going_rem_chged(AggregatedNeighbourManager aggregated_neighbour_manager, AggregatedNeighbour aggregated_neighbour)
        {
            aggregated_neighbour_colliding_going_rem_chged(aggregated_neighbour);
        }
        void repeat_sent_etp(Etp etp, NIP nip_caller, NetworkID netid_caller,
                                Gee.List<string> macs_caller, ExtendedTracerPacket pkt)
        {
            sent_etp(nip_caller, netid_caller, macs_caller, pkt);
        }
        void repeat_net_collision(Gee.List<AggregatedNeighbour> others)
        {
            net_collision(others);
        }
        void repeat_incoming_node_updated(string mac)
        {
            incoming_node_updated(mac);
        }
        void repeat_counter_hooked()
        {
            counter_hooked();
        }
        void repeat_counter_registered()
        {
            counter_registered();
        }
        void repeat_andna_hooked()
        {
            andna_hooked();
        }
        void repeat_andna_registered(string domain)
        {
            andna_registered(domain);
        }

        public virtual IncomingNodes create_IncomingNodes()
        {
            return new IncomingNodes();
        }

        public virtual MapRoute create_MapRoute(int levels, int gsize, NIP nip, GNodeID[] id_myself)
        {
            return new MapRoute(levels, gsize, nip, id_myself, this);
        }

        public virtual AggregatedNeighbourManager create_AggregatedNeighbourManager()
        {
            return new AggregatedNeighbourManager(this);
        }

        public virtual Etp create_Etp(AggregatedNeighbourManager aggregated_neighbour_manager, MapRoute maproute)
        {
            return new Etp(aggregated_neighbour_manager, maproute, this);
        }

        public virtual PeerToPeerAll create_PeerToPeerAll(AggregatedNeighbourManager aggregated_neighbour_manager, MapRoute maproute)
        {
            return new PeerToPeerAll(aggregated_neighbour_manager, maproute);
        }

        public virtual Coord create_Coord(AggregatedNeighbourManager aggregated_neighbour_manager, MapRoute maproute, PeerToPeerAll peer_to_peer_all, CoordinatorKnowledgeSet coordinator_knowledge_set)
        {
            return new Coord(aggregated_neighbour_manager, maproute, peer_to_peer_all, coordinator_knowledge_set);
        }

        public virtual Counter create_Counter(KeyPair keypair, AggregatedNeighbourManager aggregated_neighbour_manager, MapRoute maproute, PeerToPeerAll peer_to_peer_all)
        {
            return new Counter(keypair, aggregated_neighbour_manager, maproute, peer_to_peer_all);
        }

        public virtual Andna create_Andna(KeyPair keypair, Counter counter, AggregatedNeighbourManager aggregated_neighbour_manager, MapRoute maproute, PeerToPeerAll peer_to_peer_all)
        {
            return new Andna(keypair, counter, aggregated_neighbour_manager, maproute, peer_to_peer_all);
        }

        public virtual TunnelManager create_TunnelManager(string ipstr)
        {
            return new TunnelManager(ipstr);
        }

        public virtual Hook create_Hook(MapRoute maproute, Coord coord)
        {
            return new Hook(maproute, coord, this);
        }

        public virtual BorderNodesManager create_BorderNodesManager(MapRoute maproute, AggregatedNeighbourManager aggregated_neighbour_manager, Coord coord)
        {
            return new BorderNodesManager(this, maproute, aggregated_neighbour_manager, coord);
        }

        public virtual MigrationManager create_MigrationManager(MapRoute maproute, AggregatedNeighbourManager aggregated_neighbour_manager)
        {
            return new MigrationManager(this, maproute, aggregated_neighbour_manager);
        }

        public AddressManagerFakeRmt get_broadcast_client()
        {
            // prepare broadcast client for real neighbours
            ArrayList<string> _devs = new ArrayList<string>();
            foreach (NetworkInterfaceManager nic_man in nics)
                if (nic_man.to_be_managed)
                    _devs.add(nic_man.nic_name);
            string[] devs = _devs.to_array();
            var real_bc = new AddressManagerBroadcastClient(new BroadcastID(), devs);
            // prepare broadcast client for local addresses
            Gee.List<RPCDispatcher> rpcdispatchers = new ArrayList<RPCDispatcher>();
            foreach (NetworkInterfaceManager nic_man in nics)
                if (! nic_man.to_be_managed)
                {
                    GlueNetworkInterfaceManager glue_nic_man = (GlueNetworkInterfaceManager)nic_man;
                    Gee.List<Neighbour> nlist = glue_nic_man.neighbour_list();
                    foreach (Neighbour neighbour in nlist)
                        if (! neighbour.nip.is_equal(maproute.me))
                            rpcdispatchers.add(neighbour.rpcdispatcher);
                }
            var pseudo_bc = new AddressManagerPseudoBroadcastClient(rpcdispatchers);
            // use both
            return new CoupleAddressManagerFakeRmt(real_bc, pseudo_bc);
        }

        /** This method can be overriden when we want to own an address with which
          * we want to be used as a gateway only by certain nodes.
          * In its default implementation, we accept any node.
          */
        public virtual bool do_i_act_as_gateway_for(NIP nip, int nodeid, NetworkID netid)
        {
            if (! operative) return false; // change behaviour when going down.
            return true;
        }

        /** This method can be overriden when we want to own an address with which
          * we do not want to be used as a gateway by other local addresses.
          * In its default implementation, we accept them.
          */
        public virtual bool do_i_act_as_gateway_for_local()
        {
            return true;
        }

        /** This method can be overriden when we do not want to
          * be considered in forming the kernel routing tables.
          * In its default implementation returns True.
          */
        public virtual bool do_i_participate_in_routing_tables()
        {
            return true;
        }

        /** My GNode ID list
          */
        public GNodeID[] get_gid_list()
        {
            return maproute.get_gid_list();
        }

        /** My node ID
          */
        public int get_my_id()
        {
            return maproute.get_my_id();
        }

        /** The netid of the uppermost level
          */
        public NetworkID get_main_netid()
        {
            return maproute.get_main_netid();
        }

        /** Is this netid in my network
          */
        public bool is_in_my_network(NetworkID netid)
        {
            return maproute.is_in_my_network(netid);
        }

        /** Is this netid preferred over mine
          */
        public bool is_preferred_network(NetworkID netid)
        {
            return maproute.is_preferred_network(netid);
        }

        public bool would_cause_split()
        {
            // This is used in a non-primary address manager to see whether I could
            //  cause a split in a gnode when I die.
            // Search for my primary NIP.
            AddressManager? prim = null;
            foreach (AddressManager addr_man in Addresses.get_addresses_instance().get_autonomous_addresses())
                if (addr_man.is_primary) prim = addr_man;
            assert(prim != null);
            NIP prim_nip = prim.maproute.me;
            // others
            Gee.List<AggregatedNeighbour> neighbours = aggregated_neighbour_manager.neighbour_list(true);
            AggregatedNeighbour[] others = {};
            foreach (AggregatedNeighbour n in neighbours)
                if (!n.is_local() && n.is_primary)
                    others += n;
            // for each level l from nip_cmp(my_primary) down to 1 (included):
            for (int l = maproute.nip_cmp(prim_nip.get_positions()); l >= 1; l--)
            {
                // for each pair x,y such that nip_cmp(x) < l and nip_cmp(y) < l:
                for (int i = 0; i < others.length; i++)
                {
                    AggregatedNeighbour x = others[i];
                    int x_nip_cmp = maproute.nip_cmp(x.nip.get_positions());
                    if (x_nip_cmp < l)
                    {
                        for (int j = i+1; j < others.length; j++)
                        {
                            AggregatedNeighbour y = others[j];
                            int y_nip_cmp = maproute.nip_cmp(y.nip.get_positions());
                            if (y_nip_cmp < l)
                            {
                                AggregatedNeighbour gw;
                                HCoord dst;
                                if (x_nip_cmp > y_nip_cmp)
                                {
                                    gw = y;
                                    dst = maproute.nip_to_lvlid(x.nip);
                                }
                                else
                                {
                                    gw = x;
                                    dst = maproute.nip_to_lvlid(y.nip);
                                }
                                if (maproute.node_get(dst.lvl, dst.pos).route_get_by_gw(gw) == null)
                                {
                                    return true;
                                }
                            }
                        }
                    }
                }
            }
            return false;
        }

        public bool would_prevent_removal()
        {
            // This is used in a non-primary address manager to see whether I would
            //  prefer to not die. Actually, it detects whether there are active TCP connections.
            Connections kconn = Connections.get_instance();
            return kconn.active_tcp_connections(nip_to_str(maproute.levels, maproute.gsize, maproute.me));
        }

        private void impl_start_operations(int delay) throws Error
        {
            Tasklet.declare_self(@"AddressManager.start_operations(delay=$(delay))");
            aggregated_neighbour_manager.initialize();
            if (delay > 0) ms_wait(delay);
            // From now on, consider that we have a good knowledge of the network,
            //  since we should have obtained some ETPs. For example, it means we
            //  could begin a rehook if we encounter a different network of bigger
            //  size.
            is_mature = true;
            log_debug(@"Now $(maproute.me) is mature.");
            sig_is_mature(this);
            // Participate to PeerToPeer
            peer_to_peer_all.start_operations();
            // Manage border_nodes
            border_nodes_manager.start_operations();
            // Manage migrations
            migration_manager.start_operations();
            // First actual calculation of networkid
            maproute.evaluate_changed_netid();
        }

        /* Decoration of microfunc */
        private static void * helper_start_operations(void *v) throws Error
        {
            struct_helper_AddressManager_start_operations *tuple_p = (struct_helper_AddressManager_start_operations *)v;
            // The caller function has to add a reference to the ref-counted instances
            AddressManager self_save = tuple_p->self;
            // The caller function has to copy the by-value parameters
            int delay_save = tuple_p->delay;
            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            self_save.impl_start_operations(delay_save);
            // void method, return null
            return null;
        }

        public void start_operations(int delay=0)
        {
            struct_helper_AddressManager_start_operations arg = struct_helper_AddressManager_start_operations();
            arg.self = this;
            arg.delay = delay;
            Tasklet.spawn((Spawnable)helper_start_operations, &arg);
        }

        private void impl_stop_operations() throws Error
        {
            Tasklet.declare_self("AddressManager.stop_operations");
            operative = false;
            etp.stop_operations();
            peer_to_peer_all.stop_operations();
            border_nodes_manager.stop_operations();
            migration_manager.stop_operations();
            coordnode.stop_operations();
            maproute.stop_operations();
        }

        /* Decoration of microfunc */
        private static void * helper_stop_operations(void *v) throws Error
        {
            struct_helper_AddressManager_stop_operations *tuple_p = (struct_helper_AddressManager_stop_operations *)v;
            // The caller function has to add a reference to the ref-counted instances
            AddressManager self_save = tuple_p->self;
            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            self_save.impl_stop_operations();
            // void method, return null
            return null;
        }

        public void stop_operations()
        {
            struct_helper_AddressManager_stop_operations arg = struct_helper_AddressManager_stop_operations();
            arg.self = this;
            Tasklet.spawn((Spawnable)helper_stop_operations, &arg);
        }

        public void add_nic_manager(NetworkInterfaceManager nic_manager)
        {
            // things to do when an address begins to manage an NIC
            NeighbourManager neighbour_manager = nic_manager.create_neighbour_manager(this);
            neighbour_managers[nic_manager.nic_name] = neighbour_manager;
            // listen to signals of new neighbour_manager
            neighbour_manager.neighbour_new.connect(aggregated_neighbour_manager.nic_add_neighbour);
            neighbour_manager.neighbour_deleted.connect(aggregated_neighbour_manager.nic_delete_neighbour);
            neighbour_manager.neighbour_rem_chged.connect(aggregated_neighbour_manager.nic_rem_change_neighbour);
            neighbour_manager.neighbour_going_new.connect(aggregated_neighbour_manager.nic_going_add_neighbour);
            neighbour_manager.neighbour_going_deleted.connect(aggregated_neighbour_manager.nic_going_delete_neighbour);
            neighbour_manager.neighbour_going_rem_chged.connect(aggregated_neighbour_manager.nic_going_rem_change_neighbour);
            // update lists
            nics.add(nic_manager);
            nic_manager.add_address(this);
        }

        public void remove_nic_manager(NetworkInterfaceManager nic_manager)
        {
            // things to do when an address ceases to manage an NIC
            NeighbourManager neighbour_manager = neighbour_managers[nic_manager.nic_name];
            nic_manager.dispose_neighbour_manager(this, neighbour_manager);
            // ceases to handle events of this nic_manager
            neighbour_manager.neighbour_new.disconnect(aggregated_neighbour_manager.nic_add_neighbour);
            neighbour_manager.neighbour_deleted.disconnect(aggregated_neighbour_manager.nic_delete_neighbour);
            neighbour_manager.neighbour_rem_chged.disconnect(aggregated_neighbour_manager.nic_rem_change_neighbour);
            neighbour_manager.neighbour_going_new.disconnect(aggregated_neighbour_manager.nic_going_add_neighbour);
            neighbour_manager.neighbour_going_deleted.disconnect(aggregated_neighbour_manager.nic_going_delete_neighbour);
            neighbour_manager.neighbour_going_rem_chged.disconnect(aggregated_neighbour_manager.nic_going_rem_change_neighbour);
            neighbour_managers.unset(nic_manager.nic_name);
            nics.remove(nic_manager);
            nic_manager.remove_address(this);
        }

        public string to_string()
        {
            return @"<AddressManager for $(maproute.me)>";
        }
    }
}

