/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2011 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 *  Implementation of the PeerToPeer Over Ntk RFC. See {-PeerToPeerNtk-}
 */

using Gee;
using zcd;
using Tasklets;

namespace Netsukuku
{
    public errordomain PeerToPeerError {
        REGISTER,
        GENERIC
    }

    struct struct_helper_MapPeerToPeer_node_dead
    {
        public MapPeerToPeer self;
        public int lvl;
        public int pos;
    }

    public class MapPeerToPeer : Map<ParticipantNode>
    {
        private int pid;
        private bool participant;

        /** levels, gsize, me: the same of Map
          * pid: PeerToPeer id of the service associated to this map
          */
        public MapPeerToPeer(int levels, int gsize, NIP me, int pid)
        {
            base(levels, gsize, me);
            // Initialize all the nodes, except me (see explanation in overrided method node_get below)
            for (int lvl = 0; lvl < levels; lvl++)
                for (int pos = 0; pos < gsize; pos++)
                    if (me.position_at(lvl) != pos)
                        node[lvl, pos] = new ParticipantNode();
            this.pid = pid;
            participant = false;
        }

        /** Initialize map from a neighbour's data.
          */
        public void initialize_from_neighbour(NIP nip, PackedParticipantNodes packed_nodes)
        {
            int lvl = nip_cmp(nip.get_positions());
            // Initialize all the nodes, except me (see explanation in overrided method node_get below)
            for (int l = lvl; l < levels; l++)
                for (int pos = 0; pos < gsize; pos++)
                    if (me.position_at(l) != pos)
                        node[l, pos] = packed_nodes.packednodes[l][pos];
        }

        /** A new instance is created for a position that represents me. Otherwise, the usual behaviour.
          */
        public override ParticipantNode node_get(int lvl, int pos)
        {
            // The positions that represent myself, have no real data.
            // When someone looks for them, a new object is created with this data:
            //  . participant = At level 'lvl' is True iff some position at level 'lvl-1' is True.
            //                  If the node is participant, then at level 0 is True, hence all levels are True.
            //                  If the node is not participant, then at level 0 is False.
            //  . tc = TimeCapsule(0)
            if (me.position_at(lvl) == pos)
            {
                if (participant)
                    return new ParticipantNode(true, new TimeCapsule(0));
                else
                {
                    if (lvl == 0) return new ParticipantNode(false, new TimeCapsule(0));
                    else
                    {
                        for (int pos2 = 0; pos2 < gsize; pos2++)
                        {
                            if (node_get(lvl-1, pos2).participant)
                                return new ParticipantNode(true, new TimeCapsule(0));
                        }
                        return new ParticipantNode(false, new TimeCapsule(0));
                    }
                }
            }
            else return base.node_get(lvl, pos);
        }

        private void impl_node_dead(int lvl, int pos) throws Error
        {
            Tasklet.declare_self("MapPeerToPeer.node_dead");
            // A node died. So it is no more participating.
            // Its position in map_peer_to_peer is reinitialized.
            node[lvl, pos] = new ParticipantNode();
            check_node(lvl, pos);
        }

        private static void * helper_node_dead(void *v) throws Error
        {
            struct_helper_MapPeerToPeer_node_dead *tuple_p = (struct_helper_MapPeerToPeer_node_dead *)v;
            // The caller function has to add a reference to the ref-counted instances
            MapPeerToPeer self_save = tuple_p->self;
            // The caller function has to copy the value of byvalue parameters
            int lvl_save = tuple_p->lvl;
            int pos_save = tuple_p->pos;
            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            self_save.impl_node_dead(lvl_save, pos_save);
            // void method, return null
            return null;
        }

        public void node_dead(int lvl, int pos)
        {
            struct_helper_MapPeerToPeer_node_dead arg = struct_helper_MapPeerToPeer_node_dead();
            arg.self = this;
            arg.lvl = lvl;
            arg.pos = pos;
            Tasklet.spawn((Spawnable)helper_node_dead, &arg);
        }

        /** Set me to be a participant node.
          */
        public void participate()
        {
            participant = true;
        }

        /** Set me to be a non-participant node.
          */
        public void sit_out()
        {
            participant = false;
        }

        private static bool not_impl_equal_func (Object? a, Object? b)
        {
            error("MapPeerToPeer.get_packed_nodes: equal_func not implemented");
        }
        /** Prepares a packed map_peer_to_peer to be passed to refresh participations
          */
        public PackedParticipantNodes get_packed_nodes()
        {
            ArrayList<ArrayList<ParticipantNode>> l1 = new ArrayList<ArrayList<ParticipantNode>>(not_impl_equal_func);
            for (int lvl = 0; lvl < levels; lvl++)
            {
                ArrayList<ParticipantNode> l0 = new ArrayList<ParticipantNode>();
                for (int pos = 0; pos < gsize; pos++)
                {
                    l0.add(node_get(lvl, pos));
                }
                l1.add(l0);
            }
            return new PackedParticipantNodes(l1);
        }

        internal string log_service()
        {
            string ret = "{";
            for (int lvl = 0; lvl < levels; lvl++)
            {
                int tot = 0;
                for (int i = 0; i < gsize; i++)
                {
                    if (node_get(lvl, i).participant) tot++;
                }
                ret += @"$tot at level $lvl, ";
            }
            ret += "}";
            return ret;
        }
    }
    
    /** This is the class that must be inherited to create a Strict PeerToPeer module
      *  service. A strict service is a service where all the hosts connected 
      *  to Netsukuku are participant, so the MapPeerToPeer is not used here.
      */
    public class PeerToPeer : RPCDispatcher, IPeerToPeer
    {
        public weak MapRoute maproute;
        public weak AggregatedNeighbourManager aggregated_neighbour_manager;
        public int pid;
        public bool has_valid_map;

        public signal void map_peer_to_peer_validated();

        public PeerToPeer(AggregatedNeighbourManager aggregated_neighbour_manager,
                    MapRoute maproute, int pid)
        {
            this.aggregated_neighbour_manager = aggregated_neighbour_manager;
            this.maproute = maproute;
            this.pid = pid;

            // The routing algorithm of PeerToPeer messages uses 2 kinds of maps. The
            //  strict services use MapRoute, the optional services use MapPeerToPeer.
            // The informations in MapPeerToPeer become valid AFTER the retrieve_optional_services_and_participants has terminated.
            // The informations in MapRoute become valid AFTER address_manager is_mature.

            // In the particular case of a peer_to_peer message sent in the hooking phase, the
            //  first step of this message has to be sent via UDP to the neighbour (via
            //  a NeighbourClient) with which the node is hooking.
            //  In the following steps (and in all other cases) messages can be delivered
            //  to the next hop only when the map in use is ready.

            has_valid_map = false;
        }

        /** Called when the map is valid. Overridable (eg in OptionalPeerToPeer)
          */
        public virtual void start_operations()
        {
            has_valid_map = true;
            map_peer_to_peer_validated();
        }

        /** Called when address is dying. Overridable (eg in OptionalPeerToPeer)
          */
        public virtual void stop_operations()
        {
        }

        /** Returns True iff the node lvl,pos is participating
          * to the service. For a "strict PeerToPeer" it means iff the node
          * exists in maproute.
          * An inheriting class could override the function.
          */
        public virtual bool is_participant(int lvl, int pos)
        {
            RouteNode x = maproute.node_get(lvl, pos);
            return !x.is_free();
        }

        /** This is the function h:KEY-->hIP.
          *
          * You should override it with your own mapping function.
          */
        public virtual NIP h(Object key)
        {
            return (NIP)key;
        }

        /** Helper functions for "routed" methods **/

        /** Returns the AggregatedNeighbour instances of the neighbours we can use to reach
          * a given g-node, in order of efficiency.
          */
        public Gee.List<AggregatedNeighbour> neighbours_get_from_lvl_pos(int lvl, int pos)
        {
            Gee.List<Route> routes = maproute.node_get(lvl, pos).all_valid_routes();

            // TODO When we forward a packet from a source IP
            //  to a destination IP we do not know the hops that
            //  the packet (udp or tcp or other protocol) has traversed
            //  up to here. And this behaviour of the IP protocol means
            //  that we cannot figure to avoid loops at transmission-time.
            //    Note: just a tricky workaround we use the MAC of the
            //    header of lower level in collaboration with 'iptables'
            //    'mangle' table -a feature of linux- to avoid that a packet
            //    be forwarded to the same gnode it came from.
            //  The contrary happens in the forwarding of PeerToPeer messages
            //  thanks to the PeerToPeerTracerPacketList object passed to
            //  avoid loops. So, in this particular method, we could
            //  search for only the routes that don't contain
            //  any of the gnodes already touched by the message.
            //  BUT such a behaviour would result in the possibility
            //  to fail the delivery just because the hash-node could
            //  be changing during the message life-span. Considerations?

            // routes are in order of REM.
            ArrayList<AggregatedNeighbour> ret = new ArrayList<AggregatedNeighbour>(AggregatedNeighbour.equal_func);
            foreach (Route r in routes) ret.add(r.gw);
            return ret;
        }

        public Gee.List<int> list_ids(int center, int sign)
        {
            ArrayList<int> ret = new ArrayList<int>();
            for (int i = 0; i < maproute.gsize; i++)
                ret.add((center + maproute.gsize + i * sign) % maproute.gsize);
            return ret;
        }

        /** Returns the destination in hierarchical coordinates of the best
          *  approximation of the passed NIP as a participant to the service.
          * Returns null iff it's me.
          */
        public HCoord? search_participant(NIP hIP, int path_sign=1) throws PeerToPeerError
        {
            int[] H_hIP = new int[maproute.levels];
            for (int i = 0; i < maproute.levels; i++) H_hIP[i] = -1;
            int l;
            bool found = false;
            for (l = maproute.levels - 1; l >= 0; l--)
            {
                foreach (int hid in list_ids(hIP.position_at(l), path_sign))
                {
                    if (is_participant(l, hid))
                    {
                        H_hIP[l] = hid;
                        break;
                    }
                }
                if (H_hIP[l] == -1)
                {
                    throw new PeerToPeerError.GENERIC("No participants.");
                }
                // if it's not in my group, stop here
                if (H_hIP[l] != maproute.me.position_at(l))
                {
                    found = true;
                    break;
                }
            }
            if (! found) return null;
            return new HCoord(l, H_hIP[l]);
        }

        public PartialNIP search_participant_as_nip(NIP hIP, int path_sign=1) throws PeerToPeerError
        {
            int[]positions = maproute.me.get_positions();
            HCoord? lvlpos = search_participant(hIP, path_sign);
            if (lvlpos != null)
            {
                int lvl = lvlpos.lvl;
                int pos = lvlpos.pos;
                positions[lvl] = pos;
                for (int i = lvl-1; i >= 0; i--) positions[i] = -1;
            }
            return new PartialNIP(positions);
        }

        /** Sending of messages **/

        /** Start a tentative to send a message to a certain
          * perfect hip
          */
        public ISerializable msg_send(NIP sender_nip, NIP hip, RemoteCall data) throws Error
        {
            PeerToPeerTracerPacketList peer_to_peer_tpl = new PeerToPeerTracerPacketList();
            ISerializable ret;
            while (true)
            {
                try
                {
                    ret = msg_deliver(peer_to_peer_tpl, sender_nip, hip, data);
                    break;
                }
                catch(RPCError e)
                {
                    if (e is RPCError.NOT_VALID_MAP_YET)
                    {
                        // retry after a while and up to a certain timeout.
                        if (peer_to_peer_tpl.tc.is_expired())
                        {
                            log_warn("PeerToPeer routing: Too long. Giving up.");
                            throw e;
                        }
                        else
                        {
                            log_warn("PeerToPeer routing: Temporary failure. Wait a bit and try again.");
                            ms_wait(1000);
                        }
                    }
                    else
                        throw e;
                }
            }
            return ret;
        }


        /** Avoid loops, neverending stories, and the like.
          *
          * When a message is routed through the net by the peer_to_peer module,
          * we keep track of the path walked, through an instance of PeerToPeerTracerPacketList.
          * There are several functions in module peer_to_peer that try to
          * route messages, such as msg_deliver, find_nearest,
          * number_of_participants, and so on. Each function receives
          * as a parameter a PeerToPeerTracerPacketList instance,
          * calls this method execute (passing its own maproute)
          * and then passes it to the next hop.
          * The first caller instantiate it by passing no parameters or
          * can specify the timeout.
          */
        private static bool not_impl_equal_func (Object? a, Object? b)
        {
            error("PeerToPeerTracerPacketList: equal_func not implemented");
        }
        public static void execute(MapRoute maproute, PeerToPeerTracerPacketList ptptpl) throws RPCError
        {
            // check timeout
            if (ptptpl.tc.is_expired())
            {
                // drop
                throw new RPCError.DROP("PeerToPeer message timed out.");
            }
            if (ptptpl.tpl == null)
            {
                // tpl = [[n] for n in maproute.me]
                ptptpl.tpl = new ArrayList<ArrayList<int>>(not_impl_equal_func);
                foreach (int i in maproute.me.get_positions())
                {
                    ArrayList<int> tpl_1 = new ArrayList<int>();
                    tpl_1.add(i);
                    ptptpl.tpl.add(tpl_1);
                }
            }
            else
            {
                int tplsize = ptptpl.tpl.size;
                // from_nip = [ids[-1] for ids in tpl]
                int[] from_nip = new int[ptptpl.tpl.size];
                for (int i = 0; i < tplsize; i++)
                {
                    ArrayList<int> tpl_1 = ptptpl.tpl.get(i);
                    from_nip[i] = tpl_1.get(tpl_1.size - 1);
                }
                int last_lvl = maproute.nip_cmp(from_nip);
                if (last_lvl > -1)
                {
                    // unchanged = self.tpl[last_lvl:]
                    ArrayList<ArrayList<int>> unchanged = new ArrayList<ArrayList<int>>(not_impl_equal_func);
                    unchanged.add_all(ptptpl.tpl.slice(last_lvl, tplsize));
                    // detect the loop
                    ArrayList<int> last_path = unchanged.get(0);
                    if (last_path.contains(maproute.me.position_at(last_lvl)))
                    {
                        // remove the part that looped to me again
                        int pos = last_path.index_of(maproute.me.position_at(last_lvl));
                        while (last_path.size > pos) last_path.remove_at(pos);
                        ms_wait(2000);
                    }
                    // add my part
                    last_path.add(maproute.me.position_at(last_lvl));
                    // new_part = [[n] for n in maproute.me[:last_lvl]]
                    ArrayList<ArrayList<int>> new_part = new ArrayList<ArrayList<int>>(not_impl_equal_func);
                    for (int k = 0; k < last_lvl; k++)
                    {
                        int ki = maproute.me.get_positions()[k];
                        ArrayList<int> tpl_1 = new ArrayList<int>();
                        tpl_1.add(ki);
                        new_part.add(tpl_1);
                    }
                    // tpl = new_part + unchanged
                    ptptpl.tpl = new_part;
                    ptptpl.tpl.add_all(unchanged);
                }
            }
        }

        /** Delivers a msg to the nearest to hip
          */
        public ISerializable msg_deliver(PeerToPeerTracerPacketList peer_to_peer_tpl, NIP sender_nip, NIP hip, RemoteCall data) throws Error
        {
            if (! has_valid_map)
                throw new RPCError.NOT_VALID_MAP_YET(@"Not a valid map yet for $(this.get_type().name()). Request not valid.");

            // Called by any routing function in module peer_to_peer
            execute(maproute, peer_to_peer_tpl);

            ISerializable ret = new SerializableNone();
            RPCError? last_e = null;
            while (true)
            {
                bool done = false;
                HCoord? lvlpos = search_participant(hip);
                // If the node is exactly me, I execute the message
                if (lvlpos == null)
                    return msg_exec(sender_nip, data);
                else
                {
                    // route
                    int lvl = lvlpos.lvl;
                    int pos = lvlpos.pos;
                    // Obtain a list of possible next hops, instead of just the best one; this way
                    //  if the best next hop is still hooking we can try immediately with another one.
                    Gee.List<AggregatedNeighbour> list_of_n = neighbours_get_from_lvl_pos(lvl, pos);
                    if (!list_of_n.is_empty)
                    {
                        foreach (AggregatedNeighbour n in list_of_n)
                        {
                            try
                            {
                                ret = n.tcp_client.get_peer_to_peer_service(pid).msg_deliver(peer_to_peer_tpl, sender_nip, hip, data);
                                done = true;
                                break;
                            }
                            catch(RPCError e)
                            {
                                if (e is RPCError.NOT_VALID_MAP_YET)
                                {
                                    // retain all information from exception.
                                    last_e = e;
                                }
                                else
                                    throw e;
                            }
                        }
                    }
                    else
                    {
                        last_e = null;
                    }
                    if (done) break;

                    // NOT_VALID_MAP_YET or NoRoutes: retry after a while and up to a certain timeout.
                    if (peer_to_peer_tpl.tc.is_expired())
                    {
                        log_warn("PeerToPeer routing: Too many errors. Giving up.");
                        if (last_e != null) throw last_e;
                        throw new RPCError.GENERIC("Unreachable PeerToPeer destination.");
                    }
                    else
                    {
                        ms_wait(1000);
                    }
                }
            }
            return ret;
        }

        /** Execution of a msg
          */
        public ISerializable msg_exec(NIP sender_nip, RemoteCall data) throws Error
        {
            return _dispatch(null, data);
        }

        /** Search functions for registration with replica. Routed. **/

        public Gee.List<NIP> find_nearest_to_register(NIP hash_nip, int num_dupl, int? inside_gnode_level=null) throws RPCError
        {
            // If 'inside_gnode_level' is None the research is network wide.
            // Otherwise, 'inside_gnode_level' is the level of a gnode containing ourself,
            //  and the research is only inside that gnode. Note that 'hash_nip' might be outside
            //  of that gnode, as well.
            if (inside_gnode_level == null)
            {
                return find_nearest(new PeerToPeerTracerPacketList(), hash_nip, num_dupl, maproute.levels, 0);
            }
            else
            {
                int pos = (inside_gnode_level == maproute.levels) ? 0 : maproute.me.position_at(inside_gnode_level);
                return find_nearest(new PeerToPeerTracerPacketList(), hash_nip, num_dupl, inside_gnode_level, pos);
            }
        }

        public Gee.List<NIP> find_nearest(PeerToPeerTracerPacketList peer_to_peer_tpl, NIP hash_nip, int num_dupl, int lvl, int pos) throws RPCError
        {
            if (! has_valid_map)
                throw new RPCError.NOT_VALID_MAP_YET(@"Not a valid map yet for $(this.get_type().name()). Request not valid.");

            // Called by any routing function in module peer_to_peer
            execute(maproute, peer_to_peer_tpl);

            return inside_find_nearest(peer_to_peer_tpl, hash_nip, num_dupl, lvl, pos);
        }

        public Gee.List<NIP> inside_find_nearest(PeerToPeerTracerPacketList peer_to_peer_tpl, NIP hash_nip, int num_dupl, int lvl, int pos) throws RPCError
        {
            if (num_dupl == 0)
                return new ArrayList<NIP>(PartialNIP.equal_func);
            if (lvl < maproute.levels)
            {
                if (lvl == 0)
                {
                    ArrayList<NIP> ret = new ArrayList<NIP>(PartialNIP.equal_func);
                    if (is_participant(0, pos))
                    {
                        int[] positions = maproute.me.get_positions();
                        positions[0] = pos;
                        ret.add(new NIP(positions));
                    }
                    return ret;
                }
                if (! is_participant(lvl, pos))
                    return new ArrayList<NIP>(PartialNIP.equal_func);
            }
            // (lvl, pos) is a participant g-node. To know the remaining 
            // num_dupl nodes nearest to hash_nip and participating inside it,
            // if I am in this g-node I do a
            // step down in the level, otherwise I have to route the
            // question to the g-node.
            if (lvl == maproute.levels || maproute.me.position_at(lvl) == pos)
            {
                // down one level
                ArrayList<NIP> sequence = new ArrayList<NIP>(PartialNIP.equal_func);
                int new_lvl = lvl - 1;
                foreach (int new_pos in list_ids(hash_nip.position_at(new_lvl), 1))
                {
                    Gee.List<NIP> adding = inside_find_nearest(peer_to_peer_tpl, hash_nip, num_dupl-sequence.size, new_lvl, new_pos);
                    sequence.add_all(adding);
                    if (sequence.size >= num_dupl)
                        break;
                }
                return sequence;
            }
            else
            {
                // route the request
                Gee.List<NIP> ret = new ArrayList<NIP>(PartialNIP.equal_func);
                while (true)
                {
                    RPCError? last_e = null;
                    bool done = false;
                    Gee.List<AggregatedNeighbour> list_of_n = neighbours_get_from_lvl_pos(lvl, pos);
                    if (!list_of_n.is_empty)
                    {
                        foreach (AggregatedNeighbour n in list_of_n)
                        {
                            try
                            {
                                ret = n.tcp_client.get_peer_to_peer_service(pid).find_nearest(peer_to_peer_tpl, hash_nip, num_dupl, lvl, pos);
                                done = true;
                                break;
                            }
                            catch(RPCError e)
                            {
                                if (e is RPCError.NOT_VALID_MAP_YET)
                                {
                                    // retain all information from exception.
                                    last_e = e;
                                }
                                else
                                    throw e;
                            }
                        }
                    }
                    else
                    {
                        last_e = null;
                    }
                    if (done) break;

                    // NOT_VALID_MAP_YET or NoRoutes: retry after a while and up to a certain timeout.
                    if (peer_to_peer_tpl.tc.is_expired())
                    {
                        log_warn("PeerToPeer routing: Too many errors. Giving up.");
                        if (last_e != null) throw last_e;
                        throw new RPCError.GENERIC("Unreachable PeerToPeer destination.");
                    }
                    else
                    {
                        ms_wait(1000);
                    }
                }
                string nips = "";
                string nips_next = "";
                foreach (NIP one_nip in ret)
                {
                    nips += nips_next + @"$(one_nip)";
                    nips_next = ", ";
                }
                return ret;
            }
        }

        /** Search functions for hooking phase. Routed. **/

        public int get_number_of_participants(int lvl, int pos, int timeout) throws RPCError
        {
            return number_of_participants(new PeerToPeerTracerPacketList(timeout), lvl, pos);
        }

        public int number_of_participants(PeerToPeerTracerPacketList peer_to_peer_tpl, int lvl, int pos) throws RPCError
        {
            if (! has_valid_map)
                throw new RPCError.NOT_VALID_MAP_YET(@"Not a valid map yet for $(this.get_type().name()). Request not valid.");

            // Called by any routing function in module peer_to_peer
            execute(maproute, peer_to_peer_tpl);

            int ret = inside_number_of_participants(peer_to_peer_tpl, lvl, pos);
            return ret;
        }

        public int inside_number_of_participants(PeerToPeerTracerPacketList peer_to_peer_tpl, int lvl, int pos) throws RPCError
        {
            if (lvl == 0)
            {
                if (is_participant(0, pos)) return 1;
                return 0;
            }
            if (! is_participant(lvl, pos)) return 0;
            // (lvl, pos) is a participant g-node. To know the number of
            // nodes participating inside it, if I am in this g-node I do a
            // step down in the level, otherwise I have to route the
            // question to the g-node.
            if (maproute.me.position_at(lvl) == pos)
            {
                // down one level
                int new_lvl = lvl - 1;
                int ret = 0;
                for (int new_pos = 0; new_pos < maproute.gsize; new_pos++)
                {
                    ret += inside_number_of_participants(peer_to_peer_tpl, new_lvl, new_pos);
                }
                return ret;
            }
            else
            {
                // route the request
                int ret = 0;
                while (true)
                {
                    RPCError? last_e = null;
                    bool done = false;
                    Gee.List<AggregatedNeighbour> list_of_n = neighbours_get_from_lvl_pos(lvl, pos);
                    // The list of neighbours to use in order of efficiency.
                    if (!list_of_n.is_empty)
                    {
                        foreach (AggregatedNeighbour n in list_of_n)
                        {
                            try
                            {
                                ret = n.tcp_client.get_peer_to_peer_service(pid).number_of_participants(peer_to_peer_tpl, lvl, pos);
                                done = true;
                                break;
                            }
                            catch(RPCError e)
                            {
                                if (e is RPCError.NOT_VALID_MAP_YET)
                                {
                                    // retain all information from exception.
                                    last_e = e;
                                }
                                else
                                    throw e;
                            }
                        }
                    }
                    else
                    {
                        last_e = null;
                    }
                    if (done) break;

                    // NOT_VALID_MAP_YET or NoRoutes: retry after a while and up to a certain timeout.
                    if (peer_to_peer_tpl.tc.is_expired())
                    {
                        log_warn("PeerToPeer routing: Too many errors. Giving up.");
                        if (last_e != null) throw last_e;
                        throw new RPCError.GENERIC("Unreachable PeerToPeer destination.");
                    }
                    else
                    {
                        ms_wait(1000);
                    }
                }
                return ret;
            }
        }

        /** Helper functions for hooking phase. **/

        public void find_hook_peers(out int? ret_first_forward,
                                    out int? ret_first_back,
                                    out int? ret_last_back,
                                    int lvl, int num_dupl, int timeout=120000)
        {
            ret_first_forward = null;
            ret_first_back = null;
            ret_last_back = null;
            // check positions from me forward
            Gee.List<int> ids_me_forward = list_ids(maproute.me.position_at(lvl), 1);
            ids_me_forward.remove_at(0);
            foreach (int _id in ids_me_forward)
            {
                int num = 0;
                try {
                    num = get_number_of_participants(lvl, _id, timeout);
                }
                catch (RPCError e) {}
                if (num > 0)
                {
                    ret_first_forward = _id;
                    break;
                }
            }
            // Maybe no nodes at all participate. It would return 3 null.
            // Otherwise...
            if (ret_first_forward != null)
            {
                // check positions from me back
                Gee.List<int> ids_me_back = list_ids(maproute.me.position_at(lvl), -1);
                ids_me_back.remove_at(0);
                int remaining = num_dupl;
                foreach (int _id in ids_me_back)
                {
                    int num = 0;
                    try {
                        num = get_number_of_participants(lvl, _id, timeout);
                    }
                    catch (RPCError e) {}
                    if (num > 0 && ret_first_back == null) ret_first_back = _id;
                    remaining -= num;
                    if (remaining <= 0)
                    {
                        ret_last_back = _id;
                        break;
                    }
                }
                if (remaining > 0)
                {
                    // not enough nodes participate.
                    ret_first_back = null;
                    // I will reach the first forward and be a replica for any
                    // record that it has.
                }
            }
            // One possible situation is that ret_first_back == ret_last_back
            // This would mean that that gnode has a number of participants
            // greater than num_dupl.
        }
    }

    public abstract class RmtPeer : Object
    {
        private PeerToPeer peer_to_peer_service;
        private Object? key;
        private NIP? hIP;
        private AggregatedNeighbour? aggregated_neighbour;

        public RmtPeer(PeerToPeer peer_to_peer_service, Object? key=null, NIP? hIP=null, AggregatedNeighbour? aggregated_neighbour=null)
        {
            this.peer_to_peer_service = peer_to_peer_service;
            this.key = key;
            this.hIP = hIP;
            this.aggregated_neighbour = aggregated_neighbour;
        }

        public void evaluate_hash_nip() throws PeerToPeerError
        {
            if (hIP == null)
                hIP = peer_to_peer_service.h(key);
            if (hIP == null)
            {
                int pid = peer_to_peer_service.pid;
                throw new PeerToPeerError.GENERIC(@"PeerToPeer # $pid: key does not map to a IP.");
            }
        }

        public NIP get_hash_nip() throws PeerToPeerError
        {
            evaluate_hash_nip();
            return hIP;
        }

        /** Overridable. Similar to FakeRmt.rmt()
          */
        public virtual ISerializable rmt(RemoteCall data) throws Error
        {
            evaluate_hash_nip();
            if (aggregated_neighbour != null)
            {
                // We are requested to use this one as first hop via UDP.
                AddressManagerFakeRmt nclient = (AddressManagerFakeRmt)aggregated_neighbour.neighbour_client;
                IPeerToPeer caller = nclient.get_peer_to_peer_service(peer_to_peer_service.pid);
                return caller.msg_send(peer_to_peer_service.maproute.me, hIP, data);
            }
            else
            {
                // Use TCP version.
                return peer_to_peer_service.msg_send(peer_to_peer_service.maproute.me, hIP, data);
            }
        } 
    }

    struct struct_helper_OptionalPeerToPeer_participant_set
    {
        public OptionalPeerToPeer self;
        public NIP nip;
        public int lvl;
        public int pos;
        public ParticipantNode participant_node;
    }

    struct struct_helper_OptionalPeerToPeer_participant_refresh
    {
        public OptionalPeerToPeer self;
        public NIP nip;
        public PackedParticipantNodes packed_nodes;
    }

    struct struct_helper_OptionalPeerToPeer_send_refresh_periodically
    {
        public OptionalPeerToPeer self;
    }

    /** This is the class that must be inherited to create a PeerToPeer module.
      * A PeerToPeer service which is not Strict (optional) has these features:
      *  * It is not mandatory that all the nodes of the network know about
      *    this service. Hence, a node may not even have the code of this service
      *    nor know that the service-id number is in use.
      *    Nevertheless, every node will participate to the routing of messages
      *    towards the nodes participating to the service.
      *  * A node might have the code necessary to participate to the service
      *    and so be able to serve requests. A node might have the code necessary
      *    to be a client of the service and so be able to be served. It is not
      *    necessary, however, to split the code: usually a node has all the code
      *    or nothing.
      *  * A node that knows about a service (has the code) might participate to
      *    the service or not. Even if it does not participate, it is able to
      *    make requests to the service and the nodes that participate will be
      *    able to answer.
      */
    public class OptionalPeerToPeer : PeerToPeer, IOptionalPeerToPeer
    {
        public MapPeerToPeer map_peer_to_peer;
        public bool will_participate;
        private Tasklet? send_refresh_handle;

        public OptionalPeerToPeer(AggregatedNeighbourManager aggregated_neighbour_manager,
                           MapRoute maproute, int pid)
        {
            base(aggregated_neighbour_manager, maproute, pid);
            map_peer_to_peer = new MapPeerToPeer(maproute.levels, maproute.gsize,
                                maproute.me, pid);
            will_participate = false;
            maproute.node_deleted.connect(map_peer_to_peer.node_dead);
            send_refresh_handle = null;
        }

        public ParticipantNode get_my_node(int lvl)
        {
            return map_peer_to_peer.node_get(lvl, maproute.me.position_at(lvl));
        }

        private void impl_participant_set(NIP nip, int lvl, int pos, ParticipantNode participant_node) throws Error
        {
            Tasklet.declare_self("OptionalPeerToPeer.participant_set");
            // Received an information about participation of a certain
            // gnode. Handle our MapPeerToPeer instance.
            // nip: caller
            // lvl, pos: gnode as seen in the map of caller
            // participant_node: information itself
            bool participant = participant_node.participant;
            TimeCapsule tc = participant_node.tc;
            int commonlvl = maproute.nip_cmp(nip.get_positions());
            // I should not receive a request from myself!
            if (commonlvl == -1) return;
            if (lvl < commonlvl)
            {
                // this information is from a gnode that we do not see in our map,
                //  but if it is a participation then it is a valid information
                //  at the common-level.
                if (participant)
                {
                    lvl = commonlvl;
                    pos = nip.position_at(lvl);
                }
                else
                    return;
            }
            // do not process info for myself.
            if (maproute.me.position_at(lvl) == pos) return;
            ParticipantNode node = map_peer_to_peer.node_get(lvl, pos); // previous state
            RouteNode node_route = maproute.node_get(lvl, pos);
            bool change = participant != node.participant; // there was a change?
            if (! change)
            {
                if (tc.is_younger(node.tc))
                {
                    // the information received is younger than the one we had. refresh.
                    node.tc = tc;
                }
                return;
            }
            if (! node_route.is_free() && tc.is_younger(node.tc))
            {
                // remember my previous positions
                bool[] old_myself = new bool[maproute.levels];
                for (int l = 0; l < maproute.levels; l++) old_myself[l] = get_my_node(l).participant;
                node.participant = participant;
                node.tc = tc;
                // emit event
                map_peer_to_peer.check_node(lvl, pos);
                AddressManagerFakeRmt bclient = maproute.address_manager.get_broadcast_client();
                IOptionalPeerToPeer service = bclient.get_optional_peer_to_peer_service(pid);
                service.participant_set(maproute.me, lvl, pos, node);
                // check for changes in my positions
                for (int l = 0; l < maproute.levels; l++)
                {
                    if (! old_myself[l] && get_my_node(l).participant)
                    {
                        // I became participant at level l, therefore also at upper levels.
                        // I have to send this info at level l and then I can terminate.
                        service.participant_set(maproute.me,
                                    l,
                                    maproute.me.position_at(l),
                                    get_my_node(l)); // True, TimeCapsule(0)
                        return;
                    }
                    if (old_myself[l] && ! get_my_node(l).participant)
                    {
                        // I became non-participant at level l.
                        // I have to send this info at level l.
                        // Then go on to look for other changes at upper levels.
                        service.participant_set(maproute.me,
                                    l,
                                    maproute.me.position_at(l),
                                    get_my_node(l)); // False, TimeCapsule(0)
                    }
                }
            }
        }

        private static void * helper_participant_set(void *v) throws Error
        {
            struct_helper_OptionalPeerToPeer_participant_set *tuple_p = (struct_helper_OptionalPeerToPeer_participant_set *)v;
            // The caller function has to add a reference to the ref-counted instances
            OptionalPeerToPeer self_save = tuple_p->self;
            NIP nip_save = tuple_p->nip;
            ParticipantNode participant_node_save = tuple_p->participant_node;
            // The caller function has to copy the value of byvalue parameters
            int lvl_save = tuple_p->lvl;
            int pos_save = tuple_p->pos;
            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            self_save.impl_participant_set(nip_save, lvl_save, pos_save, participant_node_save);
            // void method, return null
            return null;
        }

        public void participant_set(NIP nip, int lvl, int pos, ParticipantNode participant_node)
        {
            struct_helper_OptionalPeerToPeer_participant_set arg = struct_helper_OptionalPeerToPeer_participant_set();
            arg.self = this;
            arg.nip = nip;
            arg.lvl = lvl;
            arg.pos = pos;
            arg.participant_node = participant_node;
            Tasklet.spawn((Spawnable)helper_participant_set, &arg);
        }

        private void impl_participant_refresh(NIP nip, PackedParticipantNodes packed_nodes) throws Error
        {
            Tasklet.declare_self("OptionalPeerToPeer.participant_refresh");
            // Received a packed MapPeerToPeer from a neighbour. See if
            // we lost some fresh information in the past.
            // nip: caller
            // packed_nodes.packednodes.get(lvl).get(pos): a ParticipantNode instance in caller's map
            int lvl = 0;
            foreach (ArrayList<ParticipantNode> list_lvl in packed_nodes.packednodes)
            {
                int pos = 0;
                foreach (ParticipantNode lvl_pos in list_lvl)
                {
                    participant_set(nip, lvl, pos, lvl_pos);
                    pos++;
                }
                lvl++;
            }
        }

        private static void * helper_participant_refresh(void *v) throws Error
        {
            struct_helper_OptionalPeerToPeer_participant_refresh *tuple_p = (struct_helper_OptionalPeerToPeer_participant_refresh *)v;
            // The caller function has to add a reference to the ref-counted instances
            OptionalPeerToPeer self_save = tuple_p->self;
            NIP nip_save = tuple_p->nip;
            PackedParticipantNodes packed_nodes_save = tuple_p->packed_nodes;
            // The caller function has to copy the value of byvalue parameters

            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            self_save.impl_participant_refresh(nip_save, packed_nodes_save);
            // void method, return null
            return null;
        }

        public void participant_refresh(NIP nip, PackedParticipantNodes packed_nodes)
        {
            struct_helper_OptionalPeerToPeer_participant_refresh arg = struct_helper_OptionalPeerToPeer_participant_refresh();
            arg.self = this;
            arg.nip = nip;
            arg.packed_nodes = packed_nodes;
            Tasklet.spawn((Spawnable)helper_participant_refresh, &arg);
        }

        private void impl_send_refresh_periodically() throws Error
        {
            Tasklet.declare_self("OptionalPeerToPeer.send_refresh_periodically");
            send_refresh_handle = Tasklet.self();
            while (true)
            {
                ms_wait(60000);
                AddressManagerFakeRmt bclient = maproute.address_manager.get_broadcast_client();
                bclient.get_optional_peer_to_peer_service(pid)
                        .participant_refresh(maproute.me, map_peer_to_peer.get_packed_nodes());
            }
        }

        private static void * helper_send_refresh_periodically(void *v) throws Error
        {
            struct_helper_OptionalPeerToPeer_send_refresh_periodically *tuple_p = (struct_helper_OptionalPeerToPeer_send_refresh_periodically *)v;
            // The caller function has to add a reference to the ref-counted instances
            OptionalPeerToPeer self_save = tuple_p->self;
            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            self_save.impl_send_refresh_periodically();
            // void method, return null
            return null;
        }

        public void send_refresh_periodically()
        {
            struct_helper_OptionalPeerToPeer_send_refresh_periodically arg = struct_helper_OptionalPeerToPeer_send_refresh_periodically();
            arg.self = this;
            Tasklet.spawn((Spawnable)helper_send_refresh_periodically, &arg);
        }

        /** Returns True iff the node lvl, pos is participating
          * to the service.
          */
        public override bool is_participant(int lvl, int pos)
        {
            return map_peer_to_peer.node_get(lvl, pos).participant;
        }

        public bool participant {
            get {
                return get_my_node(0).participant;
            }
        }

        // TODO Verify if the algorithm (which has been more or less implemented by temptatives)
        //      actually reflects the proposed solution in chapter 2.1 of "P2P over Netsukuku NTK RFC 0014"
        //      Find it in http://bzr.savannah.gnu.org/lh/netsukuku/original/files/head:/trunk/doc/main_doc/ntk_rfc/
        //      Ntk_p2p_over_ntk.pdf

        /** Become a participant node.
          */
        public void participate()
        {
            try
            {
                if (! participant)
                {
                    map_peer_to_peer.participate();
                    // I change state, so I must communicate my participation at level 0.
                    //  Upper levels are automatically notified.
                    AddressManagerFakeRmt bclient = maproute.address_manager.get_broadcast_client();
                    bclient.get_optional_peer_to_peer_service(pid)
                            .participant_set(maproute.me, 0, maproute.me.position_at(0), get_my_node(0));
                }
            }
            catch (RPCError ne) {}
        }

        /** Go outside and don't participate to the service.
          */
        public void sit_out()
        {
            try
            {
                if (participant)
                {
                    ArrayList<int> changes = new ArrayList<int>();
                    for (int lvl = 0; lvl < maproute.levels; lvl++)
                    {
                        if (is_participant(lvl, maproute.me.position_at(lvl)))
                        {
                            changes.add(lvl);
                        }
                    }
                    map_peer_to_peer.sit_out();
                    // I change state, so I must communicate my non-participation at level 0 AND
                    //  at any level this change applies.
                    AddressManagerFakeRmt bclient = maproute.address_manager.get_broadcast_client();
                    foreach (int lvl in changes)
                    {
                        bclient.get_optional_peer_to_peer_service(pid)
                                .participant_set(maproute.me, lvl, maproute.me.position_at(lvl), get_my_node(lvl));
                    }
                }
            }
            catch (RPCError ne) {}
        }

        public override void start_operations()
        {
            base.start_operations();
            send_refresh_periodically();
        }

        public override void stop_operations()
        {
            base.stop_operations();
            if (participant) sit_out();
            if (send_refresh_handle != null)
            {
                send_refresh_handle.abort();
                send_refresh_handle = null;
            }
            map_peer_to_peer.stop_operations();
        }

        internal string log_service()
        {
            return @"[$(pid), $(map_peer_to_peer.log_service())]";
        }

#if testing
        public void tester_impl_participant_set(NIP nip, int lvl, int pos, ParticipantNode participant_node) throws Error
        {
            impl_participant_set(nip, lvl, pos, participant_node);
        }
#endif
    }

    struct struct_helper_PeerToPeerAll_start_operations
    {
        public PeerToPeerAll self;
    }

    /** Class of all the registered peer_to_peer services
      */
    public class PeerToPeerAll : Object, IPeerToPeerAll
    {
        public AggregatedNeighbourManager aggregated_neighbour_manager {get; private set;}
        public MapRoute maproute {get; private set;}
        private bool left_all_optional_peer_to_peer;
        private HashMap<int, PeerToPeer> service;

        public PeerToPeerAll(AggregatedNeighbourManager aggregated_neighbour_manager, MapRoute maproute)
        {
            this.aggregated_neighbour_manager = aggregated_neighbour_manager;
            this.maproute = maproute;
            left_all_optional_peer_to_peer = false;
            service = new HashMap<int, PeerToPeer>();
        }

        public PeerToPeer pid_add(int pid)
        {
            service[pid] = new OptionalPeerToPeer(aggregated_neighbour_manager, maproute, pid);
            return service[pid];
        }

        public PeerToPeer pid_get(int pid)
        {
            if (service.has_key(pid)) return service[pid];
            return pid_add(pid);
        }

        public void pid_del(int pid)
        {
            if (service.has_key(pid)) service.unset(pid);
        }

        /** Used to add for the first time a PeerToPeer instance of a module in the
          * PeerToPeerAll dictionary.
          */
        public virtual void peer_to_peer_register(PeerToPeer peer_to_peer) throws PeerToPeerError
        {
            /*
             * Both PeerToPeer and OptionalPeerToPeer services register themself in PeerToPeerAll, via
             *  this function.
             * The registration of an OptionalPeerToPeer service might occur after the
             *  start_operations of PeerToPeerAll has completed. In this case the instance created from
             *  the class of the new module (here passed in parameter 'peer_to_peer') is
             *  going to replace a possibly existent instance of base class
             *  OptionalPeerToPeer. In this case we will replace the member variable
             *  map_peer_to_peer of the new instance with the one from the old instance.
             * If the inheriting class had to handle its map_peer_to_peer in a different
             *  manner, it has to consider this, and instantiate it after the
             *  call of this function.
             */
            if (service.has_key(peer_to_peer.pid))
            {
                // If it is a strict service, it cannot be registered twice.
                if (! peer_to_peer.get_type().is_a(typeof(OptionalPeerToPeer)))
                {
                    throw new PeerToPeerError.REGISTER("Strict services have to be registered once.");
                }
                OptionalPeerToPeer optional_peer_to_peer = (OptionalPeerToPeer)peer_to_peer;
                // Already registered by a base_class placeholder?
                PeerToPeer pre = pid_get(peer_to_peer.pid);
                if (peer_to_peer.get_type() != typeof(OptionalPeerToPeer))
                {
                    // conflicting pid
                    log_error(@"$(peer_to_peer.get_type().name()) is trying to use" +
                                @" pid $(peer_to_peer.pid) but it is used by $(pre.get_type().name())");
                    throw new PeerToPeerError.REGISTER("Optional services have to use unique IDs.");
                }
                OptionalPeerToPeer optional_pre = (OptionalPeerToPeer)pre;
                optional_peer_to_peer.map_peer_to_peer = optional_pre.map_peer_to_peer;
            }
            service[peer_to_peer.pid] = peer_to_peer;
        }

        public void leave_all_optional_peer_to_peer()
        {
            foreach (PeerToPeer peer_to_peer in service.values)
            {
                if (peer_to_peer.get_type().is_a(typeof(OptionalPeerToPeer)))
                {
                    OptionalPeerToPeer optional_peer_to_peer = (OptionalPeerToPeer)peer_to_peer;
                    optional_peer_to_peer.will_participate = false;
                    optional_peer_to_peer.sit_out();
                }
            }
            left_all_optional_peer_to_peer = true;
        }

        private static bool not_impl_equal_func (Object? a, Object? b)
        {
            error("PeerToPeerAll.get_optional_participants: equal_func not implemented");
        }
        public SetOptionalServiceParticipants get_optional_participants()
        {
            ArrayList<OptionalServiceParticipants> l_osp = new ArrayList<OptionalServiceParticipants>(not_impl_equal_func);
            foreach (PeerToPeer peer_to_peer in service.values)
            {
                if (peer_to_peer.get_type().is_a(typeof(OptionalPeerToPeer)))
                {
                    OptionalPeerToPeer optional_peer_to_peer = (OptionalPeerToPeer)peer_to_peer;
                    OptionalServiceParticipants osp = new OptionalServiceParticipants();
                    osp.pid = optional_peer_to_peer.pid;
                    osp.ppnodes = optional_peer_to_peer.map_peer_to_peer.get_packed_nodes();
                    l_osp.add(osp);
                }
            }
            return new SetOptionalServiceParticipants(l_osp);
        }

        /** It gets the peer_to_peer maps from our nearest neighbour.
          */
        public void retrieve_optional_services_and_participants()
        {
            Gee.List<AggregatedNeighbour> neighbours_in_net = aggregated_neighbour_manager.neighbour_list(true);
            bool got_answer = false;
            bool try_again = true;
            SetOptionalServiceParticipants? nr_optional_participants = null;
            AggregatedNeighbour? minnr = null;
            while (! got_answer && try_again)
            {
                // Find our nearest neighbour
                minnr = null;
                int minlvl = maproute.levels;
                foreach (AggregatedNeighbour nr in neighbours_in_net)
                {
                    int lvl = maproute.nip_cmp(nr.nip.get_positions());
                    if (lvl < minlvl)
                    {
                        minlvl = lvl;
                        minnr  = nr;
                    }
                }
                if (minnr == null)
                {
                    // nothing to do
                    try_again = false;
                }
                else
                {
                    try
                    {
                        nr_optional_participants = minnr.tcp_client
                                .peer_to_peer_all.get_optional_participants();
                        got_answer = true;
                    }
                    catch (Error e)
                    {
                        log_warn("PeerToPeerAll retrieve_optional_services_and_participants: " + 
                                        @"Asking to $(minnr.nip) failed: got $(e.domain) code $(e.code): $(e.message)");
                        neighbours_in_net.remove(minnr);
                    }
                }
            }
            if (got_answer)
            {
                foreach (OptionalServiceParticipants osp in nr_optional_participants.o_s_participants)
                {
                    ((OptionalPeerToPeer)pid_get(osp.pid)).map_peer_to_peer
                            .initialize_from_neighbour(minnr.nip, osp.ppnodes);
                }
            }
        }

        private string log_services()
        {
            string ret = "{";
            foreach (PeerToPeer peer_to_peer in service.values)
            {
                if (peer_to_peer.get_type().is_a(typeof(OptionalPeerToPeer)))
                {
                    OptionalPeerToPeer optional_peer_to_peer = (OptionalPeerToPeer)peer_to_peer;
                    ret += optional_peer_to_peer.log_service();
                    ret += ", ";
                }
            }
            ret += "}";
            return ret;
        }

        /** Start operations on various peer_to_peer services.
          */
        private void impl_start_operations() throws Error
        {
            Tasklet.declare_self("PeerToPeerAll.start_operations");
            // This method il called after IS_MATURE
            // So, a strict peer_to_peer has immediately a valid map.
            foreach (PeerToPeer peer_to_peer in service.values)
            {
                if (! peer_to_peer.get_type().is_a(typeof(OptionalPeerToPeer)))
                {
                    peer_to_peer.start_operations();
                }
            }
            // Retrieve valid maps.
            retrieve_optional_services_and_participants();
            // Now, a optional peer_to_peer has a valid map.
            // The fact that this node is participating itself, instead, is delayed up to a
            //  moment of choice of the specific service.
            //  Eg the andna module will call self.participate after the
            //  completion of andna_hook.
            foreach (PeerToPeer peer_to_peer in service.values)
            {
                if (peer_to_peer.get_type().is_a(typeof(OptionalPeerToPeer)))
                {
                    peer_to_peer.start_operations();
                }
            }
        }

        private static void * helper_start_operations(void *v) throws Error
        {
            struct_helper_PeerToPeerAll_start_operations *tuple_p = (struct_helper_PeerToPeerAll_start_operations *)v;
            // The caller function has to add a reference to the ref-counted instances
            PeerToPeerAll self_save = tuple_p->self;
            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            self_save.impl_start_operations();
            // void method, return null
            return null;
        }

        public void start_operations()
        {
            struct_helper_PeerToPeerAll_start_operations arg = struct_helper_PeerToPeerAll_start_operations();
            arg.self = this;
            Tasklet.spawn((Spawnable)helper_start_operations, &arg);
        }

        public void stop_operations()
        {
            foreach (PeerToPeer peer_to_peer in service.values)
            {
                peer_to_peer.stop_operations();
            }
        }
    }
}

