/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2011 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

/*   peerbuilder.in
errors:
 HookingError
  INEXISTENT_GNODE
  GENERIC
 QspnError
  NOT_YOUR_GATEWAY
  ALREADY_UP_TO_DATE
  GENERIC
 PeerRefuseServiceError
  GENERIC
 TunnelError
  GENERIC
 BorderNodesError
  WRONG_GNODE
  NOT_BORDER_NODE
  WILL_NOT_TUNNEL
  TIMEOUT
  GENERIC
 AndnaError
  GENERIC
serializables:
 TimeCapsule
 ParticipantNode
 PackedParticipantNodes
 OptionalServiceParticipants
 SetOptionalServiceParticipants
 PeerToPeerTracerPacketList
 PairNipDistance
 BnodeRecord
 BnodeList
 PairLvlNumberOfFreeNodes
 HCoord
 PartialNIP
 NIP
 REM
 NullREM
 DeadREM
 AlmostDeadREM
 RTT
 TracerPacketList
 RouteInSet
 PositionInRoutesSetPerLevel
 RoutesSetPerLevel
 RoutesSet
 ExtendedTracerPacket
 GNodeID
 NetworkID
 InfoNeighbour
 InfoRoute
 InfoNode
 QspnStats
 InfoBorderNode
 PublicKey
 AndnaServiceKey
 AndnaServerRecord
 AndnaDomainRecord
 AndnaServer
 AndnaServers
 RegisterHostnameArguments
 CounterNipRecord
 CounterSetDataResponse
 CounterCheckHostnameResponse
 AndnaConfirmPubkResponse
 AndnaRegisterMainResponse
 AndnaRegisterSpreadResponse
 AndnaGetServersResponse
 AndnaGetRegistrarResponse
 AndnaGetCacheRecordsResponse
 CoordinatorBooking
 CoordinatorMemory
 CoordinatorReservation
 CoordinatorGetCacheRecordsResponse
 BroadcastID
 UnicastID
peers:
 Coordinator
  methods:
   CoordinatorReservation reserve
    arguments:
     PartialNIP gnode
    throws:
     PeerRefuseServiceError
     HookingError
   void backup_memory
    arguments:
     CoordinatorMemory mem
    throws:
   CoordinatorGetCacheRecordsResponse get_cache_records
    arguments:
    throws:
     PeerRefuseServiceError
*/

using Gee;
using zcd;
using Tasklets;

namespace Netsukuku
{

    public interface ICoordinatorAsPeer : Object
    {
        public abstract  CoordinatorReservation
                         reserve
                         (PartialNIP gnode)
                         throws RPCError, PeerRefuseServiceError, HookingError;
        public abstract  void
                         backup_memory
                         (CoordinatorMemory mem)
                         throws RPCError;
        public abstract  CoordinatorGetCacheRecordsResponse
                         get_cache_records ()
                         throws RPCError, PeerRefuseServiceError;
    }

    public class RmtCoordinatorPeer : RmtPeer, ICoordinatorAsPeer
    {
        public RmtCoordinatorPeer
               (PeerToPeer peer_to_peer_service,
                Object? key=null,
                NIP? hIP=null,
                AggregatedNeighbour? aggregated_neighbour=null)
        {
            base(peer_to_peer_service, key, hIP, aggregated_neighbour);
        }

        public  CoordinatorReservation
                reserve
                (PartialNIP gnode)
                throws RPCError, PeerRefuseServiceError, HookingError
        {
            RemoteCall rc = new RemoteCall();
            rc.method_name = "reserve";
            rc.add_parameter(gnode);
            try {
                return (CoordinatorReservation)
                    filter_exception(
                    filter_exception_PeerRefuseServiceError(
                    filter_exception_HookingError(
                    this.rmt(rc)))
                );
            }
            catch (RPCError e) {throw e;}
            catch (PeerRefuseServiceError e) {throw e;}
            catch (HookingError e) {throw e;}
            catch (Error e)
            {
                throw new RPCError.GENERIC
                    (@"Unexpected error $(e.domain).$(e.code) '$(e.message)'");
            }
        }

        public  void
                backup_memory
                (CoordinatorMemory mem)
                throws RPCError
        {
            RemoteCall rc = new RemoteCall();
            rc.method_name = "backup_memory";
            rc.add_parameter(mem);
            try {
                filter_exception(
                    this.rmt(rc)
                );
            }
            catch (RPCError e) {throw e;}
            catch (Error e)
            {
                throw new RPCError.GENERIC
                    (@"Unexpected error $(e.domain).$(e.code) '$(e.message)'");
            }
        }

        public  CoordinatorGetCacheRecordsResponse
                get_cache_records ()
                throws RPCError, PeerRefuseServiceError
        {
            RemoteCall rc = new RemoteCall();
            rc.method_name = "get_cache_records";
            try {
                return (CoordinatorGetCacheRecordsResponse)
                    filter_exception(
                    filter_exception_PeerRefuseServiceError(
                    this.rmt(rc))
                );
            }
            catch (RPCError e) {throw e;}
            catch (PeerRefuseServiceError e) {throw e;}
            catch (Error e)
            {
                throw new RPCError.GENERIC
                    (@"Unexpected error $(e.domain).$(e.code) '$(e.message)'");
            }
        }

    }

    public class Coordinator : PeerToPeer, ICoordinatorAsPeer, ICoordinator
    {
        public static const int mypid = 1;
        public static const int DUPLICATION = 10;
        private PeerToPeerAll peer_to_peer_all;
        public bool memory_initialized;
        public signal void coordinator_ready();

        private ArrayList<LinkedList<CoordinatorBooking>> bookings;
        private ArrayList<int> last_assigned_elderships;

        public Coordinator(AggregatedNeighbourManager aggregated_neighbour_manager,
                           MapRoute maproute,
                           PeerToPeerAll peer_to_peer_all)
        {
            base(aggregated_neighbour_manager, maproute, Coordinator.mypid);

            // let's register ourself in peer_to_peer_all
            try
            {
                peer_to_peer_all.peer_to_peer_register(this);
            }
            catch (PeerToPeerError e)
            {
                error(@"Coordinator service: registering failed: $(e.message)");
            }

            empty_cache();

            // Until I have my memory initialized I don't want to answer to requests.
            memory_initialized = false;
            // Start the init_memory phase of the peer-to-peer service when my map is valid.
            this.map_peer_to_peer_validated.connect(init_memory);
        }

        void empty_cache()
        {
            bookings = new ArrayList<LinkedList<CoordinatorBooking>>();
            last_assigned_elderships = new ArrayList<int>();
            for (int i = 0; i < maproute.levels; i++)
            {
                bookings.add(new LinkedList<CoordinatorBooking>(CoordinatorBooking.equal_func));
                last_assigned_elderships.add(0);
            }
        }

        void update_cache()
        {
            for (int i = 0; i < maproute.levels; i++)
            {
                ArrayList<CoordinatorBooking> to_del = new ArrayList<CoordinatorBooking>();
                foreach (CoordinatorBooking b in bookings[i])
                {
                    if (b.tc_expire.is_expired())
                    {
                        to_del.add(b);
                    }
                }
                foreach (CoordinatorBooking b in to_del)
                {
                    bookings[i].remove(b);
                }
            }
        }

        bool bookings_contains(int level_of_nodes, int pos)
        {
            bool found = false;
            foreach (CoordinatorBooking b0 in bookings[level_of_nodes])
            {
                if (b0.pos == pos)
                {
                    found = true;
                    break;
                }
            }
            return found;
        }

        CoordinatorMemory produce_memory()
        {
            update_cache();
            CoordinatorMemory ret = new CoordinatorMemory(maproute.me);
            for (int l = 0; l < maproute.levels; l++)
            {
                PartialNIP gnode = maproute.me.get_gnode_at_level(l+1);
                ret.set_gnode(gnode, last_assigned_elderships[l], bookings[l]);
            }
            return ret;
        }

        void reproduce_memory(CoordinatorMemory mem)
        {
            foreach (PartialNIP gnode in mem.keys)
            {
                if (maproute.me.belongs_to(gnode))
                {
                    int l = gnode.level_of_gnode() -1;
                    int new_last_assigned_eldership = mem.get_last_assigned_eldership(gnode);
                    if (last_assigned_elderships[l] < new_last_assigned_eldership)
                        last_assigned_elderships[l] = new_last_assigned_eldership;
                    foreach (CoordinatorBooking b in mem.get_bookings(gnode))
                    {
                        bool found = false;
                        foreach (CoordinatorBooking b0 in bookings[l])
                        {
                            if (b0.pos == b.pos)
                            {
                                found = true;
                                if (b0.tc_expire.is_younger(b.tc_expire))
                                {
                                    bookings[l].remove(b0);
                                    bookings[l].add(b);
                                }
                                break;
                            }
                        }
                        if (!found)
                            bookings[l].add(b);
                    }
                }
            }
        }

        public RmtCoordinatorPeer
               peer
               (NIP? hIP=null,
                Object? key=null,
                AggregatedNeighbour? aggregated_neighbour=null)
        {
            assert(hIP != null || key != null);
            return new RmtCoordinatorPeer(this, key, hIP, aggregated_neighbour);
        }

        /** This method could be called *directly* for a dispatcher that does not need to transform
          * an exception into a remotable.
          */
        public override ISerializable _dispatch(Object? caller, RemoteCall data) throws Error
        {
            string[] pieces = data.method_name.split(".");
            if (pieces[0] == "reserve")
            {
                if (pieces.length != 1)
                    throw new RPCError.MALFORMED_PACKET(
                        "reserve is a function.");
                if (data.parameters.size != 1)
                    throw new RPCError.MALFORMED_PACKET(
                        "reserve wants 1 parameter.");
                ISerializable iser0 = data.parameters[0];
                if (! iser0.get_type().is_a(typeof(PartialNIP)))
                    throw new RPCError.MALFORMED_PACKET(
                        "reserve parameter 1 is not a PartialNIP.");
                PartialNIP gnode = (PartialNIP)iser0;
                return reserve(gnode);
            }
            if (pieces[0] == "backup_memory")
            {
                if (pieces.length != 1)
                    throw new RPCError.MALFORMED_PACKET(
                        "backup_memory is a function.");
                if (data.parameters.size != 1)
                    throw new RPCError.MALFORMED_PACKET(
                        "backup_memory wants 1 parameter.");
                ISerializable iser0 = data.parameters[0];
                if (! iser0.get_type().is_a(typeof(CoordinatorMemory)))
                    throw new RPCError.MALFORMED_PACKET(
                        "backup_memory parameter 1 is not a CoordinatorMemory.");
                CoordinatorMemory mem = (CoordinatorMemory)iser0;
                backup_memory(mem);
                return new SerializableNone();
            }
            if (pieces[0] == "get_cache_records")
            {
                if (pieces.length != 1)
                    throw new RPCError.MALFORMED_PACKET(
                        "get_cache_records is a function.");
                if (data.parameters.size != 0)
                    throw new RPCError.MALFORMED_PACKET(
                        "get_cache_records wants no parameters.");
                return get_cache_records();
            }
            return base._dispatch(caller, data);
        }

        private NIP nip_for_lvl_pos(int lvl, int pos)
        {
            int[] ret = maproute.me.get_positions();
            ret[lvl] = pos;
            return new NIP(ret);
        }

        private void impl_init_memory() throws Error
        {
            Tasklet.declare_self("Coordinator.init_memory");

            // clear old knowledge
            empty_cache();

            for (int lvl = 0; lvl < maproute.levels; lvl++)
            {
                int? first_forward;
                int? first_back;
                int? last_back;
                find_hook_peers(out first_forward,
                                out first_back,
                                out last_back,
                                lvl, Coordinator.DUPLICATION);
                if (first_forward == null)
                {
                    // no one in my gnode lvl+1 (except me)
                    // I am the coordinator and the knowledge is empty.
                    // I need to go up one level.
                }
                else if (first_back == null)
                {
                    // my gnode lvl+1 has some participants but not
                    // enough to satisfy our replica Coordinator.DUPLICATION.
                    // So, get all from any node and that will do.
                    NIP nip_first_forward = nip_for_lvl_pos(lvl, first_forward);
                    RmtCoordinatorPeer peer_first_forward = peer(nip_first_forward);
                    // use peer_first_forward.get_cache_records() to
                    //  obtain records and save them all to my caches
                    CoordinatorGetCacheRecordsResponse cache_first_forward =
                            peer_first_forward.get_cache_records();
                    // save knowledge
                    reproduce_memory(cache_first_forward.memory);

                    // and then no need to go up one level.
                    break;
                }
                else
                {
                    // my gnode lvl+1 has enough participant.
                    // get all from first_back and from first_forward and that will do.

                    NIP nip_first_back = nip_for_lvl_pos(lvl, first_back);
                    RmtCoordinatorPeer peer_first_back = peer(nip_first_back);
                    CoordinatorGetCacheRecordsResponse cache_first_back =
                            peer_first_back.get_cache_records();
                    reproduce_memory(cache_first_back.memory);

                    NIP nip_first_forward = nip_for_lvl_pos(lvl, first_forward);
                    RmtCoordinatorPeer peer_first_forward = peer(nip_first_forward);
                    CoordinatorGetCacheRecordsResponse cache_first_forward =
                            peer_first_forward.get_cache_records();
                    reproduce_memory(cache_first_forward.memory);

                    // Then no need to go up one level.
                    break;
                }
            }

            // Now I can answer to requests.
            memory_initialized = true;
            log_info("Coordinator service: memory ready.");
            coordinator_ready();
        }

        public void init_memory()
        {
            Tasklet.tasklet_callback(
                    () => {
                        while (true)
                        {
                            try
                            {
                                impl_init_memory();
                                break;
                            }
                            catch (Error e)
                            {
                                log_warn("Coordinator.init_memory: " +
                                         @"got $(e.domain.to_string()) $(e.code) $(e.message). " +
                                         "Trying again.\n");
                                ms_wait(100);
                            }
                        }
                    });
        }

        /** This is the function h:KEY-->hIP.
          */
        public override NIP h(Object k)
        {
            PartialNIP key = (PartialNIP)k;
            int[] positions = key.get_positions();
            for (int l = 0; l < key.level_of_gnode(); l++)
                positions[l] = 0;
            return new NIP(positions);
        }

        /** Remotables from neighbours **/

        public CoordinatorReservation reserve_into(PartialNIP gnode)
                throws HookingError
        {
            /** I want to ask a pos reserved to me or some other neighbour
              *  into gnode.
              */
            CoordinatorReservation ret;
            RmtCoordinatorPeer _peer = peer(null, gnode);
            while (true)
            {
                try
                {
                    ret = _peer.reserve(gnode);
                    break;
                }
                catch (PeerRefuseServiceError e)
                {
                    log_warn(@"Coordinator.reserve_into: got PeerRefuseServiceError $(e.message).\n");
                    ms_wait(100);
                }
                catch (RPCError e)
                {
                    log_warn(@"Coordinator.reserve_into: got RPCError $(e.message).\n");
                    ms_wait(100);
                }
            }
            return ret;
        }

        public CoordinatorInfo report_status(int lvl)
                throws RPCError
        {
            if (lvl < 1 || lvl > maproute.levels) throw new RPCError.GENERIC(@"Bad level number: $(lvl)");
            if (!memory_initialized) throw new RPCError.GENERIC("Memory not ready yet");
            update_cache();
            CoordinatorInfo ret = new CoordinatorInfo();
            ret.last_assigned_eldership = last_assigned_elderships[lvl-1];
            foreach (CoordinatorBooking b in bookings[lvl-1])
            {
                ret.add_pos(b.pos);
            }
            return ret;
        }

        /** Remotables from peers **/

        public  CoordinatorReservation
                reserve
                (PartialNIP gnode)
                throws RPCError, PeerRefuseServiceError, HookingError
        {
            if (!memory_initialized)
            {
                // My memory is not ready, so I cant answer.
                // The client is waiting for a reply.
                // After a bit I will throw an exception.
                Tasklets.Timer w = new Tasklets.Timer(10000); // 10 seconds
                while (!memory_initialized)
                {
                    if (w.is_expired())
                        throw new PeerRefuseServiceError.GENERIC("Memory not ready yet");
                    Tasklet.nap(0, 1000);
                }
            }

            update_cache();
            // First, we check if I belong to the gnode. Otherwise we throw a generic Exception.
            if (! maproute.me.belongs_to(gnode))
                throw new PeerRefuseServiceError.GENERIC("The node is not in that gnode.");
            NIP hip = h(gnode);
            // First, I check if I am the exact best approximation to hip.
            // If it doesnt, we throw a generic Exception.
            // This check doesnt need to send messages to other nodes.
            HCoord? Hhip = search_participant(hip);
            if (Hhip != null)
                throw new PeerRefuseServiceError.GENERIC("The node is not the Coordinator.");

            int level_of_gnode = gnode.level_of_gnode();
            int level_of_nodes = level_of_gnode - 1;
            int[] _fnl = maproute.free_nodes_list(level_of_nodes);
            ArrayList<int> fnl = new ArrayList<int>();
            foreach (int i in _fnl)
                if (! bookings_contains(level_of_nodes, i))
                    fnl.add(i);
            if (fnl.is_empty) throw new HookingError.GENERIC("No free positions");

            int pos = fnl[Random.int_range(0, fnl.size)];
            bookings[level_of_nodes].add(new CoordinatorBooking(pos));
            int eldership = last_assigned_elderships[level_of_nodes] + 1;
            last_assigned_elderships[level_of_nodes] = eldership;

            GNodeID[] gids = maproute.get_gid_uppermost_list(level_of_gnode);

            // start replica
            Tasklet.tasklet_callback(
                (tpar1) => {
                    NIP tasklet_hip = (NIP)tpar1;
                    Tasklet.declare_self("Coordinator replica");
                    // Here I am in a tasklet (the client has been served already)
                    Gee.List<NIP> replica_nodes =
                            find_nearest_to_register(tasklet_hip, Coordinator.DUPLICATION);
                    // For each node of the <n> nearest except myself...
                    foreach (NIP replica_node in replica_nodes) if (!replica_node.is_equal(maproute.me))
                    {
                        // ... in another tasklet...
                        Tasklet.tasklet_callback(
                            (tpar1) => {
                                NIP tonip = (NIP)tpar1;
                                Tasklet.declare_self("Coordinator replica to one");
                                // ... forward the record to the node.
                                try
                                {
                                    peer(tonip).backup_memory(produce_memory());
                                }
                                catch (RPCError e)
                                {
                                    // report the error with some info on where it happened
                                    log_warn(@"Coordinator: forwarding to $(tonip):"
                                        + @" got $(e.domain.to_string()) $(e.code) $(e.message)");
                                }
                            },
                            replica_node);
                    }
                },
                hip);

            CoordinatorReservation ret = new CoordinatorReservation();
            ret.pos = pos;
            ret.eldership = eldership;
            ret.gids = gids;
            return ret;
        }

        public void
               backup_memory
               (CoordinatorMemory mem)
               throws RPCError
        {
            if (!memory_initialized)
            {
                // This is a replica. So the client is not waiting and will not
                // request again. Thus I will wait till I get ready to memorize.
                while (!memory_initialized) Tasklet.nap(0, 1000);
            }

            reproduce_memory(mem);
        }

        public CoordinatorGetCacheRecordsResponse
               get_cache_records ()
               throws RPCError, PeerRefuseServiceError
        {
            if (!memory_initialized)
            {
                // My memory is not ready, so I cant answer.
                // The client is waiting for a reply.
                // After a bit I will throw an exception.
                Tasklets.Timer w = new Tasklets.Timer(2000); // 2 seconds
                while (!memory_initialized)
                {
                    if (w.is_expired())
                        throw new PeerRefuseServiceError.GENERIC("Memory not ready yet");
                    Tasklet.nap(0, 1000);
                }
            }

            return new CoordinatorGetCacheRecordsResponse(produce_memory());
        }
    }
}

