/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2011 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 *
 *  Base class for auxiliary address managers.
 */

using Gee;
using zcd;

namespace Netsukuku
{
    /** Instance of AddressManager for auxiliary addresses.
      */
    public class AuxiliaryAddressManager : AddressManager
    {
        public weak AddressManager autonomous_address_manager {get; private set;}

        public AuxiliaryAddressManager(int levels,
                                       int gsize,
                                       KeyPair keypair,
                                       HookInformation? hook_information,
                                       CreateNicDelegate create_new_nic,
                                       AddressManager autonomous_address_manager)
        {
            base(levels, gsize, keypair, hook_information, create_new_nic);
            this.autonomous_address_manager = autonomous_address_manager;
        }

        // Auxiliary is not Autonomous.
        public override bool is_autonomous {
            get {
                return false;
            }
        }

        public override bool is_auxiliary {
            get {
                return true;
            }
        }

        public override IncomingNodes create_IncomingNodes()
        {
            return new AuxiliaryIncomingNodes();
        }

        public override MapRoute create_MapRoute(int levels, int gsize, NIP nip, GNodeID[] id_myself)
        {
            return new AuxiliaryMapRoute(levels, gsize, nip, id_myself, this);
        }

        public override AggregatedNeighbourManager create_AggregatedNeighbourManager()
        {
            return new AuxiliaryAggregatedNeighbourManager(this);
        }

        public override Etp create_Etp(AggregatedNeighbourManager aggregated_neighbour_manager, MapRoute maproute)
        {
            return new AuxiliaryEtp(aggregated_neighbour_manager, maproute, this);
        }

        public override PeerToPeerAll create_PeerToPeerAll(AggregatedNeighbourManager aggregated_neighbour_manager, MapRoute maproute)
        {
            return new AuxiliaryPeerToPeerAll(aggregated_neighbour_manager, maproute);
        }

        public override Coordinator create_Coordinator(AggregatedNeighbourManager aggregated_neighbour_manager, MapRoute maproute, PeerToPeerAll peer_to_peer_all)
        {
            return new AuxiliaryCoordinator(aggregated_neighbour_manager, maproute, peer_to_peer_all);
        }

        public override Counter create_Counter(KeyPair keypair, AggregatedNeighbourManager aggregated_neighbour_manager, MapRoute maproute, PeerToPeerAll peer_to_peer_all)
        {
            return new AuxiliaryCounter(keypair, aggregated_neighbour_manager, maproute, peer_to_peer_all);
        }

        public override Andna create_Andna(KeyPair keypair, Counter counter, AggregatedNeighbourManager aggregated_neighbour_manager, MapRoute maproute, PeerToPeerAll peer_to_peer_all)
        {
            return new AuxiliaryAndna(keypair, counter, aggregated_neighbour_manager, maproute, peer_to_peer_all);
        }

        public override TunnelManager create_TunnelManager(string ipstr)
        {
            return new AuxiliaryTunnelManager(ipstr);
        }

        public override Hook create_Hook(MapRoute maproute)
        {
            return new AuxiliaryHook(maproute, this);
        }

        public override BorderNodesManager create_BorderNodesManager(MapRoute maproute, AggregatedNeighbourManager aggregated_neighbour_manager)
        {
            return new AuxiliaryBorderNodesManager(this, maproute, aggregated_neighbour_manager);
        }

        public override MigrationManager create_MigrationManager(MapRoute maproute, AggregatedNeighbourManager aggregated_neighbour_manager)
        {
            return new AuxiliaryMigrationManager(this, maproute, aggregated_neighbour_manager);
        }
    }

    public class AuxiliaryIncomingNodes : IncomingNodes
    {
        public AuxiliaryIncomingNodes()
        {
            base();
        }
    }

    public class AuxiliaryMapRoute : MapRoute
    {
        public AuxiliaryMapRoute(int levels, int gsize, NIP me, GNodeID[] id_myself, AddressManager address_manager)
        {
            base(levels, gsize, me, id_myself, address_manager);
        }
    }

    public class AuxiliaryAggregatedNeighbourManager : AggregatedNeighbourManager
    {
        public AuxiliaryAggregatedNeighbourManager(AddressManager address_manager)
        {
            base(address_manager);
        }
    }

    public class AuxiliaryEtp : Etp
    {
        public AuxiliaryEtp(AggregatedNeighbourManager aggregated_neighbour_manager,
                        MapRoute maproute, AddressManager address_manager)
        {
            base(aggregated_neighbour_manager, maproute, address_manager);
        }
    }

    public class AuxiliaryPeerToPeerAll : PeerToPeerAll
    {
        public AuxiliaryPeerToPeerAll(AggregatedNeighbourManager aggregated_neighbour_manager, MapRoute maproute)
        {
            base(aggregated_neighbour_manager, maproute);
        }

        public override void peer_to_peer_register(PeerToPeer peer_to_peer) throws PeerToPeerError
        {
            base.peer_to_peer_register(peer_to_peer);
            if (peer_to_peer.get_type().is_a(typeof(OptionalPeerToPeer)))
            {
                ((OptionalPeerToPeer)peer_to_peer).will_participate = false;
            }
        }
    }

    public class AuxiliaryCoordinator : Coordinator
    {
        public AuxiliaryCoordinator(AggregatedNeighbourManager aggregated_neighbour_manager,
                        MapRoute maproute, PeerToPeerAll peer_to_peer_all)
        {
            base(aggregated_neighbour_manager, maproute, peer_to_peer_all);
        }
    }

    public class AuxiliaryCounter : Counter
    {
        public AuxiliaryCounter(KeyPair keypair, AggregatedNeighbourManager aggregated_neighbour_manager,
                        MapRoute maproute, PeerToPeerAll peer_to_peer_all)
        {
            base(keypair, aggregated_neighbour_manager, maproute, peer_to_peer_all);
        }
    }

    public class AuxiliaryAndna : Andna
    {
        public AuxiliaryAndna(KeyPair keypair, Counter counter, AggregatedNeighbourManager aggregated_neighbour_manager,
                        MapRoute maproute, PeerToPeerAll peer_to_peer_all)
        {
            base(keypair, counter, aggregated_neighbour_manager, maproute, peer_to_peer_all);
        }
    }

    public class AuxiliaryTunnelManager : TunnelManager
    {
        public AuxiliaryTunnelManager(string my_addr)
        {
            base(my_addr);
        }
    }

    public class AuxiliaryHook : Hook
    {
        public AuxiliaryHook(MapRoute maproute, AddressManager addr_man)
        {
            base(maproute, addr_man);
        }
    }

    public class AuxiliaryBorderNodesManager : BorderNodesManager
    {
        public AuxiliaryBorderNodesManager(AddressManager addr_man, MapRoute maproute,
                        AggregatedNeighbourManager aggregated_neighbour_manager)
        {
            base(addr_man, maproute, aggregated_neighbour_manager);
        }
    }

    public class AuxiliaryMigrationManager : MigrationManager
    {
        public AuxiliaryMigrationManager(AddressManager address_manager, MapRoute maproute,
                        AggregatedNeighbourManager aggregated_neighbour_manager)
        {
            base(address_manager, maproute, aggregated_neighbour_manager);
        }

        // Auxiliary addresses never migrate
        public override void start_operations()
        {
            // Do nothing.
        }
        public override void stop_operations()
        {
            // Do nothing.
        }
    }
}

