/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2011 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gee;
using zcd;
using Tasklets;

namespace Netsukuku
{
    string[] read_config(string path)
    {
        string[] ret = new string[0];
        if (FileUtils.test(path, FileTest.EXISTS))
        {
            try
            {
                string contents;
                assert(FileUtils.get_contents(path, out contents));
                ret = contents.split("\n");
            }
            catch (FileError e) {error("%s: %d: %s".printf(e.domain.to_string(), e.code, e.message));}
        }
        return ret;
    }


    /** This class is instantiated once in the whole app. Keeps track of all the
      *  addresses that this node may obtain for itself.
      */
    public class Addresses : Object
    {
        private static Addresses addresses_instance;
        public static Addresses get_addresses_instance()
        {
            return addresses_instance;
        }

        public Gee.List<AddressManager> get_autonomous_addresses()
        {
            ArrayList<AddressManager> ret = new ArrayList<AddressManager>();
            ret.add_all(addresses_instance.autonomousaddresses);
            return ret;
        }

        public static void tcp_callback(CallerInfo caller,
                                              TCPRequest tcprequest,
                                              out RPCDispatcher? rpcdispatcher,
                                              out uchar[] data,
                                              out uchar[] response)
        {
            // If I get contacted via TCP on a IP that is not in the "whole network"
            //  then it could be a monitoring tool. I will answer with the AddressManager
            //  of my primary address.
            AddressManager? addr = null;
            try {
                str_to_nip(addresses_instance.levels, addresses_instance.gsize, caller.my_ip);
            } catch (InetError e) {
                if (e is InetError.NO_NIP) addr = addresses_instance.primary_address;
            }
            if (addr == null)
            {
                addr = addresses_instance
                    .get_address_manager_by_ipstr(caller.my_ip);
            }

            rpcdispatcher = null;
            data = null;
            response = null;
            if (addr != null)
            {
                rpcdispatcher = new AddressManagerDispatcher(addr);
                data = tcprequest.data.serialize();
            }
        }

        public static void udp_unicast_callback(CallerInfo caller,
                                                  UDPPayload payload,
                                                  out RPCDispatcher? rpcdispatcher,
                                                  out uchar[] data,
                                                  out Gee.List<string> devs_response)
        {
            UnicastID ucid = (UnicastID)ISerializable.deserialize(payload.ser);
            AddressManager addr = addresses_instance
                    .get_address_manager_by_nip_nodeid(
                    ucid.nip,
                    ucid.nodeid);
            rpcdispatcher = null;
            data = null;
            devs_response = null;
            if (addr != null)
            {
                rpcdispatcher = new AddressManagerDispatcher(addr);
                data = payload.data.serialize();
                devs_response = new ArrayList<string>();
                foreach (NetworkInterfaceManager nic_man in addr.nics)
                {
                    if (nic_man.to_be_managed)
                    {
                        devs_response.add(nic_man.nic_name);
                    }
                }
            }
        }

        public static void udp_broadcast_callback(CallerInfo caller,
                                                    UDPPayload payload,
                                                    out Gee.List<RPCDispatcher> rpcdispatchers,
                                                    out uchar[] data)
        {
            BroadcastID bcid = (BroadcastID)ISerializable.deserialize(payload.ser);
            NetworkID? networkid = bcid.net;
            rpcdispatchers = new ArrayList<RPCDispatcher>();
            foreach (AddressManager addr_man in addresses_instance)
            {
                if (networkid == null || addr_man.is_in_my_network(networkid))
                    rpcdispatchers.add(new AddressManagerDispatcher(addr_man));
            }
            data = payload.data.serialize();
        }

        private KeyPair keypair;
        public int levels {get; private set;}
        public int gsize {get; private set;}
        private HashMap<string, AddressManager> addrs_by_nip_nodeid;
        private HashMap<string, AddressManager> addrs_by_ipstr;
        private ArrayList<AddressManager> autonomousaddresses;
        private AddressManager? _primary_address;
        private /*IGSManager*/ Object igs_manager;
        public Addresses(KeyPair keypair)
        {
            addresses_instance = this;
            this.keypair = keypair;
            this.addrs_by_nip_nodeid = new HashMap<string, AddressManager>();
            this.addrs_by_ipstr = new HashMap<string, AddressManager>();
            this.autonomousaddresses = new ArrayList<AddressManager>();
            this._primary_address = null;
            igs_manager = new Object(); /*TODO new IGSManager(this)*/
            // listen signal PRIMARY_ADDRESS_CHANGED with adjust_ntkd
            primary_address_changed.connect(call_adjust_ntkd_1);
        }

        public signal void network_reset();
        public signal void primary_address_changed(AddressManager? old_addr, AddressManager? new_addr);
        public signal void routes_updated(AddressManager addrman, HCoord lvl_pos);
        public signal void gnode_splitted(AddressManager addrman, Gee.List<AggregatedNeighbour> passed_neighbour_list, Gee.List<int> queue_of_request_ids, GNodeID actual_gid);
        public signal void aggregated_neighbour_new(AggregatedNeighbour aggregated_neighbour);
        public signal void aggregated_neighbour_new_before(AggregatedNeighbour aggregated_neighbour);
        public signal void aggregated_neighbour_deleted(AggregatedNeighbour aggregated_neighbour);
        public signal void aggregated_neighbour_deleted_after(AggregatedNeighbour aggregated_neighbour);
        public signal void aggregated_neighbour_rem_chged(AggregatedNeighbour aggregated_neighbour, REM old_rem);
        public signal void aggregated_neighbour_rem_chged_before(AggregatedNeighbour aggregated_neighbour, REM old_rem);
        public signal void aggregated_neighbour_going_new(AggregatedNeighbour aggregated_neighbour);
        public signal void aggregated_neighbour_going_deleted(AggregatedNeighbour aggregated_neighbour);
        public signal void aggregated_neighbour_going_rem_chged(AggregatedNeighbour aggregated_neighbour);
        public signal void aggregated_neighbour_colliding_new(AggregatedNeighbour aggregated_neighbour);
        public signal void aggregated_neighbour_colliding_deleted(AggregatedNeighbour aggregated_neighbour);
        public signal void aggregated_neighbour_colliding_rem_chged(AggregatedNeighbour aggregated_neighbour, REM old_rem);
        public signal void aggregated_neighbour_colliding_going_new(AggregatedNeighbour aggregated_neighbour);
        public signal void aggregated_neighbour_colliding_going_deleted(AggregatedNeighbour aggregated_neighbour);
        public signal void aggregated_neighbour_colliding_going_rem_chged(AggregatedNeighbour aggregated_neighbour);
        public signal void net_collision(AddressManager addrman, Gee.List<AggregatedNeighbour> others);
        public signal void sent_etp(AddressManager addrman, NIP nip_caller, NetworkID netid_caller,
                                Gee.List<string> macs_caller, ExtendedTracerPacket pkt);
        public signal void incoming_node_updated(AddressManager addrman, string mac);
        public signal void counter_hooked(AddressManager addrman);
        public signal void andna_hooked(AddressManager addrman);
        public signal void counter_registered(AddressManager addrman);
        public signal void andna_registered(AddressManager addrman, string domain);
        public signal void sig_is_mature(AddressManager mature_addrman);

        public AddressManager? primary_address {
            get {
                return _primary_address;
            }
            set {
                AddressManager? old_addr = _primary_address;
                if (old_addr != value)
                {
                    if (old_addr != null) old_addr.is_primary = false;
                    if (value != null) value.is_primary = true;
                    _primary_address = value;
                    // send signal PRIMARY_ADDRESS_CHANGED(old_addr, value)
                    primary_address_changed(old_addr, value);
                }
            }
        }

        public void init_network(int levels, int gsize)
        {
            this.levels = levels;
            this.gsize = gsize;
            ArrayList<AddressManager> addrs = new ArrayList<AddressManager>();
            foreach (AddressManager addr in this) addrs.add(addr);
            foreach (AddressManager addr in addrs) remove(addr);
            primary_address = null;
            // send signal NETWORK_RESET()
            network_reset();
        }

        internal AddressManager create(HookReservation? hook_reservation)
        {
            AddressManager ret = new AddressManager(levels, gsize, keypair, hook_reservation,
                    (nic_name) => {
                        return NIC.create_instance(nic_name);
                    });
            ret.igs_manager = new IGSManagerDispatcher(ret, igs_manager);
            return ret;
        }

        void repeat_routes_updated(AddressManager addrman, HCoord lvl_pos)
        {
            routes_updated(addrman, lvl_pos);
        }
        void repeat_gnode_splitted(AddressManager addrman, Gee.List<AggregatedNeighbour> passed_neighbour_list, Gee.List<int> queue_of_request_ids, GNodeID actual_gid)
        {
            gnode_splitted(addrman, passed_neighbour_list, queue_of_request_ids, actual_gid);
        }
        void repeat_aggregated_neighbour_new(AddressManager addrman, AggregatedNeighbour aggregated_neighbour)
        {
            aggregated_neighbour_new(aggregated_neighbour);
        }
        void repeat_aggregated_neighbour_new_before(AddressManager addrman, AggregatedNeighbour aggregated_neighbour)
        {
            aggregated_neighbour_new_before(aggregated_neighbour);
        }
        void repeat_aggregated_neighbour_deleted_after(AddressManager addrman, AggregatedNeighbour aggregated_neighbour)
        {
            aggregated_neighbour_deleted_after(aggregated_neighbour);
        }
        void repeat_aggregated_neighbour_deleted(AddressManager addrman, AggregatedNeighbour aggregated_neighbour)
        {
            aggregated_neighbour_deleted(aggregated_neighbour);
        }
        void repeat_aggregated_neighbour_rem_chged(AddressManager addrman, AggregatedNeighbour aggregated_neighbour, REM old_rem)
        {
            aggregated_neighbour_rem_chged(aggregated_neighbour, old_rem);
        }
        void repeat_aggregated_neighbour_rem_chged_before(AddressManager addrman, AggregatedNeighbour aggregated_neighbour, REM old_rem)
        {
            aggregated_neighbour_rem_chged_before(aggregated_neighbour, old_rem);
        }
        void repeat_aggregated_neighbour_going_new(AddressManager addrman, AggregatedNeighbour aggregated_neighbour)
        {
            aggregated_neighbour_going_new(aggregated_neighbour);
        }
        void repeat_aggregated_neighbour_going_deleted(AddressManager addrman, AggregatedNeighbour aggregated_neighbour)
        {
            aggregated_neighbour_going_deleted(aggregated_neighbour);
        }
        void repeat_aggregated_neighbour_going_rem_chged(AddressManager addrman, AggregatedNeighbour aggregated_neighbour)
        {
            aggregated_neighbour_going_rem_chged(aggregated_neighbour);
        }
        void repeat_aggregated_neighbour_colliding_new(AddressManager addrman, AggregatedNeighbour aggregated_neighbour)
        {
            aggregated_neighbour_colliding_new(aggregated_neighbour);
        }
        void repeat_aggregated_neighbour_colliding_deleted(AddressManager addrman, AggregatedNeighbour aggregated_neighbour)
        {
            aggregated_neighbour_colliding_deleted(aggregated_neighbour);
        }
        void repeat_aggregated_neighbour_colliding_rem_chged(AddressManager addrman, AggregatedNeighbour aggregated_neighbour, REM old_rem)
        {
            aggregated_neighbour_colliding_rem_chged(aggregated_neighbour, old_rem);
        }
        void repeat_aggregated_neighbour_colliding_going_new(AddressManager addrman, AggregatedNeighbour aggregated_neighbour)
        {
            aggregated_neighbour_colliding_going_new(aggregated_neighbour);
        }
        void repeat_aggregated_neighbour_colliding_going_deleted(AddressManager addrman, AggregatedNeighbour aggregated_neighbour)
        {
            aggregated_neighbour_colliding_going_deleted(aggregated_neighbour);
        }
        void repeat_aggregated_neighbour_colliding_going_rem_chged(AddressManager addrman, AggregatedNeighbour aggregated_neighbour)
        {
            aggregated_neighbour_colliding_going_rem_chged(aggregated_neighbour);
        }
        void repeat_net_collision(AddressManager addrman, Gee.List<AggregatedNeighbour> others)
        {
            net_collision(addrman, others);
        }
        void repeat_sent_etp(AddressManager addrman, NIP nip_caller, NetworkID netid_caller,
                                Gee.List<string> macs_caller, ExtendedTracerPacket pkt)
        {
            sent_etp(addrman, nip_caller, netid_caller, macs_caller, pkt);
        }
        void repeat_incoming_node_updated(AddressManager addrman, string mac)
        {
            incoming_node_updated(addrman, mac);
        }
        void repeat_counter_hooked(AddressManager addrman)
        {
            counter_hooked(addrman);
        }
        void repeat_andna_hooked(AddressManager addrman)
        {
            andna_hooked(addrman);
        }
        void repeat_counter_registered(AddressManager addrman)
        {
            counter_registered(addrman);
        }
        void repeat_andna_registered(AddressManager addrman, string domain)
        {
            andna_registered(addrman, domain);
        }
        void repeat_is_mature(AddressManager addrman, AddressManager mature_addrman)
        {
            sig_is_mature(mature_addrman);
        }

        public void add(AddressManager address_manager)
        {
            // Repeat signals
            address_manager.routes_updated.connect(repeat_routes_updated);
            address_manager.gnode_splitted.connect(repeat_gnode_splitted);
            address_manager.aggregated_neighbour_new.connect(repeat_aggregated_neighbour_new);
            address_manager.aggregated_neighbour_new_before.connect(repeat_aggregated_neighbour_new_before);
            address_manager.aggregated_neighbour_deleted.connect(repeat_aggregated_neighbour_deleted);
            address_manager.aggregated_neighbour_deleted_after.connect(repeat_aggregated_neighbour_deleted_after);
            address_manager.aggregated_neighbour_rem_chged.connect(repeat_aggregated_neighbour_rem_chged);
            address_manager.aggregated_neighbour_rem_chged_before.connect(repeat_aggregated_neighbour_rem_chged_before);
            address_manager.aggregated_neighbour_going_new.connect(repeat_aggregated_neighbour_going_new);
            address_manager.aggregated_neighbour_going_deleted.connect(repeat_aggregated_neighbour_going_deleted);
            address_manager.aggregated_neighbour_going_rem_chged.connect(repeat_aggregated_neighbour_going_rem_chged);
            address_manager.aggregated_neighbour_colliding_new.connect(repeat_aggregated_neighbour_colliding_new);
            address_manager.aggregated_neighbour_colliding_deleted.connect(repeat_aggregated_neighbour_colliding_deleted);
            address_manager.aggregated_neighbour_colliding_rem_chged.connect(repeat_aggregated_neighbour_colliding_rem_chged);
            address_manager.aggregated_neighbour_colliding_going_new.connect(repeat_aggregated_neighbour_colliding_going_new);
            address_manager.aggregated_neighbour_colliding_going_deleted.connect(repeat_aggregated_neighbour_colliding_going_deleted);
            address_manager.aggregated_neighbour_colliding_going_rem_chged.connect(repeat_aggregated_neighbour_colliding_going_rem_chged);
            address_manager.net_collision.connect(repeat_net_collision);
            address_manager.sent_etp.connect(repeat_sent_etp);
            address_manager.incoming_node_updated.connect(repeat_incoming_node_updated);
            address_manager.counter_hooked.connect(repeat_counter_hooked);
            address_manager.andna_hooked.connect(repeat_andna_hooked);
            address_manager.counter_registered.connect(repeat_counter_registered);
            address_manager.andna_registered.connect(repeat_andna_registered);
            address_manager.sig_is_mature.connect(repeat_is_mature);
            // Listen to signals
            address_manager.aggregated_neighbour_new.connect(call_adjust_ntkd_2);
            address_manager.auxiliary_address_manager_new.connect(auxiliary_added);
            address_manager.auxiliary_address_manager_deleted.connect(auxiliary_deleted);
            address_manager.address_manager_new.connect(autonomous_to_add);
            address_manager.request_be_primary.connect(primary_change_requested);
            address_manager.address_manager_delete.connect(autonomous_to_delete);
            // Add to the collections
            NIP nip = address_manager.maproute.me;
            int nodeid = address_manager.get_my_id();
            string tuple_nip_nodeid = @"$(nip)-$(nodeid)";
            addrs_by_nip_nodeid[tuple_nip_nodeid] = address_manager;
            string ipstr = nip_to_str(levels, gsize, nip);
            addrs_by_ipstr[ipstr] = address_manager;
            if (address_manager.is_autonomous) autonomousaddresses.add(address_manager);
        }

        public void remove(AddressManager address_manager)
        {
            int local_levels = address_manager.maproute.levels;
            int local_gsize = address_manager.maproute.gsize;
            NIP nip = address_manager.maproute.me;
            int nodeid = address_manager.get_my_id();
            string ipstr = nip_to_str(local_levels, local_gsize, nip);
            // stop managing
            log_debug(@"Addresses: start tasklet stopping operations for $(ipstr)");
            address_manager.stop_operations();
            // send goodbye messages gracefully
            log_debug(@"Addresses: broadcast declare_dead for $(ipstr)");
            try
            {
                address_manager.get_broadcast_client().aggregated_neighbour_manager.declare_dead(nip, nodeid);
            }
            catch (RPCError e) {}
            // wait a little (gives time to stop_operations and broadcast sending)
            log_debug(@"Addresses: wait a bit...");
            ms_wait(500);
            // remove nics (deactivation here)
            log_debug(@"Addresses: start deactivation of $(ipstr) in my NICs");
            ArrayList<NetworkInterfaceManager> nics = new ArrayList<NetworkInterfaceManager>();
            nics.add_all(address_manager.nics);
            foreach (NetworkInterfaceManager nic_man in nics)
            {
                address_manager.remove_nic_manager(nic_man);
            }
            // Stop: Repeat signals
            address_manager.routes_updated.disconnect(repeat_routes_updated);
            address_manager.gnode_splitted.disconnect(repeat_gnode_splitted);
            address_manager.aggregated_neighbour_new.disconnect(repeat_aggregated_neighbour_new);
            address_manager.aggregated_neighbour_new_before.disconnect(repeat_aggregated_neighbour_new_before);
            address_manager.aggregated_neighbour_deleted.disconnect(repeat_aggregated_neighbour_deleted);
            address_manager.aggregated_neighbour_deleted_after.disconnect(repeat_aggregated_neighbour_deleted_after);
            address_manager.aggregated_neighbour_rem_chged.disconnect(repeat_aggregated_neighbour_rem_chged);
            address_manager.aggregated_neighbour_rem_chged_before.disconnect(repeat_aggregated_neighbour_rem_chged_before);
            address_manager.aggregated_neighbour_going_new.disconnect(repeat_aggregated_neighbour_going_new);
            address_manager.aggregated_neighbour_going_deleted.disconnect(repeat_aggregated_neighbour_going_deleted);
            address_manager.aggregated_neighbour_going_rem_chged.disconnect(repeat_aggregated_neighbour_going_rem_chged);
            address_manager.aggregated_neighbour_colliding_new.disconnect(repeat_aggregated_neighbour_colliding_new);
            address_manager.aggregated_neighbour_colliding_deleted.disconnect(repeat_aggregated_neighbour_colliding_deleted);
            address_manager.aggregated_neighbour_colliding_rem_chged.disconnect(repeat_aggregated_neighbour_colliding_rem_chged);
            address_manager.aggregated_neighbour_colliding_going_new.disconnect(repeat_aggregated_neighbour_colliding_going_new);
            address_manager.aggregated_neighbour_colliding_going_deleted.disconnect(repeat_aggregated_neighbour_colliding_going_deleted);
            address_manager.aggregated_neighbour_colliding_going_rem_chged.disconnect(repeat_aggregated_neighbour_colliding_going_rem_chged);
            address_manager.net_collision.disconnect(repeat_net_collision);
            address_manager.sent_etp.disconnect(repeat_sent_etp);
            address_manager.incoming_node_updated.disconnect(repeat_incoming_node_updated);
            address_manager.counter_hooked.disconnect(repeat_counter_hooked);
            address_manager.andna_hooked.disconnect(repeat_andna_hooked);
            address_manager.counter_registered.disconnect(repeat_counter_registered);
            address_manager.andna_registered.disconnect(repeat_andna_registered);
            address_manager.sig_is_mature.disconnect(repeat_is_mature);
            // Stop: Listen to signals
            address_manager.aggregated_neighbour_new.disconnect(call_adjust_ntkd_2);
            address_manager.auxiliary_address_manager_new.disconnect(auxiliary_added);
            address_manager.auxiliary_address_manager_deleted.disconnect(auxiliary_deleted);
            address_manager.address_manager_new.disconnect(autonomous_to_add);
            address_manager.request_be_primary.disconnect(primary_change_requested);
            address_manager.address_manager_delete.disconnect(autonomous_to_delete);
            // Remove from the collections
            string tuple_nip_nodeid = @"$(nip)-$(nodeid)";
            addrs_by_nip_nodeid.unset(tuple_nip_nodeid);
            addrs_by_ipstr.unset(ipstr);
            if (address_manager.is_autonomous) autonomousaddresses.remove(address_manager);
            log_debug(@"Addresses: $(ipstr) fully removed.");
        }

        void auxiliary_added(AddressManager address_manager)
        {
            add(address_manager);
        }

        void auxiliary_deleted(AddressManager address_manager)
        {
            remove(address_manager);
        }

        void autonomous_to_add(HookReservation hook_reservation, AddressManager old_addr_man, ref AddressManager return_value)
        {
            AddressManager address_manager = create(hook_reservation);
            add(address_manager);
            foreach (NetworkInterfaceManager nic_man in old_addr_man.nics)
            {
                if (nic_man.to_be_copied) address_manager.add_nic_manager(nic_man);
            }
            address_manager.start_operations(20000);
            // This delay replaces the delay between HOOKED and HOOKED_STABLE: that is, wait
            // for the first ETP to come in from some neighbour.

            // Returning a value from an signal handler (bad hack)
            return_value = address_manager;
        }

        void primary_change_requested(AddressManager address_manager)
        {
            primary_address = address_manager;
        }

        void autonomous_to_delete(AddressManager address_manager)
        {
            remove(address_manager);
        }

        public Iterator<AddressManager> iterator()
        {
            return addrs_by_nip_nodeid.values.iterator();
        }

        public AddressManager? get_address_manager_by_nip_nodeid(NIP nip, int nodeid)
        {
            string tuple_nip_nodeid = @"$(nip)-$(nodeid)";
            if (! addrs_by_nip_nodeid.has_key(tuple_nip_nodeid)) return null;
            return addrs_by_nip_nodeid[tuple_nip_nodeid];
        }

        public AddressManager? get_address_manager_by_ipstr(string ipstr)
        {
            if (! addrs_by_ipstr.has_key(ipstr)) return null;
            return addrs_by_ipstr[ipstr];
        }

        void call_adjust_ntkd_1(AddressManager? old_addr, AddressManager? new_addr)
        {
            adjust_ntkd();
        }
        void call_adjust_ntkd_2(AddressManager addrman, AggregatedNeighbour aggregated_neighbour)
        {
            adjust_ntkd();
        }
        public void adjust_ntkd()
        {
            // This has to be called on any change signaled by aggregated_neighbour_manager
            //  and after its initialize().
            foreach (AddressManager addr_man in this)
            {
                foreach (AggregatedNeighbour aggregated_neighbour in addr_man.aggregated_neighbour_manager.neighbour_list())
                {
                    aggregated_neighbour.activate_tcp(primary_address.is_in_my_network(aggregated_neighbour.netid));
                }
            }
        }

        /** Gathers all available paths and places them in instances of CondensedPathChooser
          *  which makes it easy to find the best path (optionally not passing through a
          *  neighbour of ours)
          */
        public CondensedMapRoute get_condensed_map_route()
        {
            /* Since we have <n> maproutes, that is <n> addresses:
             *  E.g. we have [4,3,2,1], [7,6,5,2] and [7,8,9,2]
             *  At level 3 we have ids [1] and [2]; hence we do not have
             *  to store paths towards 1.* and 2.*. We do have to store
             *  paths towards 5.* for instance. We have to look for them in all
             *  the maproutes that we have got.
             *  At level 2, we have 2 sets of addresses: the first contains the
             *  addresses prefixed with [1] and the second one contains those prefixed
             *  with [2]. For instance, in the second set we have 2 ids, [5,2] and [9,2]
             *  Thus, we do not have to store paths towards [5,2] and [9,2]. We do
             *  have to store paths towards [8,2] for instance. We have to look for
             *  them in the maproutes prefixed with [2]
             * So, abstracting, at each level from the uppermost to the lowermost,
             *  for each destination that is not in our own ids, we take all the paths
             *  available from all the maproutes and populate one "PathChooser".
             * Furthermore we do not consider the paths that have as gateway one of
             *  our own addresses. And we do consider only one path per gateway per destination.
             */
            CondensedMapRoute ret_value = new CondensedMapRoute();
            get_structure(ref ret_value);
            return ret_value;
        }
        private void get_structure(ref CondensedMapRoute ret_value, PartialNIP? _prefix=null, int? _lvl=null, Gee.List<MapRoute>? _maproutes=null)
        {
            PartialNIP? prefix = _prefix;
            int lvl = _lvl==null ? levels-1 : _lvl;
            Gee.List<MapRoute>? maproutes = _maproutes;
            if (prefix == null)
            {
                int[] positions = new int[levels];
                for (int i = 0; i < levels; i++)
                    positions[i] = -1;
                prefix = new PartialNIP(positions);
            }
            if (maproutes == null)
            {
                maproutes = new ArrayList<MapRoute>();
                foreach (AddressManager addr in this)
                    if (addr.do_i_participate_in_routing_tables())
                        if (addr.is_mature)
                            maproutes.add(addr.maproute);
            }
            // how many different addresses we have in this level?
            HashMap<int, ArrayList<MapRoute>> distinct_pos = new HashMap<int, ArrayList<MapRoute>>();
            foreach (MapRoute mr in maproutes)
            {
                int pos = mr.me.position_at(lvl);
                if (! distinct_pos.has_key(pos)) distinct_pos[pos] = new ArrayList<MapRoute>();
                distinct_pos[pos].add(mr);
            }
            // for each pos that is NOT in my addresses in this level...
            for (int pos = 0; pos < gsize; pos++)
                if (! distinct_pos.has_key(pos))
                {
                    int[] key_pos = prefix.get_positions();
                    key_pos[lvl] = pos;
                    PartialNIP key = new PartialNIP(key_pos);
                    // search routes in passed maproutes
                    HashMap<NIP, CloneRoute> newpaths = new HashMap<NIP, CloneRoute>(PartialNIP.hash_func, PartialNIP.equal_func);
                    // key is nip of gateway, value is a Route to lvl,pos
                    foreach (MapRoute mr in maproutes)
                    {
                        foreach (Route route in mr.node_get(lvl, pos).all_valid_routes())
                        {
                            // this is a route passing through a neighbour,
                            // but is it a real neighbour or it is just another
                            // address of mine?
                            AggregatedNeighbour aggregated_neighbour = route.gw;
                            if (! aggregated_neighbour.is_local())
                            {
                                if (newpaths.has_key(aggregated_neighbour.nip))
                                {
                                    CloneRoute old_route = newpaths[aggregated_neighbour.nip];
                                    if (old_route.rem.compare_to(route.rem) < 0)
                                        newpaths[aggregated_neighbour.nip] = route.clone();
                                }
                                else
                                {
                                    newpaths[aggregated_neighbour.nip] = route.clone();
                                }
                            }
                        }
                    }
                    // if there are any, add them to ret_value
                    if (! newpaths.is_empty)
                    {
                        CondensedPathChooser x = new CondensedPathChooser();
                        x.paths.add_all(newpaths.values);
                        ret_value.add(key, x);
                    }
                }
            if (lvl > 0)
            {
                // for each different address in this level go inner...
                foreach (int pos in distinct_pos.keys)
                {
                    int[] newprefix_pos = prefix.get_positions();
                    newprefix_pos[lvl] = pos;
                    PartialNIP newprefix = new PartialNIP(newprefix_pos);
                    get_structure(ref ret_value, newprefix, lvl-1, distinct_pos[pos]);
                }
            }
        }

        /** Gathers all available real neighbours in my network
          */
        public CondensedNeighbours get_condensed_neighbours()
        {
            HashMap<string, CondensedNeighbour> neighbours = new HashMap<string, CondensedNeighbour>();
            // key is tnip_nodeid_key(nip, nodeid)
            foreach(AddressManager addr in this)
            {
                Gee.List<AggregatedNeighbour> nl = addr.aggregated_neighbour_manager.neighbour_list(true);
                foreach (AggregatedNeighbour an in nl)
                {
                    // Is it one of my addresses?
                    if (! an.is_local())
                    {
                        string key = tnip_nodeid_key(an.nip, an.nodeid);
                        // Did I already meet this address?
                        if (neighbours.has_key(key))
                        {
                            CondensedNeighbour condensed_neighbour = neighbours[key];
                            condensed_neighbour.add_data(an.macs, an.dev, an.rem);
                        }
                        else
                        {
                            neighbours[key] = new CondensedNeighbour(an.levels, an.gsize,
                                    an.nip, an.macs, an.dev, an.rem);
                        }
                    }
                }
            }
            CondensedNeighbours ret = new CondensedNeighbours();
            ret.neighbours_list.add_all(neighbours.values);
            return ret;
        }
        private string tnip_nodeid_key(NIP key_nip, int key_nodeid)
        {
            string ret = "";
            int[] nip = key_nip.get_positions();
            foreach (int i in nip) ret += "%d_".printf(i);
            ret += "%d".printf(key_nodeid);
            return ret;
        }

        /** Gathers all available real incoming nodes in my network
          */
        public CondensedIncomingNodeContainer get_condensed_incoming_nodes()
        {
            HashMap<string, CondensedIncomingNode> incomings = new HashMap<string, CondensedIncomingNode>();
            // key is tnip_nodeid_key(nip, nodeid)
            foreach(AddressManager addr in this)
            {
                foreach (IncomingNode incoming_node in addr.incoming_nodes)
                {
                    // Is it one of my addresses?
                    bool false_friend = false;
                    foreach(AddressManager addr2 in this)
                        if (addr2.maproute.me.is_equal(incoming_node.nip))
                        {
                            false_friend = true;
                            break;
                        }
                    if (! false_friend)
                    {
                        // Did I already meet this address?
                        string key = tnip_nodeid_key(incoming_node.nip, incoming_node.nodeid);
                        if (incomings.has_key(key))
                        {
                            CondensedIncomingNode condensed_incoming = incomings[key];
                            condensed_incoming.add_data(incoming_node.macs);
                        }
                        else
                        {
                            incomings[key] = new CondensedIncomingNode(incoming_node.nip, incoming_node.macs);
                        }
                    }
                }
            }
            CondensedIncomingNodeContainer ret = new CondensedIncomingNodeContainer();
            ret.nodes_list.add_all(incomings.values);
            return ret;
        }
    }

    public class CondensedMapRoute : Object
    {
        public HashMap<PartialNIP, CondensedPathChooser> routes;
        public CondensedMapRoute()
        {
            routes = new HashMap<PartialNIP, CondensedPathChooser>(PartialNIP.hash_func, PartialNIP.equal_func);
        }
        public void add(PartialNIP key, CondensedPathChooser val)
        {
            routes[key] = val;
        }
    }

    /** List of paths towards a certain destination.
      */
    public class CondensedPathChooser : Object
    {
        public ArrayList<CloneRoute> paths;
        public CondensedPathChooser()
        {
            paths = new ArrayList<CloneRoute>(CloneRoute.equal_func);
        }
        public void sort()
        {
            // reversed sort
            CompareDataFunc<CloneRoute> cmp = (a,b) => {return b.compare_to(a);};
            paths.sort(cmp);
        }
        public CloneRoute? best_path()
        {
            sort();
            if (paths.is_empty) return null;
            return paths[0];
        }
        public CloneRoute? best_path_without(NIP nip)
        {
            sort();
            foreach (CloneRoute p in paths)
            {
                if (p.mymap_me.is_equal(nip)) continue;
                HCoord hop = p.nip_to_lvlid(nip);
                if (! p.contains(hop))
                    return p;
            }
            return null;
        }
        public CloneRoute? best_path_without_any(Gee.List<NIP> nips)
        {
            sort();
            foreach (CloneRoute p in paths)
            {
                bool valid = true;
                foreach (NIP nip in nips)
                {
                    if (p.mymap_me.is_equal(nip)) continue;
                    HCoord hop = p.nip_to_lvlid(nip);
                    if (p.contains(hop))
                    {
                        valid = false;
                        break;
                    }
                }
                if (valid)
                    return p;
            }
            return null;
        }
        public Gee.List<CloneRoute> all_paths()
        {
            sort();
            ArrayList<CloneRoute> ret = new ArrayList<CloneRoute>(CloneRoute.equal_func);
            foreach (CloneRoute p in paths)
            {
                if (p.rem.get_type() != typeof(DeadREM))
                    ret.add(p);
            }
            return ret;
        }
        public Gee.List<CloneRoute> all_paths_without(NIP nip)
        {
            sort();
            ArrayList<CloneRoute> ret = new ArrayList<CloneRoute>(CloneRoute.equal_func);
            foreach (CloneRoute p in paths)
            {
                if (p.rem.get_type() != typeof(DeadREM))
                {
                    if (p.mymap_me.is_equal(nip)) continue;
                    HCoord hop = p.nip_to_lvlid(nip);
                    if (! p.contains(hop))
                        ret.add(p);
                }
            }
            return ret;
        }
        public Gee.List<CloneRoute> all_paths_without_any(Gee.List<NIP> nips)
        {
            sort();
            ArrayList<CloneRoute> ret = new ArrayList<CloneRoute>(CloneRoute.equal_func);
            foreach (CloneRoute p in paths)
            {
                if (p.rem.get_type() != typeof(DeadREM))
                {
                    bool valid = true;
                    foreach (NIP nip in nips)
                    {
                        if (p.mymap_me.is_equal(nip)) continue;
                        HCoord hop = p.nip_to_lvlid(nip);
                        if (p.contains(hop))
                        {
                            valid = false;
                            break;
                        }
                    }
                    if (valid)
                        ret.add(p);
                }
            }
            return ret;
        }
    }

    /** A list of all the real neighbours (not my addresses) and for
      * each nip just one CondensedNeighbour with all the macs.
      */
    public class CondensedNeighbours : Object
    {
        public ArrayList<CondensedNeighbour> neighbours_list;
        public CondensedNeighbours()
        {
            neighbours_list = new ArrayList<CondensedNeighbour>(CondensedNeighbour.equal_func);
        }
        public Gee.List<CondensedNeighbour> get_neighbours_with_mac(string mac)
        {
            // returns all the CondensedNeighbour which have a mac
            ArrayList<CondensedNeighbour> ret = new ArrayList<CondensedNeighbour>(CondensedNeighbour.equal_func);
            foreach (CondensedNeighbour condensed_neighbour in neighbours_list)
                if (condensed_neighbour.macs.contains(mac))
                    ret.add(condensed_neighbour);
            return ret;
        }
        public Gee.List<string> get_all_macs()
        {
            // returns all the MACs of my real neighbours
            ArrayList<string> ret = new ArrayList<string>();
            foreach (CondensedNeighbour condensed_neighbour in neighbours_list)
                foreach (string mac in condensed_neighbour.macs)
                    if (! ret.contains(mac))
                        ret.add(mac);
            return ret;
        }
    }

    public class CondensedNeighbour : Object
    {
        public int levels;
        public int gsize;
        public NIP nip;
        public ArrayList<string> macs;
        public string bestdev;
        public REM bestrem;
        public CondensedNeighbour(int levels, int gsize, NIP nip, Gee.List<string> macs, string dev, REM rem)
        {
            try
            {
                this.levels = levels;
                this.gsize = gsize;
                this.nip = (NIP)ISerializable.deserialize(nip.serialize());
                this.macs = new ArrayList<string>();
                this.macs.add_all(macs);
                bestdev = dev;
                bestrem = (REM)ISerializable.deserialize(rem.serialize());
            }
            catch (SerializerError e)
            {
                error(@"CondensedNeighbour: Caught exception while deserializing members $(e.domain) code $(e.code): $(e.message)");
            }
        }
        public void add_data(Gee.List<string> macs, string dev, REM rem)
        {
            try
            {
                foreach (string mac in macs)
                    if (! this.macs.contains(mac))
                        this.macs.add(mac);
                if (rem.compare_to(bestrem) > 0)
                {
                    bestdev = dev;
                    bestrem = (REM)ISerializable.deserialize(rem.serialize());
                }
            }
            catch (SerializerError e)
            {
                error(@"CondensedNeighbour.add_data(): Caught exception while deserializing members $(e.domain) code $(e.code): $(e.message)");
            }
        }

        public static bool equal_func(CondensedNeighbour? a, CondensedNeighbour? b)
        {
            if (a == b) return true;
            if (a == null || b == null) return false;
            string a_list = @"$(a.levels)_$(a.gsize)_$(a.nip)_$(a.bestdev)_$(a.bestrem)_";
            foreach (string mac in a.macs) a_list += @"$(mac)_";
            string b_list = @"$(b.levels)_$(b.gsize)_$(b.nip)_$(b.bestdev)_$(b.bestrem)_";
            foreach (string mac in b.macs) b_list += @"$(mac)_";
            if (a_list != b_list) return false;
            return true;
        }
    }

    /** A list of all the incoming nodes and for
      * each nip just one CondensedIncomingNode with all the macs.
      */
    public class CondensedIncomingNodeContainer : Object
    {
        public ArrayList<CondensedIncomingNode> nodes_list;
        public CondensedIncomingNodeContainer()
        {
            nodes_list = new ArrayList<CondensedIncomingNode>(CondensedIncomingNode.equal_func);
        }
        public Gee.List<CondensedIncomingNode> get_nodes_with_mac(string mac)
        {
            // returns all the incoming nodes which have a mac
            ArrayList<CondensedIncomingNode> ret = new ArrayList<CondensedIncomingNode>(CondensedIncomingNode.equal_func);
            foreach (CondensedIncomingNode condensed_incoming_node in nodes_list)
                if (condensed_incoming_node.macs.contains(mac))
                    ret.add(condensed_incoming_node);
            return ret;
        }
        public Gee.List<string> get_all_macs()
        {
            // returns all the MACs of my incoming nodes
            ArrayList<string> ret = new ArrayList<string>();
            foreach (CondensedIncomingNode condensed_incoming_node in nodes_list)
                foreach (string mac in condensed_incoming_node.macs)
                    if (! ret.contains(mac))
                        ret.add(mac);
            return ret;
        }
    }

    public class CondensedIncomingNode : Object
    {
        public NIP nip;
        public ArrayList<string> macs;
        public CondensedIncomingNode(NIP nip, Gee.List<string> macs)
        {
            try
            {
                this.nip = (NIP)ISerializable.deserialize(nip.serialize());
                this.macs = new ArrayList<string>();
                this.macs.add_all(macs);
            }
            catch (SerializerError e)
            {
                error(@"CondensedIncomingNode: Caught exception while deserializing members $(e.domain) code $(e.code): $(e.message)");
            }
        }
        public void add_data(Gee.List<string> macs)
        {
            foreach (string mac in macs)
                if (! this.macs.contains(mac))
                    this.macs.add(mac);
        }

        public static bool equal_func(CondensedIncomingNode? a, CondensedIncomingNode? b)
        {
            if (a == b) return true;
            if (a == null || b == null) return false;
            string a_list = @"$(a.nip)_";
            foreach (string mac in a.macs) a_list += @"$(mac)_";
            string b_list = @"$(b.nip)_";
            foreach (string mac in b.macs) b_list += @"$(mac)_";
            if (a_list != b_list) return false;
            return true;
        }
    }
}

