/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2011 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gee;
using zcd;
using Tasklets;

namespace Netsukuku
{
    struct struct_helper_MigrationManager_primary_evaluate_migration
    {
        public MigrationManager self;
    }

    struct struct_helper_MigrationManager_secondary_evaluate_removal
    {
        public MigrationManager self;
    }

    /** Each node periodically evaluates whether it is
      *  convenient or not to migrate into another gnode.
      */
    public class MigrationManager : Object
    {
        public weak AddressManager address_manager {get; private set;}
        public weak AggregatedNeighbourManager aggregated_neighbour_manager {get; private set;}
        public weak MapRoute maproute {get; private set;}
        private string ipstr;
        public REM threshold {get; private set;}
        private Tasklet? primary_evaluate_migration_handle = null;
        private Tasklet? secondary_evaluate_removal_handle = null;
        public MigrationManager(AddressManager address_manager, MapRoute maproute, AggregatedNeighbourManager aggregated_neighbour_manager)
        {
            this.address_manager = address_manager;
            this.address_manager.is_primary_changed.connect(impl_is_primary_changed);
            this.maproute = maproute;
            ipstr = nip_to_str(maproute.levels, maproute.gsize, maproute.me);
            this.aggregated_neighbour_manager = aggregated_neighbour_manager;
            int settings_migration_capability = Settings.MIGRATION_CAPABILITY;
            if (settings_migration_capability == 1) threshold = new RTT(8000);
            else if (settings_migration_capability == 2) threshold = new RTT(800);
            else if (settings_migration_capability == 3) threshold = new RTT(80);
            else
            {
                log_warn("MigrationManager: Invalid value for MIGRATION_CAPABILITY. Using default.");
                threshold = new RTT(80);
            }
        }

        /** Called after address_manager is_mature
          */
        public virtual void start_operations()
        {
            if (address_manager.is_primary) start_primary_evaluate_migration();
            else start_secondary_evaluate_removal();
        }

        public virtual void stop_operations()
        {
            log_debug(@"Etp: stopping operations for $(ipstr)");
            ms_wait(100);  // to make sure the micros have started
            stop_secondary_evaluate_removal();
            stop_primary_evaluate_migration();
        }

        public void impl_is_primary_changed()
        {
            if (address_manager.is_primary) start_primary_evaluate_migration();
        }

        public void start_primary_evaluate_migration()
        {
            if (primary_evaluate_migration_handle == null)
            {
                primary_evaluate_migration();
                while (primary_evaluate_migration_handle == null) ms_wait(1);
            }
        }

        public void stop_primary_evaluate_migration()
        {
            if (primary_evaluate_migration_handle != null)
            {
                log_debug(@"MigrationManager: aborting primary_evaluate_migration for $(ipstr)");
                primary_evaluate_migration_handle.abort();
                primary_evaluate_migration_handle = null;
            }
        }

        private void impl_primary_evaluate_migration() throws Error
        {
            Tasklet.declare_self("MigrationManager.primary_evaluate_migration");
            stop_secondary_evaluate_removal();
            primary_evaluate_migration_handle = Tasklet.self();

            log_debug(@"MigrationManager: for $(ipstr): primary_evaluate_migration started.");
            while (true)
            {
                try
                {
                    ms_wait(60000);
                    log_debug(@"MigrationManager: for $(ipstr): start to evaluate possible migrations");
                    for (int lvl = maproute.levels-1; lvl > 0; lvl--)
                    {
                        log_debug(@"MigrationManager: for $(ipstr): Is there any free position in $(maproute.me.get_gnode_at_level(lvl+1))?");
                        if (maproute.free_nodes_nb(lvl) > 0)
                        {
                            REM my_worst = maproute.worst_internal_bestrem(lvl);
                            log_debug(@"MigrationManager: for $(ipstr): Yes... My worst_internal_bestrem at level" +
                                    @" $(lvl) is $(my_worst). Shall I migrate?");
                            if (my_worst.compare_to(threshold) < 0)
                            {
                                log_debug(@"MigrationManager: for $(ipstr): Yes... Hook inside $(maproute.me.get_gnode_at_level(lvl+1))");
                                // obtain a reservation
                                HookReservation hook_reservation = address_manager.coordnode.enter_into(lvl+1, maproute.me, null);
                                // send an event and obtain the new AddressManager instance
                                AddressManager new_addr_man = null;
                                address_manager.address_manager_new(hook_reservation, address_manager, ref new_addr_man);
                                // wait for the new address_manager to be mature
                                while (! new_addr_man.is_mature) ms_wait(1000);
                                // now use the new instance to request to become primary
                                new_addr_man.request_be_primary(new_addr_man);
                                // That has restarted primary_evaluate_migration in the new primary address_manager
                                // Now we are non_primary, so:
                                start_secondary_evaluate_removal();
                                log_debug(@"MigrationManager: for $(ipstr): exiting beacuse we are not primary.");
                                return;
                            }
                            else
                            {
                                log_debug(@"MigrationManager: for $(ipstr): No.");
                            }
                        }
                        else
                        {
                            log_debug(@"MigrationManager: for $(ipstr): No.");
                        }
                        // Gather neighbours which belong to my gnode lvl+1 but not to my gnode lvl
                        ArrayList<AggregatedNeighbour> neighbours = new ArrayList<AggregatedNeighbour>(AggregatedNeighbour.equal_func);
                        foreach (AggregatedNeighbour n in aggregated_neighbour_manager.neighbour_list(true))
                        {
                            if (! n.is_local() && n.is_primary && maproute.nip_cmp(n.nip.get_positions()) == lvl)
                                neighbours.add(n);
                        }
                        log_debug(@"MigrationManager: for $(ipstr): Number of direct neighbours" +
                                @" which are in $(maproute.me.get_gnode_at_level(lvl+1)) but not" +
                                @" in $(maproute.me.get_gnode_at_level(lvl)) is $(neighbours.size)");

                        REM best_choice_worstrem = new AlmostDeadREM();
                        AggregatedNeighbour? best_choice_neighbour = null;
                        AddressManager? best_choice_secondary_address_manager = null;
                        foreach(AggregatedNeighbour aggregated_neighbour in neighbours)
                        {
                            try
                            {
                                PartialNIP candidate_gnode = aggregated_neighbour.nip.get_gnode_at_level(lvl);
                                log_debug(@"MigrationManager: for $(ipstr): Since we see $(aggregated_neighbour.nip)" +
                                        @" we evaluate migrating into $(candidate_gnode)");
                                AddressManager? secondary_address_manager = null;
                                foreach (AddressManager addr_man in Addresses.get_addresses_instance().get_autonomous_addresses())
                                {
                                    if (addr_man != address_manager && maproute.nip_cmp(addr_man.maproute.me.get_positions()) == lvl)
                                    {
                                        secondary_address_manager = addr_man;
                                        break;
                                    }
                                }
                                if (secondary_address_manager != null)
                                {
                                    // We have a secondary address exactly where we want to go/examine.
                                    //  So we do not need to check for place availability.
                                    log_debug(@"MigrationManager: for $(ipstr): We already have a secondary into that: $(secondary_address_manager.maproute.me)");
                                    REM candidate_worstrem = secondary_address_manager.maproute.worst_internal_bestrem(lvl);
                                    log_debug(@"MigrationManager: for $(ipstr): Its worst_internal_bestrem is $(candidate_worstrem)");
                                    if (best_choice_worstrem.compare_to(candidate_worstrem) < 0)
                                    {
                                        log_debug(@"MigrationManager: for $(ipstr): Interesting.");
                                        best_choice_worstrem = candidate_worstrem;
                                        best_choice_neighbour = null;
                                        best_choice_secondary_address_manager = secondary_address_manager;
                                    }
                                    else
                                    {
                                        log_debug(@"MigrationManager: for $(ipstr): Not interesting.");
                                    }
                                }
                                else
                                {
                                    // We do need to check for place availability.
                                    log_debug(@"MigrationManager: for $(ipstr): Ask to $(aggregated_neighbour.nip)" +
                                            @" if there is room in $(candidate_gnode)");
                                    bool confirm = false;
                                    foreach (PairLvlNumberOfFreeNodes candidate in
                                                aggregated_neighbour.neighbour_client.hook.list_non_saturated_levels())
                                    {
                                        int candidate_lvl = candidate.lvl;
                                        int candidate_fn = candidate.number_of_free_nodes;
                                        if (candidate_lvl <= lvl && candidate_fn > 0)
                                        {
                                            confirm = true;
                                            break;
                                        }
                                    }
                                    if (confirm)
                                    {
                                        log_debug(@"MigrationManager: for $(ipstr): Yes... Ask to him the worst_internal_bestrem.");
                                        REM candidate_worstrem = aggregated_neighbour.neighbour_client.maproute.worst_internal_bestrem(lvl);
                                        log_debug(@"MigrationManager: for $(ipstr): Got $(candidate_worstrem)");
                                        if (best_choice_worstrem.compare_to(candidate_worstrem) < 0)
                                        {
                                            log_debug(@"MigrationManager: for $(ipstr): Interesting.");
                                            best_choice_worstrem = candidate_worstrem;
                                            best_choice_neighbour = aggregated_neighbour;
                                            best_choice_secondary_address_manager = null;
                                        }
                                        else
                                        {
                                            log_debug(@"MigrationManager: for $(ipstr): Not interesting.");
                                        }
                                    }
                                }
                            }
                            catch (Error e)
                            {
                                log_debug(@"Caught exception while trying to evaluate migration through $(aggregated_neighbour.nip): $(e.domain) code $(e.code): $(e.message)");
                            }
                        }
                        if (best_choice_neighbour != null)
                        {
                            log_debug(@"MigrationManager: for $(ipstr): Best alternative is neighbour " +
                                    @"$(best_choice_neighbour.nip)" +
                                    @" which rem to me is $(best_choice_neighbour.rem) and" +
                                    @" which worstrem in level $(lvl) is $(best_choice_worstrem)");
                            REM my_worst = maproute.worst_internal_bestrem(lvl);
                            log_debug(@"MigrationManager: for $(ipstr): My worst_internal_bestrem at level $(lvl)" +
                                    @" is $(my_worst). Shall I migrate?");
                            REM compare = threshold.add_segment(best_choice_worstrem).add_segment(best_choice_neighbour.rem);
                            if (my_worst.compare_to(compare) < 0)
                            {
                                log_debug(@"MigrationManager: for $(ipstr): Yes... Hook to him.");
                                // obtain a reservation
                                Gee.List<AggregatedNeighbour> nlist = new ArrayList<AggregatedNeighbour>(AggregatedNeighbour.equal_func);
                                nlist.add(best_choice_neighbour);
                                HookReservation hook_reservation = address_manager.hook.hook(nlist);
                                // send an event and obtain the new AddressManager instance
                                AddressManager new_addr_man = null;
                                address_manager.address_manager_new(hook_reservation, address_manager, ref new_addr_man);
                                // wait for the new address_manager to be mature
                                while (! new_addr_man.is_mature) ms_wait(1000);
                                // now use the new instance to request to become primary
                                new_addr_man.request_be_primary(new_addr_man);
                                // That has restarted primary_evaluate_migration in the new primary address_manager
                                // Now we are non_primary, so:
                                start_secondary_evaluate_removal();
                                log_debug(@"MigrationManager: for $(ipstr): exiting beacuse we are not primary.");
                                return;
                            }
                            else
                            {
                                log_debug(@"MigrationManager: for $(ipstr): No.");
                            }

                        }
                        else if (best_choice_secondary_address_manager != null)
                        {
                            log_debug(@"MigrationManager: for $(ipstr): Best alternative is secondary " +
                                    @"$(best_choice_secondary_address_manager.maproute.me)" +
                                    @" which worstrem in level $(lvl) is $(best_choice_worstrem)");
                            REM my_worst = maproute.worst_internal_bestrem(lvl);
                            log_debug(@"MigrationManager: for $(ipstr): My worst_internal_bestrem at level $(lvl)" +
                                    @" is $(my_worst). Shall I migrate?");
                            REM compare = threshold.add_segment(best_choice_worstrem);
                            if (my_worst.compare_to(compare) < 0)
                            {
                                log_debug(@"MigrationManager: for $(ipstr): Yes... use secondary.");
                                // now secondary requests to become primary
                                best_choice_secondary_address_manager.request_be_primary(best_choice_secondary_address_manager);
                                // That has restarted primary_evaluate_migration in the new primary address_manager
                                // Now we are non_primary, so:
                                start_secondary_evaluate_removal();
                                log_debug(@"MigrationManager: for $(ipstr): exiting beacuse we are not primary.");
                                return;
                            }
                            else
                            {
                                log_debug(@"MigrationManager: for $(ipstr): No.");
                            }
                        }
                    }
                }
                catch (Error e)
                {
                    log_error(@"Caught exception while primary_evaluate_migration $(e.domain) code $(e.code): $(e.message)");
                }
            }
        }

        private static void * helper_primary_evaluate_migration(void *v) throws Error
        {
            struct_helper_MigrationManager_primary_evaluate_migration *tuple_p = (struct_helper_MigrationManager_primary_evaluate_migration *)v;
            // The caller function has to add a reference to the ref-counted instances
            MigrationManager self_save = tuple_p->self;
            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            self_save.impl_primary_evaluate_migration();
            // void method, return null
            return null;
        }

        public void primary_evaluate_migration()
        {
            struct_helper_MigrationManager_primary_evaluate_migration arg = struct_helper_MigrationManager_primary_evaluate_migration();
            arg.self = this;
            Tasklet.spawn((FunctionDelegate)helper_primary_evaluate_migration, &arg);
        }

        public void start_secondary_evaluate_removal()
        {
            if (secondary_evaluate_removal_handle == null)
            {
                secondary_evaluate_removal();
                while (secondary_evaluate_removal_handle == null) ms_wait(1);
            }
        }

        public void stop_secondary_evaluate_removal()
        {
            if (secondary_evaluate_removal_handle != null)
            {
                log_debug(@"MigrationManager: aborting secondary_evaluate_removal for $(ipstr)");
                secondary_evaluate_removal_handle.abort();
                secondary_evaluate_removal_handle = null;
            }
        }

        private void impl_secondary_evaluate_removal() throws Error
        {
            Tasklet.declare_self("MigrationManager.secondary_evaluate_removal");
            secondary_evaluate_removal_handle = Tasklet.self();

            log_debug(@"MigrationManager: for $(ipstr): secondary_evaluate_removal started.");
            Tasklets.Timer timeout_active_connections = new Tasklets.Timer(7200000);
            Tasklets.Timer timeout_gnode_would_split  = new Tasklets.Timer(600000);
            while (true)
            {
                try
                {
                    ms_wait(120000);
                    // Do I want to make sure that there are no active TCP connections?
                    //  I occupy a precious resource, so after a
                    //  maximum time I want to free the address.
                    if (timeout_active_connections.is_expired())
                        break;
                    // Are there active TCP connections on this address?
                    bool prevent = false;
                    try
                    {
                        prevent = address_manager.would_prevent_removal();
                    }
                    catch (Error e)
                    {
                        log_error(@"Caught exception while secondary_evaluate_removal (prevent) $(e.domain) code $(e.code): $(e.message); continue anyway.");
                    }
                    if (! prevent)
                    {
                        // No active TCP connections. Do I want to make sure that gnode won't split?
                        //  Anyway split is not sure, and I occupy a precious resource, so after a
                        //  maximum time I want to free the address.
                        if (timeout_gnode_would_split.is_expired())
                            break;
                        // On the other hand I can be sure that there won't be any split.
                        //  In this case, immediately free the address.
                        if (! address_manager.would_cause_split())
                            break;
                    }
                }
                catch (Error e)
                {
                    log_error(@"Caught exception while secondary_evaluate_removal $(e.domain) code $(e.code): $(e.message); continue anyway.");
                }
            }
            // We must die.
            address_manager.address_manager_delete(address_manager);
        }

        private static void * helper_secondary_evaluate_removal(void *v) throws Error
        {
            struct_helper_MigrationManager_secondary_evaluate_removal *tuple_p = (struct_helper_MigrationManager_secondary_evaluate_removal *)v;
            // The caller function has to add a reference to the ref-counted instances
            MigrationManager self_save = tuple_p->self;
            // schedule back to the spawner; this will probably invalidate *v and *tuple_p.
            Tasklet.schedule_back();
            // The actual call
            self_save.impl_secondary_evaluate_removal();
            // void method, return null
            return null;
        }

        public void secondary_evaluate_removal()
        {
            struct_helper_MigrationManager_secondary_evaluate_removal arg = struct_helper_MigrationManager_secondary_evaluate_removal();
            arg.self = this;
            Tasklet.spawn((FunctionDelegate)helper_secondary_evaluate_removal, &arg);
        }
    }
}

