/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2011 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gee;
using zcd;

namespace Netsukuku
{
    public class IncomingNode : Object
    {
        public NIP nip {get; private set;}
        public int nodeid {get; private set;}
        public Tasklets.Timer tcapsule {get; private set;}
        public Gee.List<string> macs {get; private set;}
        private TreeMap<int, RoutesSet> knowledge;

        public IncomingNode(NIP nip, int nodeid, Gee.List<string> macs, int? ttl=null)
        {
            this.nip = nip;
            this.nodeid = nodeid;
            this.macs = macs;
            int _ttl = 60000;
            if (ttl != null) _ttl = ttl;
            this.tcapsule = new Tasklets.Timer(_ttl);
            this.knowledge = new TreeMap<int, RoutesSet>();
        }

        public void update(Gee.List<string> macs, int? ttl=null)
        {
            this.macs = macs;
            int _ttl = 60000;
            if (ttl != null) _ttl = ttl;
            this.tcapsule = new Tasklets.Timer(_ttl);
        }

        public void log_knowledge(int my_seq_num, RoutesSet destinations)
        {
            // max 3 elements
            if (! knowledge.is_empty)
            {
                if (knowledge.has_key(my_seq_num))
                    return;
                int min_key = knowledge.ascending_keys.first();
                int max_key = knowledge.ascending_keys.last();
                if (my_seq_num < max_key)
                    return;
                if (knowledge.size >= 3)
                    knowledge.unset(min_key);
            }
            knowledge[my_seq_num] = destinations;
        }

        public RoutesSet? get_knowledge(int prev_seq_num)
        {
            if (knowledge.has_key(prev_seq_num))
                return knowledge[prev_seq_num];
            else return null;
        }
    }

    public errordomain IncomingNodesError {
        KEY_NOT_FOUND,
        GENERIC
    }

    public class IncomingNodes : Object
    {
        private HashMap<string, IncomingNode> cont;

        // This event is emitted at each kind of variation (new, delete)
        //  about a MAC. The argument is the mac.
        // E.g.: an update, when the same nip,nodeid was present AND
        //       the mac was already there: no UPDATED event.
        // E.g.: a new nip,nnodeid pair: one UPDATED for each mac.
        public signal void incoming_node_updated(string mac);

        public IncomingNodes()
        {
            cont = new HashMap<string, IncomingNode>();
        }

        public Iterator<IncomingNode> iterator()
        {
            return cont.values.iterator();
        }

        private string tnip_nodeid_key(NIP key_nip, int key_nodeid)
        {
            string ret = "";
            int[] nip = key_nip.get_positions();
            foreach (int i in nip) ret += "%d_".printf(i);
            ret += "%d".printf(key_nodeid);
            return ret;
        }
        private void key_tnip_nodeid(string key, out NIP key_nip, out int key_nodeid)
        {
            string[] nums = key.split("_");
            int[] pos = new int[nums.length - 1];
            for (int i = 0; i < pos.length; i++) pos[i] = int.parse(nums[i]);
            key_nodeid = int.parse(nums[nums.length - 1]);
            key_nip = new NIP(pos);
        }

        public bool contains(NIP nip, int nodeid)
        {
            // Does this node exist?
            return cont.has_key(tnip_nodeid_key(nip, nodeid));
        }

        public Gee.List<string> get_macs(NIP nip, int nodeid) throws IncomingNodesError
        {
            // Retrieves the MACs of this node
            string k = tnip_nodeid_key(nip, nodeid);
            if (! cont.has_key(k))
                throw new IncomingNodesError.KEY_NOT_FOUND("IncomingNode not present");
            return cont[k].macs;
        }

        public void update(NIP nip, int nodeid, Gee.List<string> macs, int? ttl=null)
        {
            // adds or updates node nipX,nodeidX with these MACs
            string k = tnip_nodeid_key(nip, nodeid);
            ArrayList<string> to_send_event = new ArrayList<string>();
            if (cont.has_key(k))
            {
                // already there... an event for each mac new or missing
                foreach (string mac in macs)
                    if (! cont[k].macs.contains(mac))
                        to_send_event.add(mac);
                foreach (string mac in cont[k].macs)
                    if (! macs.contains(mac))
                        to_send_event.add(mac);
                ArrayList<string> macs_clone = new ArrayList<string>();
                foreach (string mac in macs) macs_clone.add(mac);
                cont[k].update(macs_clone, ttl);
            }
            else
            {
                // an event for each mac
                foreach (string mac in macs)
                    to_send_event.add(mac);
                ArrayList<string> macs_clone = new ArrayList<string>();
                foreach (string mac in macs) macs_clone.add(mac);
                cont[k] = new IncomingNode(nip, nodeid, macs_clone, ttl);
            }
            // structures are updated. now send the event.
            foreach (string mac in to_send_event)
                incoming_node_updated(mac);
        }

        public void purge()
        {
            // clean up
            ArrayList<string> to_send_event = new ArrayList<string>();
            ArrayList<string> cont_keys = new ArrayList<string>();
            foreach (string k in cont.keys) cont_keys.add(k);
            foreach (string k in cont_keys)
            {
                if (cont[k].tcapsule.is_expired())
                {
                    foreach (string mac in cont[k].macs)
                        if (! to_send_event.contains(mac))
                            to_send_event.add(mac);
                    cont.unset(k);
                }
            }
            // structures are updated. now send the event.
            foreach (string mac in to_send_event)
                incoming_node_updated(mac);
        }

        public void log_knowledge(NIP nip, int nodeid, int my_seq_num, RoutesSet destinations) throws IncomingNodesError
        {
            string k = tnip_nodeid_key(nip, nodeid);
            if (! cont.has_key(k))
                throw new IncomingNodesError.KEY_NOT_FOUND("IncomingNode not present");
            cont[k].log_knowledge(my_seq_num, destinations);
        }

        public RoutesSet? get_knowledge(NIP nip, int nodeid, int prev_seq_num) throws IncomingNodesError
        {
            string k = tnip_nodeid_key(nip, nodeid);
            if (! cont.has_key(k))
                throw new IncomingNodesError.KEY_NOT_FOUND("IncomingNode not present");
            return cont[k].get_knowledge(prev_seq_num);
        }
    }
}

