/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2014 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gee;
using zcd;
using Netsukuku;

namespace Andns
{
    namespace RpcAndns
    {
        public void init()
        {
            // Register serializable types
            typeof(AndnaQuery).class_peek();
            typeof(AndnaResponse).class_peek();
            typeof(AndnaResponseAnswer).class_peek();
        }
    }

    /** This is the interface for a root-dispatcher class
      */
    public interface IAndnaResolveRootDispatcher : Object
    {
        public abstract IAndnaQueryHandler get_query_handler(int request_id);
    }

    /** This is the interface for the remotable methods of the class Directory
      */
    public interface IAndnaQueryHandler : Object
    {
        public abstract AndnaResponse resolve(AndnaQuery query) throws RPCError;
    }

    public enum AndnaQueryType { NAME_TO_IP = 0, IP_TO_NAME = 1, GLOBAL = 2 }
    public enum AndnaQueryProtocol { TCP = 0, UDP = 1 }
    public enum AndnaQueryIPVersion { IPV4 = 0, IPV6 = 1 }
    public enum AndnaQueryRealm { NTK = 1, INT = 2 }
    public enum AndnaResponseCode {
        NO_ERROR = 0,
        MALFORMED = 1,
        SERV_FAIL = 2,
        NO_DOMAIN = 3,
        NOT_IMPL = 4,
        REFUSED = 5
    }

    /** A query.
      */
    public class AndnaQuery : Object, ISerializable
    {
        public AndnaQueryType query_type {get; private set;}
        public bool recursion {get; private set;}
        public bool compression {get; private set;}
        public AndnaQueryProtocol protocol {get; private set;}
        public AndnaQueryIPVersion ip_version {get; private set;}
        public AndnaQueryRealm ip_realm {
                    get;
                    private set;
                    /* Here default is necessary because 0 is not a valid value */
                    default=AndnaQueryRealm.NTK;}

        /* Only for AndnaQueryType.NAME_TO_IP && AndnaQueryRealm.NTK */
        public string hashed_domain {get; private set;}
        public AndnaServiceKey service {get; private set;}

        /* Only for AndnaQueryType.NAME_TO_IP && AndnaQueryRealm.INT */
        public string inet_hostname {get; private set;}
        public int inet_service {get; private set;}   // 0 or 25, TCP implied.

        /* Only for AndnaQueryType.IP_TO_NAME && AndnaQueryIPVersion.IPV4 */
        public uint8 ipv4_ip[4];
        // ipv4_ip[0] is the most significant byte.
        //  {0,0,0,1} means "0.0.0.1" and {1,2,3,4} means "1.2.3.4"
        //  no matter the endianness of the host.

        /* Only for AndnaQueryType.IP_TO_NAME && AndnaQueryIPVersion.IPV6 */
        public uint8 ipv6_ip[16];
        // ipv6_ip[0] is the most significant byte.
        //  {0,0,...,1} means "::1"
        //  {32,1,0,...,1} means "2001::1"
        //  no matter the endianness of the host.

        /* Only for AndnaQueryType.GLOBAL */
        public string global_hashed_domain {get; private set;}

        private AndnaQuery.priv(AndnaQueryType query_type,
                                bool recursion,
                                bool compression,
                                AndnaQueryProtocol protocol,
                                AndnaQueryIPVersion ip_version,
                                AndnaQueryRealm ip_realm)
        {
            this.query_type = query_type;
            this.recursion = recursion;
            this.compression = compression;
            this.protocol = protocol;
            this.ip_version = ip_version;
            this.ip_realm = ip_realm;
            // fields that we need to initialize no matters what
            hashed_domain = "";
            service = AndnaServiceKey.NULL_SERV_KEY;
            inet_hostname = "";
            global_hashed_domain = "";
        }

        public AndnaQuery.name_to_ip_ntk(
                                bool recursion,
                                bool compression,
                                AndnaQueryProtocol protocol,
                                AndnaQueryIPVersion ip_version,
                                string hashed_domain,
                                AndnaServiceKey service)
        {
            this.priv ( AndnaQueryType.NAME_TO_IP,
                        recursion,
                        compression,
                        protocol,
                        ip_version,
                        AndnaQueryRealm.NTK);
            this.hashed_domain = hashed_domain;
            this.service = service;
        }

        public AndnaQuery.ip_to_name_ntk(
                                bool compression,
                                AndnaQueryProtocol protocol,
                                AndnaQueryIPVersion ip_version,
                                uint8[] ip)
        {
            this.priv ( AndnaQueryType.IP_TO_NAME,
                        false,
                        compression,
                        protocol,
                        ip_version,
                        AndnaQueryRealm.NTK);
            if (ip_version == AndnaQueryIPVersion.IPV4)
            {
                for (int i = 0; i < ipv4_ip.length; i++)
                    ipv4_ip[i] = ip[i];
            }
            else
            {
                for (int i = 0; i < ipv6_ip.length; i++)
                    ipv6_ip[i] = ip[i];
            }
        }

        public Variant serialize_to_variant()
        {
            int i_query_type = (int)query_type;
            int i_protocol = (int)protocol;
            int i_ip_version = (int)ip_version;
            int i_ip_realm = (int)ip_realm;
            int i_recursion = recursion ? 1 : 0;
            int i_compression = compression ? 1 : 0;
            SerializableBuffer b_ipv4_ip = new SerializableBuffer(ipv4_ip);
            SerializableBuffer b_ipv6_ip = new SerializableBuffer(ipv6_ip);

            Variant v0 = Serializer.int_to_variant(i_query_type);
            Variant v1 = Serializer.int_to_variant(i_protocol);
            Variant v2 = Serializer.int_to_variant(i_ip_version);
            Variant v3 = Serializer.int_to_variant(i_ip_realm);
            Variant v4 = Serializer.int_to_variant(i_recursion);
            Variant v5 = Serializer.int_to_variant(i_compression);
            Variant v6 = Serializer.string_to_variant(hashed_domain);
            Variant v7 = service.serialize_to_variant();
            Variant v8 = Serializer.string_to_variant(inet_hostname);
            Variant v9 = Serializer.int_to_variant(inet_service);
            Variant vA = b_ipv4_ip.serialize_to_variant();
            Variant vB = b_ipv6_ip.serialize_to_variant();
            Variant vC = Serializer.string_to_variant(global_hashed_domain);

            Variant vtemp0 = Serializer.tuple_to_variant_5(v0, v1, v2, v3, v4);
            Variant vtemp1 = Serializer.tuple_to_variant_5(vtemp0, v5, v6, v7, v8);
            Variant vret = Serializer.tuple_to_variant_5(vtemp1, v9, vA, vB, vC);
            return vret;
        }

        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            Variant v0;
            Variant v1;
            Variant v2;
            Variant v3;
            Variant v4;
            Variant v5;
            Variant v6;
            Variant v7;
            Variant v8;
            Variant v9;
            Variant vA;
            Variant vB;
            Variant vC;
            Variant vtemp0;
            Variant vtemp1;
            Serializer.variant_to_tuple_5(v, out vtemp1, out v9, out vA, out vB, out vC);
            Serializer.variant_to_tuple_5(vtemp1, out vtemp0, out v5, out v6, out v7, out v8);
            Serializer.variant_to_tuple_5(vtemp0, out v0, out v1, out v2, out v3, out v4);

            int i_query_type = Serializer.variant_to_int(v0);
            int i_protocol = Serializer.variant_to_int(v1);
            int i_ip_version = Serializer.variant_to_int(v2);
            int i_ip_realm = Serializer.variant_to_int(v3);
            int i_recursion = Serializer.variant_to_int(v4);
            int i_compression = Serializer.variant_to_int(v5);
            hashed_domain = Serializer.variant_to_string(v6);
            service = (AndnaServiceKey)Object.new(typeof(AndnaServiceKey));
            service.deserialize_from_variant(v7);
            inet_hostname = Serializer.variant_to_string(v8);
            inet_service = Serializer.variant_to_int(v9);
            {
                uint8 *buf = ipv4_ip;
                SerializableBuffer buf_ser =
                    (SerializableBuffer)Object.new(typeof(SerializableBuffer));
                buf_ser.deserialize_from_variant(vA);
                for (int i = 0; i < buf_ser.buffer.length; i++)
                {
                    buf[i] = buf_ser.buffer[i];
                }
            }
            {
                uint8 *buf = ipv6_ip;
                SerializableBuffer buf_ser =
                    (SerializableBuffer)Object.new(typeof(SerializableBuffer));
                buf_ser.deserialize_from_variant(vB);
                for (int i = 0; i < buf_ser.buffer.length; i++)
                {
                    buf[i] = buf_ser.buffer[i];
                }
            }
            global_hashed_domain = Serializer.variant_to_string(vC);

            query_type = (AndnaQueryType)i_query_type;
            protocol = (AndnaQueryProtocol)i_protocol;
            ip_version = (AndnaQueryIPVersion)i_ip_version;
            ip_realm = (AndnaQueryRealm)i_ip_realm;
            recursion = i_recursion == 1;
            compression = i_compression == 1;
        }
    }

    /** A response (list of answers).
      */
    public class AndnaResponse : Object, ISerializable
    {
        public AndnaResponseCode rcode {get; private set;}
        public ArrayList<AndnaResponseAnswer> answers {get; private set;}
        // answers.size <=> ANCOUNT

        public AndnaResponse(AndnaResponseCode _rcode,
                             ArrayList<AndnaResponseAnswer> _answers)
        {
            answers = new ArrayList<AndnaResponseAnswer>(AndnaResponseAnswer.equal_func);
            answers.add_all(_answers);
            rcode = _rcode;
        }

        public Variant serialize_to_variant()
        {
            int i_rcode = (int)rcode;
            Variant v0 = Serializer.int_to_variant(i_rcode);

            Variant v1;
            {
                ListISerializable lst = new ListISerializable();
                foreach (AndnaResponseAnswer o in answers) lst.add(o);
                v1 = lst.serialize_to_variant();
            }

            Variant vret = Serializer.tuple_to_variant(v0, v1);
            return vret;
        }

        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            Variant v0;
            Variant v1;
            Serializer.variant_to_tuple(v, out v0, out v1);

            int i_rcode = Serializer.variant_to_int(v0);
            rcode = (AndnaResponseCode)i_rcode;

            answers = new ArrayList<AndnaResponseAnswer>(AndnaResponseAnswer.equal_func);
            {
                ListISerializable lst = (ListISerializable)Object.new(typeof(ListISerializable));
                lst.deserialize_from_variant(v1);
                Gee.List<AndnaResponseAnswer> typed_lst = (Gee.List<AndnaResponseAnswer>)lst.backed;
                answers.add_all(typed_lst);
            }
        }
    }

    /** A answer.
      */
    public class AndnaResponseAnswer : Object, ISerializable
    {
        /* Only for AndnaQueryType.NAME_TO_IP */
        public bool is_ip {get; private set;}
        public int weight {get; private set;}
        public int priority {get; private set;}
        public uint8 ipv4_ip[4];
        public uint8 ipv6_ip[16];
        public string hashed_alias {get; private set;}
        public uint16 port_number {get; private set;}
        /* Only for AndnaQueryType.IP_TO_NAME */
        public string hostname {get; private set;}
        /* Only for AndnaQueryType.GLOBAL */
        public bool gl_is_main {get; private set;}
        public bool gl_is_ip {get; private set;}
        public AndnaServiceKey gl_service {get; private set;}
        public int gl_weight {get; private set;}
        public int gl_priority {get; private set;}
        public uint8 gl_ipv4_ip[4];
        public uint8 gl_ipv6_ip[16];
        public string gl_hashed_alias {get; private set;}
        public uint16 gl_port_number {get; private set;}

        private AndnaResponseAnswer.priv()
        {
            // fields that we need to initialize no matters what
            hashed_alias = "";
            gl_service = AndnaServiceKey.NULL_SERV_KEY;
            hostname = "";
            gl_hashed_alias = "";
        }

        public AndnaResponseAnswer.name_to_ip(
                                int weight,
                                int priority,
                                uint8[] ipv4_ip,
                                uint16 port_number)
        {
            this.priv ();
            this.is_ip = true;
            this.weight = weight;
            this.priority = priority;
            assert(this.ipv4_ip.length == ipv4_ip.length);
            for (int i = 0; i < ipv4_ip.length; i++)
            {
                this.ipv4_ip[i] = ipv4_ip[i];
            }
            this.port_number = port_number;
        }

        public AndnaResponseAnswer.name_to_ip_v6(
                                int weight,
                                int priority,
                                uint8[] ipv6_ip,
                                uint16 port_number)
        {
            this.priv ();
            this.is_ip = true;
            this.weight = weight;
            this.priority = priority;
            assert(this.ipv6_ip.length == ipv6_ip.length);
            for (int i = 0; i < ipv6_ip.length; i++)
            {
                this.ipv6_ip[i] = ipv6_ip[i];
            }
            this.port_number = port_number;
        }

        public AndnaResponseAnswer.name_to_alias(
                                int weight,
                                int priority,
                                string hashed_alias,
                                uint16 port_number)
        {
            this.priv ();
            this.is_ip = false;
            this.weight = weight;
            this.priority = priority;
            this.hashed_alias = hashed_alias;
            this.port_number = port_number;
        }

        public AndnaResponseAnswer.ip_to_name(
                                string hostname)
        {
            this.priv ();
            this.hostname = hostname;
        }

        public Variant serialize_to_variant()
        {
            int i_is_ip = is_ip ? 1 : 0;
            int i_gl_is_ip = gl_is_ip ? 1 : 0;
            int i_gl_is_main = gl_is_main ? 1 : 0;
            SerializableBuffer b_ipv4_ip = new SerializableBuffer(ipv4_ip);
            SerializableBuffer b_ipv6_ip = new SerializableBuffer(ipv6_ip);
            SerializableBuffer b_gl_ipv4_ip = new SerializableBuffer(gl_ipv4_ip);
            SerializableBuffer b_gl_ipv6_ip = new SerializableBuffer(gl_ipv6_ip);

            Variant v0 = Serializer.int_to_variant(i_is_ip);
            Variant v1 = Serializer.int_to_variant(weight);
            Variant v2 = Serializer.int_to_variant(priority);
            Variant v3 = b_ipv4_ip.serialize_to_variant();
            Variant v4 = b_ipv6_ip.serialize_to_variant();
            Variant v5 = Serializer.int_to_variant(0);  // UNUSED
            Variant v6 = Serializer.string_to_variant(hashed_alias);
            Variant v7 = Serializer.int64_to_variant((int64)port_number);
            Variant v8 = Serializer.string_to_variant(hostname);
            Variant v9 = Serializer.int_to_variant(i_gl_is_main);
            Variant vA = Serializer.int_to_variant(i_gl_is_ip);
            Variant vB = gl_service.serialize_to_variant();
            Variant vC = Serializer.int_to_variant(gl_weight);
            Variant vD = Serializer.int_to_variant(gl_priority);
            Variant vE = b_gl_ipv4_ip.serialize_to_variant();
            Variant vF = b_gl_ipv6_ip.serialize_to_variant();
            Variant vG = Serializer.int_to_variant(0);  // UNUSED
            Variant vH = Serializer.string_to_variant(gl_hashed_alias);
            Variant vJ = Serializer.int64_to_variant((int64)gl_port_number);
            Variant vK = Serializer.int_to_variant(0);  // UNUSED

            Variant vtemp0 = Serializer.tuple_to_variant_5(v0, v1, v2, v3, v4);
            Variant vtemp1 = Serializer.tuple_to_variant_5(v5, v6, v7, v8, v9);
            Variant vtemp2 = Serializer.tuple_to_variant_5(vA, vB, vC, vD, vE);
            Variant vtemp3 = Serializer.tuple_to_variant_5(vF, vG, vH, vJ, vK);
            Variant vret = Serializer.tuple_to_variant_4(vtemp0, vtemp1, vtemp2, vtemp3);
            return vret;
        }

        public void deserialize_from_variant(Variant v) throws SerializerError
        {
            Variant v0;
            Variant v1;
            Variant v2;
            Variant v3;
            Variant v4;
            Variant v5;
            Variant v6;
            Variant v7;
            Variant v8;
            Variant v9;
            Variant vA;
            Variant vB;
            Variant vC;
            Variant vD;
            Variant vE;
            Variant vF;
            Variant vG;
            Variant vH;
            Variant vJ;
            Variant vK;
            Variant vtemp0;
            Variant vtemp1;
            Variant vtemp2;
            Variant vtemp3;
            Serializer.variant_to_tuple_4(v, out vtemp0, out vtemp1, out vtemp2, out vtemp3);
            Serializer.variant_to_tuple_5(vtemp3, out vF, out vG, out vH, out vJ, out vK);
            Serializer.variant_to_tuple_5(vtemp2, out vA, out vB, out vC, out vD, out vE);
            Serializer.variant_to_tuple_5(vtemp1, out v5, out v6, out v7, out v8, out v9);
            Serializer.variant_to_tuple_5(vtemp0, out v0, out v1, out v2, out v3, out v4);

            int i_is_ip = Serializer.variant_to_int(v0);
            weight = Serializer.variant_to_int(v1);
            priority = Serializer.variant_to_int(v2);
            {
                uint8 *buf = ipv4_ip;
                SerializableBuffer buf_ser =
                    (SerializableBuffer)Object.new(typeof(SerializableBuffer));
                buf_ser.deserialize_from_variant(v3);
                for (int i = 0; i < buf_ser.buffer.length; i++)
                {
                    buf[i] = buf_ser.buffer[i];
                }
            }
            {
                uint8 *buf = ipv6_ip;
                SerializableBuffer buf_ser =
                    (SerializableBuffer)Object.new(typeof(SerializableBuffer));
                buf_ser.deserialize_from_variant(v4);
                for (int i = 0; i < buf_ser.buffer.length; i++)
                {
                    buf[i] = buf_ser.buffer[i];
                }
            }
            // v5 UNUSED
            hashed_alias = Serializer.variant_to_string(v6);
            port_number = (int16)Serializer.variant_to_int64(v7);
            hostname = Serializer.variant_to_string(v8);
            int i_gl_is_main = Serializer.variant_to_int(v9);
            int i_gl_is_ip = Serializer.variant_to_int(vA);
            gl_service = (AndnaServiceKey)Object.new(typeof(AndnaServiceKey));
            gl_service.deserialize_from_variant(vB);
            gl_weight = Serializer.variant_to_int(vC);
            gl_priority = Serializer.variant_to_int(vD);
            {
                uint8 *buf = gl_ipv4_ip;
                SerializableBuffer buf_ser =
                    (SerializableBuffer)Object.new(typeof(SerializableBuffer));
                buf_ser.deserialize_from_variant(vE);
                for (int i = 0; i < buf_ser.buffer.length; i++)
                {
                    buf[i] = buf_ser.buffer[i];
                }
            }
            {
                uint8 *buf = gl_ipv6_ip;
                SerializableBuffer buf_ser =
                    (SerializableBuffer)Object.new(typeof(SerializableBuffer));
                buf_ser.deserialize_from_variant(vF);
                for (int i = 0; i < buf_ser.buffer.length; i++)
                {
                    buf[i] = buf_ser.buffer[i];
                }
            }
            // vG UNUSED
            gl_hashed_alias = Serializer.variant_to_string(vH);
            gl_port_number = (int16)Serializer.variant_to_int64(vJ);
            // vK UNUSED

            is_ip = i_is_ip == 1;
            gl_is_ip = i_gl_is_ip == 1;
            gl_is_main = i_gl_is_main == 1;
        }

        public static bool equal_func(AndnaResponseAnswer a, AndnaResponseAnswer b)
        {
            if (a.is_ip != b.is_ip) return false;
            if (a.weight != b.weight) return false;
            if (a.priority != b.priority) return false;
            {
                uint8[] a_buf = a.ipv4_ip;
                uint8[] b_buf = b.ipv4_ip;
                assert(a_buf.length == b_buf.length);
                for (int i = 0; i < a_buf.length; i++)
                {
                    if (a_buf[i] != b_buf[i]) return false;
                }
            }
            {
                uint8[] a_buf = a.ipv6_ip;
                uint8[] b_buf = b.ipv6_ip;
                assert(a_buf.length == b_buf.length);
                for (int i = 0; i < a_buf.length; i++)
                {
                    if (a_buf[i] != b_buf[i]) return false;
                }
            }
            if (a.hashed_alias != b.hashed_alias) return false;
            if (a.port_number != b.port_number) return false;
            if (a.hostname != b.hostname) return false;
            if (a.gl_is_main != b.gl_is_main) return false;
            if (a.gl_is_ip != b.gl_is_ip) return false;
            if (! AndnaServiceKey.equal_func(a.gl_service, b.gl_service)) return false;
            if (a.gl_weight != b.gl_weight) return false;
            if (a.gl_priority != b.gl_priority) return false;
            {
                uint8[] a_buf = a.gl_ipv4_ip;
                uint8[] b_buf = b.gl_ipv4_ip;
                assert(a_buf.length == b_buf.length);
                for (int i = 0; i < a_buf.length; i++)
                {
                    if (a_buf[i] != b_buf[i]) return false;
                }
            }
            {
                uint8[] a_buf = a.gl_ipv6_ip;
                uint8[] b_buf = b.gl_ipv6_ip;
                assert(a_buf.length == b_buf.length);
                for (int i = 0; i < a_buf.length; i++)
                {
                    if (a_buf[i] != b_buf[i]) return false;
                }
            }
            if (a.gl_hashed_alias != b.gl_hashed_alias) return false;
            if (a.gl_port_number != b.gl_port_number) return false;
            return true;
        }
    }
}

