/*
 *  This file is part of Netsukuku.
 *  (c) Copyright 2014 Luca Dionisi aka lukisi <luca.dionisi@gmail.com>
 *
 *  Netsukuku is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Netsukuku is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Netsukuku.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gee;
using zcd;
using Netsukuku;

namespace Netsukuku
{
    namespace RpcNtk
    {
        public void init()
        {
            // Register serializable types
            typeof(AndnaServiceKey).class_peek();
        }
    }
    public class AndnaServiceKey : Object, ISerializable
    {
        // A Service Key has a name and a protocol. E.g. the
        // service key name="imaps" and protocol="tcp" is the
        // equivalent of the SRV record that identifies a host
        // serving as secure IMAP server for a certain domain.
        // The command "dig _imaps._tcp.gmail.com SRV"
        // will return that record for the domain gmail.com
        // The answer is imap.gmail.com.
        public string? name {get; private set;}
        public string? proto {get; private set;}

        public AndnaServiceKey(string name, string proto)
        {
            this.name = name;
            this.proto = proto;
        }

        private AndnaServiceKey.make_null()
        {
            this.name = null;
            this.proto = null;
        }

        // The serv_key to use for the main registration of the hostname
        private static AndnaServiceKey _NULL_SERV_KEY;

        // get the NULL_SERV_KEY
        public static AndnaServiceKey NULL_SERV_KEY {
            get {
                if (_NULL_SERV_KEY == null) _NULL_SERV_KEY = new AndnaServiceKey.make_null();
                return _NULL_SERV_KEY;
            }
        }

        public Variant serialize_to_variant()
        {
            Variant v0;
            if (name == null) v0 = Serializer.int_to_variant(0);
            else v0 = Serializer.string_to_variant(name);
            Variant v1;
            if (proto == null) v1 = Serializer.int_to_variant(0);
            else v1 = Serializer.string_to_variant(proto);
            return Serializer.tuple_to_variant(v0, v1);
        }
        
        public void deserialize_from_variant(Variant v)
        {
            Variant v0;
            Variant v1;
            Serializer.variant_to_tuple(v, out v0, out v1);
            string vt0 = v0.get_type_string();
            if (vt0 == "i") name = null;
            else name = Serializer.variant_to_string(v0);
            string vt1 = v1.get_type_string();
            if (vt1 == "i") proto = null;
            else proto = Serializer.variant_to_string(v1);
        }

        public static bool equal_func(AndnaServiceKey? a, AndnaServiceKey? b)
        {
            if (a == b) return true;
            if (a == null || b == null) return false;
            return a.name == b.name && a.proto == b.proto;
        }

        public static uint hash_func(AndnaServiceKey a)
        {
            if (a.name == null) return 0;
            return a.name.hash();
        }

        public string to_string()
        {
            if (name == null) return "<AndnaServiceKey NULL_SERV_KEY>";
            return @"<AndnaServiceKey _$(name)._$(proto)>";
        }
    }
}

/**  Stub classes generated by rpcdesign.
  */

namespace Andns
{
    /** The following classes are used to perform
      *  RPC call using the following form:
      *
      *    var x = remote_instance.property1.property2.method0(p1, p2, p3);
      *
      *  instead of:
      *
      *    RemoteCall rc = new RemoteCall();
      *    rc.method_name = "property1.property2.method0";
      *    rc.add_parameter(p1);
      *    rc.add_parameter(p2);
      *    rc.add_parameter(p3);
      *    var x = (SerClass)remote_instance.rmt(rc);
      *
      * The actual implementation of method rmt is a duty of a derived class.
      */

    public abstract class AndnaResolveFakeRmt : Object, IAndnaResolveRootDispatcher, FakeRmt
    {
        public IAndnaQueryHandler get_query_handler(int request_id)
        {
            RemoteAndnaQueryHandler ret = new RemoteAndnaQueryHandler();
            ret.root = this;
            ret.accumulated = @"get_query_handler($(request_id))";
            return ret;
        }

        public abstract ISerializable rmt(RemoteCall data) throws RPCError;
    }

    public class RemoteAndnaQueryHandler : Object, IAndnaQueryHandler
    {
        public weak FakeRmt root;
        public string accumulated = "";

        public AndnaResponse resolve(AndnaQuery query) throws RPCError
        {
            RemoteCall rc = new RemoteCall();
            rc.method_name = accumulated + ".resolve";
            rc.add_parameter(query);
            return (AndnaResponse)
                filter_exception(
                root.rmt(rc)
            );
        }

    }

    /** An implementation of AndnaResolveFakeRmt that sends a message via TCP.
      */
    public class AndnaResolveTCPClient : AndnaResolveFakeRmt
    {
        private TCPClient inner;
        public AndnaResolveTCPClient(string dest_addr, uint16? dest_port=null, string? my_addr=null, bool wait_response=true)
        {
            inner = new TCPClient(dest_addr, dest_port, my_addr, wait_response);
        }

        public override ISerializable rmt(RemoteCall data) throws RPCError
        {
            return inner.rmt(data);
        }

        public uint16 dest_port {
            get {
                return inner.dest_port;
            }
        }

        public string dest_addr {
            get {
                return inner.dest_addr;
            }
        }

        public bool calling {
            get {
                return inner.calling;
            }
        }

        public bool retry_connect {
            get {
                return inner.retry_connect;
            }
            set {
                inner.retry_connect = value;
            }
        }

        public void close() throws RPCError
        {
            inner.close();
        }
    }
}

