/*
 * L_buffer.c -  implements logging to a circular buffer.
 *
 * Improvements: it might be worthwhile to implement a compression 
 *   scheme for the buffer, decompression could be handled by the 
 *   L_buffer_dump routine. 
 * 
 * Copyright (c) 1997 Phil Maker
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: L_buffer.c,v 1.8 1997/05/26 10:20:40 pjm Exp $
 */

#include <stdio.h>
#include <stddef.h>
#include <stdlib.h>
#include <stdarg.h>
#include <ctype.h>

#include <I.h>
#include <L_buffer.h>

/*
 * L_buffer_create - malloc's and initialises a buffer with space for 
 *     size bytes worth of messages. Returns NULL on failure.
 *
 *     The buffer is initialised to '\0' so we know which locations haven't
 *     been written to yet. An alternate implementation would use a
 *     counter for this but this way seems easier and cuts down the overhead
 *     in the working functions (even if only by a couple of instructions).
 */

L_BUFFER *L_buffer_create(size_t size) { 
     L_BUFFER *p = malloc(sizeof(L_BUFFER));
     if(p != NULL) {
	  p->size = size;
	  p->data = malloc(size);
	  if(p->data != NULL) {
	       int i;
	       for(i = 0; i < size; i++) {
		    p->data[i] = '\0';
	       }
	       return p;
	  }
     }
     return NULL;
}

/*
 * L_buffer_putchar - copies a single char into the buffer.
 */
  
inline void L_buffer_putchar(L_BUFFER *b, char c) {
     b->data[b->free++] = c;
     if(b->free == b->size) {
	  b->free = 0;
     }
}

/*
 * L_buffer_puts - copies the argument into the buffer.
 */

inline void L_buffer_puts(L_BUFFER *b, const char *s) {
     int i;

     /* just copy into the buffer with wraparound */
     for(i = 0; s[i] != '\0'; i++) {
	  b->data[b->free] = s[i];
	  b->free++;
	  if(b->free == b->size) {
	       b->free = 0;
	  }
     }
}


/*
 * L_buffer_printf - does a standard printf operation into the buffer.
 *
 * 
 * Currently we use vs[n]printf to write it to a buffer before 
 * before finally copying it off into our L_BUFFER using L_buffer_puts.
 *
 */

void L_buffer_printf(L_BUFFER *b, const char *format, ...) {

     char buffer[512]; 
     int n;
     va_list ap;
     

     I(b != NULL);

     /* printf the arguments into buffer[0...] */
     va_start(ap, format); 
#if HAVE_VSNPRINTF == 1
     n = vsnprintf(&buffer[0], sizeof(buffer), format, ap);
#elif HAVE_VSPRINTF
     n = vsprintf(&buffer[0], format, ap);
#else
#error We need either vsprintf or vsnprintf for this one
       Sorry perhaps you should install libiberty
#endif

     va_end(ap);

     L_buffer_puts(b, &buffer[0]);
}

/*
 * L_buffer_dump - dumps the buffer out from the oldest to the
 *      newest; we also check for non-printing characters.
 */

void L_buffer_dump(L_BUFFER *b, FILE *fp) {
     int i;
     char c;

     I(b != NULL);

     /* print the banner */
     fprintf(fp, "* L_buffer_dump =\n");

     /* skip '\0' bytes from free upwards to find the oldest message */
     for(i = b->free+1; ; i++) { 
	  if(i == b->size) { /* wraparound at end of buffer */ 
	       i = 0;
	  }
	  if(i == b->free) { /* been through the entire buffer of '\0' */
	       fprintf(fp, "buffer was empty\n");
	       goto end;
	  }
	  if(b->data[i] != '\0') { /* found oldest one */
	       break;
	  }
     }

     /* now print out in time order */
     do {
	  c = b->data[i];
	  if(isprint(c) || isspace(c)) {
	       fputc(c, fp);
	  } else { /* can't be printed so print it in hex */
	       fprintf(fp, "* non-printable character 0x%x\n", c);
	  }
	  i++;
	  if(i == b->size) {
	       i = 0;
	  }
     } while(b->data[i] != '\0' && i != b->free);
end:     
     fprintf(fp, "* end of dump\n");
}

/*
 * L_buffer_clear - reinitialise the buffer.
 */

void L_buffer_clear(L_BUFFER *b) {
     int i;

     I(b != NULL);
     for(i = 0; i < b->size; i++) {
	  b->data[i] = '\0';
     }
     b->free = 0;
}

/*
 * L_buffer_delete - free up all the memory.
 */

void L_buffer_delete(L_BUFFER *b) {
     if(b != NULL) {
	  free(b->data);
	  free(b);
     }
}

