/*
# Copyright (C) 2008-2009 
# Raffaele Granito <raffaele.granito@tiscali.it>
#
# This file is part of myTCPClient:
# Universal TCP Client
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/ 

 
/*
.--------------.
| headers LIBC |
'--------------' */
#include <string.h>
#include <stdio.h>

/*
.--------------------.
| header Client      |
'--------------------' */
#include "common.h"
#include "output.h"

/*
.--------------------.
| header Frontend    |
'--------------------' */
#include "conf.h"

/*
.--------------------.
| header Locale      |
'--------------------' */
#include "param.h" 
#include "param_messages.h"



/*
.-----------------------------------------------------------------------------------------.
|                                                                                         |
|  Funzione   : ParametriCheck                                                            |
|                                                                                         |
|  Descrizione: Converte i parametri passati dal formato ARRAY [**argv[]] al formato      |
|               così come si aspetta di riceverli la libreria client.                     |
|               La funzione non effettua controlli sulla qualità dei parametri passati    |
|               o manipolazioni di alcun tipo [ad esempio applicazione default] se il     |
|               parametro obbligatorio non risulta inserito o se non è corretto.          |
|               La funziona si limita ad identificare i parametri correttamente per       | 
|               svolgere il suo lavoro di conversione. La logica deve stare nel client in |
|               maniera tale da non doverla replicare in ogni frontend implementato.      | 
|                                                                                         |
|  TODO       : * spostare la logica fuori eliminarla da qui.                             |
|                                                                                         |
|               * spostare i messaggi in un file XML quindi non più tramite la tabella    |
|                 di memoria memorizzata nel file output.c. Prevedere il multilingua.     | 
|                                                                                         |
|               * la lista dei parametri validi dovrebbe essere non cablata nel codice.   |
|                                                                                         |
|               * i parametri dovrebbero essere letti anche da file, se un parametro è    |
|                 presente sia nel file che sulla linea di comando prevale la linea di    |
|                 comando.                                                                |
|                                                                                         |
|                                                                                         | 
'-----------------------------------------------------------------------------------------'*/
extern struct tParametriCheck ParametriCheck(int nParametri, char **lParametri, char *frontend, char *frontend_version )
{

    /*
    .------------------------------------------.
    | Dichiarazione delle STRUTTURE            |
    '------------------------------------------' */
    struct tParametriCheck  rParametriCheck;
    struct tParametriCheck *pParametriCheck;

    struct tUri2SingleFieldConnect  rUri2SingleFieldConnect ; 
    struct tUri2SingleFieldConnect *pUri2SingleFieldConnect ; 

    /*
    .------------------------------------------.
    | Dichiarazione delle STRUTTURE * LIB Conf |
    '------------------------------------------' */
    tLoadConf  rLoadConf;
    tLoadConf *pLoadConf;

    char buffer[__DIM_MAX_COMANDO__];

    int DEBUG = 0 ;

 
    /*
    .--------------------------------------------------.
    | Alloca lo spazio per contenere la struttura dati |
    | che va a riempire la funzione ClientConnect. La  |
    | seconda istruzione serve per dedicare lo spazio  |
    | allocato alla struttura dati. Successivamente    |
    | inizializza l'intera struttura con NULL ed il    |
    | campo return_code a ZERO.                        |
    '--------------------------------------------------' */
    pParametriCheck = malloc(sizeof(struct tParametriCheck));
    pParametriCheck = &rParametriCheck;
    memset((void*)&rParametriCheck,0,sizeof(rParametriCheck));
    rParametriCheck.return_code = 0;

    pUri2SingleFieldConnect = malloc(sizeof(struct tUri2SingleFieldConnect));
    pUri2SingleFieldConnect = &rUri2SingleFieldConnect;
    memset((void*)&rUri2SingleFieldConnect,0,sizeof(rUri2SingleFieldConnect));
    rUri2SingleFieldConnect.return_code = 0;

    pLoadConf = malloc(sizeof(struct tLoadConf));
    pLoadConf = &rLoadConf;
    memset((void*)&rLoadConf,0,sizeof(rLoadConf));
    rLoadConf.return_code = 0;


    /*
    .--------------------------------------.
    | Inizializzazione struttura di OUTPUT |
    '--------------------------------------' */
    rParametriCheck.rClientConnectParameter.frontend                  = 0 ;
    rParametriCheck.rClientConnectParameter.frontend_version          = 0 ;
    rParametriCheck.rClientConnectParameter.protocollo                = 0 ;
    rParametriCheck.rClientConnectParameter.host                      = 0 ;
    rParametriCheck.rClientConnectParameter.porta                     = 0 ;
    rParametriCheck.rClientConnectParameter.risorsa                   = 0 ;
    rParametriCheck.rClientConnectParameter.queryStringDimTabVariable = 0 ; 
    rParametriCheck.rClientConnectParameter.queryStringLen            = 0 ; 
    rParametriCheck.rClientConnectParameter.auth_user                 = 0 ;
    rParametriCheck.rClientConnectParameter.auth_password             = 0 ;
    rParametriCheck.rClientConnectParameter.keystore                  = 0 ;
    rParametriCheck.rClientConnectParameter.password                  = 0 ;
    rParametriCheck.rClientConnectParameter.truststore                = 0 ;
    rParametriCheck.rClientConnectParameter.ciphersuite               = 0 ;
    rParametriCheck.rClientConnectParameter.sslversion                = 0 ;
    rParametriCheck.rClientConnectParameter.type_operation            = 0 ; 
    rParametriCheck.rClientConnectParameter.message_language          = 0 ;
    rParametriCheck.rClientConnectParameter.message_type              = 0 ;
    rParametriCheck.rClientConnectParameter.message                   = 0 ; 
    rParametriCheck.rClientConnectParameter.SSLCipherSuiteServerScan  = 0 ; 
    rParametriCheck.rClientConnectParameter.MaxRequest                = 0 ;
    rParametriCheck.rClientConnectParameter.OutputFile                = 0 ;
    rParametriCheck.rClientConnectParameter.debug                     = 0 ;
    rParametriCheck.return_code                                       = 0 ;

    /*
    .------------------------------------------.
    |  Determina il livello di DEBUG           |
    '------------------------------------------' */
    int debug_level = GetDebugLevel ( "+++param", "param" ) ; 

    /*
    .--------------------------------------------------.
    | Nessun parametro: visualizza banner              |
    '--------------------------------------------------' */
    if (nParametri == 1)
    {
        PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__BANNER" );
        PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__FORMATO");
        rParametriCheck.return_code = +1;                   
        return rParametriCheck;
    };

    int indice = 1;
    int selected_parameter = 0;
    while ( indice < nParametri )
    {
          /*
          .--------------------------------------------------.
          | Legge il parametro successivo                    |
          '--------------------------------------------------' */
          strncpy (buffer, lParametri[indice], __DIM_MAX_COMANDO__ );
          buffer[__DIM_MAX_COMANDO__ - 1] = 0;


          /*
          .-----------------------------------------------------------------------------------------------.
          |                                                                                               |
          |                                                                                               |
          |                                          V A L O R I                                          |
          |                                                                                               |
          |                                                                                               |
          '-----------------------------------------------------------------------------------------------' */
 
          /*
          .------------------------------------------------------------------------.
          | Se è stato trovato uno dei parametri supportati ma il parse dei        |
          | parametri si aspetta di chiudere un parametro ancora in fase di        | 
          | definizione il software esce con un errore perchè si aspettava altr.   |
          '------------------------------------------------------------------------' */
          if (( strcmp(buffer,"-c")  == 0 || strcmp(buffer,"--connect")                     == 0 || 
                strcmp(buffer,"-U")  == 0 || strcmp(buffer,"--auth-user")                   == 0 ||
                strcmp(buffer,"-P")  == 0 || strcmp(buffer,"--auth-password")               == 0 ||
                strcmp(buffer,"-k")  == 0 || strcmp(buffer,"--keystore")                    == 0 || 
                strcmp(buffer,"-p")  == 0 || strcmp(buffer,"--password")                    == 0 || 
                strcmp(buffer,"-c")  == 0 || strcmp(buffer,"--connect")                     == 0 || 
                strcmp(buffer,"-t")  == 0 || strcmp(buffer,"--truststore")                  == 0 || 
                strcmp(buffer,"-C")  == 0 || strcmp(buffer,"--ciphersuite")                 == 0 || 
                strcmp(buffer,"-s")  == 0 || strcmp(buffer,"--ssl-version")                 == 0 || 
                strcmp(buffer,"-T")  == 0 || strcmp(buffer,"--type-operation")              == 0 ||
                strcmp(buffer,"-Ml") == 0 || strcmp(buffer,"--message-language")            == 0 ||
                strcmp(buffer,"-Mt") == 0 || strcmp(buffer,"--message-type")                == 0 ||
                strcmp(buffer,"-M")  == 0 || strcmp(buffer,"--message")                     == 0 ||
                strcmp(buffer,"-S")  == 0 || strcmp(buffer,"--ssl-ciphersuite-server-scan") == 0 || 
                strcmp(buffer,"-m")  == 0 || strcmp(buffer,"--max-request")                 == 0 || 
                strcmp(buffer,"-o")  == 0 || strcmp(buffer,"--output-file")                 == 0 ||
                strcmp(buffer,"-f")  == 0 || strcmp(buffer,"--config")                      == 0 ||
                strcmp(buffer,"-d")  == 0 || strcmp(buffer,"--debug")                       == 0 ) && selected_parameter > 0 )
          {
                switch (selected_parameter)
                {
                    case  1 : PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_ATTESO_PARAMETRO_CONNECT"                     ); break; 
                    case  2 : PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_ATTESO_PARAMETRO_AUTH_USER"                   ); break;
                    case  3 : PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_ATTESO_PARAMETRO_AUTH_PASSWORD"               ); break; 
                    case  4 : PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_ATTESO_PARAMETRO_KEYSTORE"                    ); break; 
                    case  5 : PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_ATTESO_PARAMETRO_KEYSTORE_PASSWORD"           ); break; 
                    case  6 : PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_ATTESO_PARAMETRO_TRUSTSTORE"                  ); break; 
                    case  7 : PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_ATTESO_PARAMETRO_CIPHERSUITE"                 ); break; 
                    case  8 : PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_ATTESO_PARAMETRO_VERSION_SSL"                 ); break;
                    case  9 : PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_ATTESO_PARAMETRO_TYPE_OPERATION"              ); break;
                    case 10 : PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_ATTESO_PARAMETRO_MESSAGE_LANGUAGE"            ); break;
                    case 11 : PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_ATTESO_PARAMETRO_MESSAGE_TYPE"                ); break;
                    case 12 : PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_ATTESO_PARAMETRO_MESSAGE"                     ); break;
                    case 13 : PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_ATTESO_PARAMETRO_SSL_CIPHERSUITE_SERVER_SCAN" ); break;
                    case 14 : PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_ATTESO_PARAMETRO_MAX_REQUEST"                 ); break; 
                    case 15 : PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_ATTESO_PARAMETRO_OUTPUT_FILE"                 ); break; 
                    case 16 : PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_ATTESO_PARAMETRO_CONFIGURATION_FILE"          ); break;
                    case 17 : PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_ATTESO_PARAMETRO_DEBUG"                       ); break;
                    default : PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_UNKNOWN_PARAMETRO"                            ); 
                };
                PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__BANNER"  );
                PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__FORMATO" );
                rParametriCheck.return_code = -1;                    
                return rParametriCheck;
          };

          /*
          .--------------------------------------------------.
          | Se il parametro è -h|--help i precedenti         |
          | parametri ed anche i successivi ancora da        |
          | leggere *non* vengono considerati.               |
          '--------------------------------------------------' */
          if ( strcmp(buffer,"-h") == 0 || strcmp(buffer,"--help") == 0 ) 
          { 
               PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__BANNER"          );
               PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__FORMATO"         );
               PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__REMARKS"         );
               PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__LISTA_PARAMETRI" );
               PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__RETURN_CODE"     );
               PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__BUG_REPORT"      );
               rParametriCheck.return_code = +1;                   
               return rParametriCheck;
          };

          /*
          .--------------------------------------------------.
          | Se il parametro è -v|--version i precedenti      |
          | parametri ed anche i successivi ancora da        |
          | leggere *non* vengono considerati.               |
          '--------------------------------------------------' */
          if ( strcmp(buffer,"-v") == 0 || strcmp(buffer,"--version")     == 0 )
          { 
               PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__BANNER"  );
               PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__FORMATO" );
               PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__REMARKS" );
               PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__VERSION" );
               rParametriCheck.return_code = +1;               
               return rParametriCheck;
          };

          /*
          .-------------------------------------------------.
          | Connect                                         |
          '-------------------------------------------------' */
          if ( strcmp(buffer,"-c") == 0 || strcmp(buffer,"--connect")     == 0 )
          {
               if ( rParametriCheck.rClientConnectParameter.protocollo  != NULL || 
                    rParametriCheck.rClientConnectParameter.host        != NULL || 
                    rParametriCheck.rClientConnectParameter.porta       != 0    ||  
                    rParametriCheck.rClientConnectParameter.risorsa     != NULL  ) { 
                    PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_DUPLICATE_PARAMETRO_CONNECT");
                    PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__BANNER"                            );
                    PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__FORMATO"                           );
                    rParametriCheck.return_code = -1;             
                    return rParametriCheck; };

               selected_parameter = 1 ;
          };

          /*
          .-------------------------------------------------.
          | Auth-User                                       |
          '-------------------------------------------------' */
          if ( strcmp(buffer,"-U") == 0 || strcmp(buffer,"--auth-user")    == 0 )
          { 
               if ( rParametriCheck.rClientConnectParameter.auth_user   != NULL ) { 
                    PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_DUPLICATE_PARAMETRO_AUTH_USER");
                    PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__BANNER"                              );
                    PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__FORMATO"                             );
                    rParametriCheck.return_code = -1;            
                    return rParametriCheck; };

               selected_parameter = 2 ;
          };

          /*
          .-------------------------------------------------.
          | Auth-Password                                   |
          '-------------------------------------------------' */
          if ( strcmp(buffer,"-P") == 0 || strcmp(buffer,"--auth-password")    == 0 )
          { 
               if ( rParametriCheck.rClientConnectParameter.auth_password    != NULL ) { 
                    PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_DUPLICATE_PARAMETRO_AUTH_PASSWORD");
                    PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__BANNER"                                  );
                    PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__FORMATO"                                 );
                    rParametriCheck.return_code = -1;           
                    return rParametriCheck; };

               selected_parameter = 3 ;
          };

          /*
          .-------------------------------------------------.
          | Keystore                                        |
          '-------------------------------------------------' */
          if ( strcmp(buffer,"-k") == 0 || strcmp(buffer,"--keystore")    == 0 )
          {
               if ( rParametriCheck.rClientConnectParameter.keystore    != NULL ) { 
                    PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_DUPLICATE_PARAMETRO_KEYSTORE"   );
                    PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__BANNER"                                );
                    PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__FORMATO"                               );
                    rParametriCheck.return_code = -1;          
                    return rParametriCheck; };
 
               selected_parameter = 4 ;
          };

          /*
          .-------------------------------------------------.
          | Password                                        |
          '-------------------------------------------------' */
          if ( strcmp(buffer,"-p") == 0 || strcmp(buffer,"--password")    == 0 )
          {
               if ( rParametriCheck.rClientConnectParameter.password    != NULL ) { 
                    PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_DUPLICATE_PARAMETRO_KEYSTORE_PASSWORD");
                    PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__BANNER"                                      );
                    PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__FORMATO"                                     );
                    rParametriCheck.return_code = -1;            
                    return rParametriCheck; };

               selected_parameter = 5 ;
          };

          /*
          .-------------------------------------------------.
          | Truststore                                      |
          '-------------------------------------------------' */
          if ( strcmp(buffer,"-t") == 0 || strcmp(buffer,"--truststore")  == 0 )
          {
               if ( rParametriCheck.rClientConnectParameter.truststore  != NULL ) {
                    PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_DUPLICATE_PARAMETRO_TRUSTSTORE"      );
                    PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__BANNER"                                     );
                    PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__FORMATO"                                    );
                    rParametriCheck.return_code = -1;             
                    return rParametriCheck; };
 
               selected_parameter = 6 ;
          };

          /*
          .-------------------------------------------------.
          | Chiper Suite                                    |
          '-------------------------------------------------' */
          if ( strcmp(buffer,"-C") == 0 || strcmp(buffer,"--ciphersuite") == 0 )
          {
               if ( rParametriCheck.rClientConnectParameter.ciphersuite != NULL ) { 
                    PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_DUPLICATE_PARAMETRO_CIPHERSUITE"    );
                    PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__BANNER"                                    );
                    PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__FORMATO"                                   );
                    rParametriCheck.return_code = -1;              
                    return rParametriCheck; };

               selected_parameter = 7 ;
          };

          /*
          .-------------------------------------------------.
          | Configuration File. L'ultimo definito (in caso  |
          | di doppi) sostituisce il precedente. Vengono    |
          | aggiornati anche eventuali parametri definiti   |
          | in precedenza sulla linea di comando.           |
          '-------------------------------------------------' */
          if ( strcmp(buffer,"-s") == 0 || strcmp(buffer,"--ssl-version")      == 0 )
          {
               if ( rParametriCheck.rClientConnectParameter.sslversion != NULL ) {
                    PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_DUPLICATE_PARAMETRO_SSLVERSION"    );
                    PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__BANNER"                                   );
                    PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__FORMATO"                                  );
                    rParametriCheck.return_code = -1;               
                    return rParametriCheck; };

               selected_parameter = 8 ;
          };

          /*
          .-------------------------------------------------.
          | type-operation                                  |
          '-------------------------------------------------' */ 
          if ( strcmp(buffer,"-T") == 0 || strcmp(buffer,"--type-operation") == 0 )
          {
               if ( rParametriCheck.rClientConnectParameter.type_operation != 0 ) {
                    PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_DUPLICATE_PARAMETRO_TYPE_OPERATION");
                    PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__BANNER"                                   );
                    PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__FORMATO"                                  );
                    rParametriCheck.return_code = -1;
                    return rParametriCheck; };
               
               selected_parameter = 9 ;
          };

          /*
          .-------------------------------------------------.
          | message-language                                |
          '-------------------------------------------------' */
          if ( strcmp(buffer,"-Ml") == 0 || strcmp(buffer,"--message-language") == 0 )
          {
               if ( rParametriCheck.rClientConnectParameter.message_language != 0 ) {
                    PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_DUPLICATE_PARAMETRO_MESSAGE_LANGUAGE"  );
                    PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__BANNER"                                       );
                    PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__FORMATO"                                      );
                    rParametriCheck.return_code = -1;
                    return rParametriCheck; };

               selected_parameter = 10 ;
          };

          /*
          .-------------------------------------------------.
          | message-type                                    |
          '-------------------------------------------------' */
          if ( strcmp(buffer,"-Mt") == 0 || strcmp(buffer,"--message-type") == 0 )
          {
               if ( rParametriCheck.rClientConnectParameter.message_type != 0 ) {
                    PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_DUPLICATE_PARAMETRO_MESSAGE_TYPE"  );
                    PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__BANNER"                                   );
                    PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__FORMATO"                                  );
                    rParametriCheck.return_code = -1;
                    return rParametriCheck; };

               selected_parameter = 11 ;
          };

          /*
          .-------------------------------------------------.
          | message                                         |
          '-------------------------------------------------' */
          if ( strcmp(buffer,"-M") == 0 || strcmp(buffer,"--message") == 0 )
          {
               if ( rParametriCheck.rClientConnectParameter.message != 0 ) {
                    PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_DUPLICATE_PARAMETRO_MESSAGE"      );
                    PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__BANNER"                                  );
                    PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__FORMATO"                                 );
                    rParametriCheck.return_code = -1;
                    return rParametriCheck; };

               selected_parameter = 12 ;
          };

          /*
          .-------------------------------------------------.
          | ssl-ciphersuite-server-scan                     |
          '-------------------------------------------------' */
          if ( strcmp(buffer,"-S") == 0 || strcmp(buffer,"--ssl-ciphersuite-server-scan") == 0 )
          {
               if ( rParametriCheck.rClientConnectParameter.SSLCipherSuiteServerScan != NULL ) {
                    PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_DUPLICATE_PARAMETRO_SSL_CIPHERSUITE_SERVER_SCAN" );
                    PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__BANNER"                                                 );
                    PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__FORMATO"                                                );
                    rParametriCheck.return_code = -1;
                    return rParametriCheck; };

               selected_parameter = 13 ;
          };

          /*
          .-------------------------------------------------.
          | max-request                                     | 
          '-------------------------------------------------' */
          if ( strcmp(buffer,"-m") == 0 || strcmp(buffer,"--max-request") == 0 ) 
          {
               if ( rParametriCheck.rClientConnectParameter.MaxRequest != 0 ) {
                    PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_DUPLICATE_PARAMETRO_MAX_REQUEST" );
                    PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__BANNER"                                 );
                    PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__FORMATO"                                );
                    rParametriCheck.return_code = -1;                      
                    return rParametriCheck; };

               selected_parameter = 14 ;
          };

          /*
          .-------------------------------------------------.
          | output-file                                     |   
          '-------------------------------------------------' */
          if ( strcmp(buffer,"-o") == 0 || strcmp(buffer,"--output-file") == 0 ) 
          {
               if ( rParametriCheck.rClientConnectParameter.OutputFile != NULL ) {
                    PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_DUPLICATE_PARAMETRO_OUTPUT_FILE");
                    PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__BANNER"                                );
                    PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__FORMATO"                               );
                    rParametriCheck.return_code = -1;                      
                    return rParametriCheck; };

               selected_parameter = 15 ;
          };

          /*
          .-------------------------------------------------.
          | Configuration File. L'ultimo definito (in caso  |
          | di doppi) sostituisce il precedente. Vengono    |
          | aggiornati anche eventuali parametri definiti   |
          | in precedenza sulla linea di comando.           |
          '-------------------------------------------------' */
          if ( strcmp(buffer,"-f") == 0 || strcmp(buffer,"--config")      == 0 )
          {
               selected_parameter = 16 ;
          };

          /*
          .-------------------------------------------------.
          | debug                                           |  
          '-------------------------------------------------' */
          if ( strcmp(buffer,"-d") == 0 || strcmp(buffer,"--debug") == 0 )
          {
               if ( rParametriCheck.rClientConnectParameter.debug != NULL ) {
                    PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_DUPLICATE_PARAMETRO_DEBUG");
                    PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__BANNER"                                );
                    PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__FORMATO"                               );
                    rParametriCheck.return_code = -1;
                    return rParametriCheck; };

               selected_parameter = 17 ;
          };


          /*
          .-----------------------------------------------------------------------------------------------.
          |                                                                                               |
          |                                                                                               |
          |                                       P A R A M E T E R                                       |
          |                                                                                               |
          |                                                                                               |
          '-----------------------------------------------------------------------------------------------' */

          if (( strcmp(buffer,"-c")  != 0 && strcmp(buffer,"--connect")                           != 0 ) && 
              ( strcmp(buffer,"-U")  != 0 && strcmp(buffer,"--auth-user")                         != 0 ) && 
              ( strcmp(buffer,"-P")  != 0 && strcmp(buffer,"--auth-password")                     != 0 ) && 
              ( strcmp(buffer,"-k")  != 0 && strcmp(buffer,"--keystore")                          != 0 ) && 
              ( strcmp(buffer,"-p")  != 0 && strcmp(buffer,"--password")                          != 0 ) && 
              ( strcmp(buffer,"-c")  != 0 && strcmp(buffer,"--connect")                           != 0 ) && 
              ( strcmp(buffer,"-t")  != 0 && strcmp(buffer,"--truststore")                        != 0 ) && 
              ( strcmp(buffer,"-C")  != 0 && strcmp(buffer,"--ciphersuite")                       != 0 ) && 
              ( strcmp(buffer,"-s")  != 0 && strcmp(buffer,"--ssl-version")                       != 0 ) &&
              ( strcmp(buffer,"-T")  != 0 && strcmp(buffer,"--type-operation")                    != 0 ) && 
              ( strcmp(buffer,"-Ml") != 0 && strcmp(buffer,"--message-language")                  != 0 ) &&
              ( strcmp(buffer,"-Mt") != 0 && strcmp(buffer,"--message-type")                      != 0 ) &&
              ( strcmp(buffer,"-M")  != 0 && strcmp(buffer,"--message")                           != 0 ) && 
              ( strcmp(buffer,"-m")  != 0 && strcmp(buffer,"--ssl-ciphersuite-server-scan")       != 0 ) &&
              ( strcmp(buffer,"-m")  != 0 && strcmp(buffer,"--max-request")                       != 0 ) && 
              ( strcmp(buffer,"-o")  != 0 && strcmp(buffer,"--output-file")                       != 0 ) && 
              ( strcmp(buffer,"-f")  != 0 && strcmp(buffer,"--config")                            != 0 ) &&
              ( strcmp(buffer,"-d")  != 0 && strcmp(buffer,"--debug")                             != 0 )) 
          {
              switch (selected_parameter)
              {
                  /*
                  .-----------------------.
                  |   Parameter Unknown   |
                  '-----------------------' */
                  case 0: 
                          /*
                          .--------------------------------------------------.
                          | Se il parametro non è conosciuto visualizza      |
                          | l'help ed esce. In futuro è da aggiungere un     |
                          | messaggio di errore in cui si evedenzia il fatto |
                          | che è stato trovato un parametro non supportato. |
                          '--------------------------------------------------' */
                          PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_UNKNOWN_PARAMETRO"); 
                          PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__BANNER"                  );
                          PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__FORMATO"                 );
                          rParametriCheck.return_code = -1;           
                          return rParametriCheck;
                          break;

                  /*
                  .-----------------------.
                  |   Connect Parameter   | 
                  '-----------------------' */
                  case 1:  
                          /*
                          .--------------------------------------------------.
                          | Se il primo parametro è una -c|--connect         |
                          | Verifica la stringa di connessione               |
                          '--------------------------------------------------' */
                          strncpy (buffer, lParametri[indice], __DIM_MAX_COMANDO__);
                          buffer[__DIM_MAX_COMANDO__ - 1] = 0;

                          /*
                          .-----------------------------------------------------------------------------------------.
                          |  Verifica e spezzettamento stringa di connessione.                                      |
                          +-----------------------------------------------------------------------------------------+
                          |                                                                                         |
                          |  Return Code:  -1 errore: Protocollo  * Non Specificato (len 0)                         |
                          |                -2 errore: Protocollo  * Missing Separatore con Hostname (://)           |
                          |                -3 errore: Hostname    * Non Specificato (len 0)                         |
                          |                -4 errore: Porta       * Valore non numerico                             |
                          |                -5 errore: Porta       * Valore non valido ( ! 1..65535 )                |
                          |                -6 errore: QueryString * Missing separatore URL/Query [?]                |
                          |                -7 errore: QueryString * Missing QueryString (len 0)                     |
                          |                -8 errore: QueryString * Lunga più di __DIM_QUERY_STRING__ caratteri     |
                          |                -9 errore: QueryString * Missing separatore variabile                    |
                          |               -10 errore: QueryString * Missing variable (len 0)                        |
                          |               -11 errore: QueryString * Missing carattere di assegnazione valore        |
                          |                                                                                         |
                          '-----------------------------------------------------------------------------------------' */
                          rUri2SingleFieldConnect = Uri2SingleFieldConnect(buffer);
                          switch ( rUri2SingleFieldConnect.return_code ) 
                          {
                              case  0 : 
                                        /*
                                        .------------------------------------------.
                                        |                protocollo                |
                                        '------------------------------------------' */
                                        rParametriCheck.rClientConnectParameter.protocollo = malloc(strlen(rUri2SingleFieldConnect.protocollo)+1); 
                                        memset(rParametriCheck.rClientConnectParameter.protocollo,0,
                                                     strlen(rParametriCheck.rClientConnectParameter.protocollo)+1); 
                                        strcpy(rParametriCheck.rClientConnectParameter.protocollo, rUri2SingleFieldConnect.protocollo);

                                        /*
                                        .------------------------------------------.
                                        |                   host                   |
                                        '------------------------------------------' */
                                        rParametriCheck.rClientConnectParameter.host = malloc(strlen(rUri2SingleFieldConnect.host)+1);
                                        memset(rParametriCheck.rClientConnectParameter.host,0,
                                                     strlen(rParametriCheck.rClientConnectParameter.host)+1); 
                                        strcpy(rParametriCheck.rClientConnectParameter.host, rUri2SingleFieldConnect.host);

                                        /*
                                        .-----------------------------------------.
                                        |                   porta                 |
                                        '-----------------------------------------' */
                                        rParametriCheck.rClientConnectParameter.porta = rUri2SingleFieldConnect.porta;

                                        /*
                                        .-----------------------------------------.
                                        |                path risorsa             |
                                        '-----------------------------------------' */
                                        rParametriCheck.rClientConnectParameter.risorsa_path = 
                                                     malloc(strlen(rUri2SingleFieldConnect.risorsa_path)+1);
                                        memset(rParametriCheck.rClientConnectParameter.risorsa_path,0,
                                                     strlen(rParametriCheck.rClientConnectParameter.risorsa_path)+1);
                                        strcpy(rParametriCheck.rClientConnectParameter.risorsa_path, rUri2SingleFieldConnect.risorsa_path);

                                        /* 
                                        .-----------------------------------------.
                                        |                  risorsa                |
                                        '-----------------------------------------' */ 
                                        rParametriCheck.rClientConnectParameter.risorsa = malloc(strlen(rUri2SingleFieldConnect.risorsa)+1);
                                        memset(rParametriCheck.rClientConnectParameter.risorsa,0,
                                                     strlen(rParametriCheck.rClientConnectParameter.risorsa)+1);
                                        strcpy(rParametriCheck.rClientConnectParameter.risorsa, 
                                                     rUri2SingleFieldConnect.risorsa);

                                        /*
                                        .-----------------------------------------.
                                        |                 Parametri               |
                                        '-----------------------------------------' */
                                        int indice = 0 ;
                                        while ( indice < rUri2SingleFieldConnect.queryStringDimTabVariable )
                                        {  
                                                /*
                                                .-------------------------------------------.
                                                | Variabile                                 |
                                                '-------------------------------------------' */
                                                rParametriCheck.rClientConnectParameter.queryStringTabVar[indice].variable = 
                                                           malloc(strlen(rUri2SingleFieldConnect.queryStringTabVariable[indice].variable)+1);

                                                memset(rParametriCheck.rClientConnectParameter.queryStringTabVar[indice].variable,0,
                                                           strlen(rUri2SingleFieldConnect.queryStringTabVariable[indice].variable)+1);

                                                strcpy(rParametriCheck.rClientConnectParameter.queryStringTabVar[indice].variable,
                                                           rUri2SingleFieldConnect.queryStringTabVariable[indice].variable);

                                                /*
                                                .-------------------------------------------.
                                                | Value                                     |
                                                '-------------------------------------------' */
                                                rParametriCheck.rClientConnectParameter.queryStringTabVar[indice].value =
                                                           malloc(strlen(rUri2SingleFieldConnect.queryStringTabVariable[indice].value)+1);

                                                memset(rParametriCheck.rClientConnectParameter.queryStringTabVar[indice].value,0,
                                                           strlen(rUri2SingleFieldConnect.queryStringTabVariable[indice].value)+1);

                                                strcpy(rParametriCheck.rClientConnectParameter.queryStringTabVar[indice].value,
                                                           rUri2SingleFieldConnect.queryStringTabVariable[indice].value);

                                                indice++; 
                                        };

                                        /*
                                        .---------------------------------------.
                                        | Dimensione della tabella              |
                                        '---------------------------------------' */
                                        rParametriCheck.rClientConnectParameter.queryStringDimTabVariable =  
                                                        rUri2SingleFieldConnect.queryStringDimTabVariable ; 

                                        /*
                                        .--------------------------------------------.
                                        | Lunghezza totale della Query String        |
                                        '--------------------------------------------' */ 
                                        rParametriCheck.rClientConnectParameter.queryStringLen = 
                                                        rUri2SingleFieldConnect.queryStringLen ; 

                                        break;

                              case  -1 : PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_STRINGA_CONNESSIONE_PROTOCOLLO"       ); break;
                              case  -2 : PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_STRINGA_CONNESSIONE_PROTOCOLLO_NOSEP" ); break;
                              case  -3 : PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_STRINGA_CONNESSIONE_HOSTNAME"         ); break;
                              case  -4 : PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_STRINGA_CONNESSIONE_PORTA_NONUM"      ); break;
                              case  -5 : PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_STRINGA_CONNESSIONE_PORTA_NORANGE"    ); break;
                              case  -6 : PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_STRINGA_CONNESSIONE_QUERY_NOSEPURL"   ); break;
                              case  -7 : PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_STRINGA_CONNESSIONE_QUERY_LEN0"       ); break;
                              case  -8 : PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_STRINGA_CONNESSIONE_QUERY_NOTLENMAX"  ); break;
                              case  -9 : PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_STRINGA_CONNESSIONE_QUERY_NOSEPVAR"   ); break;
                              case -10 : PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_STRINGA_CONNESSIONE_QUERY_VARLEN0"    ); break;
                              case -11 : PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_STRINGA_CONNESSIONE_QUERY_NOASSIGN"   ); break;
                              default  : PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_STRINGA_CONNESSIONE_UNKNOWN"          ); break;
                          };
 
                          if ( rUri2SingleFieldConnect.return_code < 0 )
                          {
                               PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__BANNER" );
                               PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__FORMATO");
                               rParametriCheck.return_code = -1;              
                               return rParametriCheck;
                          };

                          selected_parameter = 0; 
                          break; 

                  /*
                  .-------------------------------.
                  |  Auth User Parameter          | 
                  '-------------------------------' */
                  case 2:
                          rParametriCheck.rClientConnectParameter.auth_user = malloc(strlen(buffer)+1);
                          strcpy(rParametriCheck.rClientConnectParameter.auth_user, buffer);
                          selected_parameter = 0;
                          break;

                  /*
                  .-------------------------------.
                  |  Auth Password Parameter      | 
                  '-------------------------------' */
                  case 3:
                          rParametriCheck.rClientConnectParameter.auth_password = malloc(strlen(buffer)+1);
                          strcpy(rParametriCheck.rClientConnectParameter.auth_password, buffer);
                          selected_parameter = 0;
                          break;

                  /*
                  .----------------------.
                  |  Keystore Parameter  |
                  '----------------------' */
                  case 4: 
                          rParametriCheck.rClientConnectParameter.keystore = malloc(strlen(buffer)+1);
                          strcpy(rParametriCheck.rClientConnectParameter.keystore, buffer);
                          selected_parameter = 0; 
                          break; 

                  /*
                  .-------------------------------.
                  |  Password Keystore Parameter  |
                  '-------------------------------' */ 
                  case 5: 
                          rParametriCheck.rClientConnectParameter.password = malloc(strlen(buffer)+1);
                          strcpy(rParametriCheck.rClientConnectParameter.password, buffer);
                          selected_parameter = 0; 
                          break; 

                  /*
                  .-----------------------.
                  |  Truststore Parameter |
                  '-----------------------' */ 
                  case 6: 
                          rParametriCheck.rClientConnectParameter.truststore=malloc(strlen(buffer)+1);
                          strcpy(rParametriCheck.rClientConnectParameter.truststore, buffer);
                          selected_parameter = 0; 
                          break; 

                  /*
                  .------------------------.
                  |  Chipersuite Parameter |
                  '------------------------' */
                  case 7: 
                          rParametriCheck.rClientConnectParameter.ciphersuite=malloc(strlen(buffer)+1);
                          strcpy(rParametriCheck.rClientConnectParameter.ciphersuite, buffer);
                          selected_parameter = 0; 
                          break; 

                  /*
                  .------------------------.
                  |  SSL Version Parameter |
                  '------------------------' */
                  case 8:
                          rParametriCheck.rClientConnectParameter.sslversion=malloc(strlen(buffer)+1);
                          strcpy(rParametriCheck.rClientConnectParameter.sslversion, buffer);
                          selected_parameter = 0;
                          break;

                  /* 
                  .--------------------------------.
                  |  Type Operation                |
                  '--------------------------------' */
                  case 9:
                          rParametriCheck.rClientConnectParameter.type_operation=malloc(strlen(buffer)+1);
                          strcpy(rParametriCheck.rClientConnectParameter.type_operation, buffer);
                          selected_parameter = 0;
                          break;

                  /*
                  .--------------------------------.
                  |  Message Language              |
                  '--------------------------------' */
                  case 10:
                          rParametriCheck.rClientConnectParameter.message_language=malloc(strlen(buffer)+1);
                          strcpy(rParametriCheck.rClientConnectParameter.message_language, buffer);
                          selected_parameter = 0;
                          break;

                  /*
                  .--------------------------------.
                  |  Message Type                  |
                  '--------------------------------' */
                  case 11:
                          rParametriCheck.rClientConnectParameter.message_type=malloc(strlen(buffer)+1);
                          strcpy(rParametriCheck.rClientConnectParameter.message_type, buffer);
                          selected_parameter = 0;
                          break;

                  /* 
                  .--------------------------------.
                  |  Message                       |
                  '--------------------------------' */
                  case 12:
                          rParametriCheck.rClientConnectParameter.message=malloc(strlen(buffer)+1);
                          strcpy(rParametriCheck.rClientConnectParameter.message, buffer);
                          selected_parameter = 0;
                          break;

                  /*
                  .--------------------------------.
                  |  SSL Ciphersuite Server Scan   |
                  '--------------------------------' */
                  case 13:
                          rParametriCheck.rClientConnectParameter.SSLCipherSuiteServerScan=malloc(strlen(buffer)+1); 
                          strcpy(rParametriCheck.rClientConnectParameter.SSLCipherSuiteServerScan, buffer); 
                          selected_parameter = 0;
                          break;

                  /*
                  .--------------------------------.
                  |  Max Request                   |
                  '--------------------------------' */
                  case 14:
                          rParametriCheck.rClientConnectParameter.MaxRequest = atol(buffer);
                          selected_parameter = 0;
                          break;

                  /*
                  .--------------------------------.
                  |  Output File                   |
                  '--------------------------------' */
                  case 15:
                          rParametriCheck.rClientConnectParameter.OutputFile=malloc(strlen(buffer)+1);
                          strcpy(rParametriCheck.rClientConnectParameter.OutputFile, buffer);
                          selected_parameter = 0;
                          break;

                  /*
                  .--------------------------------.
                  |  Configuration File Parameter  |
                  '--------------------------------' */
                  case 16: 

                          /*
                          .--------------------------.
                          | Load configuration file  | 
                          '--------------------------' */
                          rLoadConf = LoadConf (buffer);
                          if (rLoadConf.return_code == -1) {
                              PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_CONFIGURATION_FILE_NOTFOUND");
                              PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__BANNER"                            );
                              PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__FORMATO"                           );
                              rParametriCheck.return_code = -1;        
                              return rParametriCheck; };

                          /*
                          .---------------------------------------.
                          | Get parameter from configuration file | 
                          '---------------------------------------' */
                          PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__BANNER");

                          printf ("\nEnvironment configuration file [%s]\n", buffer);
                          printf ("\n.--------------.---------------------------------------------------------.");
                          printf ("\n| parameter    | value                                                   |");
                          printf ("\n.--------------.---------------------------------------------------------.");
                          printf ("\n| -connect     | [%70s] |", getVar(rLoadConf,"default","connect")          );
                          printf ("\n| -keystore    | [%70s] |", getVar(rLoadConf,"default","keystore")         );
                          printf ("\n| -password    | [%70s] |", getVar(rLoadConf,"default","password")         );
                          printf ("\n| -truststore  | [%70s] |", getVar(rLoadConf,"default","truststore")       );
                          printf ("\n| -ciphersuite | [%70s] |", getVar(rLoadConf,"default","ciphersuite")      );
                          printf ("\n'--------------'---------------------------------------------------------'");
                          printf ("\n");
                          printf ("\nThe parameter Configuration File (-config) is not implemented.");
                          printf ("\n\n");
                          selected_parameter = 0; 

                          rParametriCheck.return_code = -1;          
                          return rParametriCheck;
                          break; 

                  /*
                  .--------------------------------.
                  |  Debug                         |
                  '--------------------------------' */
                  case 17:
                          rParametriCheck.rClientConnectParameter.debug=malloc(strlen(buffer)+1);
                          strcpy(rParametriCheck.rClientConnectParameter.debug, buffer);
                          selected_parameter = 0;
                          break;


                  default: 
                          PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_UNKNOWN_PARAMETRO"); 
                          PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__BANNER"                  );
                          PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__FORMATO"                 );
                          rParametriCheck.return_code = -1;    
                          return rParametriCheck;
              };
          }; 

          indice++;
    };

    /*
    .------------------------------------------------------------------------.
    | Se il parse ha concluso l'analisi del comando e la variabile           |
    | selected_parameter *NON* è uguale a ZERO bensì contiene un valore      |
    | maggiore di ZERO significa che il parse stava analizzando un parametro |
    | di quelli supportati e si aspettava di trovare qualcosa ancora per     |
    | poterlo completamente definire. Quindi il parse si ferma con un errore |                                      
    '------------------------------------------------------------------------' */
    if ( selected_parameter > 0 ) 
    {
         switch ( selected_parameter )
         {
              case  1 : PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_ATTESO_PARAMETRO_CONNECT"                      ); break;
              case  2 : PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_ATTESO_PARAMETRO_AUTH_USER"                    ); break;
              case  3 : PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_ATTESO_PARAMETRO_AUTH_PASSWORD"                ); break;
              case  4 : PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_ATTESO_PARAMETRO_KEYSTORE"                     ); break;
              case  5 : PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_ATTESO_PARAMETRO_KEYSTORE_PASSWORD"            ); break;
              case  6 : PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_ATTESO_PARAMETRO_TRUSTSTORE"                   ); break;
              case  7 : PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_ATTESO_PARAMETRO_CIPHERSUITE"                  ); break;
              case  8 : PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_ATTESO_PARAMETRO_VERSION_SSL"                  ); break;
              case  9 : PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_ATTESO_PARAMETRO_TYPE_OPERATION"               ); break;
              case 10 : PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_ATTESO_PARAMETRO_MESSAGE_LANGUAGE"             ); break;
              case 11 : PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_ATTESO_PARAMETRO_MESSAGE_TYPE"                 ); break;
              case 12 : PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_ATTESO_PARAMETRO_MESSAGE"                      ); break;
              case 13 : PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_ATTESO_PARAMETRO_SSL_CIPHERSUITE_SERVER_SCAN"  ); break;
              case 14 : PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_ATTESO_PARAMETRO_MAX_REQUEST"                  ); break;
              case 15 : PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_ATTESO_PARAMETRO_OUTPUT_FILE"                  ); break;
              case 16 : PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_ATTESO_PARAMETRO_CONFIGURATION_FILE"           ); break;
              case 17 : PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_ATTESO_PARAMETRO_DEBUG"                        ); break;
              default : PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_UNKNOWN_PARAMETRO"                             );
         };
         PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__BANNER"          );
         PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__FORMATO"         );
         rParametriCheck.return_code = -1;                    
         return rParametriCheck;
    };

//====================================================================================================
// Il legame logico tra i parametri va spostato nella libreria client. il frontend deve essere
// svuotato completamente di logica. La logica va centralizzata perchè ci potrebbero essere 
// più di un frontend che insiste sullo stessa libreria client.
//====================================================================================================
    /*
    .------------------------------------------------------------------.
    | Se son valorizzati parametri legati alla CONNECT e non risulta   |
    | impostato il parametro CONNECT segnala l'errore...               |
    '------------------------------------------------------------------' */
    if ((rParametriCheck.rClientConnectParameter.auth_user     != NULL ||
         rParametriCheck.rClientConnectParameter.auth_password != NULL || 
         rParametriCheck.rClientConnectParameter.keystore      != NULL || 
         rParametriCheck.rClientConnectParameter.password      != NULL ||
         rParametriCheck.rClientConnectParameter.truststore    != NULL || 
         rParametriCheck.rClientConnectParameter.ciphersuite   != NULL || 
         rParametriCheck.rClientConnectParameter.sslversion    != NULL ) && 
         rParametriCheck.rClientConnectParameter.protocollo    == NULL ) { 
         PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__ERRORE_ATTESO_PARAMETRO_CONNECT");
         PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__BANNER"                         );
         PrintMessage(debug_level, PARAM_TABMSG, "MY_CLIENT_TCP__FORMATO"                        );
    };

//=====================================================================================================

    /*
    .------------------------------------------------.
    | frontend utilizzato                            |
    '------------------------------------------------' */
    rParametriCheck.rClientConnectParameter.frontend = malloc(strlen(frontend)+1) ;
    strcpy(rParametriCheck.rClientConnectParameter.frontend, frontend);

    /*
    .------------------------------------------------.
    | frontend utilizzato (version)                  |
    '------------------------------------------------' */
    rParametriCheck.rClientConnectParameter.frontend_version = malloc(strlen(frontend_version)+1);
    strcpy(rParametriCheck.rClientConnectParameter.frontend_version, frontend_version); 

    /*
    .------------------------------------------------------------------.
    | Visualizza i parametri impostati analizzati dalla routine per    |
    | attività di DEBUG. (DEBUG = 1|0 * Attiva|Non Attiva).            |
    '------------------------------------------------------------------' */ 
    if ( DEBUG == 1 )
    {
         int indice;    
          
         printf ("\n.-----------------------------------------------------------------------------.");
         printf ("\n|                 Funzione ParametriCheck DUMP DEBUG                          |");
         printf ("\n'-----------------------------------------------------------------------------'\n\n");
       
         printf ("\nX------------------------- Valori di Input -----------------------------------X\n\n");

         printf ("\nfrontend              : [%s]" , frontend         );
         printf ("\nfrontend version      : [%s]" , frontend_version );
         indice = 0;
         while ( indice < nParametri )
         {
                 printf ("\nParametro [%d]         : [%s]" , indice, lParametri[indice]);
                 indice++;
         }; 
         printf ("\nnumero parametri      : [%d]\n\n" , nParametri );


         printf ("\nX------------------------- Valori di Output ----------------------------------X\n\n");

         printf ("\nfrontend              : [%s]" , rParametriCheck.rClientConnectParameter.frontend         );
         printf ("\nfrontend version      : [%s]" , rParametriCheck.rClientConnectParameter.frontend_version );
         printf ("\nprotocollo            : [%s]" , rParametriCheck.rClientConnectParameter.protocollo       );
         printf ("\nhost                  : [%s]" , rParametriCheck.rClientConnectParameter.host             );
         printf ("\nporta                 : [%d]" , rParametriCheck.rClientConnectParameter.porta            );
         printf ("\nrisorsa path          : [%s]" , rParametriCheck.rClientConnectParameter.risorsa_path     );
         printf ("\nrisorsa               : [%s]" , rParametriCheck.rClientConnectParameter.risorsa          );

         printf ("\nLunghezza QueryString : [%d]" , rParametriCheck.rClientConnectParameter.queryStringLen            );
         printf ("\nNumero Variabili      : [%d]" , rParametriCheck.rClientConnectParameter.queryStringDimTabVariable );
         printf ("\n");
         indice = 0 ;
         while ( indice < rParametriCheck.rClientConnectParameter.queryStringDimTabVariable )
         {
                 printf ("\n[%3d.]  [%30s] = [%30s]",
                         indice,
                         rParametriCheck.rClientConnectParameter.queryStringTabVar[indice].variable,
                         rParametriCheck.rClientConnectParameter.queryStringTabVar[indice].value );

                 indice++;
         }
         printf("\n");

         printf ("\nauth user                   : [%s]" , rParametriCheck.rClientConnectParameter.auth_user                   );
         printf ("\nauth password               : [%s]" , rParametriCheck.rClientConnectParameter.auth_password               );
         printf ("\nkeystore                    : [%s]" , rParametriCheck.rClientConnectParameter.keystore                    ); 
         printf ("\npassword                    : [%s]" , rParametriCheck.rClientConnectParameter.password                    );
         printf ("\ntruststore                  : [%s]" , rParametriCheck.rClientConnectParameter.truststore                  );
         printf ("\nciphersuite                 : [%s]" , rParametriCheck.rClientConnectParameter.ciphersuite                 ); 
         printf ("\nssl version                 : [%s]" , rParametriCheck.rClientConnectParameter.sslversion                  );
         printf ("\ntype operation              : [%s]" , rParametriCheck.rClientConnectParameter.type_operation              ); 
         printf ("\nmessage_language            : [%s]" , rParametriCheck.rClientConnectParameter.message_language            );
         printf ("\nmessage_type                : [%s]" , rParametriCheck.rClientConnectParameter.message_type                );
         printf ("\nmessage                     : [%s]" , rParametriCheck.rClientConnectParameter.message                     ); 
         printf ("\nssl ciphersuite server scan : [%s]" , rParametriCheck.rClientConnectParameter.SSLCipherSuiteServerScan    );
         printf ("\nmax Request                 : [%ld]", rParametriCheck.rClientConnectParameter.MaxRequest                  );  
         printf ("\noutput file                 : [%s]" , rParametriCheck.rClientConnectParameter.OutputFile                  );
         printf ("\ndebug                       : [%s]" , rParametriCheck.rClientConnectParameter.debug                       );
         printf ("\nreturn_code                 : [%d]" , rParametriCheck.return_code                                         );
         printf ("\n\n");
    };

    return rParametriCheck;

};
 

// .: ___EOF___ :. 

