/*
# Copyright (C) 2008-2009 
# Raffaele Granito <raffaele.granito@tiscali.it> 
#
# This file is part of myTCPClient:
# Universal TCP Client 
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

/*
.--------------.
| headers LIBC |
'--------------' */
#include <sys/time.h>

/*
.--------------------.
| header Common      |
'--------------------' */
#include "common.h"
#include "output.h" 

/*
.--------------------.
| header protocolli  | 
'--------------------' */
#include "tcp.h"
#include "ssl.h"
#include "http.h"
#include "ems.h"
#include "tems.h"

/*
.--------------------.
| header locale      |
'--------------------' */
#include "client.h"
#include "client_messages.h"

/*
.----------------------------------------------------------.
|                                                          |
|  Funzione   : clientConnect                              |
|                                                          |
|  Descrizione: Effettua una connessione SSL               |
|                                                          |
'----------------------------------------------------------'*/
extern struct tClientConnect
ClientConnect ( struct tClientConnectParameter rClientConnectParameter )
{
    /*
    .------------------------------------------.
    | Dichiarazione delle STRUTTURE            |
    '------------------------------------------' */
    struct tClientConnect  rClientConnect;
    struct tClientConnect *pClientConnect;

    /*
    .------------------------------------.
    | Allocazione Spazio                 |
    '------------------------------------' */
    pClientConnect = malloc(sizeof(struct tClientConnect));

    /*
    .------------------------------------.
    | Utilizzo dello spazio allocato     |
    '------------------------------------' */
    pClientConnect = &rClientConnect;

    /*
    .-----------------------------.
    | Inizializzazione Strutture  |
    '-----------------------------' */
    memset((void*)&rClientConnect, 0, sizeof(rClientConnect));

    /*
    .-----------------------------.
    | Inizializzazione Strutture  |
    '-----------------------------' */
    rClientConnect.return_code = 0;


    /*
    .------------------------------------------.
    |  Determina il livello di DEBUG           |
    '------------------------------------------' */
    int debug_level = GetDebugLevel ( rClientConnectParameter.debug, "client" ) ; 


    /*
    .----------------------------------------------------------.
    |                                                          |
    |                    Banner Client TCP                     |
    |                                                          |
    '----------------------------------------------------------' */
  //PrintMessage(debug_level, CLIENT_TABMSG, "MY_CLIENT_TCP__BANNER");
    PrintMessage(1,           CLIENT_TABMSG, "MY_CLIENT_TCP__BANNER");

    /*
    .---------------------------------------------------.
    |                                                   |
    |                  Tempo Iniziale                   |
    |                 (intera sessione)                 |
    |                                                   |
    '---------------------------------------------------' */
    struct timeval before;
    gettimeofday (&before, 0);
    long long tempo_iniziale_intera_sessione;
    tempo_iniziale_intera_sessione = 1000000LL * before.tv_sec + before.tv_usec;

    /*
    .--------------------------------------------------.
    |                                                  |
    |                 Inizializzazioni                 |
    |                                                  |
    '--------------------------------------------------' */ 
    int flag_unsupported_protocol = 0;
    int sequence = 0 ;
    int contaPacchettiInviati = 0 ;
    int contaPacchettiRicevuti = 0 ;
    double tempo_min_singola_connessione = 0 ;
    double tempo_max_singola_connessione = 0 ;

    if ( rClientConnectParameter.SSLCipherSuiteServerScan == NULL ) { 
         rClientConnectParameter.SSLCipherSuiteServerScan = malloc(strlen("Off\0")+1);
         strcpy(rClientConnectParameter.SSLCipherSuiteServerScan, "Off\0"); 
    };
 

    /*
    .-------------------------------------------------------.
    | Numero di Richieste se non impostato assume 1         |
    | come valore di default.                               |
    '-------------------------------------------------------' */
    if ( rClientConnectParameter.MaxRequest == 0 )
         rClientConnectParameter.MaxRequest = 1 ; 


    while ( sequence < rClientConnectParameter.MaxRequest || 
                ( strcasecmp(rClientConnectParameter.SSLCipherSuiteServerScan,"on") == 0 && rClientConnect.SSLCipherSuite_NoUsed_Num > 0 ) )
    {
            sequence++;

            /*
            .---------------------------------------------------.
            |                                                   |
            |                                                   |
            |                  Tempo Iniziale                   |
            |               (Singola connessione)               |
            |                                                   |
            |                                                   |
            '---------------------------------------------------' */
            struct timeval before;
            gettimeofday (&before, 0);
            long long tempo_iniziale_singola_connessione;
            tempo_iniziale_singola_connessione = 1000000LL * before.tv_sec + before.tv_usec;
 
            /* 
            .---------------------------------------------------.
            |                 Protocollo TCP                    |
            '---------------------------------------------------' */ 
            if ( strcmp ( rClientConnectParameter.protocollo , "tcp" ) == 0 )
                 rClientConnect = TCPClientConnect ( rClientConnectParameter );

            /*
            .---------------------------------------------------.
            |                 Protocollo SSL                    |
            '---------------------------------------------------' */
            if ( strcmp ( rClientConnectParameter.protocollo , "ssl" ) == 0 )
                 rClientConnect = SSLClientConnect ( rClientConnectParameter );
            
            /*
            .---------------------------------------------------.
            |              Protocollo HTTP/HTTPS                |
            '---------------------------------------------------' */
            if ( strcmp ( rClientConnectParameter.protocollo , "http"  ) == 0 || 
                 strcmp ( rClientConnectParameter.protocollo , "https" ) == 0 )  
                 rClientConnect = HTTPClientConnect ( rClientConnectParameter ); 

            /*
            .---------------------------------------------------.
            |                Protocollo EMS/EMSS                |
            '---------------------------------------------------' */
            if ( strcmp ( rClientConnectParameter.protocollo , "ems"  ) == 0 || 
                 strcmp ( rClientConnectParameter.protocollo , "emss" ) == 0 )  
                 rClientConnect = EMSClientConnect ( rClientConnectParameter );

            /*
            .---------------------------------------------------.
            |               Protocollo TIBCO EMS                |
            '---------------------------------------------------' */
            if ( strcmp ( rClientConnectParameter.protocollo , "tems" ) == 0 ||
                 strcmp ( rClientConnectParameter.protocollo , "temss") == 0 )  
                 rClientConnect = TEMSClientConnect ( rClientConnectParameter );

            /*
            .---------------------------------------------------.
            |               Protocollo UNSUPPORTED              |
            '---------------------------------------------------' */
            if ( strcmp ( rClientConnectParameter.protocollo , "tcp"   ) != 0 && 
                 strcmp ( rClientConnectParameter.protocollo , "ssl"   ) != 0 && 
                 strcmp ( rClientConnectParameter.protocollo , "http"  ) != 0 && 
                 strcmp ( rClientConnectParameter.protocollo , "https" ) != 0 && 
                 strcmp ( rClientConnectParameter.protocollo , "ems"   ) != 0 && 
                 strcmp ( rClientConnectParameter.protocollo , "emss"  ) != 0 && 
                 strcmp ( rClientConnectParameter.protocollo , "tems"  ) != 0 && 
                 strcmp ( rClientConnectParameter.protocollo , "temss" ) != 0 )
                 flag_unsupported_protocol = 1;

            /*
            .---------------------------------------------------.
            |                                                   |
            |             Gestione Scan CipherSuite             |
            |                                                   |
            '---------------------------------------------------' */
            if ( strcasecmp(rClientConnectParameter.SSLCipherSuiteServerScan,"on") == 0 )
            {
                 /*
                 .------------------------------------------------------------.
                 | La connessione SSL è andata male per nessuna ciphersuite   |
                 | col server [???] nessuna di quelle ancora usate son        |
                 | considerate valide per il nostro interlocutore. Pertanto   |
                 | è inutile andare avanti. Le ciphersuite son state provate  |
                 | [???] tutte quindi si può impostare la variabile           |
                 | SSLCipherSuite_NoUsed_Num a 0, quindi terminare lo scan.   |
                 | E' stato effettuato l'ulteriore test sul numero di         |
                 | ciphersuite non ancora utilizzate, se nessuna o meno per   |
                 | ulteriori sviluppi futuri. Al momento tale ulteriore test  |
                 | non ha senso perchè l'azione scaturita è la medesima.      |
                 '------------------------------------------------------------' */ 
                 /* quà è necessario verificare bene (nel limite del possibile)
                    che la connessione sia andata male perchè il server non 
                    ha considerato valida nessuna ciphersuite che il client gli ha proposto
                    In ogni caso l'importante è comunque fornire le informazioni
                    necessarie all'utente ad interpretare il fenomeno e proseguire */
                 if ( rClientConnect.SSLConnectReturn != 0 )
                 { 
                      if ( rClientConnect.SSLCipherSuite_NoUsed_Num > 0 )
                      { 
                           rClientConnect.SSLCipherSuite_NoUsed_Num = 0 ; 
                           /* quà è necessario utilizzare quella routine che analizza ed elimina 
                              la ciphesuite utilizzata dall'array e farla funzionare in maniera
                              tale che se non gli viene passata la ciphersuite da eliminare
                              elimina la prima trovata nell'array, restituendo ciphersuite 
                              eliminata e array nuovo senza tale ciphersuite in maniera tale 
                              da consumare l'array ed uscire in maniera naturale con 
                              la conclusione dello scanning */
                      }

                      if ( rClientConnect.SSLCipherSuite_NoUsed_Num == 0 )
                      {                       
                           rClientConnect.SSLCipherSuite_NoUsed_Num = 0 ;
                           /* quà bisogna far in modo di visualizzare solo quella utilizzata, che è anche l'ultima
                              che andava provata, e poi uscire pr fine naturale dello scanning */
                      }
                 };
 
                 /*
                 .------------------------------------------------------------.
                 | La connessione SSL è andata bene, quindi si può            |
                 | tranquillamente proseguire con lo scanning delle           |
                 | ciphersuite verso il server, sempre che ce ne siano ancora |
                 | da provare.                                                |
                 | E' stato effettuato l'ulteriore test sul numero di         |
                 | ciphersuite non ancora utilizzate, se nessuna o meno per   |
                 | ulteriori sviluppi futuri. Al momento tale ulteriore test  |
                 | non ha senso se il numero di ciphersuite non utilizzate è  |
                 | ZERO. Essendo questa la fine naturale dello scanning non   |
                 | viene eseguita nessuna azione.                             |
                 '------------------------------------------------------------' */ 
                 if ( rClientConnect.SSLConnectReturn == 0 ) 
                 {  
                      /*
                      .-------------------------------------------------------.
                      | Se l'ultima connessione è andata bene e ci sono       |
                      | ancora ciphersuite non usate allora la routine        |
                      | fornisce in input per la prossimo test di connessione |
                      | le ciphersuite non ancora utilizzate dal server       |
                      '-------------------------------------------------------' */
                      if ( rClientConnect.SSLCipherSuite_NoUsed_Num > 0 ) 
                      { 
                           rClientConnectParameter.ciphersuite = malloc(strlen(rClientConnect.SSLCipherSuite_NoUsed)+1);
                           strcpy ( rClientConnectParameter.ciphersuite, rClientConnect.SSLCipherSuite_NoUsed );
                      }; 

                      /* questa condizione credo vada eliminata */
                      if ( rClientConnect.SSLCipherSuite_NoUsed_Num == 0 )
                      {
                      }
                 };
            }; 

            /*
            .---------------------------------------------------.
            |                                                   |
            |                                                   |
            |                   Tempo Finale                    |
            |               (Singola connessione)               |
            |                                                   |
            |                                                   |
            '---------------------------------------------------' */ 
            struct timeval after;
            gettimeofday (&after, 0);
            long long tempo_finale_singola_connessione ;
            tempo_finale_singola_connessione = 1000000LL * after.tv_sec + after.tv_usec;

            double tempo_totale_singola_connessione = (double) (tempo_finale_singola_connessione - tempo_iniziale_singola_connessione)/1000 ;

            if ( tempo_totale_singola_connessione < tempo_min_singola_connessione || sequence == 1 )
                 tempo_min_singola_connessione = tempo_totale_singola_connessione ;

            if ( tempo_totale_singola_connessione > tempo_max_singola_connessione )
                 tempo_max_singola_connessione = tempo_totale_singola_connessione ;
  


            /*
            .---------------------------------------------------.
            |                                                   |
            |    Costruisce e Visualizza la stringa testata     |
            |                                                   |
            '---------------------------------------------------' */
            if ( sequence == 1 )
            { 
                 printf ("PING %s://%s:%d%s%s (%s) %d(%d) bytes of data.\n", 
                          rClientConnectParameter.protocollo,
                          rClientConnectParameter.host,
                          rClientConnect.TCPPort,
                          rClientConnectParameter.risorsa_path,
                          rClientConnectParameter.risorsa,
                          rClientConnect.TCPIp,
                          rClientConnect.ServerRequestBufferLen,
                          rClientConnect.ServerRequestBufferLen ); 
            };

            /*
            .---------------------------------------------------.
            |                                                   |
            |     Costruisce e Visualizza la riga dettaglio     | 
            |                                                   |
            '---------------------------------------------------' */
            printf ("%06d byte from %s:%d %s_seq=%03d time=%.3g ms",
                        rClientConnect.ServerResponseBufferLen,  
                        rClientConnectParameter.host,
                        rClientConnect.TCPPort,
                        rClientConnectParameter.protocollo,
                        sequence,
                        tempo_totale_singola_connessione ); 

            /*
            .------------------------------------------------------------.
            | Print SSL Parameter Handshake (negoziati)                  |
            +------------------------------------------------------------+
            | Visualizza i parametri concordati tra C/S durante          |
            | l'handshake ssl. Se non si sta utilizzando SSL oppure se   |
            | la negoziazzione fallisce non visualizza nulla chiaramente |
            | per i campi non sono valorizzati.                          |
            '------------------------------------------------------------' */
            if ( rClientConnect.SSLCipherSuite_AlgSim_LenKey > 0 )
            {
                 printf ("\t [ %11s | %20s | %3d ]" ,
                     rClientConnect.SSLCipherSuite_AlgSim_SSLVersion,   
                     rClientConnect.SSLCipherSuite,   
                     rClientConnect.SSLCipherSuite_AlgSim_LenKey );
            };

            /*
            .------------------------------------------------------------.
            | Print Error                                                |
            +------------------------------------------------------------+
            | Visualizza il primo messaggio di errore trovato.           |
            '------------------------------------------------------------' */ 
            if ( rClientConnect.TCPConnectReturn              != 0 ) 
                 printf ("\t [%ld] %s" , rClientConnect.TCPConnectMsg,
                                         rClientConnect.TCPConnectMsgDesc );

            if ( rClientConnect.TCPConnectReturn              == 0 && 
                 rClientConnect.SSLConnectReturn              != 0 ) 
                 printf ("\t [%ld] %s" , rClientConnect.SSLConnectMsg, 
                                         rClientConnect.SSLConnectMsgDesc );

            if ( rClientConnect.TCPConnectReturn              == 0 &&
                 rClientConnect.SSLConnectReturn              == 0 && 
                 rClientConnect.SSLServerVerifyIdentityReturn <  0 )
                 printf ("\t [%ld] %s" , rClientConnect.SSLServerVerifyIdentityMsg,
                                         rClientConnect.SSLServerVerifyIdentityMsgDesc );

            if ( rClientConnect.TCPConnectReturn              == 0 && 
                 rClientConnect.SSLConnectReturn              == 0 && 
                 rClientConnect.SSLServerVerifyIdentityReturn >= 0 && 
                 rClientConnect.ServerConnectReturn           != 0 ) 
                 printf ("\t [%ld] %s" , rClientConnect.ServerConnectMsg,
                                         rClientConnect.ServerConnectMsgDesc );

            if ( rClientConnect.TCPConnectReturn              == 0 &&
                 rClientConnect.SSLConnectReturn              == 0 &&
                 rClientConnect.SSLServerVerifyIdentityReturn >= 0 &&
                 rClientConnect.ServerConnectReturn           == 0 && 
                 rClientConnect.ServerRequestReturn           != 0 ) 
                 printf ("\t [%ld] %s" , rClientConnect.ServerRequestMsg,
                                         rClientConnect.ServerRequestMsgDesc );

            if ( rClientConnect.TCPConnectReturn              == 0 &&
                 rClientConnect.SSLConnectReturn              == 0 &&
                 rClientConnect.SSLServerVerifyIdentityReturn >= 0 &&
                 rClientConnect.ServerConnectReturn           == 0 && 
                 rClientConnect.ServerRequestReturn           == 0 && 
                 rClientConnect.ServerResponseReturn          != 0 ) 
                 printf ("\t [%ld] %s" , rClientConnect.ServerResponseMsg,
                                         rClientConnect.ServerResponseMsgDesc );

            if ( rClientConnect.TCPConnectReturn              == 0 &&
                 rClientConnect.SSLConnectReturn              == 0 &&
                 rClientConnect.SSLServerVerifyIdentityReturn >= 0 &&
                 rClientConnect.ServerConnectReturn           == 0 &&
                 rClientConnect.ServerRequestReturn           == 0 && 
                 rClientConnect.ServerResponseReturn          == 0 && 
                 rClientConnect.ServerDisconnectReturn        != 0 ) 
                 printf ("\t [%ld] %s" , rClientConnect.ServerDisconnectMsg,
                                         rClientConnect.ServerDisconnectMsgDesc );

            if ( rClientConnect.TCPConnectReturn              == 0 &&
                 rClientConnect.SSLConnectReturn              == 0 &&
                 rClientConnect.SSLServerVerifyIdentityReturn >= 0 &&
                 rClientConnect.ServerConnectReturn           == 0 &&
                 rClientConnect.ServerRequestReturn           == 0 &&
                 rClientConnect.ServerResponseReturn          == 0 && 
                 rClientConnect.ServerDisconnectReturn        == 0 && 
                 rClientConnect.SSLDisconnectReturn           != 0 ) 
                 printf ("\t [%ld] %s" , rClientConnect.SSLDisconnectMsg,
                                         rClientConnect.SSLDisconnectMsgDesc );

            if ( rClientConnect.TCPConnectReturn              == 0 &&
                 rClientConnect.SSLConnectReturn              == 0 &&
                 rClientConnect.SSLServerVerifyIdentityReturn >= 0 &&
                 rClientConnect.ServerConnectReturn           == 0 &&
                 rClientConnect.ServerRequestReturn           == 0 &&
                 rClientConnect.ServerResponseReturn          == 0 &&
                 rClientConnect.ServerDisconnectReturn        == 0 && 
                 rClientConnect.SSLDisconnectReturn           == 0 && 
                 rClientConnect.TCPDisconnectReturn           != 0 ) 
                 printf ("\t [%ld] %s" , rClientConnect.TCPDisconnectMsg,
                                         rClientConnect.TCPDisconnectMsgDesc );

            if ( flag_unsupported_protocol == 1 ) { 
                 printf ("\t [%d] %s" , -1, "CLIENT.PROTOCOL.USED.IS.NOT.SUPPORTED" );
                 rClientConnect.return_code = -1; 
            };

            /*
            .------------------------------------------------------------.
            | Return Code                                                |
            +------------------------------------------------------------+
            | Forza a -2 il return-code della funzione client se il      |
            | il tentativo di connessione per qualche ragione è fallito  |
            | oltre a visualizzare sulla riga il messaggio di "fail".    |
            '------------------------------------------------------------' */
            if ( rClientConnect.return_code <   0 ) {
                 printf (" fail.\n");
            };

            if ( rClientConnect.return_code ==  0 )
                 printf (" \n");

            /*
            .---------------------------------------------------.
            |                                                   |
            |             Conta i pacchetti inviati             |
            |                                                   |
            '---------------------------------------------------' */
            if ( rClientConnect.ServerRequestBufferLen > 0 ) 
                 contaPacchettiInviati++;

            /*
            .---------------------------------------------------.
            |                                                   |
            |             Conta i pacchetti inviati             |
            |                                                   |
            '---------------------------------------------------' */
            if ( rClientConnect.ServerResponseBufferLen > 0 ) 
                 contaPacchettiRicevuti++ ;

            /*
            .---------------------------------------------------.
            |                                                   |
            |                       Pausa                       |
            |                                                   |
            '---------------------------------------------------' */
            sleep(1);
    };


    /*
    .---------------------------------------------------.
    |                                                   |
    |                   Tempo Finale                    |
    |                 (intera sessione)                 |
    |                                                   |
    '---------------------------------------------------' */
    struct timeval after;
    gettimeofday (&after, 0);
    long long tempo_finale_intera_sessione ;
    tempo_finale_intera_sessione = 1000000LL * after.tv_sec + after.tv_usec;


    /*
    .----------------------------------------------------------------------.
    |                                                                      |
    |                      Visualizza le statistiche                       |
    |        ( solamente se viene effettuata piu di una richiesta )        | 
    |                                                                      | 
    |                                                                      |
    '----------------------------------------------------------------------' */
    if ( sequence > 1 )
    {
         printf ("--- %s://%s:%d%s%s ping statistics ---\n",
                             rClientConnectParameter.protocollo,
                             rClientConnectParameter.host,
                             rClientConnect.TCPPort,
                             rClientConnectParameter.risorsa_path,
                             rClientConnectParameter.risorsa ); 

         double tempo_totale_intera_sessione = (double) (tempo_finale_intera_sessione - tempo_iniziale_intera_sessione)/1000 ;
         printf ("%d packets transmitted, %d received, 0%% packet loss, time %.3gms\n", 
                             contaPacchettiInviati, contaPacchettiRicevuti, tempo_totale_intera_sessione ) ; 
 
         printf ("rtt min/avg/max/mdev = %.3g/0.000/%.3g/0.000 ms\n",  
                             tempo_min_singola_connessione, 
                             tempo_max_singola_connessione ); 
    };

    return rClientConnect; 
}



// __EOF__
