/***************************************************************************
 * Author:             Manuele Vaggelli [manuele.vaggelli@member.fsf.org]
 * Creation_Date:      Tuesday, May 16, 2006
 * Last_Modified_Date: Friday, August 18, 2006
 * Program:            MySPwizard
 * File:               myspwizard.c
 * Description:        Stored Procedure generator for  MySQL database
 * Note:               Version 1.0
 *
 * Copyright (C) 2006 Manuele Vaggelli
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc.,51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 ***************************************************************************/

#include <unistd.h>
#include <time.h>
#include <errno.h>
#include <string.h>
#include <mysql.h>
#include <mysql_com.h>
#include "concat.h"

#if defined(_WIN32) || defined(_WIN64)
#include <windows.h>
#include <winsock.h>
#endif

#undef DEBUG
#define PROG_NAME "MySPwizard"
#define PROG_VERSION "1.0"
#define PROG_AUTHOR "Manuele Vaggelli [manuele.vaggelli@member.fsf.org]"
#define PARAM_PREFIX "_"
#define DELIMITER "$$\n\n"
#define INSERT "insert"
#define DELETE "delete"
#define UPDATE "update"
#define SELECT "select"

/* PROTOTYPE DECLARATIONS */
void save_spfile(char *, char *);
void show_usage(void);
char *get_sp_start(void);
char *get_sp_end(void);
char *get_sp_comments(void);
char *get_sp_parameters(void);
char *get_sql_select(void);
char *get_sql_insert(void);
char *get_sql_update(void);
char *get_sql_delete(void);
char *get_fdate(void);
char *get_curr_statement(char *);

/* VARIABLE DECLARATIONS */

// The handles to a database connection
MYSQL mysql;
MYSQL *sock;

// Rapresentation of one row of data
MYSQL_ROW row;

// The result of a query (select)
MYSQL_RES *result;

// The program name
char *prog_name = NULL;

// The number of fields per row
unsigned int num_fields;

// The number of rows retrieved
my_ulonglong num_rows;

// The current row number
my_ulonglong cur_rows;

// SQL buffer to send to the server
char *sql_buf = "";

// SQL buffer for stored procedure text
char *sp_buf = "";

// The current statement (select, insert, update, delete)
char *currstmnt = "select";

// The stored procedure name
char *sp_name = "";

// The output file containing the stored procedure
char *sp_fname = "";

// The Table name
char *table = NULL;

// The Database name
char *db = NULL;

// Server Host
char *host = NULL;

// Password to use when connecting to server
char *passwd = "";

// User to use when connecting to server
char *user = NULL;

// Port number to use for connection
unsigned int port = 0;


int
main(int argc, char *argv[])
{
  int index;
  int c;
  
  opterr = 0;
  prog_name = argv[0];
  
  while ((c = getopt (argc, argv, "h:p:P:u:s:")) != -1)
    switch (c)
      {
      case 'h':
	host = optarg;
	break;
      case 'p':
	passwd = optarg;
	break;
      case 'P':
	port = (unsigned int)atoi(optarg);
	break;
      case 'u':
	user = optarg;
	break;
      case 's':
	currstmnt = get_curr_statement(optarg);
	break;	
      case '?':
	if (isprint (optopt))
	  fprintf (stderr, "Unknown option `-%c'.\n", optopt);
	else
	  fprintf (stderr, "Unknown option character `\\x%x'.\n", optopt);
	show_usage();
	return 1;
      default:
	abort ();
      }

  for (index = optind; index < argc; index++)
    { 
      if(db == NULL)
	db = argv[index];
      else
	table = argv[index];
    }
       
  if( (db == NULL) || (table == NULL) )
    {
      show_usage();
      exit(2);
    }

  // Debug Information
#ifdef DEBUG  
  printf ("db = %s\nhost = %s\npasswd = %s\nport = %d\ntable = %s\nuser = %s\ncurrstmnt = %s\n",
	  db,host,passwd,port,table,user,currstmnt);
#endif

  
  // Initilize the connection handler
  mysql_init(&mysql);

  // Attempt to estabilish a connection to the MySQL database
  if (!(sock = mysql_real_connect(&mysql,host,user,passwd,db,port,NULL,0)))
    {
      error(EXIT_FAILURE,
	    errno,
	    "Failed to connect to database\n%s", mysql_error(&mysql));
    }
	
  // Try reconnecting to the server before giving up
  mysql.reconnect= 1;

  sql_buf = concat(sql_buf,"DESCRIBE ", table, NULL);

  // Set the Stored Procedure Name
  sp_name = concat(sp_name, db, "_", currstmnt, "_", table, NULL);

  // Set the Stored Procedure File Name
  sp_fname = concat(sp_fname, sp_name, ".sql", NULL);

     
  if (mysql_query(&mysql, sql_buf))
    {
      error(EXIT_FAILURE,
	    errno,
	    "Failed executing query: Error: %s\n", mysql_error(&mysql));
    } 
  else 
    {
      result = mysql_store_result(&mysql);
		
      if (result) 
	{	  
	  num_fields = mysql_num_fields(result);
	  num_rows = mysql_num_rows(result);
	  
	  sp_buf = concat(sp_buf, get_sp_start(), NULL);
	  
	  if (strcmp(currstmnt, INSERT) == 0)
	    sp_buf = concat(sp_buf, get_sql_insert(), NULL);	
	  else if (strcmp(currstmnt, UPDATE) == 0)
	    sp_buf = concat(sp_buf, get_sql_update(), NULL);
	  else if (strcmp(currstmnt, DELETE) == 0)
	    sp_buf = concat(sp_buf, get_sql_delete(), NULL);
	  else 
	    sp_buf = concat(sp_buf, get_sql_select(), NULL);
	
	  sp_buf = concat(sp_buf, get_sp_end(), NULL);
	  
	  mysql_free_result(result);
	       
	  // Saves the Stored Procedure in a file
	  save_spfile(sp_buf, sp_fname);
	} 
      else 
	{
	  if (mysql_errno(&mysql))
	    {
	      error(EXIT_FAILURE,
		    errno,
		    "Error: %s\n", mysql_error(&mysql));
	    }
	}
    }

  mysql_close(sock);

  printf("\nRESULT OK:\n%s succesfully created in file %s\n\n", sp_name, sp_fname);
  
  return 0;
}

/***********************************************************
 * save_spfile:
 *  
 *  Creates the file containing the SQL statement to 
 *  generate the stored procedure
 ***********************************************************/
void
save_spfile(char *sp_sql, char *sp_fname)
{
  FILE *sp_file;

  sp_file = fopen(sp_fname, "w");
  if(sp_file == NULL) {
    error(EXIT_FAILURE,
	  errno,
	  "Couldn't save file %s;\n", sp_fname);
  }

  fprintf(sp_file, sp_sql);	
  fclose(sp_file);
}


/***********************************************************
 * show_usage:
 *
 *  Composes and print the usage message
 ***********************************************************/
void
show_usage(void)
{
  printf("\n");
  printf("%s - Version: %s", PROG_NAME, PROG_VERSION);
  printf("\n\n");
  printf("Copyright (C) 2006 %s\n", PROG_AUTHOR);  
  printf("This software comes with ABSOLUTELY NO WARRANTY. This is free software,\n");
  printf("and you are welcome to modify and redistribute it under the GPL license\n");
  printf("\nUsage: %s [OPTIONS] DATABASE TABLE\n", prog_name);
  printf("\n");
  printf("-h   Connect to host\n");
  printf("     (if it is not given 'localhost' is assumed)\n");
  printf("-p   Password to use when connecting to the server\n");
  printf("-P   Port number to use for connection\n");
  printf("     (if it is not given '3306' is assumed)\n");
  printf("-u   User to use when connecting to the server\n");
  printf("     (if it is not given the current user is assumed)\n");
  printf("-s   The SQL Statement Type\n");
  printf("     one of: INSERT || UPDATE || DELETE || SELECT\n");
  printf("     (if it is not given 'SELECT' is assumed)\n");
  printf("\n");
}


/***********************************************************
 * get_sp_start:
 *
 *  Composes and returns the initial SQL stataments 
 *  for the stored procedure
 ***********************************************************/
char *
get_sp_start(void)
{
  char *snew = "";
   
  sp_buf = concat(sp_buf, get_sp_comments(), NULL);
  sp_buf = concat(sp_buf, "USE `", db, "`;\n\n", NULL);
  sp_buf = concat(sp_buf, "DELIMITER ", DELIMITER, NULL);
  sp_buf = concat(sp_buf, "DROP PROCEDURE IF EXISTS `",db,"`.`",sp_name,"` ",NULL);
  sp_buf = concat(sp_buf, DELIMITER, NULL);
  sp_buf = concat(sp_buf, "CREATE PROCEDURE `",db,"`.`",sp_name,"`", NULL);
  sp_buf = concat(sp_buf, get_sp_parameters(), NULL);
  sp_buf = concat(sp_buf, "BEGIN\n", NULL);
  
  return snew;
}


/***********************************************************
 * get_sp_end:
 *
 *  Composes and returns the final SQL stataments 
 *  for the stored procedure
 ***********************************************************/
char *
get_sp_end(void)
{
  char *snew = "";

  sp_buf = concat(sp_buf, "\nEND ", DELIMITER, NULL);
  sp_buf = concat(sp_buf, "DELIMITER ;\n", NULL);   
  
  return snew;
}


/***********************************************************
 * get_sp_comments:
 *
 *  Composes and returns the comments to print 
 *  in the header of the stored procedure
 ***********************************************************/
char *
get_sp_comments(void)
{
  char *snew = "";

  snew = concat(snew, "/*\n", NULL);
  snew = concat(snew, "\tStored Procedure generated by: ", NULL);
  snew = concat(snew, PROG_NAME, NULL);
  snew = concat(snew, " - Version:", NULL);
  snew = concat(snew, PROG_VERSION, NULL);
  snew = concat(snew, "\n\t", NULL);
  snew = concat(snew, user, "@", host, NULL);
  snew = concat(snew, "\n\t", NULL);
  snew = concat(snew, get_fdate(), NULL);
  snew = concat(snew, "\n*/\n\n", NULL);

  return snew;
}


/***********************************************************
 * get_sp_parameters:
 *
 *  Composes and returns the parameters 
 *  for the stored procedure
 ***********************************************************/
char *
get_sp_parameters()
{
  unsigned int i;
  char *snew = "";

  cur_rows = 0;
  snew = concat(snew, "(\n", NULL);

  while ((row = mysql_fetch_row(result)))
    {
      cur_rows++;

      for(i = 0; i < num_fields; i++)
	{
	  switch(i)
	    {
	    case 0:// Field 
	      snew = concat(snew, "\t", NULL);
	      snew = concat(snew, PARAM_PREFIX, NULL);
	      snew = concat(snew, row[i], NULL);
	      snew = concat(snew, "\t", NULL);
	      break;
	    case 1:// Type
	      snew = concat(snew, row[i], NULL);
	      snew = concat(snew, ((num_rows != cur_rows) ? "," : ""), NULL);
	      snew = concat(snew, "\t", NULL);
	      break;
	    case 2:// Null
	      snew = concat(snew, "/*\tallow_null:", NULL);
	      snew = concat(snew, row[i], NULL);
	      snew = concat(snew, " - ", NULL);
	      break;
	    case 3:// Key
	      snew = concat(snew, "key_type:", NULL);
	      snew = concat(snew, row[i], NULL);
	      snew = concat(snew, " - ", NULL);
	      break;
	    case 4: // Default
	      snew = concat(snew, "default_value:", NULL);
	      snew = concat(snew, row[i] ? row[i] : "NULL", NULL);
	      snew = concat(snew, " - ", NULL);
	      break;
	    case 5: // Extra
	      snew = concat(snew, "extra:", NULL);
	      snew = concat(snew, row[i], NULL);
	      snew = concat(snew, "\t*/\n", NULL);
	      break;
	    default:
	      break;
	    }
	}
    }

  snew = concat(snew, ")\n", NULL);

  return snew;
}


/***********************************************************
 * get_sp_select:
 *
 *  Composes and returns the SQL statement 
 *  for a stored procedure of type SELECT
 ***********************************************************/
char *
get_sql_select(void)
{
  char *snew;
  cur_rows = 0;

  snew = (char *) calloc(1, (sizeof(char) * 1));

  // Returns to the first row
  mysql_data_seek(result, cur_rows);			

  snew = concat(snew, "\n\tSELECT ", NULL);
	
  while ((row = mysql_fetch_row(result)))
    {
      cur_rows++;
      snew = concat(snew, row[0], NULL);
      snew = concat(snew, ((num_rows != cur_rows) ? ",\n\t\t" : ""), NULL);
    }

  snew = concat(snew, "\n\tFROM ", NULL);
  snew = concat(snew, table, NULL);

  //	Returns to the first row
  cur_rows = 0;
  mysql_data_seek(result, cur_rows);			

  snew = concat(snew, "\n\tWHERE ", NULL);

  while ((row = mysql_fetch_row(result)))
    {
      cur_rows++;
      snew = concat(snew, row[0], NULL);
      snew = concat(snew, " = ", NULL);
      snew = concat(snew, PARAM_PREFIX, NULL);
      snew = concat(snew, row[0], NULL);
      snew = concat(snew, ((num_rows != cur_rows) ? "\n\t\tAND " : ""), NULL);
    }			

  snew = concat(snew, ";\n", NULL);

  return snew;
}


/***********************************************************
 * get_sp_insert:
 *
 *  Composes and returns the SQL statement 
 *  for a stored procedure of type INSERT
 ***********************************************************/
char *
get_sql_insert(void)
{
  char *snew = "";
  cur_rows = 0;

  // Returns to the first row
  mysql_data_seek(result, cur_rows);			
	
  snew = concat(snew, "\n\tINSERT INTO ", NULL);
  snew = concat(snew, table, NULL);
  snew = concat(snew, "\t(\n", NULL);

  while ((row = mysql_fetch_row(result))){
    cur_rows++;		
    snew = concat(snew, "\t\t", NULL);
    snew = concat(snew, row[0], NULL);
    snew = concat(snew, ((num_rows != cur_rows) ? ",\n" : ""), NULL);		
  }
	
  snew = concat(snew, "\n\t)", NULL);
  snew = concat(snew, " VALUES ", NULL);
  snew = concat(snew, "(\n", NULL);

  // Returns to the first row
  cur_rows = 0;
  mysql_data_seek(result, cur_rows);			

  while ((row = mysql_fetch_row(result)))
    {
      cur_rows++;
      snew = concat(snew, "\t\t", NULL);
      snew = concat(snew, PARAM_PREFIX, NULL);
      snew = concat(snew, row[0], NULL);
      snew = concat(snew, ((num_rows != cur_rows) ? ",\n" : ""), NULL);
    }

  snew = concat(snew, "\n\t);\n", NULL);

  return snew;
}


/***********************************************************
 * get_sp_update:
 *
 *  Composes and returns the SQL statement 
 *  for a stored procedure of type UPDATE
 ***********************************************************/
char *
get_sql_update(void)
{	
  char *snew;
  cur_rows = 0;

  snew = (char *) calloc(1, (sizeof(char) * 1));

  // Returns to the first row
  mysql_data_seek(result, cur_rows);			
	
  snew = concat(snew, "\n\tUPDATE ", NULL);
  snew = concat(snew, table, NULL);
  snew = concat(snew, "\n\tSET", NULL);

  while ((row = mysql_fetch_row(result)))
    {
      cur_rows++;		
      snew = concat(snew, "\t", NULL);
      snew = concat(snew, row[0], NULL);
      snew = concat(snew, " = ", NULL);
      snew = concat(snew, PARAM_PREFIX, NULL);
      snew = concat(snew, row[0], NULL);
      snew = concat(snew, ((num_rows != cur_rows) ? ",\n\t\t" : ""), NULL);	
    }
	
  // Returns to the first row
  cur_rows = 0;
  mysql_data_seek(result, cur_rows);			

  snew = concat(snew, "\n\tWHERE ", NULL);

  while ((row = mysql_fetch_row(result)))
    {
      cur_rows++;
      snew = concat(snew, row[0], NULL);
      snew = concat(snew, " = ", NULL);
      snew = concat(snew, PARAM_PREFIX, NULL);
      snew = concat(snew, row[0], NULL);
      snew = concat(snew, ((num_rows != cur_rows) ? "\n\t\tAND " : ""), NULL);
    }			

  snew = concat(snew, ";\n", NULL);

  return snew;
}


/***********************************************************
 * get_sp_delete:
 *
 *  Composes and returns the SQL statement 
 *  for a stored procedure of type DELETE
 ***********************************************************/
char *
get_sql_delete(void)
{
  char *snew;
  cur_rows = 0;

  snew = (char *) calloc(1, (sizeof(char) * 1));

  // Returns to the first row
  mysql_data_seek(result, cur_rows);			

  snew = concat(snew, "\n\tDELETE FROM ", NULL);
  snew = concat(snew, table, NULL);			
  snew = concat(snew, "\n\tWHERE ", NULL);

  while ((row = mysql_fetch_row(result)))
    {
      cur_rows++;
      snew = concat(snew, row[0], NULL);
      snew = concat(snew, " = ", NULL);
      snew = concat(snew, PARAM_PREFIX, NULL);
      snew = concat(snew, row[0], NULL);
      snew = concat(snew, ((num_rows != cur_rows) ? "\n\t\tAND " : ""), NULL);
    }			

  snew = concat(snew, ";\n", NULL);

  return snew;
}


/***********************************************************
 * get_fdate:
 *
 *  Composes, formats and returns the current datetime as string
 ***********************************************************/
char *
get_fdate(void)
{
  char *tm_stmp;
  time_t timer;
  timer = time(NULL);

  tm_stmp = asctime(localtime(&timer));
  tm_stmp[strlen(tm_stmp)-1]= '\0';

  return tm_stmp;
}


/***********************************************************
 * get_curr_statement
 *
 *  Determines the statement to use to build the stored procedure
 *  The statement can be: insert, update, delete, select
 *  It is provided by the user by the -s option.
 *  This function validates the value provided by the user.
 ***********************************************************/
char * 
get_curr_statement(char *stmnt)
{
  unsigned int i = 0;
	
  while(stmnt[i++] = tolower(stmnt[i]));

  if (strcmp(stmnt, INSERT) == 0)
    return stmnt;

  else if (strcmp(stmnt, UPDATE) == 0)
    return stmnt;

  else if (strcmp(stmnt, DELETE) == 0)
    return stmnt;

  else if (strcmp(stmnt, SELECT) == 0)
    return stmnt;	

  else
    {
      fprintf(stderr, "%s is not a valid argument for -s [SQL Statement]\n", stmnt);
      show_usage();
      exit(2);
    }
}
