/* myermerge.c - Merge compilation units into master list -*-coding: latin-1-*-

    2003 by Jonathan Yavner.  GPL license; see file COPYING for details.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <assert.h>

#include "hashtab.h"

#include "myer.h"

static SRC_t **CurSIDs;
static DECL_t **CurUIDs, **CurFakeUIDs;
static int   NextAllUID;


/* ----------------------------------------------------------------------------
   Transfer a DECL from the current file to the master.
*/
static void
Transfer_UID( DECL_t *d )
{
	TOKEN_HASH_t *th = Token_Hash_add( AllHash, d->name, 0, INSERT );

	d->name = th->key;
	if (d->uid < 0) {
		CurFakeUIDs[-d->uid] = 0;
	} else {
		CurUIDs[d->uid] = 0;
	}
	d->uid = ++NextAllUID;
	AddUID( &AllUIDs, d->uid, d );
	d->wasmerged = 1;
} /* Transfer_UID */


/* -------------------------------------------------------------------------
 */
static void
Transfer_or_merge_UID( DECL_t *d )
{
	if (d->wasmerged) {
		return;
	}
	if (d->ispublic) {
		TOKEN_HASH_t *th;
		th = Token_Hash_add( AllHash, d->name, d->class, NO_INSERT );
		assert( th );
		MergeUID( (DECL_t *) th->data, d );
	} else {
		Transfer_UID( d );
	}
} /* Transfer_or_merge_UID */


/* ----------------------------------------------------------------------------
   Transfer a chain of SPOT_t from current file to master.
*/
static void
Transfer_Spots( CHAIN_t *sc )
{
	SPOT_t *s;

	while ((s = sc->spot)) {
		switch (s->kind) {
		case SPOT_Skip:
			/* Has no pointer to be updated */
			break;
		case SPOT_IncludeFile:
			s->ref.sid = CurSIDs[s->ref.sid]->sid;
			break;
		default:
			Transfer_or_merge_UID( s->decl );
		}
		sc = sc->next;
	}
} /* Transfer_Spots */


/* ----------------------------------------------------------------------------
   Moves a spot to a different chain or different position in its chain.  If
   this was the defining spot for its decl, updates the backlink.
*/
static void
MoveSpot( CHAIN_t *old, CHAIN_t *new )
{
	SPOT_t  *x = UnlinkChain( old );
	CHAIN_t *y = InsertChain( new, x );

	if (x->decl && x->decl->def == old) {
		x->decl->def = y;
	}
} /* MoveSpot */


/* ----------------------------------------------------------------------------
   Match up a declaration in a compilation unit with one in the master.  If
   the decl has the same definition, merge it. If the master has several
   coextensive spots, the decl may match a later one.

   Result is the master spot where the decl matched (usually same as
   `master' argument, or 0 for no match.
*/
static CHAIN_t *
Match_Decls( CHAIN_t *master, CHAIN_t *one )
{
	SPOT_t *m = master->spot;
	SPOT_t *o = one->spot;
	DECL_t *md = m->decl;
	DECL_t *od = o->decl;

	if (od->isglobal && md->isglobal
	    && !strcmp( od->name, md->name )) {
		/* Ref to like-named globals */
		if (o->kind == SPOT_Decl
		    && m->kind == SPOT_Ref
		    && md->def) {
			/* Ref to a function or blind struct not defined in
			   this file, but some other file does have a def
			   for it */
			o->kind = m->kind;
			MergeUID( md, od );
			return master;
		}
		if (m->kind == SPOT_Decl
			   && (o->kind == SPOT_Ref || o->kind == SPOT_Decl)
			   && od->def
			   && od->def->spot->kind == SPOT_Def) {
			/* Ref to a defined struct that was blind in all
			   previous occurrences of this file. */
			m->kind = o->kind;
			MergeUID( md, od );
			return master;
		}
	}

	if (od->def && md->def &&
	    od->defsid == md->defsid &&
	    od->def->spot->line == md->def->spot->line &&
	    od->def->spot->col == md->def->spot->col) {
		/* The decls are defined at same spot in same file */
		MergeUID( md, od );
		return master;
	}

	/* Maybe this decl matches some later coextensive spot */
	SPOT_t *try = master->next->spot;
	if (try &&
	    try->line == m->line &&
	    try->col  == m->col &&
	    try->endline == m->endline &&
	    try->endcol  == m->endcol &&
	    Match_Decls( master->next, one )) {
		/* Bubble-sort to move matching spot to proper place in chain */
		MoveSpot( master->next, master );
		return master->prev;
	}

	/* No match */
	return 0;
} /* Match_Decls */


/* ----------------------------------------------------------------------------
   A file in a compilation unit is already in the master list.  Merge the
   information from the two compilations.
*/
static void
myerMerge_File( SRC_t *master, SRC_t *one )
{
	CHAIN_t *match, *mc = master->spots.next, *oc = one->spots.next;
	SPOT_t *prev = 0;
	static const SPOT_t Empty = {
		SPOT_Skip, 0x7fffffff, 0, 0x7fffffff, 0, 0, 0.0, 0.0, {.sid = 0}
	};

	while (mc != &master->spots || oc != &one->spots) {
		SPOT_t *m = mc->spot, *o = oc->spot;

		if (!m) {
			m = (SPOT_t *) &Empty;
		}
		if (!o) {
			o = (SPOT_t *) &Empty;
		}

		if (m->endline < o->line ||
		    (m->line == o->line
		     && m->endline == o->endline
		     && m->endcol < o->col)) {
			/* Master token precedes new token */
			mc = mc->next;
			continue;
		}
		if (o->endline < m->line ||
		    (o->line == m->line
		     && o->endline == m->endline
		     && o->endcol < m->col)) {
			/* New token precedes master token */
			oc = oc->next;
			if (o->decl) {
				Transfer_or_merge_UID( o->decl );
			}
			MoveSpot( oc->prev, mc );
			continue;
		}
		if (m->kind == SPOT_Skip
		    && m->line <= o->line
		    && m->endline >= o->line) {
			/* Master had skip, new has real data.  Delete skip */
			mc = mc->next;
			free( UnlinkChain(mc->prev) );
			continue;
		}
		if (o->kind == SPOT_Skip
		    && o->line <= m->line
		    && o->endline >= m->line) {
			/* New has skip, master had real data.  Delete skip */
			oc = oc->next;
			continue;
		}
		if ((m->line < o->line ||
		     (m->line == o->line && m->col < o->col))
		    && (m->endline > o->endline ||
			(m->endline == o->endline && m->endcol > o->endcol))) {
			/* Master token encloses new token */
			mc = mc->next;
			continue;
		}
		if ((o->line < m->line ||
		     (o->line == m->line && o->col < m->col))
		    && (o->endline > m->endline ||
			(o->endline == m->endline && o->endcol > m->endcol))) {
			/* New token encloses master token */
			oc = oc->next;
			if (o->decl) {
				Transfer_or_merge_UID( o->decl );
			}
			MoveSpot( oc->prev, mc );
			continue;
		}
		match = mc;
		if (o->kind == SPOT_IncludeFile) {
			if (prev != o) {
				/* Update the SID, but only once.  The same
				   o might be seen several times if we're
				   skipping over MASTER items not mentioned
				   in the ONE file */
				o->ref.sid = CurSIDs[o->ref.sid]->sid;
				prev = o;
			}
		} else if (m->decl && o->decl) {
			match = Match_Decls( mc, oc );
			if (match) {
				/* Master spot might have been bubble-sorted */
				m = mc->spot;
				mc = match;
			}
		}

		if (m->kind != o->kind
		    || m->line != o->line
		    || m->endline != o->endline
		    || m->col != o->col
		    || o->endcol != o->endcol) {
			/* Mismatch */
			break;
		}

		oc = oc->next;
		if (match) {
			mc = mc->next;
		} else {
			/* Keep both */
			if (o->decl) {
				Transfer_or_merge_UID( o->decl );
			}
			MoveSpot( oc->prev, mc );
		}
	}

	if (mc == &master->spots && oc == &one->spots) {
		return; /* Successful merge */
	}

	/* This file didn't compile the same way in all compilation units */
	master->baseline = INCONSISTENT_CONTENTS; /* Prohibit HTML display */

	/* Transfer the remainder of the current file as a concatenation to
	   the master. */
	Transfer_Spots( oc );
	while (one->spots.next != oc) {
		if (one->spots.next->spot->kind != SPOT_IncludeFile &&
		    one->spots.next->spot->decl &&
		    one->spots.next->spot->decl->def == one->spots.next) {
			/* Deleting the def-spot.  If this comes up later,
			   just pretend it was never defined. */
			one->spots.next->spot->decl->def = 0;
		}
		SPOT_t *s = UnlinkChain(one->spots.next);
		if (s->decl) {
			UnlinkChain( s->ref.dlink );
		}
		free( s );
	}
	master->spots.prev->next = one->spots.next;
	one->spots.next->prev    = master->spots.prev;
	one->spots.prev->next    = &master->spots;
	master->spots.prev       = one->spots.prev;
	InitChain( &one->spots );
} /* myerMerge_File */


/* ----------------------------------------------------------------------------
   Merge one compilation unit into the master list.
*/
static void
myerMerge( COMPILATION_UNIT_t cu )
{
	TOKEN_HASH_t *th;
	SRC_t *src;
	int j, sid;

	CurSIDs = cu.sids;
	CurUIDs = cu.uids;
	CurFakeUIDs = cu.fake_uids;

	/* Translate all source-file ID numbers. Hash the filenames to find
	   duplicates.  FIXME: This is inconsistent for the dev/ino
	   duplicate-detection method used elsewhere in Myer. */
	for (j = 1; j < NUM_IDS(CurSIDs); ++j) {
		src = CurSIDs[j];
		if (!src) {
			continue;
		}
		th = Token_Hash_add( AllHash,src->name,TOKEN_FILEPUSH,INSERT );
		src->name = th->key;
		if (!th->data) {
			* (int *) &th->data = ++NextAllSID;
		}
		src->sid = * (int *) &th->data;
	}

	/* For all decls, translate SIDs and merge "public" things */
	for (int j = -NUM_IDS(cu.fake_uids)+1; j < NUM_IDS(cu.uids); ++j) {
		DECL_t *d;
		if (j > 0) {
			d = CurUIDs[j];
		} else if (j < 0) {
			d = CurFakeUIDs[-j];
		} else {
			d = 0;
		}
		if (!d || d->uid != j) {
			/* Skip unused or duplicate UID slots */
			continue;
		}
		if (d->defsid) {
			d->defsid = CurSIDs[d->defsid]->sid;
		}
		for (CHAIN_t *ic=d->decls.next; ic!=&d->decls; ic=ic->next) {
			ic->sid = CurSIDs[ic->sid]->sid;
		}

		if (d->ispublic) {
			th = Token_Hash_add( AllHash,d->name,d->class,INSERT );
			if (!th->data) {
				Transfer_UID( d );
				th->data = d;
			}
		}
	}

	/* Process each source file */
	for (int j = 1; j < NUM_IDS(CurSIDs); ++j) {
		src = CurSIDs[j];
		if (!src) {
			continue;
		}
		sid = src->sid;
		if (sid < NUM_IDS(AllSIDs) && AllSIDs[sid]) {
			src->baseline = 0;
			myerMerge_File( AllSIDs[sid], src );
		} else { /* Transfer this source file */
			AddSID( &AllSIDs, src );
			src->baseline = -1;
			Transfer_Spots( src->spots.next );
		}
	}

	for (int j = 1; j < NUM_IDS(CurSIDs); ++j) {
		if (CurSIDs[j] && CurSIDs[j]->baseline) {
			/* This one was transfered to the merged cu, so
			   don't free it when the cu is freed. */
			CurSIDs[j] = 0;
		}
	}
} /* myerMerge */



/* ----------------------------------------------------------------------------
   API for phase 3.
*/
int
myerMerge_From_File( char **fnamep )
{
	COMPILATION_UNIT_t cu;
	char *filename = *fnamep;
	char *extension = strrchr( filename, '.' );

	if (extension && !strcmp( extension, ".myer3" )) {
		assert( !AllHash );
		if (Verbose) {
			fprintf( stderr, "[3] < %s\n", filename );
		}
		cu = myerToken_Read( filename, EOF );
		AllSIDs = cu.sids;
		AllUIDs = cu.uids;
		AllHash = cu.names;
		NextAllSID = NUM_IDS(AllSIDs);
		assert( !cu.fake_uids );
		return 3;
	} else {
		if (!AllHash) { /* First call */
			AllHash = Token_Hash_init();
		}
		cu = myerToken_From_File( fnamep );
		if (!cu.names) {
			/* File has been skipped */
			return 2;
		}
		if (Verbose) {
			fprintf( stderr, "[3] = %s\n", *fnamep );
		}
		myerMerge( cu );
		Common_Free( cu );
		return 2;
	}
} /* myerMerge_From_File */
